per.
	 *
	 * @var Current_Page_Helper
	 */
	protected $current_page;

	/**
	 * The image helper.
	 *
	 * @var Image_Helper
	 */
	protected $image;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	protected $options;

	/**
	 * The URL helper.
	 *
	 * @var Url_Helper
	 */
	protected $url;

	/**
	 * The user helper.
	 *
	 * @var User_Helper
	 */
	protected $user;

	/**
	 * The indexable helper.
	 *
	 * @var Indexable_Helper
	 */
	protected $indexable_helper;

	/**
	 * The permalink helper.
	 *
	 * @var Permalink_Helper
	 */
	protected $permalink_helper;

	/**
	 * The values helper.
	 *
	 * @var Values_Helper
	 */
	protected $values_helper;

	/**
	 * Sets the generator dependencies.
	 *
	 * @required
	 *
	 * @param Schema_Generator            $schema_generator            The schema generator.
	 * @param Open_Graph_Locale_Generator $open_graph_locale_generator The Open Graph locale generator.
	 * @param Open_Graph_Image_Generator  $open_graph_image_generator  The Open Graph image generator.
	 * @param Twitter_Image_Generator     $twitter_image_generator     The Twitter image generator.
	 * @param Breadcrumbs_Generator       $breadcrumbs_generator       The breadcrumbs generator.
	 *
	 * @return void
	 */
	public function set_generators(
		Schema_Generator $schema_generator,
		Open_Graph_Locale_Generator $open_graph_locale_generator,
		Open_Graph_Image_Generator $open_graph_image_generator,
		Twitter_Image_Generator $twitter_image_generator,
		Breadcrumbs_Generator $breadcrumbs_generator
	) {
		$this->schema_generator            = $schema_generator;
		$this->open_graph_locale_generator = $open_graph_locale_generator;
		$this->open_graph_image_generator  = $open_graph_image_generator;
		$this->twitter_image_generator     = $twitter_image_generator;
		$this->breadcrumbs_generator       = $breadcrumbs_generator;
	}

	/**
	 * Used by dependency injection container to inject the helpers.
	 *
	 * @required
	 *
	 * @param Image_Helper        $image        The image helper.
	 * @param Options_Helper      $options      The options helper.
	 * @param Current_Page_Helper $current_page The current page helper.
	 * @param Url_Helper          $url          The URL helper.
	 * @param User_Helper         $user         The user helper.
	 * @param Indexable_Helper    $indexable    The indexable helper.
	 * @param Permalink_Helper    $permalink    The permalink helper.
	 * @param Values_Helper       $values       The values helper.
	 *
	 * @return void
	 */
	public function set_helpers(
		Image_Helper $image,
		Options_Helper $options,
		Current_Page_Helper $current_page,
		Url_Helper $url,
		User_Helper $user,
		Indexable_Helper $indexable,
		Permalink_Helper $permalink,
		Values_Helper $values
	) {
		$this->image            = $image;
		$this->options          = $options;
		$this->current_page     = $current_page;
		$this->url              = $url;
		$this->user             = $user;
		$this->indexable_helper = $indexable;
		$this->permalink_helper = $permalink;
		$this->values_helper    = $values;
	}

	/**
	 * Gets the permalink from the indexable or generates it if dynamic permalinks are enabled.
	 *
	 * @return string The permalink.
	 */
	public function generate_permalink() {
		if ( $this->indexable_helper->dynamic_permalinks_enabled() ) {
			return $this->permalink_helper->get_permalink_for_indexable( $this->model );
		}

		if ( \is_date() ) {
			return $this->current_page->get_date_archive_permalink();
		}

		if ( \is_attachment() ) {
			global $wp;
			return \trailingslashit( \home_url( $wp->request ) );
		}

		return $this->model->permalink;
	}

	/**
	 * Generates the title.
	 *
	 * @return string The title.
	 */
	public function generate_title() {
		if ( $this->model->title ) {
			return $this->model->title;
		}

		return '';
	}

	/**
	 * Generates the meta description.
	 *
	 * @return string The meta description.
	 */
	public function generate_meta_description() {
		if ( $this->model->description ) {
			return $this->model->description;
		}

		return '';
	}

	/**
	 * Generates the robots value.
	 *
	 * @return array The robots value.
	 */
	public function generate_robots() {
		$robots = $this->get_base_robots();

		return $this->filter_robots( $robots );
	}

	/**
	 * Gets the base robots value.
	 *
	 * @return array The base robots value.
	 */
	protected function get_base_robots() {
		return [
			'index'             => ( $this->model->is_robots_noindex === true ) ? 'noindex' : 'index',
			'follow'            => ( $this->model->is_robots_nofollow === true ) ? 'nofollow' : 'follow',
			'max-snippet'       => 'max-snippet:-1',
			'max-image-preview' => 'max-image-preview:large',
			'max-video-preview' => 'max-video-preview:-1',
		];
	}

	/**
	 * Run the robots output content through the `wpseo_robots` filter.
	 *
	 * @param array $robots The meta robots values to filter.
	 *
	 * @return array The filtered meta robots values.
	 */
	protected function filter_robots( $robots ) {
		// Remove values that are only listened to when indexing.
		if ( $robots['index'] === 'noindex' ) {
			$robots['imageindex']        = null;
			$robots['archive']           = null;
			$robots['snippet']           = null;
			$robots['max-snippet']       = null;
			$robots['max-image-preview'] = null;
			$robots['max-video-preview'] = null;
		}

		$robots_string = \implode( ', ', \array_filter( $robots ) );

		/**
		 * Filter: 'wpseo_robots' - Allows filtering of the meta robots output of Yoast SEO.
		 *
		 * @param string                 $robots       The meta robots directives to be echoed.
		 * @param Indexable_Presentation $presentation The presentation of an indexable.
		 */
		$robots_filtered = \apply_filters( 'wpseo_robots', $robots_string, $this );

		// Convert the robots string back to an array.
		if ( \is_string( $robots_filtered ) ) {
			$robots_values = \explode( ', ', $robots_filtered );
			$robots_new    = [];

			foreach ( $robots_values as $value ) {
				$key = $value;

				// Change `noindex` to `index.
				if ( \strpos( $key, 'no' ) === 0 ) {
					$key = \substr( $value, 2 );
				}
				// Change `max-snippet:-1` to `max-snippet`.
				$colon_position = \strpos( 