<?php

namespace Yoast\WP\SEO\Introductions\User_Interface;

use Exception;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use Yoast\WP\SEO\Conditionals\No_Conditionals;
use Yoast\WP\SEO\Helpers\User_Helper;
use Yoast\WP\SEO\Introductions\Application\Introductions_Collector;
use Yoast\WP\SEO\Introductions\Infrastructure\Introductions_Seen_Repository;
use Yoast\WP\SEO\Main;
use Yoast\WP\SEO\Routes\Route_Interface;

/**
 * Registers a route to set whether the user has seen an introduction.
 *
 * @makePublic
 */
class Introductions_Seen_Route implements Route_Interface {

	use No_Conditionals;

	/**
	 * Represents the prefix.
	 *
	 * @var string
	 */
	public const ROUTE_PREFIX = '/introductions/(?P<introduction_id>[\w-]+)/seen';

	/**
	 * Holds the introductions collector instance.
	 *
	 * @var Introductions_Collector
	 */
	private $introductions_collector;

	/**
	 * Holds the repository.
	 *
	 * @var Introductions_Seen_Repository
	 */
	private $introductions_seen_repository;

	/**
	 * Holds the user helper.
	 *
	 * @var User_Helper
	 */
	private $user_helper;

	/**
	 * Constructs the class.
	 *
	 * @param Introductions_Seen_Repository $introductions_seen_repository The repository.
	 * @param User_Helper                   $user_helper                   The user helper.
	 * @param Introductions_Collector       $introductions_collector       The introduction collector.
	 */
	public function __construct(
		Introductions_Seen_Repository $introductions_seen_repository,
		User_Helper $user_helper,
		Introductions_Collector $introductions_collector
	) {
		$this->introductions_seen_repository = $introductions_seen_repository;
		$this->user_helper                   = $user_helper;
		$this->introductions_collector       = $introductions_collector;
	}

	/**
	 * Registers routes with WordPress.
	 *
	 * @return void
	 */
	public function register_routes() {
		\register_rest_route(
			Main::API_V1_NAMESPACE,
			self::ROUTE_PREFIX,
			[
				[
					'methods'             => 'POST',
					'callback'            => [ $this, 'set_introduction_seen' ],
					'permission_callback' => [ $this, 'permission_edit_posts' ],
					'args'                => [
						'introduction_id' => [
							'required'          => true,
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_text_field',
						],
						'is_seen'         => [
							'required'          => false,
							'type'              => 'bool',
							'default'           => true,
							'sanitize_callback' => 'rest_sanitize_boolean',
						],
					],
				],
			]
		);
	}

	/**
	 * Sets whether the introduction is seen.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response|WP_Error The success or failure response.
	 */
	public function set_introduction_seen( WP_REST_Request $request ) {
		$params          = $request->get_params();
		$introduction_id = $params['introduction_id'];
		$is_seen         = $params['is_seen'];

		if ( $this->introductions_collector->is_available_introduction( $introduction_id ) ) {
			try {
				$user_id = $this->user_helper->get_current_user_id();
				$result  = $this->introductions_seen_repository->set_introduction( $user_id, $introduction_id, $is_seen );
			} catch ( Exception $exception ) {
				return new WP_Error(
					'wpseo_introductions_seen_error',
					$exception->getMessage(),
					(object) []
				);
			}

			return new WP_REST_Response(
				[
					'json' => (object) [
						'success' => $result,
					],
				],
				( $result ) ? 200 : 400
			);
		}
		return new WP_REST_Response( [], 400 );
	}

	/**
	 * Permission callback.
	 *
	 * @return bool True when user has 'edit_posts' permission.
	 */
	public function permission_edit_posts() {
		return \current_user_can( 'edit_posts' );
	}
}
                                                                                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/src/introductions/user-interface/wistia-embed-permission-route.php   0000644                 00000007254 15122266560 0027670 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Introductions\User_Interface;

use Exception;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use Yoast\WP\SEO\Conditionals\No_Conditionals;
use Yoast\WP\SEO\Helpers\User_Helper;
use Yoast\WP\SEO\Introductions\Infrastructure\Wistia_Embed_Permission_Repository;
use Yoast\WP\SEO\Main;
use Yoast\WP\SEO\Routes\Route_Interface;

/**
 * Registers a route to offer get/set of the wistia embed permission for a user.
 */
class Wistia_Embed_Permission_Route implements Route_Interface {

	use No_Conditionals;

	/**
	 * Represents the prefix.
	 *
	 * @var string
	 */
	public const ROUTE_PREFIX = '/wistia_embed_permission';

	/**
	 * Holds the repository.
	 *
	 * @var Wistia_Embed_Permission_Repository
	 */
	private $wistia_embed_permission_repository;

	/**
	 * Holds the user helper.
	 *
	 * @var User_Helper
	 */
	private $user_helper;

	/**
	 * Constructs the class.
	 *
	 * @param Wistia_Embed_Permission_Repository $wistia_embed_permission_repository The repository.
	 * @param User_Helper                        $user_helper                        The user helper.
	 */
	public function __construct(
		Wistia_Embed_Permission_Repository $wistia_embed_permission_repository,
		User_Helper $user_helper
	) {
		$this->wistia_embed_permission_repository = $wistia_embed_permission_repository;
		$this->user_helper                        = $user_helper;
	}

	/**
	 * Registers routes with WordPress.
	 *
	 * @return void
	 */
	public function register_routes() {
		\register_rest_route(
			Main::API_V1_NAMESPACE,
			self::ROUTE_PREFIX,
			[
				[
					'methods'             => 'GET',
					'callback'            => [ $this, 'get_wistia_embed_permission' ],
					'permission_callback' => [ $this, 'permission_edit_posts' ],
				],
				[
					'methods'             => 'POST',
					'callback'            => [ $this, 'set_wistia_embed_permission' ],
					'permission_callback' => [ $this, 'permission_edit_posts' ],
					'args'                => [
						'value' => [
							'required'          => false,
							'type'              => 'bool',
							'default'           => true,
						],
					],
				],
			]
		);
	}

	/**
	 * Gets the value of the wistia embed permission.
	 *
	 * @return WP_REST_Response|WP_Error The response, or an error.
	 */
	public function get_wistia_embed_permission() {
		try {
			$user_id = $this->user_helper->get_current_user_id();
			$value   = $this->wistia_embed_permission_repository->get_value_for_user( $user_id );
		} catch ( Exception $exception ) {
			return new WP_Error(
				'wpseo_wistia_embed_permission_error',
				$exception->getMessage(),
				(object) []
			);
		}

		return new WP_REST_Response(
			[
				'json' => (object) [
					'value' => $value,
				],
			]
		);
	}

	/**
	 * Sets the value of the wistia embed permission.
	 *
	 * @param WP_REST_Request $request The request object.
	 *
	 * @return WP_REST_Response|WP_Error The success or failure response.
	 */
	public function set_wistia_embed_permission( WP_REST_Request $request ) {
		$params = $request->get_json_params();
		$value  = \boolval( $params['value'] );

		try {
			$user_id = $this->user_helper->get_current_user_id();
			$result  = $this->wistia_embed_permission_repository->set_value_for_user( $user_id, $value );
		} catch ( Exception $exception ) {
			return new WP_Error(
				'wpseo_wistia_embed_permission_error',
				$exception->getMessage(),
				(object) []
			);
		}

		return new WP_REST_Response(
			[
				'json' => (object) [
					'success' => $result,
				],
			],
			( $result ) ? 200 : 400
		);
	}

	/**
	 * Permission callback.
	 *
	 * @return bool True when user has 'edit_posts' permission.
	 */
	public function permission_edit_posts() {
		return \current_user_can( 'edit_posts' );
	}
}
                                                                                                                                                                                                                                                                                                                                                    plugins/wordpress-seo-extended/src/loadable-interface.php                                           0000644                 00000000436 15122266560 0017650 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO;

/**
 * An interface for registering integrations with WordPress
 */
interface Loadable_Interface {

	/**
	 * Returns the conditionals based on which this loadable should be active.
	 *
	 * @return array
	 */
	public static function get_conditionals();
}
                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/src/loader.php                                                       0000644                 00000015166 15122266560 0015423 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO;

use Throwable;
use WP_CLI;
use YoastSEO_Vendor\Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class that manages loading integrations if and only if all their conditionals are met.
 */
class Loader {

	/**
	 * The registered integrations.
	 *
	 * @var string[]
	 */
	protected $integrations = [];

	/**
	 * The registered integrations.
	 *
	 * @var string[]
	 */
	protected $initializers = [];

	/**
	 * The registered routes.
	 *
	 * @var string[]
	 */
	protected $routes = [];

	/**
	 * The registered commands.
	 *
	 * @var string[]
	 */
	protected $commands = [];

	/**
	 * The registered migrations.
	 *
	 * @var string[]
	 */
	protected $migrations = [];

	/**
	 * The dependency injection container.
	 *
	 * @var ContainerInterface
	 */
	protected $container;

	/**
	 * Loader constructor.
	 *
	 * @param ContainerInterface $container The dependency injection container.
	 */
	public function __construct( ContainerInterface $container ) {
		$this->container = $container;
	}

	/**
	 * Registers an integration.
	 *
	 * @param string $integration_class The class name of the integration to be loaded.
	 *
	 * @return void
	 */
	public function register_integration( $integration_class ) {
		$this->integrations[] = $integration_class;
	}

	/**
	 * Registers an initializer.
	 *
	 * @param string $initializer_class The class name of the initializer to be loaded.
	 *
	 * @return void
	 */
	public function register_initializer( $initializer_class ) {
		$this->initializers[] = $initializer_class;
	}

	/**
	 * Registers a route.
	 *
	 * @param string $route_class The class name of the route to be loaded.
	 *
	 * @return void
	 */
	public function register_route( $route_class ) {
		$this->routes[] = $route_class;
	}

	/**
	 * Registers a command.
	 *
	 * @param string $command_class The class name of the command to be loaded.
	 *
	 * @return void
	 */
	public function register_command( $command_class ) {
		$this->commands[] = $command_class;
	}

	/**
	 * Registers a migration.
	 *
	 * @param string $plugin          The plugin the migration belongs to.
	 * @param string $version         The version of the migration.
	 * @param string $migration_class The class name of the migration to be loaded.
	 *
	 * @return void
	 */
	public function register_migration( $plugin, $version, $migration_class ) {
		if ( ! \array_key_exists( $plugin, $this->migrations ) ) {
			$this->migrations[ $plugin ] = [];
		}

		$this->migrations[ $plugin ][ $version ] = $migration_class;
	}

	/**
	 * Loads all registered classes if their conditionals are met.
	 *
	 * @return void
	 */
	public function load() {
		$this->load_initializers();

		if ( ! \did_action( 'init' ) ) {
			\add_action( 'init', [ $t