*
	 * @return array The excluded post types.
	 */
	public function get_excluded_post_types_for_indexables() {
		/**
		 * Filter: 'wpseo_indexable_excluded_post_types' - Allows excluding posts of a certain post
		 * type from being saved to the indexable table.
		 *
		 * @param array $excluded_post_types The currently excluded post types that indexables will not be created for.
		 */
		$excluded_post_types = \apply_filters( 'wpseo_indexable_excluded_post_types', [] );

		// Failsafe, to always make sure that `excluded_post_types` is an array.
		if ( ! \is_array( $excluded_post_types ) ) {
			return [];
		}

		return $excluded_post_types;
	}

	/**
	 * Checks if the post type is excluded.
	 *
	 * @param string $post_type The post type to check.
	 *
	 * @return bool If the post type is exclude.
	 */
	public function is_excluded( $post_type ) {
		return \in_array( $post_type, $this->get_excluded_post_types_for_indexables(), true );
	}

	/**
	 * Checks if the post type with the given name has an archive page.
	 *
	 * @param WP_Post_Type|string $post_type The name of the post type to check.
	 *
	 * @return bool True when the post type has an archive page.
	 */
	public function has_archive( $post_type ) {
		if ( \is_string( $post_type ) ) {
			$post_type = \get_post_type_object( $post_type );
		}

		return ( ! empty( $post_type->has_archive ) );
	}

	/**
	 * Returns the post types that should be indexed.
	 *
	 * @return array The post types that should be indexed.
	 */
	public function get_indexable_post_types() {
		$public_post_types   = $this->get_public_post_types();
		$excluded_post_types = $this->get_excluded_post_types_for_indexables();

		$included_post_types = \array_diff( $public_post_types, $excluded_post_types );

		return $this->filter_included_post_types( $included_post_types );
	}

	/**
	 * Returns all indexable post types with archive pages.
	 *
	 * @return array All post types which are indexable and have archive pages.
	 */
	public function get_indexable_post_archives() {
		return \array_filter( $this->get_indexable_post_type_objects(), [ $this, 'has_archive' ] );
	}

	/**
	 * Filters the post types that are included to be indexed.
	 *
	 * @param array $included_post_types The post types that are included to be indexed.
	 *
	 * @return array The filtered post types that are included to be indexed.
	 */
	protected function filter_included_post_types( $included_post_types ) {
		/**
		 * Filter: 'wpseo_indexable_forced_included_post_types' - Allows force including posts of a certain post
		 * type to be saved to the indexable table.
		 *
		 * @param array $included_post_types The currently included post types that indexables will be created for.
		 */
		$filtered_included_post_types = \apply_filters( 'wpseo_indexable_forced_included_post_types', $included_post_types );

		if ( ! \is_array( $filtered_included_post_types ) ) {
			// If the filter got misused, let's return the unfiltered array.
			return \array_values( $included_post_types );
		}

		// Add sanity check to make sure everything is an actual post type.
		foreach ( $filtered_included_post_types as $key => $post_type ) {
			if ( ! \post_type_exists( $post_type ) ) {
				unset( $filtered_included_post_types[ $key ] );
			}
		}

		// `array_values`, to make sure that the keys are reset.
		return \array_values( $filtered_included_post_types );
	}

	/**
	 * Checks if the given post type should be indexed.
	 *
	 * @param string $post_type The post type that is checked.
	 *
	 * @return bool
	 */
	public function is_of_indexable_post_type( $post_type ) {
		$public_types = $this->get_indexable_post_types();
		if ( ! \in_array( $post_type, $public_types, true ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Checks if the archive of a post type is indexable.
	 *
	 * @param string $post_type The post type to check.
	 *
	 * @return bool if the archive is indexable.
	 */
	public function is_post_type_archive_indexable( $post_type ) {
		$public_type_objects = $this->get_indexable_post_archives();
		$public_types        = \array_map(
			static function