ar Indexable_Repository
	 */
	protected $repository;

	/**
	 * Represents the options helper.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * Represents the environment helper.
	 *
	 * @var Environment_Helper
	 */
	protected $environment_helper;

	/**
	 * Represents the indexing helper.
	 *
	 * @var Indexing_Helper
	 */
	protected $indexing_helper;

	/**
	 * Default values of certain columns.
	 *
	 * @var array
	 */
	protected $default_values = [
		'title'                  => [
			'default_value'   => null,
		],
		'description'            => [
			'default_value'   => null,
		],
		'open_graph_title'       => [
			'default_value'   => null,
		],
		'open_graph_description' => [
			'default_value'   => null,
		],
		'twitter_title'          => [
			'default_value'   => null,
		],
		'twitter_description'    => [
			'default_value'   => null,
		],
		'canonical'              => [
			'default_value'   => null,
		],
		'primary_focus_keyword'  => [
			'default_value'   => null,
		],
		'is_robots_noindex'      => [
			'default_value'   => null,
		],
		'is_robots_nofollow'     => [
			'default_value'   => false,
		],
		'is_robots_noarchive'    => [
			'default_value'   => null,
		],
		'is_robots_noimageindex' => [
			'default_value'   => null,
		],
		'is_robots_nosnippet'    => [
			'default_value'   => null,
		],
	];

	/**
	 * Indexable_Helper constructor.
	 *
	 * @param Options_Helper     $options_helper     The options helper.
	 * @param Environment_Helper $environment_helper The environment helper.
	 * @param Indexing_Helper    $indexing_helper    The indexing helper.
	 */
	public function __construct( Options_Helper $options_helper, Environment_Helper $environment_helper, Indexing_Helper $indexing_helper ) {
		$this->options_helper     = $options_helper;
		$this->environment_helper = $environment_helper;
		$this->indexing_helper    = $indexing_helper;
	}

	/**
	 * Sets the indexable repository. Done to avoid circular dependencies.
	 *
	 * @required
	 *
	 * @param Indexable_Repository $repository The indexable repository.
	 *
	 * @return void
	 */
	public function set_indexable_repository( Indexable_Repository $repository ) {
		$this->repository = $repository;
	}

	/**
	 * Returns the page type of an indexable.
	 *
	 * @param Indexable $indexable The indexable.
	 *
	 * @return string|false The page type. False if it could not be determined.
	 */
	public function get_page_type_for_indexable( $indexable ) {
		switch ( $indexable->object_type ) {
			case 'post':
				$front_page_id = (int) \get_option( 'page_on_front' );
				if ( $indexable->object_id === $front_page_id ) {
					return 'Static_Home_Page';
				}
				$posts_page_id = (int) \get_option( 'page_for_posts' );
				if ( $indexable->object_id === $posts_page_id ) {
					return 'Static_Posts_Page';
				}

				return 'Post_Type';
			case 'term':
				return 'Term_Archive';
			case 'user':
				return 'Author_Archive';
			case 'home-page':
				return 'Home_Page';
			case 'post-type-archive':
				return 'Post_Type_Archive';
			case 'date-archive':
				return 'Date_Archive';
			case 'system-page':
				if ( $indexable->object_sub_type === 'search-result' ) {
					return 'Search_Result_Page';
				}
				if ( $indexable->object_sub_type === '404' ) {
					return 'Error_Page';
				}
		}

		return false;
	}

	/**
	 * Resets the permalinks of the indexables.
	 *
	 * @param string|null $type    The type of the indexable.
	 * @param string|null $subtype The subtype. Can be null.
	 * @param string      $reason  The reason that the permalink has been changed.
	 *
	 * @return void
	 */
	public function reset_permalink_indexables( $type = null, $subtype = null, $reason = Indexing_Reasons::REASON_PERMALINK_SETTINGS ) {
		$result = $this->repository->reset_permalink( $type, $subtype );

		$this->indexing_helper->set_reason( $reason );

		if ( $result !== false && $result > 0 ) {
			\delete_transient( Indexable_Post_Indexation_Action::UNINDEXED_COUNT_TRANSIENT );
			\delete_transient( Indexable_Post_Type_Archive_Indexation_Action::UNINDEXED_COUNT_TRANSIENT );
			\delete_transient( Indexable_Term_Indexation_Action::UNINDEXED_COUNT_TRANSIENT );
		}
	}

	/**
	 * Determines whether indexing indexables is appropriate at this time.
	 *
	 * @return bool Whether the indexables should be indexed.
	 */
	public function should_index_indexables() {
		// Currently, the only reason to index is when we're on a production website.
		$should_index = $this->environment_helper->is_production_mode();

		/**
		 * Filter: 'Yoast\WP\SEO\should_index_indexables' - Allow developers to enable / disable
		 * creating indexables. Warning: overriding
		 * the intended action may cause problems when moving from a staging to a
		 * production environment because indexable permalinks may get set incorrectly.
		 *
		 * @since 18.2
		 *
		 * @param bool $should_index Whether the site's indexables should be created.
		 */
		return (bool) \apply_filters( 'Yoast\WP\SEO\should_index_indexables', $should_index );
	}

	/**
	 * Returns whether or not dynamic permalinks should be used.
	 *
	 * @return bool Whether or not the dynamic permalinks should be used.
	 */
	public function dynamic_permalinks_enabled() {
		/**
		 * Filters the value of the `dynamic_permalinks` option.
		 *
		 * @param bool $value The value of the `dynamic_permalinks` option.
		 */
		return (bool) \apply_filters( 'wpseo_dynamic_permalinks_enabled', $this->options_helper->get( 'dynamic_permalinks', false ) );
	}

	/**
	 * Sets a boolean to indicate that the indexing of the indexables has completed.
	 *
	 * @return void
	 */
	public function finish_indexing() {
		$this->options_helper->set( 'indexables_indexing_completed', true );
	}

	/**
	 * Checks whether the indexable has default values in given fields.
	 *
	 * @param Indexable $indexable The Yoast indexable that we're checking.
	 * @param array     $fields    The Yoast indexable fields that we're checking against.
	 *
	 * @return bool Whether the indexable has default values.
	 */
	public function check_if_default_indexable( $indexable, $fields ) {
		foreach ( $fields as $field ) {
			$is_default = $this->check_if_default_field( $indexable, $field );
			if ( ! $is_default ) {
				break;
			}
		}

		return $is_default;
	}

	/**
	 * Checks if an indexable field contains the default value.
	 *
	 * @param Indexable $indexable The Yoast indexable that we're checking.
	 * @param string    $field     The field that we're checking.
	 *
	 * @return bool True if default value.
	 */
	public function check_if_default_field( $indexable, $field ) {
		$defaults = $this->default_values;
		if ( ! isset( $defaults[ $field ] ) ) {
			return false;
		}

		if ( $indexable->$field === $defaults[ $field ]['default_value'] ) {
			return true;
		}

		return false;
	}
}
                                                                                                                                                                                                                                                                                                                                                                             plugins/wordpress-seo-extended/src/helpers/indexable-to-postmeta-helper.php                         0000644                 00000015407 15122266560 0023277 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Helpers;

use Yoast\WP\SEO\Models\Indexable;

/**
 * A helper object to map indexable data to postmeta.
 */
class Indexable_To_Postmeta_Helper {

	/**
	 * The Meta helper.
	 *
	 * @var Meta_Helper
	 */
	public $meta;

	/**
	 * The map of yoast to post meta.
	 *
	 * @var array
	 */
	protected $yoast_to_postmeta = [
		'title'                  => [
			'post_meta_key' => 'title',
			'map_method'    => 'simple_map',
		],
		'description'            => [
			'post_meta_key' => 'metadesc',
			'map_method'    => 'simple_map',
		],
		'open_graph_title'       => [
			'post_meta_key' => 'opengraph-title',
			'map_method'    => 'simple_map',
		],
		'open_graph_description' => [
			'post_meta_key' => 'opengraph-description',
			'map_method'    => 'simple_map',
		],
		'twitter_title'          => [
			'post_meta_key' => 'twitter-title',
			'map_method'    => 'simple_map',
		],
		'twitter_description'    => [
			'post_meta_key' => 'twitter-description',
			'map_method'    => 'simple_map',
		],
		'canonical'              => [
			'post_meta_key' => 'canonical',
			'map_method'    => 'simple_map',
		],
		'primary_focus_keyword'  => [
			'post_meta_key' => 'focuskw',
			'map_method'    => 'simple_map',
		],
		'open_graph_image'       => [
			'post_meta_key' => 'opengraph-image',
			'map_method'    => 'social_image_map',
		],
		'open_graph_image_id'    => [
			'post_meta_key' => 'opengraph-image-id',
			'map_method'    => 'social_image_map',
		],
		'twitter_image'          => [
			'post_meta_key' => 'twitter-image',
			'map_method'    => 'social_image_map',
		],
		'twitter_image_id'       => [
			'post_meta_key' => 'twitter-image-id',
			'map_method'    => 'social_image_map',
		],
		'is_robots_noindex'      => [
			'post_meta_key' => 'meta-robots-noindex',
			'map_method'    => 'noindex_map',
		],
		'is_robots_nofollow'     => [
			'post_meta_key' => 'meta-robots-nofollow',
			'map_method'    => 'nofollow_map',
		],
		'meta_robots_adv'        => [
			'post_meta_key' => 'meta-robots-adv',
			'map_method'    => 'robots_adv_map',
		],
	];

	/**
	 * Indexable_To_Postmeta_Helper constructor.
	 *
	 * @param Meta_Helper $meta The Meta helper.
	 */
	public function __construct( Meta_Helper $meta ) {
		$this->meta = $meta;
	}

	/**
	 * Creates postmeta from a Yoast indexable.
	 *
	 * @param Indexable $indexable The Yoast indexable.
	 *
	 * @return void
	 */
	public function map_to_postmeta( $indexable ) {
		foreach ( $this->yoast_to_postmeta as $indexable_column => $map_info ) {
			\call_user_func( [ $this, $map_info['map_method'] ], $indexable, $map_info['post_meta_key'], $indexable_column );
		}
	}

	/**
	 * Uses a simple set_value for non-empty data.
	 *
	 * @param Indexable $indexable        The Yoast indexable.
	 * @param string    $post_meta_key    The post_meta key that will be populated.
	 * @param string    $indexable_column The indexable data that will be mapped to post_meta.
	 *
	 * @return void
	 */
	public function simple_map( $indexable, $post_meta_key, $indexable_column ) {
		if ( empty( $indexable->{$indexable_column} ) ) {
			return;
		}

		$this->meta->set_value( $post_meta_key, $indexable->{$indexable_column}, $indexable->object_id );
	}

	/**
	 * Map social image data only if social image is explicitly set.
	 *
	 * @param Indexable $indexable        The Yoast indexable.
	 * @param string    $post_meta_key    The post_meta key that will be populated.
	 * @param string    $indexable_column The indexable data that will be mapped to post_meta.
	 *
	 * @return void
	 */
	public function social_image_map( $indexable, $post_meta_key, $indexable_column ) {
		if ( empty( $indexable->{$indexable_column} ) ) {
			return;
		}

		switch ( $indexable_column ) {
			case 'open_graph_image':
			case 'open_graph_image_id':
				$source = $indexable->open_graph_image_source;
				break;
			case 'twitter_image':
			case 'twitter_image_id':
				$source = $indexable->twitter_image_source;
				break;
		}

		// Map the social image data only when the social image is explicitly set.
		if ( $source === 'set-by-user' || $source === 'imported' ) {
			$value = (string) $indexable->{$indexable_column};

			$this->meta->set_value( $post_meta_key, $value, $indexable->object_id );
		}
	}

	/**
	 * Deletes the noindex post_meta key if no noindex in the indexable. Populates the post_meta key appropriately if there is noindex in the indexable.
	 *
	 * @param Indexable $indexable     The Yoast indexable.
	 * @param string    $post_meta_key The post_meta key that will be populated.
	 *
	 * @return void
	 */
	public function noindex_map( $indexable, $post_meta_key ) {
		if ( \is_null( $indexable->is_robots_noindex ) ) {
			$this->meta->delete( $post_meta_key, $indexable->object_id );
			return;
		}

		if ( $indexable->is_robots_noindex === false ) {
			$this->meta->set_value( $post_meta_key, 2, $indexable->object_id );
		}

		if ( $indexable->is_robots_noindex === true ) {
			$this->meta->set_value( $post_meta_key, 1, $indexable->object_id );
		}
	}

	/**
	 * Deletes the nofollow post_meta key if no nofollow in the indexable or if nofollow is false. Populates the post_meta key appropriately if there is a true nofollow in the indexable.
	 *
	 * @param Indexable $indexable     The Yoast indexable.
	 * @param string    $post_meta_key The post_meta key that will be populated.
	 *
	 * @return void
	 */
	public function nofollow_map( $indexable, $post_meta_key ) {
		if ( \is_null( $indexable->is_robots_nofollow ) || $indexable->is_robots_nofollow === false ) {
			$this->meta->delete( $post_meta_key, $indexable->object_id );
		}

		if ( $indexable->is_robots_nofollow === true ) {
			$this->meta->