class ORM implements ArrayAccess {

	/*
	 * --- CLASS CONSTANTS ---
	 */

	public const CONDITION_FRAGMENT = 0;

	public const CONDITION_VALUES = 1;

	/*
	 * --- INSTANCE PROPERTIES ---
	 */

	/**
	 * Holds the class name. Wrapped find_one and find_many classes will return an instance or instances of this class.
	 *
	 * @var string
	 */
	protected $class_name;

	/**
	 * Holds the name of the table the current ORM instance is associated with.
	 *
	 * @var string
	 */
	protected $table_name;

	/**
	 * Holds the alias for the table to be used in SELECT queries.
	 *
	 * @var string
	 */
	protected $table_alias = null;

	/**
	 * Values to be bound to the query.
	 *
	 * @var array
	 */
	protected $values = [];

	/**
	 * Columns to select in the result.
	 *
	 * @var array
	 */
	protected $result_columns = [ '*' ];

	/**
	 * Are we using the default result column or have these been manually changed?
	 *
	 * @var bool
	 */
	protected $using_default_result_columns = true;

	/**
	 * Holds the join sources.
	 *
	 * @var array
	 */
	protected $join_sources = [];

	/**
	 * Should the query include a DISTINCT keyword?
	 *
	 * @var bool
	 */
	protected $distinct = false;

	/**
	 * Is this a raw query?
	 *
	 * @var bool
	 */
	protected $is_raw_query = false;

	/**
	 * The raw query.
	 *
	 * @var string
	 */
	protected $raw_query = '';

	/**
	 * The raw query parameters.
	 *
	 * @var array
	 */
	protected $raw_parameters = [];

	/**
	 * Array of WHERE clauses.
	 *
	 * @var array
	 */
	protected $where_conditions = [];

	/**
	 * LIMIT.
	 *
	 * @var int
	 */
	protected $limit = null;

	/**
	 * OFFSET.
	 *
	 * @var int
	 */
	protected $offset = null;

	/**
	 * ORDER BY.
	 *
	 * @var array
	 */
	protected $order_by = [];

	/**
	 * GROUP BY.
	 *
	 * @var array
	 */
	protected $group_by = [];

	/**
	 * HAVING.
	 *
	 * @var array
	 */
	protected $having_conditions = [];

	/**
	 * The data for a hydrated instance of the class.
	 *
	 * @var array
	 */
	protected $data = [];

	/**
	 * Lifetime of the object.
	 *
	 * @var array
	 */
	protected $dirty_fields = [];

	/**
	 * Fields that are to be inserted in the DB raw.
	 *
	 * @var array
	 */
	protected $expr_fields = [];

	/**
	 * Is this a new object (has create() been called)?
	 *
	 * @var bool
	 */
	protected $is_new = false;

	/**
	 * Name of the column to use as the primary key for
	 * this instance only. Overrides the config settings.
	 *
	 * @var string
	 */
	protected $instance_id_column = null;

	/*
	 * --- STATIC METHODS ---
	 */

	/**
	 * Factory method, return an instance of this class bound to the supplied
	 * table name.
	 *
	 * A repeat of content in parent::for_table, so that created class is ORM.
	 *
	 * @param string $table_name The table to create instance for.
	 *
	 * @return ORM Instance of the ORM.
	 */
	public static function for_table( $table_name ) {
		return new static( $table_name, [] );
	}

	/**
	 * Executes a raw query as a wrapper for wpdb::query.
	 * Useful for queries that can't be accomplished through Idiorm,
	 * particularly those using engine-specific features.
	 *
	 * @example raw_execute('INSERT OR REPLACE INTO `widget` (`id`, `name`) SELECT `id`, `name` FROM `other_table`')
	 * @example raw_execute('SELECT `name`, AVG(`order`) FROM `customer` GROUP BY `name` HAVING AVG(`order`) > 10')
	 *
	 * @param string $query      The raw SQL query.
	 * @param array  $parameters Optional bound parameters.
	 *
	 * @return bool Success.
	 */
	public static function raw_execute( $query, $parameters = [] ) {
		return self::execute( $query, $parameters );
	}

	/**
	 * Internal helper method for executing statements.
	 *
	 * @param string $query      The query.
	 * @param array  $parameters An array of parameters to be bound in to the query.
	 *
	 * @return bool|int Response of wpdb::query
	 */
	protected static function execute( $query, $parameters = [] ) {
		/**
		 * The global WordPress database variable.
		 *
		 * @var wpdb
		 */
		global $wpdb;

		$show_errors = $wpdb->show_errors;

		if ( \YoastSEO()->classes->get( Migration_Status::class )->get_error( 'free' ) ) {
			$wpdb->show_errors = false;
		}

		$parameters = \array_filter(
			$parameters,
			static function ( $parameter ) {
				return $parameter !== null;
			}
		);
		if ( ! empty( $parameters ) ) {
			$query = $wpdb->prepare( $query, $parameters );
		}

		$result = $wpdb->query( $query );

		$wpdb->show_errors = $show_errors;

		return $result;
	}

	/*
	 * --- INSTANCE METHODS ---
	 */

	/**
	 * "Private" constructor; shouldn't be called directly.
	 * Use the ORM::for_table factory method instead.
	 *
	 * @param string $table_name Table name.
	 * @param array  $data       Data to populate table.
	 */
	protected function __construct( $table_name, $data = [] ) {
		$this->table_name = $table_name;
		$this->data       = $data;
	}

	/**
	 * Sets the name of the class which the wrapped methods should return instances of.
	 *
	 * @param string $class_name The classname to set.
	 *
	 * @return void
	 */
	public function set_class_name( $class_name ) {
		$this->class_name = $class_name;
	}

	/**
	 * Creates a new, empty instance of the class. Used to add a new row to your database. May optionally be passed an
	 * associative array of data to populate the instance. If so, all fields will be flagged as dirty so all will be
	 * saved to the database when save() is called.
	 *
	 * @param array|null $data Data to populate table.
	 *
	 * @return bool|Model|ORM
	 */
	public function create( $data = null ) {
		$this->is_new = true;
		if ( ! \is_null( $data ) ) {
			$this->hydrate( $data )->force_all_dirty();
		}

		return $this->create_model_instance( $this );
	}

	/**
	 * Specifies the ID column to use for this instance or array of instances only.
	 * This overrides the id_column and id_column_overrides settings.
	 *
	 * This is mostly useful for libraries built on top of Idiorm, and will not normally be used in manually built
	 * queries. If you don't know why you would want to use this, you should probably just ignore it.
	 *
	 * @param string $id_column The ID column.
	 *
	 * @return ORM
	 */
	public function use_id_column( $id_column ) {
		$this->instance_id_column = $id_column;

		return $this;
	}

	/**
	 * Creates an ORM instance from the given row (an associative array of data fetched from the database).
	 *
	 * @param array $row A row from the database.
	 *
	 * @return bool|Model
	 */
	protected function create_instance_from_row( $row ) {
		$instance = self::for_table( $this->table_name );
		$instance->use_id_column( $this->instance_id_column );
		$instance->hydrate( $row );

		return $this->create_model_instance( $instance );
	}

	/**
	 * Tells the ORM that you are expecting a single result back from your query, and execute it. Will return a single
	 * instance of the ORM class, or false if no rows were returned. As a shortcut, you may supply an ID as a parameter
	 * to this method. This will perform a primary key lookup on the table.
	 *
	 * @param int|null $id An (optional) ID.
	 *
	 * @return bool|Model
	 */
	public function find_one( $id = null ) {
		if ( ! \is_null( $id ) ) {
			$this->where_id_is( $id );
		}
		$this->limit( 1 );
		$rows = $this->run();
		if ( empty( $rows ) ) {
			return false;
		}

		return $this->create_instance_from_row( $rows[0] );
	}

	/**
	 * Tells the ORM that you are expecting multiple results from your query, and execute it. Will return an array of
	 * instances of the ORM class, or an empty array if no rows were returned.
	 *
	 * @return array
	 */
	public function find_many() {
		$rows = $this->run();

		if ( $rows === false ) {
			return [];
		}

		return \array_map( [ $this, 'create_instance_from_row' ], $rows );
	}

	/**
	 * Creates an instance of the model class associated with this wrapper and populate it with the supplied Idiorm
	 * instance.
	 *
	 * @param ORM $orm The ORM used by model.
	 *
	 * @return bool|Model Instance of the model class.
	 */
	protected function create_model_instance( $orm ) {
		if ( $orm === false ) {
			return false;
		}

		/**
		 * An instance of Model is being made.
		 *
		 * @var Model $model
		 */
		$model = new $this->class_name();
		$model->set_orm( $orm );

		return $model;
	}

	/**
	 * Tells the ORM that you are expecting multiple results from your query, and execute it. Will return an array, or
	 * an empty array if no rows were returned.
	 *
	 * @return array The query results.
	 */
	public function find_array() {
		return $this->run();
	}

	/**
	 * Tells the ORM that you wish to execute a COUNT query.
	 *
	 * @param string $column The table column.
	 *
	 * @return float|int An integer representing the number of rows returned.
	 */
	public function count( $column = '*' ) {
		return $this->call_aggregate_db_function( __FUNCTION__, $column );
	}

	/**
	 * Tells the ORM that you wish to execute a MAX query.
	 *
	 * @param string $column The table column.
	 *
	 * @return float|int The max value of the chosen column.
	 */
	public function max( $column ) {
		return $this->call_aggregate_db_function( __FUNCTION__, $column );
	}

	/**
	 * Tells the ORM that you wish to execute a MIN query.
	 *
	 * @param string $column The table column.
	 *
	 * @return float|int The min value of the chosen column.
	 */
	public function min( $column ) {
		return $this->call_aggregate_db_function( __FUNCTION__, $column );
	}

	/**
	 * Tells the ORM that you wish to execute a AVG query.
	 *
	 * @param string $column The table column.
	 *
	 * @return float|int The average value of the chosen column.
	 */
	public function avg( $column ) {
		return $this->call_aggregate_db_function( __FUNCTION__, $column );
	}

	/**
	 * Tells the ORM that you wish to execute a SUM query.
	 *
	 * @param string $column The table column.
	 *
	 * @return float|int The sum of the chosen column.
	 */
	public function sum( $column ) {
		return $this->call_aggregate_db_function( __FUNCTION__, $column );
	}

	/**
	 * Returns the select query as SQL.
	 *
	 * @return string The select query in SQL.
	 */
	public function get_sql() {
		return $this->build_select();
	}

	/**
	 * Returns the update query as SQL.
	 *
	 * @return string The update query in SQL.
	 */
	public function get_update_sql() {
		return $this->build_update();
	}

	/**
	 * Executes an aggregate query on the current connection.
	 *
	 * @param string $sql_function The aggregate function to call eg. MIN, COUNT, etc.
	 * @param string $column       The column to execute the aggregate query against.
	 *
	 * @return int
	 */
	protected function call_aggregate_db_function( $sql_function, $column ) {
		$alias        = \strtolower( $sql_function );
		$sql_function = \strtoupper( $sql_function );
		if ( $column !== '*' ) {
			$column = $this->quote_identifier( $column );
		}
		$result_columns       = $this->result_columns;
		$this->result_columns = [];
		$this->select_expr( "{$sql_function}({$column})", $alias );
		$result               = $this->find_one();
		$this->result_columns = $result_columns;
		$return_value         = 0;
		if ( $result !== false && isset( $result->{$alias} ) ) {
			if ( ! \is_numeric( $result->{$alias} ) ) {
				$return_value = $result->{$alias};
			}
			// phpcs:ignore Universal.Operators.StrictComparisons -- Reason: This loose comparison seems intentional.
			elseif ( (int) $result->{$alias} == (float) $result->{$alias} ) {
				$return_value = (int) $result->{$alias};
			}
			else {
				$return_value = (float) $result->{$alias};
			}
		}

		return $return_value;
	}

	/**
	 * Hydrates (populate) this instance of the class from an associative array of data. This will usually be called
	 * only from inside the class, but it's public in case you need to call it directly.
	 *
	 * @param array $data Data to populate table.
	 *
	 * @return ORM
	 */
	public function hydrate( $data = [] ) {
		$this->data = $data;

		return $this;
	}

	/**
	 * Forces the ORM to flag all the fields in the $data array as "dirty" and therefore update them when save() is
	 * called.
	 *
	 * @return ORM
	 */
	public function force_all_dirty() {
		$this->dirty_fields = $this->data;

		return $this;
	}

	/**
	 * Performs a raw query. The query can contain placeholders in either named or question mark style. If placeholders
	 * are used, the parameters should be an array of values which will be bound to the placeholders in the query.
	 * If this method is called, all other query building methods will be ignored.
	 *
	 * @param array $query      The query.
	 * @param array $parameters The parameters. Defaults to an empty array.
	 *
	 * @return ORM
	 */
	public function raw_query( $query, $parameters = [] ) {
		$this->is_raw_query   = true;
		$this->raw_query      = $query;
		$this->raw_parameters = $parameters;

		return $this;
	}

	/**
	 * Adds an alias for the main table to be used in SELECT queries.
	 *
	 * @param string $alias The alias.
	 *
	 * @return ORM
	 */
	public function table_alias( $alias ) {
		$this->table_alias = $alias;

		return $this;
	}

	/**
	 * Adds an unquoted expression to the set of columns returned by the SELECT query. Internal method.
	 *
	 * @param string      $expr  The expression.
	 * @param string|null $alias The alias to return the expression as. Defaults to null.
	 *
	 * @return ORM
	 */
	protected function add_result_column( $expr, $alias = null ) {
		if ( ! \is_null( $alias ) ) {
			$expr .= ' AS ' . $this->quote_identifier( $alias );
		}
		if ( $this->using_default_result_columns ) {
			$this->result_columns               = [ $expr ];
			$this->using_default_result_columns = false;
		}
		else {
			$this->result_columns[] = $expr;
		}

		return $this;
	}

	/**
	 * Counts the number of columns that belong to the primary key and their value is null.
	 *
	 * @return int The amount of null columns.
	 *
	 * @throws Exception Primary key ID contains null value(s).
	 * @throws Exception Primary key ID missing from row or is null.
	 */
	public function count_null_id_columns() {
		if ( \is_array( $this->get_id_column_name() ) ) {
			return \count( \array_filter( $this->id(), 'is_null' ) );
		}
		else {
			return \is_null( $this->id() ) ? 1 : 0;
		}
	}

	/**
	 * Adds a column to the list of columns returned by the SELECT query.
	 *
	 * @param string      $column The column. Defaults to '*'.
	 * @param string|null $alias  The alias to return the column as. Defaults to null.
	 *
	 * @return ORM
	 */
	public function select( $column, $alias = null ) {
		$column = $this->quote_identifier( $column );

		return $this->add_result_column( $column, $alias );
	}

	/**
	 * Adds an unquoted expression to the list of columns returned by the SELECT query.
	 *
	 * @param string      $expr  The expression.
	 * @param string|null $alias The alias to return the column as. Defaults to null.
	 *
	 * @return ORM
	 */
	public function select_expr( $expr, $alias = null ) {
		return $this->add_result_column( $expr, $alias );
	}

	/**
	 * Adds columns to the list of columns returned by the SELECT query.
	 *
	 * This defaults to '*'.
	 * Many columns can be supplied as either an array or as a list of parameters to the method.
	 * Note that the alias must not be numeric - if you want a numeric alias then prepend it with some alpha chars. eg.
	 * a1.
	 *
	 * @example select_many(array('column', 'column2', 'column3'), 'column4', 'column5');
	 * @example select_many(array('alias' => 'column', 'column2', 'alias2' => 'column3'), 'column4', 'column5');
	 * @example select_many('column', 'column2', 'column3');
	 *
	 * @return ORM
	 */
	public function select_many() {
		$columns = \func_get_args();
		if ( ! empty( $columns ) ) {
			$columns = $this->normalise_select_many_columns( $columns );
			foreach ( $columns as $alias => $column ) {
				if ( \is_numeric( $alias ) ) {
					$alias = null;
				}
				$this->select( $column, $alias );
			}
		}

		return $this;
	}

	/**
	 * Adds an unquoted expression to the list of columns returned by the SELECT query.
	 *
	 * Many columns can be supplied as either an array or as a list of parameters to the method.
	 * Note that the alias must not be numeric - if you want a numeric alias then prepend it with some alpha chars. eg.
	 * a1
	 *
	 * @example select_many_expr(array('alias' => 'column', 'column2', 'alias2' => 'column3'), 'column4', 'column5')
	 * @example select_many_expr('column', 'column2', 'column3')
	 * @example select_many_expr(array('column', 'column2', 'column3'), 'column4', 'column5')
	 *
	 * @return ORM
	 */
	public function select_many_expr() {
		$columns = \func_get_args();
		if ( ! empty( $columns ) ) {
			$columns = $this->normalise_select_many_columns( $columns );
			foreach ( $columns as $alias => $column ) {
				if ( \is_numeric( $alias ) ) {
					$alias = null;
				}
				$this->select_expr( $column, $alias );
			}
		}

		return $this;
	}

	/**
	 * Takes a column specification for the select many methods and convert it into a normalised array of columns and
	 * aliases.
	 *
	 * It is designed to turn the following styles into a normalised array:
	 * array(array('alias' => 'column', 'column2', 'alias2' => 'column3'), 'column4', 'column5'))
	 *
	 * @param array $columns The columns.
	 *
	 * @return array
	 */
	protected function normalise_select_many_columns( $columns ) {
		$return = [];
		foreach ( $columns as $column ) {
			if ( \is_array( $column ) ) {
				foreach ( $column as $key => $value ) {
					if ( ! \is_numeric( $key ) ) {
						$return[ $key ] = $value;
					}
					else {
						$return[] = $value;
					}
				}
			}
			else {
				$return[] = $column;
			}
		}

		return $return;
	}

	/**
	 * Adds a DISTINCT keyword before the list of columns in the SELECT query.
	 *
	 * @return ORM
	 */
	public function distinct() {
		$this->distinct = true;

		return $this;
	}

	/**
	 * Add a JOIN source to the query. Internal method.
	 *
	 * The join_operator should be one of INNER, LEFT OUTER, CROSS etc - this
	 * will be prepended to JOIN.
	 *
	 * The table should be the name of the table to join to.
	 *
	 * The constraint may be either a string or an array with three elements. If it
	 * is a string, it will be compiled into the query as-is, with no escaping. The
	 * recommended way to supply the constraint is as an array with three elements:
	 *
	 * first_column, operator, second_column
	 *
	 * Example: array('user.id', '=', 'profile.user_id')
	 *
	 * will compile to
	 *
	 * ON `user`.`id` = `profile`.`user_id`
	 *
	 * The final (optional) argument specifies an alias for the joined table.
	 *
	 * @param string      $join_operator The join_operator should be one of INNER, LEFT OUTER, CROSS etc - this will be
	 *                                   prepended to JOIN.
	 * @param string      $table         The table should be the name of the table to join to.
	 * @param string      $constraint    The constraint.
	 * @param string|null $table_alias   The alias for the joined table. Defaults to null.
	 *
	 * @return ORM
	 */
	protected function add_join_source( $join_operator, $table, $constraint, $table_alias = null ) {
		$join_operator = \trim( "{$join_operator} JOIN" );
		$table         = $this->quote_identifier( $table );
		// Add table alias if present.
		if ( ! \is_null( $table_alias ) ) {
			$table_alias = $this->quote_identifier( $table_alias );
			$table      .= " {$table_alias}";
		}
		// Build the constraint.
		if ( \is_array( $constraint ) ) {
			list( $first_column, $operator, $second_column ) = $constraint;

			$first_column  = $this->quote_identifier( $first_column );
			$second_column = $this->quote_identifier( $second_column );
			$constraint    = "{$first_column} {$operator} {$second_column}";
		}
		$this->join_sources[] = "{$join_operator} {$table} ON {$constraint}";

		return $this;
	}

	/**
	 * Adds a RAW JOIN source to the query.
	 *
	 * @param string $table       The table name.
	 * @param string $constraint  The constraint.
	 * @param string $table_alias The table alias.
	 * @param array  $parameters  The parameters. Defaults to an empty array.
	 *
	 * @return ORM
	 */
	public function raw_join( $table, $constraint, $table_alias, $parameters = [] ) {
		// Add table alias if present.
		if ( ! \is_null( $table_alias ) ) {
			$table_alias = $this->quote_identifier( $table_alias );
			$table      .= " {$table_alias}";
		}
		$this->values = \array_merge( $this->values, $parameters );
		// Build the constraint.
		if ( \is_array( $constraint ) ) {
			list( $first_column, $operator, $second_column ) = $constraint;

			$first_column  = $this->quote_identifier( $first_column );
			$second_column = $this->quote_identifier( $second_column );
			$constraint    = "{$first_column} {$operator} {$second_column}";
		}
		$this->join_sources[] = "{$table} ON {$constraint}";

		return $this;
	}

	/**
	 * Adds a simple JOIN source to the query.
	 *
	 * @param string      $table       The table name.
	 * @param string      $constraint  The constraint.
	 * @param string|null $table_alias The table alias. Defaults to null.
	 *
	 * @return ORM
	 */
	public function join( $table, $constraint, $table_alias = null ) {
		return $this->add_join_source( '', $table, $constraint, $table_alias );
	}

	/**
	 * Adds an INNER JOIN source to the query.
	 *
	 * @param string      $table       The table name.
	 * @param string      $constraint  The constraint.
	 * @param string|null $table_alias The table alias. Defaults to null.
	 *
	 * @return ORM
	 */
	public function inner_join( $table, $constraint, $table_alias = null ) {
		return $this->add_join_source( 'INNER', $table, $constraint, $table_alias );
	}

	/**
	 * Adds a LEFT OUTER JOIN source to the query.
	 *
	 * @param string      $table       The table name.
	 * @param string      $constraint  The constraint.
	 * @param string|null $table_alias The table alias. Defaults to null.
	 *
	 * @return ORM
	 */
	public function left_outer_join( $table, $constraint, $table_alias = null ) {
		return $this->add_join_source( 'LEFT OUTER', $table, $constraint, $table_alias );
	}

	/**
	 * Adds a RIGHT OUTER JOIN source to the query.
	 *
	 * @param string      $table       The table name.
	 * @param string      $constraint  The constraint.
	 * @param string|null $table_alias The table alias. Defaults to null.
	 *
	 * @return ORM
	 */
	public function right_outer_join( $table, $constraint, $table_alias = null ) {
		return $this->add_join_source( 'RIGHT OUTER', $table, $constraint, $table_alias );
	}

	/**
	 * Adds a FULL OUTER JOIN source to the query.
	 *
	 * @param string      $table       The table name.
	 * @param string      $constraint  The constraint.
	 * @param string|null $table_alias The table alias. Defaults to null.
	 *
	 * @return ORM
	 */
	public function full_outer_join( $table, $constraint, $table_alias = null ) {
		return $this->add_join_source( 'FULL OUTER', $table, $constraint, $table_alias );
	}

	/**
	 * Adds a HAVING condition to the query. Internal method.
	 *
	 * @param string $fragment The fragment.
	 * @param array  $values   The values. Defaults to an empty array.
	 *
	 * @return ORM
	 */
	protected function add_having( $fragment, $values = [] ) {
		return $this->add_condition( 'having', $fragment, $values );
	}

	/**
	 * Adds a HAVING condition to the query. Internal method.
	 *
	 * @param string $column_name The table column.
	 * @param string $separator   The separator.
	 * @param mixed  $value       The value.
	 *
	 * @return ORM
	 */
	protected function add_simple_having( $column_name, $separator, $value ) {
		return $this->add_simple_condition( 'having', $column_name, $separator, $value );
	}

	/**
	 * Adds a HAVING clause with multiple values (like IN and NOT IN). Internal method.
	 *
	 * @param string|array $column_name The table column.
	 * @param string       $separator   The separator.
	 * @param array        $values      The values.
	 *
	 * @return ORM
	 */
	public function add_having_placeholder( $column_name, $separator, $values ) {
		if ( ! \is_array( $column_name ) ) {
			$data = [ $column_name => $values ];
		}
		else {
			$data = $column_name;
		}
		$result = $this;
		foreach ( $data as $key => $val ) {
			$column       = $result->quote_identifier( $key );
			$placeholders = $result->create_placeholders( $val );
			$result       = $result->add_having( "{$column} {$separator} ({$placeholders})", $val );
		}

		return $result;
	}

	/**
	 * Adds a HAVING clause with no parameters(like IS NULL and IS NOT NULL). Internal method.
	 *
	 * @param string $column_name The column name.
	 * @param string $operator    The operator.
	 *
	 * @return ORM
	 */
	public function add_having_no_value( $column_name, $operator ) {
		$conditions = \is_array( $column_name ) ? $column_name : [ $column_name ];
		$result     = $this;
		foreach ( $conditions as $column ) {
			$column = $this->quote_identifier( $column );
			$result = $result->add_having( "{$column} {$operator}" );
		}

		return $result;
	}

	/**
	 * Adds a WHERE condition to the query. Internal method.
	 *
	 * @param string $fragment The fragment.
	 * @param array  $values   The values. Defaults to an empty array.
	 *
	 * @return ORM
	 */
	protected function add_where( $fragment, $values = [] ) {
		return $this->add_condition( 'where', $fragment, $values );
	}

	/**
	 * Adds a WHERE condition to the query. Internal method.
	 *
	 * @param string|array $column_name The table column.
	 * @param string       $separator   The separator.
	 * @param mixed        $value       The value.
	 *
	 * @return ORM
	 */
	protected function add_simple_where( $column_name, $separator, $value ) {
		return $this->add_simple_condition( 'where', $column_name, $separator, $value );
	}

	/**
	 * Adds a WHERE clause with multiple values (like IN and NOT IN).
	 *
	 * @param string|array $column_name The table column.
	 * @param string       $separator   The separator.
	 * @param array        $values      The values.
	 *
	 * @return ORM
	 */
	public function add_where_placeholder( $column_name, $separator, $values ) {
		if ( ! \is_array( $column_name ) ) {
			$data = [ $column_name => $values ];
		}
		else {
			$data = $column_name;
		}
		$result = $this;
		foreach ( $data as $key => $val ) {
			$column       = $result->quote_identifier( $key );
			$placeholders = $result->create_placeholders( $val );
			$result       = $result->add_where( "{$column} {$separator} ({$placeholders})", $val );
		}

		return $result;
	}

	/**
	 * Adds a WHERE clause with no parameters(like IS NULL and IS NOT NULL).
	 *
	 * @param string $column_name The column name.
	 * @param string $operator    The operator.
	 *
	 * @return ORM
	 */
	public function add_where_no_value( $column_name, $operator ) {
		$conditions = \is_array( $column_name ) ? $column_name : [ $column_name ];
		$result     = $this;
		foreach ( $conditions as $column ) {
			$column = $this->quote_identifier( $column );
			$result = $result->add_where( "{$column} {$operator}" );
		}

		return $result;
	}

	/**
	 * Adds a HAVING or WHERE condition to the query. Internal method.
	 *
	 * @param string $type     The type.
	 * @param string $fragment The fragment.
	 * @param array  $values   The values. Defaults to empty array.
	 *
	 * @return ORM
	 */
	protected function add_condition( $type, $fragment, $values = [] ) {
		$conditions_class_property_name = "{$type}_conditions";
		if ( ! \is_array( $values ) ) {
			$values = [ $values ];
		}
		\array_push(
			$this->{$conditions_class_property_name},
			[
				self::CONDITION_FRAGMENT => $fragment,
				self::CONDITION_VALUES   => $values,
			]
		);

		return $this;
	}

	/**
	 * Compiles a simple COLUMN SEPARATOR VALUE style HAVING or WHERE condition into a string and value ready to be
	 * passed to the add_condition method.
	 *
	 * Avoids duplication of the call to quote_identifier.
	 * If column_name is an associative array, it will add a condition for each column.
	 *
	 * @param string       $type        The type.
	 * @param string|array $column_name The table column.
	 * @param string       $separator   The separator.
	 * @param mixed        $value       The value.
	 *
	 * @return ORM
	 */
	protected function add_simple_condition( $type, $column_name, $separator, $value ) {
		$multiple = \is_array( $column_name ) ? $column_name : [ $column_name => $value ];
		$result   = $this;
		foreach ( $multiple as $key => $val ) {
			// Add the table name in case of ambiguous columns.
			if ( \count( $result->join_sources ) > 0 && \strpos( $key, '.' ) === false ) {
				$table = $result->table_name;
				if ( ! \is_null( $result->table_alias ) ) {
					$table = $result->table_alias;
				}
				$key = "{$table}.{$key}";
			}
			$key         = $result->quote_identifier( $key );
			$placeholder = ( $val === null ) ? 'NULL' : '%s';
			$result      = $result->add_condition( $type, "{$key} {$separator} {$placeholder}", $val );
		}

		return $result;
	}

	/**
	 * Returns a string containing the given number of question marks, separated by commas. Eg "?, ?, ?".
	 *
	 * @param array $fields Fields to create placeholder for.
	 *
	 * @return string
	 */
	protected function create_placeholders( $fields ) {
		if ( ! empty( $fields ) ) {
			$db_fields = [];
			foreach ( $fields as $key => $value ) {
				// Process expression fields directly into the query.
				if ( \array_key_exists( $key, $this->expr_fields ) ) {
					$db_fields[] = $value;
				}
				else {
					$db_fields[] = ( $value === null ) ? 'NULL' : '%s';
				}
			}

			return \implode( ', ', $db_fields );
		}

		return '';
	}

	/**
	 * Filters a column/value array returning only those columns that belong to a compound primary key.
	 *
	 * If the key contains a column that does not exist in the given array, a null value will be returned for it.
	 *
	 * @param mixed $value The value.
	 *
	 * @return array
	 */
	protected function get_compound_id_column_values( $value ) {
		$filtered = [];
		foreach ( $this->get_id_column_name() as $key ) {
			$filtered[ $key ] = ( $value[ $key ] ?? null );
		}

		return $filtered;
	}

	/**
	 * Filters an array containing compound column/value arrays.
	 *
	 * @param array $values The values.
	 *
	 * @return array
	 */
	protected function get_compound_id_column_values_array( $values ) {
		$filtered = [];
		foreach ( $values as $value ) {
			$filtered[] = $this->get_compound_id_column_values( $value );
		}

		return $filtered;
	}

	/**
	 * Add a WHERE column = value clause to your query. Each time this is called in the chain, an additional WHERE will
	 * be added, and these will be ANDed together when the final query is built.
	 *
	 * If you use an array in $column_name, a new clause will be added for each element. In this case, $value is
	 * ignored.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where( $column_name, $value = null ) {
		return $this->where_equal( $column_name, $value );
	}

	/**
	 * More explicitly named version of for the where() method. Can be used if preferred.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_equal( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, '=', $value );
	}

	/**
	 * Add a WHERE column != value clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_not_equal( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, '!=', $value );
	}

	/**
	 * Queries the table by its primary key. Special method.
	 *
	 * If primary key is compound, only the columns that belong to they key will be used for the query.
	 *
	 * @param string $id The ID.
	 *
	 * @return ORM
	 */
	public function where_id_is( $id ) {
		return \is_array( $this->get_id_column_name() ) ? $this->where( $this->get_compound_id_column_values( $id ), null ) : $this->where( $this->get_id_column_name(), $id );
	}

	/**
	 * Allows adding a WHERE clause that matches any of the conditions specified in the array. Each element in the
	 * associative array will be a different condition, where the key will be the column name.
	 *
	 * By default, an equal operator will be used against all columns, but it can be overriden for any or every column
	 * using the second parameter.
	 *
	 * Each condition will be ORed together when added to the final query.
	 *
	 * @param array  $values   The values.
	 * @param string $operator The operator.
	 *
	 * @return ORM
	 */
	public function where_any_is( $values, $operator = '=' ) {
		$data  = [];
		$query = [ '((' ];
		$first = true;
		foreach ( $values as $value ) {
			if ( $first ) {
				$first = false;
			}
			else {
				$query[] = ') OR (';
			}
			$firstsub = true;
			foreach ( $value as $key => $item ) {
				$op = \is_string( $operator ) ? $operator : ( $operator[ $key ] ?? '=' );
				if ( $op === '=' && $item === null ) {
					$op = 'IS';
				}
				if ( $firstsub ) {
					$firstsub = false;
				}
				else {
					$query[] = 'AND';
				}
				$query[] = $this->quote_identifier( $key );
				$data[]  = $item;
				$query[] = $op;
				$query[] = ( ( $item === null ) ? 'NULL' : '%s' );
			}
		}
		$query[] = '))';

		return $this->where_raw( \implode( ' ', $query ), $data );
	}

	/**
	 * Queries the table by its primary key.
	 *
	 * Similar to where_id_is() but allowing multiple primary keys.
	 * If primary key is compound, only the columns that belong to they key will be used for the query.
	 *
	 * @param string[] $ids The IDs.
	 *
	 * @return ORM
	 */
	public function where_id_in( $ids ) {
		return \is_array( $this->get_id_column_name() ) ? $this->where_any_is( $this->get_compound_id_column_values_array( $ids ) ) : $this->where_in( $this->get_id_column_name(), $ids );
	}

	/**
	 * Adds a WHERE ... LIKE clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_like( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, 'LIKE', $value );
	}

	/**
	 * Adds where WHERE ... NOT LIKE clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_not_like( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, 'NOT LIKE', $value );
	}

	/**
	 * Adds a WHERE ... > clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_gt( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, '>', $value );
	}

	/**
	 * Adds a WHERE ... < clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_lt( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, '<', $value );
	}

	/**
	 * Adds a WHERE ... >= clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_gte( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, '>=', $value );
	}

	/**
	 * Adds a WHERE ... <= clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function where_lte( $column_name, $value = null ) {
		return $this->add_simple_where( $column_name, '<=', $value );
	}

	/**
	 * Adds a WHERE ... IN clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param array        $values      The values.
	 *
	 * @return ORM
	 */
	public function where_in( $column_name, $values ) {
		return $this->add_where_placeholder( $column_name, 'IN', $values );
	}

	/**
	 * Adds a WHERE ... NOT IN clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param array        $values      The values.
	 *
	 * @return ORM
	 */
	public function where_not_in( $column_name, $values ) {
		return $this->add_where_placeholder( $column_name, 'NOT IN', $values );
	}

	/**
	 * Adds a WHERE column IS NULL clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function where_null( $column_name ) {
		return $this->add_where_no_value( $column_name, 'IS NULL' );
	}

	/**
	 * Adds a WHERE column IS NOT NULL clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function where_not_null( $column_name ) {
		return $this->add_where_no_value( $column_name, 'IS NOT NULL' );
	}

	/**
	 * Adds a raw WHERE clause to the query. The clause should contain question mark placeholders, which will be bound
	 * to the parameters supplied in the second argument.
	 *
	 * @param string $clause     The clause that should contain question mark placeholders.
	 * @param array  $parameters The parameters to include in the query.
	 *
	 * @return ORM
	 */
	public function where_raw( $clause, $parameters = [] ) {
		return $this->add_where( $clause, $parameters );
	}

	/**
	 * Adds a LIMIT to the query.
	 *
	 * @param int $limit The limit.
	 *
	 * @return ORM
	 */
	public function limit( $limit ) {
		$this->limit = $limit;

		return $this;
	}

	/**
	 * Adds an OFFSET to the query.
	 *
	 * @param int $offset The offset.
	 *
	 * @return ORM
	 */
	public function offset( $offset ) {
		$this->offset = $offset;

		return $this;
	}

	/**
	 * Adds an ORDER BY clause to the query.
	 *
	 * @param string $column_name The column name.
	 * @param string $ordering    The ordering. DESC or ASC.
	 *
	 * @return ORM
	 */
	protected function add_order_by( $column_name, $ordering ) {
		$column_name      = $this->quote_identifier( $column_name );
		$this->order_by[] = "{$column_name} {$ordering}";

		return $this;
	}

	/**
	 * Adds an ORDER BY column DESC clause.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function order_by_desc( $column_name ) {
		return $this->add_order_by( $column_name, 'DESC' );
	}

	/**
	 * Adds an ORDER BY column ASC clause.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function order_by_asc( $column_name ) {
		return $this->add_order_by( $column_name, 'ASC' );
	}

	/**
	 * Adds an unquoted expression as an ORDER BY clause.
	 *
	 * @param string $clause The clause.
	 *
	 * @return ORM
	 */
	public function order_by_expr( $clause ) {
		$this->order_by[] = $clause;

		return $this;
	}

	/**
	 * Adds a column to the list of columns to GROUP BY.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function group_by( $column_name ) {
		$column_name      = $this->quote_identifier( $column_name );
		$this->group_by[] = $column_name;

		return $this;
	}

	/**
	 * Adds an unquoted expression to the list of columns to GROUP BY.
	 *
	 * @param string $expr The expression.
	 *
	 * @return ORM
	 */
	public function group_by_expr( $expr ) {
		$this->group_by[] = $expr;

		return $this;
	}

	/**
	 * Adds a HAVING column = value clause to your query.
	 *
	 * Each time this is called in the chain, an additional HAVING will be added, and these will be ANDed together when
	 * the final query is built.
	 *
	 * If you use an array in $column_name, a new clause will be added for each element. In this case, $value is
	 * ignored.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value.
	 *
	 * @return ORM
	 */
	public function having( $column_name, $value = null ) {
		return $this->having_equal( $column_name, $value );
	}

	/**
	 * Adds a having equal to your query.
	 *
	 * More explicitly named version of for the having() method. Can be used if preferred.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value.
	 *
	 * @return ORM
	 */
	public function having_equal( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, '=', $value );
	}

	/**
	 * Adds a HAVING column != value clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed|null   $value       The value.
	 *
	 * @return ORM
	 */
	public function having_not_equal( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, '!=', $value );
	}

	/**
	 * Queries the table by its primary key. Special method.
	 *
	 * If primary key is compound, only the columns that belong to they key will be used for the query.
	 *
	 * @param string $id The ID.
	 *
	 * @return ORM
	 */
	public function having_id_is( $id ) {
		return \is_array( $this->get_id_column_name() ) ? $this->having( $this->get_compound_id_column_values( $id ), null ) : $this->having( $this->get_id_column_name(), $id );
	}

	/**
	 * Adds a HAVING ... LIKE clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param string|null  $value       The value.
	 *
	 * @return ORM
	 */
	public function having_like( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, 'LIKE', $value );
	}

	/**
	 * Adds where HAVING ... NOT LIKE clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param string|null  $value       The value.
	 *
	 * @return ORM
	 */
	public function having_not_like( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, 'NOT LIKE', $value );
	}

	/**
	 * Adds a HAVING ... > clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed        $value       The value.
	 *
	 * @return ORM
	 */
	public function having_gt( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, '>', $value );
	}

	/**
	 * Adds a HAVING ... < clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed        $value       The value.
	 *
	 * @return ORM
	 */
	public function having_lt( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, '<', $value );
	}

	/**
	 * Adds a HAVING ... >= clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed        $value       The value. Defaults to null.
	 *
	 * @return ORM
	 */
	public function having_gte( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, '>=', $value );
	}

	/**
	 * Adds a HAVING ... <= clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param mixed        $value       The value.
	 *
	 * @return ORM
	 */
	public function having_lte( $column_name, $value = null ) {
		return $this->add_simple_having( $column_name, '<=', $value );
	}

	/**
	 * Adds a HAVING ... IN clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param array|null   $values      The values. Defaults to null.
	 *
	 * @return ORM
	 */
	public function having_in( $column_name, $values = null ) {
		return $this->add_having_placeholder( $column_name, 'IN', $values );
	}

	/**
	 * Adds a HAVING ... NOT IN clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 * @param array|null   $values      The values. Defaults to null.
	 *
	 * @return ORM
	 */
	public function having_not_in( $column_name, $values = null ) {
		return $this->add_having_placeholder( $column_name, 'NOT IN', $values );
	}

	/**
	 * Adds a HAVING column IS NULL clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function having_null( $column_name ) {
		return $this->add_having_no_value( $column_name, 'IS NULL' );
	}

	/**
	 * Adds a HAVING column IS NOT NULL clause to your query.
	 *
	 * @param string|array $column_name The table column.
	 *
	 * @return ORM
	 */
	public function having_not_null( $column_name ) {
		return $this->add_having_no_value( $column_name, 'IS NOT NULL' );
	}

	/**
	 * Adds a raw HAVING clause to the query. The clause should contain question mark placeholders, which will be bound
	 * to the parameters supplied in the second argument.
	 *
	 * @param string $clause     The clause that should contain question mark placeholders.
	 * @param array  $parameters The parameters to include in the query.
	 *
	 * @return ORM
	 */
	public function having_raw( $clause, $parameters = [] ) {
		return $this->add_having( $clause, $parameters );
	}

	/**
	 * Builds a SELECT statement based on the clauses that have been passed to this instance by chaining method calls.
	 *
	 * @return string
	 */
	protected function build_select() {
		// If the query is raw, just set the $this->values to be the raw query parameters and return the raw query.
		if ( $this->is_raw_query ) {
			$this->values = $this->raw_parameters;

			return $this->raw_query;
		}

		// Build and return the full SELECT statement by concatenating the results of calling each separate builder method.
		return $this->join_if_not_empty(
			' ',
			[
				$this->build_select_start(),
				$this->build_join(),
				$this->build_where(),
				$this->build_group_by(),
				$this->build_having(),
				$this->build_order_by(),
				$this->build_limit(),
				$this->build_offset(),
			]
		);
	}

	/**
	 * Builds the start of the SELECT statement.
	 *
	 * @return string
	 */
	protected function build_select_start() {
		$fragment       = 'SELECT ';
		$result_columns = \implode( ', ', $this->result_columns );
		if ( $this->distinct ) {
			$result_columns = 'DISTINCT ' . $result_columns;
		}
		$fragment .= "{$result_columns} FROM " . $this->quote_identifier( $this->table_name );
		if ( ! \is_null( $this->table_alias ) ) {
			$fragment .= ' ' . $this->quote_identifier( $this->table_alias );
		}

		return $fragment;
	}

	/**
	 * Builds the JOIN sources.
	 *
	 * @return string
	 */
	protected function build_join() {
		if ( \count( $this->join_sources ) === 0 ) {
			return '';
		}

		return \implode( ' ', $this->join_sources );
	}

	/**
	 * Builds the WHERE clause(s).
	 *
	 * @return string
	 */
	protected function build_where() {
		return $this->build_conditions( 'where' );
	}

	/**
	 * Build the HAVING clause(s)
	 *
	 * @return string
	 */
	protected function build_having() {
		return $this->build_conditions( 'having' );
	}

	/**
	 * Builds GROUP BY.
	 *
	 * @return string
	 */
	protected function build_group_by() {
		if ( \count( $this->group_by ) === 0 ) {
			return '';
		}

		return 'GROUP BY ' . \implode( ', ', $this->group_by );
	}

	/**
	 * Builds a WHERE or HAVING clause.
	 *
	 * @param string $type Where or having.
	 *
	 * @return string
	 */
	protected function build_conditions( $type ) {
		$conditions_class_property_name = "{$type}_conditions";
		// If there are no clauses, return empty string.
		if ( \count( $this->{$conditions_class_property_name} ) === 0 ) {
			return '';
		}
		$conditions = [];
		foreach ( $this->{$conditions_class_property_name} as $condition ) {
			$conditions[] = $condition[ self::CONDITION_FRAGMENT ];
			$this->values = \array_merge( $this->values, $condition[ self::CONDITION_VALUES ] );
		}

		return \strtoupper( $type ) . ' ' . \implode( ' AND ', $conditions );
	}

	/**
	 * Builds ORDER BY.
	 *
	 * @return string
	 */
	protected function build_order_by() {
		if ( \count( $this->order_by ) === 0 ) {
			return '';
		}

		return 'ORDER BY ' . \implode( ', ', $this->order_by );
	}

	/**
	 * Builds LIMIT.
	 *
	 * @return string
	 */
	protected function build_limit() {
		if ( ! \is_null( $this->limit ) ) {
			return "LIMIT {$this->limit}";
		}

		return '';
	}

	/**
	 * Builds OFFSET.
	 *
	 * @return string
	 */
	protected function build_offset() {
		if ( ! \is_null( $this->offset ) ) {
			return 'OFFSET ' . $this->offset;
		}

		return '';
	}

	/**
	 * Joins strings if they are not empty.
	 *
	 * @param string   $glue   Glue.
	 * @param string[] $pieces Pieces to join.
	 *
	 * @return string
	 */
	protected function join_if_not_empty( $glue, $pieces ) {
		$filtered_pieces = [];
		foreach ( $pieces as $piece ) {
			if ( \is_string( $piece ) ) {
				$piece = \trim( $piece );
			}
			if ( ! empty( $piece ) ) {
				$filtered_pieces[] = $piece;
			}
		}

		return \implode( $glue, $filtered_pieces );
	}

	/**
	 * Quotes a string that is used as an identifier (table names, column names etc).
	 * This method can also deal with dot-separated identifiers eg table.column.
	 *
	 * @param string|string[] $identifier One or more identifiers.
	 *
	 * @return string
	 */
	protected function quote_one_identifier( $identifier ) {
		$parts = \explode( '.', $identifier );
		$parts = \array_map( [ $this, 'quote_identifier_part' ], $parts );

		return \implode( '.', $parts );
	}

	/**
	 * Quotes a string that is used as an identifier (table names, column names etc) or an array containing multiple
	 * identifiers. This method can also deal with dot-separated identifiers eg table.column.
	 *
	 * @param string|string[] $identifier One or more identifiers.
	 *
	 * @return string
	 */
	protected function quote_identifier( $identifier ) {
		if ( \is_array( $identifier ) ) {
			$result = \array_map( [ $this, 'quote_one_identifier' ], $identifier );

			return \implode( ', ', $result );
		}
		else {
			return $this->quote_one_identifier( $identifier );
		}
	}

	/**
	 * Quotes a single part of an identifier, using the identifier quote character specified in the config
	 * (or autodetected).
	 *
	 * @param string $part The part to quote.
	 *
	 * @return string
	 */
	protected function quote_identifier_part( $part ) {
		if ( $part === '*' ) {
			return $part;
		}
		$quote_character = '`';

		// Double up any identifier quotes to escape them.
		return $quote_character . \str_replace( $quote_character, $quote_character . $quote_character, $part ) . $quote_character;
	}

	/**
	 * Executes the SELECT query that has been built up by chaining methods on this class.
	 * Return an array of rows as associative arrays.
	 *
	 * @return array|false The result rows. False if the query failed.
	 */
	protected function run() {
		global $wpdb;

		$query   = $this->build_select();
		$success = self::execute( $query, $this->values );

		if ( $success === false ) {
			// If the query fails run the migrations and try again.
			// Action is intentionally undocumented and should not be used by third-parties.
			\do_action( '_yoast_run_migrations' );
			$success = self::execute( $query, $this->values );
		}

		$this->reset_idiorm_state();

		if ( $success === false ) {
			return false;
		}

		$rows = [];
		foreach ( $wpdb->last_result as $row ) {
			$rows[] = \get_object_vars( $row );
		}

		return $rows;
	}

	/**
	 * Resets the Idiorm instance state.
	 *
	 * @return void
	 */
	private function reset_idiorm_state() {
		$this->values                       = [];
		$this->result_columns               = [ '*' ];
		$this->using_default_result_columns = true;
	}

	/**
	 * Returns the raw data wrapped by this ORM instance as an associative array. Column names may optionally be
	 * supplied as arguments, if so, only those keys will be returned.
	 *
	 * @return array Associative array of the raw data.
	 */
	public function as_array() {
		if ( \func_num_args() === 0 ) {
			return $this->data;
		}
		$args = \func_get_args();

		return \array_intersect_key( $this->data, \array_flip( $args ) );
	}

	/**
	 * Returns the value of a property of this object (database row) or null if not present.
	 *
	 * If a column-names array is passed, it will return a associative array with the value of each column or null if
	 * it is not present.
	 *
	 * @param string|array $key Key.
	 *
	 * @return array|mixed|null
	 */
	public function get( $key ) {
		if ( \is_array( $key ) ) {
			$result = [];
			foreach ( $key as $column ) {
				$result[ $column ] = ( $this->data[ $column ] ?? null );
			}

			return $result;
		}
		else {
			return ( $this->data[ $key ] ?? null );
		}
	}

	/**
	 * Returns the name of the column in the database table which contains the primary key ID of the row.
	 *
	 * @return string The primary key ID of the row.
	 */
	protected function get_id_column_name() {
		if ( ! \is_null( $this->instance_id_column ) ) {
			return $this->instance_id_column;
		}

		return 'id';
	}

	/**
	 * Gets the primary key ID of this object.
	 *
	 * @param bool $disallow_null Whether to allow null IDs.
	 *
	 * @return array|mixed|null
	 *
	 * @throws Exception Primary key ID contains null value(s).
	 * @throws Exception Primary key ID missing from row or is null.
	 */
	public function id( $disallow_null = false ) {
		$id = $this->get( $this->get_id_column_name() );
		if ( $disallow_null ) {
			if ( \is_array( $id ) ) {
				foreach ( $id as $id_part ) {
					if ( $id_part === null ) {
						throw new Exception( 'Primary key ID contains null value(s)' );
					}
				}
			}
			elseif ( $id === null ) {
				throw new Exception( 'Primary key ID missing from row or is null' );
			}
		}

		return $id;
	}

	/**
	 * Sets a property to a particular value on this object.
	 *
	 * To set multiple properties at once, pass an associative array as the first parameter and leave out the second
	 * parameter. Flags the properties as 'dirty' so they will be saved to the database when save() is called.
	 *
	 * @param string|array $key   Key.
	 * @param string|null  $value Value.
	 *
	 * @return ORM
	 */
	public function set( $key, $value = null ) {
		return $this->set_orm_property( $key, $value );
	}

	/**
	 * Set a property to a particular value on this object as expression.
	 *
	 * To set multiple properties at once, pass an associative array as the first parameter and leave out the second
	 * parameter. Flags the properties as 'dirty' so they will be saved to the database when save() is called.
	 *
	 * @param string|array $key   Key.
	 * @param string|null  $value Value.
	 *
	 * @return ORM
	 */
	public function set_expr( $key, $value = null ) {
		return $this->set_orm_property( $key, $value, true );
	}

	/**
	 * Sets a property on the ORM object.
	 *
	 * @param string|array $key   Key.
	 * @param string|null  $value Value.
	 * @param bool         $expr  Expression.
	 *
	 * @return ORM
	 */
	protected function set_orm_property( $key, $value = null, $expr = false ) {
		if ( ! \is_array( $key ) ) {
			$key = [ $key => $value ];
		}
		foreach ( $key as $field => $value ) {
			$this->data[ $field ]         = $value;
			$this->dirty_fields[ $field ] = $value;
			if ( $expr === false && isset( $this->expr_fields[ $field ] ) ) {
				unset( $this->expr_fields[ $field ] );
			}
			elseif ( $expr === true ) {
				$this->expr_fields[ $field ] = true;
			}
		}

		return $this;
	}

	/**
	 * Checks whether the given field has been changed since this object was saved.
	 *
	 * @param mixed $key Key.
	 *
	 * @return bool
	 */
	public function is_dirty( $key ) {
		return \array_key_exists( $key, $this->dirty_fields );
	}

	/**
	 * Checks whether the model was the result of a call to create() or not.
	 *
	 * @return bool
	 */
	public function is_new() {
		return $this->is_new;
	}

	/**
	 * Saves any fields which have been modified on this object to the database.
	 *
	 * @return bool True on success.
	 *
	 * @throws Exception Primary key ID contains null value(s).
	 * @throws Exception Primary key ID missing from row or is null.
	 */
	public function save() {
		global $wpdb;

		// Remove any expression fields as they are already baked into the query.
		$values = \array_values( \array_diff_key( $this->dirty_fields, $this->expr_fields ) );
		if ( ! $this->is_new ) {
			// UPDATE.
			// If there are no dirty values, do nothing.
			if ( empty( $values ) && empty( $this->expr_fields ) ) {
				return true;
			}
			$query = \implode( ' ', [ $this->build_update(), $this->add_id_column_conditions() ] );

			$id = $this->id( true );
			if ( \is_array( $id ) ) {
				$values = \array_merge( $values, \array_values( $id ) );
			}
			else {
				$values[] = $id;
			}
		}
		else {
			// INSERT.
			$query = $this->build_insert();
		}
		$success = self::execute( $query, $values );
		// If we've just inserted a new record, set the ID of this object.
		if ( $this->is_new ) {
			$this->is_new = false;
			if ( $this->count_null_id_columns() !== 0 ) {
				$column = $this->get_id_column_name();
				// If the primary key is compound, assign the last inserted id to the first column.
				if ( \is_array( $column ) ) {
					$column = \reset( $column );
				}
				// Explicitly cast to int to make dealing with Id's simpler.
				$this->data[ $column ] = (int) $wpdb->insert_id;
			}
		}
		$this->dirty_fields = [];
		$this->expr_fields  = [];

		return $success;
	}

	/**
	 * Extracts and gathers all dirty column names from the given model instances.
	 *
	 * @param array $models Array of model instances to be inserted.
	 *
	 * @return array The distinct set of columns that are dirty in at least one of the models.
	 *
	 * @throws InvalidArgumentException Instance to be inserted is not a new one.
	 */
	public function get_dirty_column_names( $models ) {
		$dirty_column_names = [];

		foreach ( $models as $model ) {
			if ( ! $model->orm->is_new() ) {
				throw new InvalidArgumentException( 'Instance to be inserted is not a new one' );
			}

			// Remove any expression fields as they are already baked into the query.
			$dirty_fields       = \array_diff_key( $model->orm->dirty_fields, $model->orm->expr_fields );
			$dirty_column_names = \array_merge( $dirty_column_names, $dirty_fields );
		}

		$dirty_column_names = \array_keys( $dirty_column_names );

		return $dirty_column_names;
	}

	/**
	 * Inserts multiple rows in a single query. Expects new rows as it's a strictly insert function, not an update one.
	 *
	 * @example From the Indexable_Link_Builder class: $this->seo_links_repository->query()->insert_many( $links );
	 *
	 * @param array $models Array of model instances to be inserted.
	 *
	 * @return bool True for successful insert, false for failed.
	 *
	 * @throws InvalidArgumentException Invalid instances to be inserted.
	 * @throws InvalidArgumentException Instance to be inserted is not a new one.
	 */
	public function insert_many( $models ) {
		// Validate the input first.
		if ( ! \is_array( $models ) ) {
			throw new InvalidArgumentException( 'Invalid instances to be inserted' );
		}

		if ( empty( $models ) ) {
			return true;
		}

		$success = true;

		/**
		 * Filter: 'wpseo_chunk_bulked_insert_queries' - Allow filtering the chunk size of each bulked INSERT query.
		 *
		 * @param int $chunk_size The chunk size of the bulked INSERT queries.
		 */
		$chunk = \apply_filters( 'wpseo_chunk_bulk_insert_queries', 100 );
		$chunk = ! \is_int( $chunk ) ? 100 : $chunk;
		$chunk = ( $chunk <= 0 ) ? 100 : $chunk;

		$chunked_models = \array_chunk( $models, $chunk );
		foreach ( $chunked_models as $models_chunk ) {
			$values = [];

			// First, we'll gather all the dirty fields throughout the models to be inserted.
			$dirty_column_names = $this->get_dirty_column_names( $models_chunk );

			// Now, we're creating all dirty fields throughout the models and
			// setting them to null if they don't exist in each model.
			foreach ( $models_chunk as $model ) {
				$model_values = [];

				foreach ( $dirty_column_names as $dirty_column ) {
					// Set the value to null if it hasn't been set already.
					if ( ! isset( $model->orm->dirty_fields[ $dirty_column ] ) ) {
						$model->orm->dirty_fields[ $dirty_column ] = null;
					}

					// Only register the value if it is not null.
					if ( ! \is_null( $model->orm->dirty_fields[ $dirty_column ] ) ) {
						$model_values[] = $model->orm->dirty_fields[ $dirty_column ];
					}
				}
				$values = \array_merge( $values, $model_values );
			}

			// We now have the same set of dirty columns in all our models and also gathered all values.
			$query   = $this->build_insert_many( $models_chunk, $dirty_column_names );
			$success = $success && (bool) self::execute( $query, $values );
		}

		return $success;
	}

	/**
	 * Updates many records in the database.
	 *
	 * @return int|bool The number of rows changed if the query was succesful. False otherwise.
	 */
	public function update_many() {
		// Remove any expression fields as they are already baked into the query.
		$values = \array_values( \array_diff_key( $this->dirty_fields, $this->expr_fields ) );

		// UPDATE.
		// If there are no dirty values, do nothing.
		if ( empty( $values ) && empty( $this->expr_fields ) ) {
			return true;
		}

		$query = $this->join_if_not_empty( ' ', [ $this->build_update(), $this->build_where() ] );

		$success            = self::execute( $query, \array_merge( $values, $this->values ) );
		$this->dirty_fields = [];
		$this->expr_fields  = [];

		return $success;
	}

	/**
	 * Adds a WHERE clause for every column that belongs to the primary key.
	 *
	 * @return string The where part of the query.
	 */
	public function add_id_column_conditions() {
		$query   = [];
		$query[] = 'WHERE';
		$keys    = \is_array( $this->get_id_column_name() ) ? $this->get_id_column_name() : [ $this->get_id_column_name() ];
		$first   = true;
		foreach ( $keys as $key ) {
			if ( $first ) {
				$first = false;
			}
			else {
				$query[] = 'AND';
			}
			$query[] = $this->quote_identifier( $key );
			$query[] = '= %s';
		}

		return \implode( ' ', $query );
	}

	/**
	 * Builds an UPDATE query.
	 *
	 * @return string The update query.
	 */
	protected function build_update() {
		$query      = [];
		$query[]    = "UPDATE {$this->quote_identifier($this->table_name)} SET";
		$field_list = [];
		foreach ( $this->dirty_fields as $key => $value ) {
			if ( ! \array_key_exists( $key, $this->expr_fields ) ) {
				$value = ( $value === null ) ? 'NULL' : '%s';
			}
			$field_list[] = "{$this->quote_identifier($key)} = {$value}";
		}
		$query[] = \implode( ', ', $field_list );

		return \implode( ' ', $query );
	}

	/**
	 * Builds an INSERT query.
	 *
	 * @return string The insert query.
	 */
	protected function build_insert() {
		$query        = [];
		$query[]      = 'INSERT INTO';
		$query[]      = $this->quote_identifier( $this->table_name );
		$field_list   = \array_map( [ $this, 'quote_identifier' ], \array_keys( $this->dirty_fields ) );
		$query[]      = '(' . \implode( ', ', $field_list ) . ')';
		$query[]      = 'VALUES';
		$placeholders = $this->create_placeholders( $this->dirty_fields );
		$query[]      = "({$placeholders})";

		return \implode( ' ', $query );
	}

	/**
	 * Builds a bulk INSERT query.
	 *
	 * @param array $models             Array of model instances to be inserted.
	 * @param array $dirty_column_names Array of dirty fields to be used in INSERT.
	 *
	 * @return string The insert query.
	 */
	protected function build_insert_many( $models, $dirty_column_names ) {
		$example_model      = $models[0];
		$total_placeholders = '';

		$query      = [];
		$query[]    = 'INSERT INTO';
		$query[]    = $this->quote_identifier( $example_model->orm->table_name );
		$field_list = \array_map( [ $this, 'quote_identifier' ], $dirty_column_names );
		$query[]    = '(' . \implode( ', ', $field_list ) . ')';
		$query[]    = 'VALUES';

		// We assign placeholders per model for dirty fields that have values and NULL for dirty fields that don't.
		foreach ( $models as $model ) {
			$placeholder = [];
			foreach ( $dirty_column_names as $dirty_field ) {
				$placeholder[] = ( $model->orm->dirty_fields[ $dirty_field ] === null ) ? 'NULL' : '%s';
			}
			$placeholders        = \implode( ', ', $placeholder );
			$total_placeholders .= "({$placeholders}),";
		}

		$query[] = \rtrim( $total_placeholders, ',' );
		return \implode( ' ', $query );
	}

	/**
	 * Deletes this record from the database.
	 *
	 * @return string The delete query.
	 *
	 * @throws Exception Primary key ID contains null value(s).
	 * @throws Exception Primary key ID missing from row or is null.
	 */
	public function delete() {
		$query = [ 'DELETE FROM', $this->quote_identifier( $this->table_name ), $this->add_id_column_conditions() ];

		return self::execute( \implode( ' ', $query ), \is_array( $this->id( true ) ) ? \array_values( $this->id( true ) ) : [ $this->id( true ) ] );
	}

	/**
	 * Deletes many records from the database.
	 *
	 * @return bool|int Response of wpdb::query.
	 */
	public function delete_many() {
		// Build and return the full DELETE statement by concatenating
		// the results of calling each separate builder method.
		$query = $this->join_if_not_empty(
			' ',
			[
				'DELETE FROM',
				$this->quote_identifier( $this->table_name ),
				$this->build_where(),
			]
		);

		return self::execute( $query, $this->values );
	}

	/*
	 * ---  ArrayAccess  ---
	 */

	/**
	 * Checks whether the data has the key.
	 *
	 * @param mixed $offset Key.
	 *
	 * @return bool Whether the data has the key.
	 */
	#[ReturnTypeWillChange]
	public function offsetExists( $offset ) {
		return \array_key_exists( $offset, $this->data );
	}

	/**
	 * Retrieves the value of the key.
	 *
	 * @param mixed $offset Key.
	 *
	 * @return array|mixed|null The value.
	 */
	#[ReturnTypeWillChange]
	public function offsetGet( $offset ) {
		return $this->get( $offset );
	}

	/**
	 * Sets the value of the key.
	 *
	 * @param string|int $offset Key.
	 * @param mixed      $value  Value.
	 *
	 * @return void
	 */
	#[ReturnTypeWillChange]
	public function offsetSet( $offset, $value ) {
		if ( \is_null( $offset ) ) {
			return;
		}
		$this->set( $offset, $value );
	}

	/**
	 * Removes the given key from the data.
	 *
	 * @param mixed $offset Key.
	 *
	 * @return void
	 */
	#[ReturnTypeWillChange]
	public function offsetUnset( $offset ) {
		unset( $this->data[ $offset ] );
		unset( $this->dirty_fields[ $offset ] );
	}

	/*
	 * --- MAGIC METHODS ---
	 */

	/**
	 * Handles magic get via offset.
	 *
	 * @param mixed $key Key.
	 *
	 * @return array|mixed|null The value in the offset.
	 */
	public function __get( $key ) {
		return $this->offsetGet( $key );
	}

	/**
	 * Handles magic set via offset.
	 *
	 * @param string|int $key   Key.
	 * @param mixed      $value Value.
	 *
	 * @return void
	 */
	public function __set( $key, $value ) {
		$this->offsetSet( $key, $value );
	}

	/**
	 * Handles magic unset via offset.
	 *
	 * @param mixed $key Key.
	 *
	 * @return void
	 */
	public function __unset( $key ) {
		$this->offsetUnset( $key );
	}

	/**
	 * Handles magic isset via offset.
	 *
	 * @param mixed $key Key.
	 *
	 * @return bool Whether the offset has the key.
	 */
	public function __isset( $key ) {
		return $this->offsetExists( $key );
	}
}
                                                                                                                                                                                                                                     plugins/wordpress-seo-extended/license.txt                                                          0000644                 00000101047 15122266557 0015040 0                                                                                                    ustar 00                                                                                                                                                                                                                                                                           GNU GENERAL PUBLIC LICENSE
                       Version 3, 29 June 2007

 Copyright (C) 2007 Free Software Foundation, Inc. <http://fsf.org/>
 Everyone is permitted to copy and distribute verbatim copies
 of this license document, but changing it is not allowed.

                            Preamble

  The GNU General Public License is a free, copyleft license for
software and other kinds of works.

  The licenses for most software and other practical works are designed
to take away your freedom to share and change the works.  By contrast,
the GNU General Public License is intended to guarantee your freedom to
share and change all versions of a program--to make sure it remains free
software for all its users.  We, the Free Software Foundation, use the
GNU General Public License for most of our software; it applies also to
any other work released this way by its authors.  You can apply it to
your programs, too.

  When we speak of free software, we are referring to freedom, not
price.  Our General Public Licenses are designed to make sure that you
have the freedom to distribute copies of free software (and charge for
them if you wish), that you receive source code or can get it if you
want it, that you can change the software or use pieces of it in new
free programs, and that you know you can do these things.

  To protect your rights, we need to prevent others from denying you
these rights or asking you to surrender the rights.  Therefore, you have
certain responsibilities if you distribute copies of the software, or if
you modify it: responsibilities to respect the freedom of others.

  For example, if you distribute copies of such a program, whether
gratis or for a fee, you must pass on to the recipients the same
freedoms that you received.  You must make sure that they, too, receive
or can get the source code.  And you must show them these terms so they
know their rights.

  Developers that use the GNU GPL protect your rights with two steps:
(1) assert copyright on the software, and (2) offer you this License
giving you legal permission to copy, distribute and/or modify it.

  For the developers' and authors' protection, the GPL clearly explains
that there is no warranty for this free software.  For both users' and
authors' sake, the GPL requires that modified versions be marked as
changed, so that their problems will not be attributed erroneously to
authors of previous versions.

  Some devices are designed to deny users access to install or run
modified versions of the software inside them, although the manufacturer
can do so.  This is fundamentally incompatible with the aim of
protecting users' freedom to change the software.  The systematic
pattern of such abuse occurs in the area of products for individuals to
use, which is precisely where it is most unacceptable.  Therefore, we
have designed this version of the GPL to prohibit the practice for those
products.  If such problems arise substantially in other domains, we
stand ready to extend this provision to those domains in future versions
of the GPL, as needed to protect the freedom of users.

  Finally, every program is threatened constantly by software patents.
States should not allow patents to restrict development and use of
software on general-purpose computers, but in those that do, we wish to
avoid the special danger that patents applied to a free program could
make it effectively proprietary.  To prevent this, the GPL assures that
patents cannot be used to render the program non-free.

  The precise terms and conditions for copying, distribution and
modification follow.

                       TERMS AND CONDITIONS

  0. Definitions.

  "This License" refers to version 3 of the GNU General Public License.

  "Copyright" also means copyright-like laws that apply to other kinds of
works, such as semiconductor masks.

  "The Program" refers to any copyrightable work licensed under this
License.  Each licensee is addressed as "you".  "Licensees" and
"recipients" may be individuals or organizations.

  To "modify" a work means to copy from or adapt all or part of the work
in a fashion requiring copyright permission, other than the making of an
exact copy.  The resulting work is called a "modified version" of the
earlier work or a work "based on" the earlier work.

  A "covered work" means either the unmodified Program or a work based
on the Program.

  To "propagate" a work means to do anything with it that, without
permission, would make you directly or secondarily liable for
infringement under applicable copyright law, except executing it on a
computer or modifying a private copy.  Propagation includes copying,
distribution (with or without modification), making available to the
public, and in some countries other activities as well.

  To "convey" a work means any kind of propagation that enables other
parties to make or receive copies.  Mere interaction with a user through
a computer network, with no transfer of a copy, is not conveying.

  An interactive user interface displays "Appropriate Legal Notices"
to the extent that it includes a convenient and prominently visible
feature that (1) displays an appropriate copyright notice, and (2)
tells the user that there is no warranty for the work (except to the
extent that warranties are provided), that licensees may convey the
work under this License, and how to view a copy of this License.  If
the interface presents a list of user commands or options, such as a
menu, a prominent item in the list meets this criterion.

  1. Source Code.

  The "source code" for a work means the preferred form of the work
for making modifications to it.  "Object code" means any non-source
form of a work.

  A "Standard Interface" means an interface that either is an official
standard defined by a recognized standards body, or, in the case of
interfaces specified for a particular programming language, one that
is widely used among developers working in that language.

  The "System Libraries" of an executable work include anything, other
than the work as a whole, that (a) is included in the normal form of
packaging a Major Component, but which is not part of that Major
Component, and (b) serves only to enable use of the work with that
Major Component, or to implement a Standard Interface for which an
implementation is available to the public in source code form.  A
"Major Component", in this context, means a major essential component
(kernel, window system, and so on) of the specific operating system
(if any) on which the executable work runs, or a compiler used to
produce the work, or an object code interpreter used to run it.

  The "Corresponding Source" for a work in object code form means all
the source code needed to generate, install, and (for an executable
work) run the object code and to modify the work, including scripts to
control those activities.  However, it does not include the work's
System Libraries, or general-purpose tools or generally available free
programs which are used unmodified in performing those activities but
which are not part of the work.  For example, Corresponding Source
includes interface definition files associated with source files for
the work, and the source code for shared libraries and dynamically
linked subprograms that the work is specifically designed to require,
such as by intimate data communication or control flow between those
subprograms and other parts of the work.

  The Corresponding Source need not include anything that users
can regenerate automatically from other parts of the Corresponding
Source.

  The Corresponding Source for a work in source code form is that
same work.

  2. Basic Permissions.

  All rights granted under this License are granted for the term of
copyright on the Program, and are irrevocable provided the stated
conditions are met.  This License explicitly affirms your unlimited
permission to run the unmodified Program.  The output from running a
covered work is covered by this License only if the output, given its
content, constitutes a covered work.  This License acknowledges your
rights of fair use or other equivalent, as provided by copyright law.

  You may make, run and propagate covered works that you do not
convey, without conditions so long as your license otherwise remains
in force.  You may convey covered works to others for the sole purpose
of having them make modifications exclusively for you, or provide you
with facilities for running those works, provided that you comply with
the terms of this License in conveying all material for which you do
not control copyright.  Those thus making or running the covered works
for you must do so exclusively on your behalf, under your direction
and control, on terms that prohibit them from making any copies of
your copyrighted material outside their relationship with you.

  Conveying under any other circumstances is permitted solely under
the conditions stated below.  Sublicensing is not allowed; section 10
makes it unnecessary.

  3. Protecting Users' Legal Rights From Anti-Circumvention Law.

  No covered work shall be deemed part of an effective technological
measure under any applicable law fulfilling obligations under article
11 of the WIPO copyright treaty adopted on 20 December 1996, or
similar laws prohibiting or restricting circumvention of such
measures.

  When you convey a covered work, you waive any legal power to forbid
circumvention of technological measures to the extent such circumvention
is effected by exercising rights under this License with respect to
the covered work, and you disclaim any intention to limit operation or
modification of the work as a means of enforcing, against the work's
users, your or third parties' legal rights to forbid circumvention of
technological measures.

  4. Conveying Verbatim Copies.

  You may convey verbatim copies of the Program's source code as you
receive it, in any medium, provided that you conspicuously and
appropriately publish on each copy an appropriate copyright notice;
keep intact all notices stating that this License and any
non-permissive terms added in accord with section 7 apply to the code;
keep intact all notices of the absence of any warranty; and give all
recipients a copy of this License along with the Program.

  You may charge any price or no price for each copy that you convey,
and you may offer support or warranty protection for a fee.

  5. Conveying Modified Source Versions.

  You may convey a work based on the Program, or the modifications to
produce it from the Program, in the form of source code under the
terms of section 4, provided that you also meet all of these conditions:

    a) The work must carry prominent notices stating that you modified
    it, and giving a relevant date.

    b) The work must carry prominent notices stating that it is
    released under this License and any conditions added under section
    7.  This requirement modifies the requirement in section 4 to
    "keep intact all notices".

    c) You must license the entire work, as a whole, under this
    License to anyone who comes into possession of a copy.  This
    License will therefore apply, along with any applicable section 7
    additional terms, to the whole of the work, and all its parts,
    regardless of how they are packaged.  This License gives no
    permission to license the work in any other way, but it does not
    invalidate such permission if you have separately received it.

    d) If the work has interactive user interfaces, each must display
    Appropriate Legal Notices; however, if the Program has interactive
    interfaces that do not display Appropriate Legal Notices, your
    work need not make them do so.

  A compilation of a covered work with other separate and independent
works, which are not by their nature extensions of the covered work,
and which are not combined with it such as to form a larger program,
in or on a volume of a storage or distribution medium, is called an
"aggregate" if the compilation and its resulting copyright are not
used to limit the access or legal rights of the compilation's users
beyond what the individual works permit.  Inclusion of a covered work
in an aggregate does not cause this License to apply to the other
parts of the aggregate.

  6. Conveying Non-Source Forms.

  You may convey a covered work in object code form under the terms
of sections 4 and 5, provided that you also convey the
machine-readable Corresponding Source under the terms of this License,
in one of these ways:

    a) Convey the object code in, or embodied in, a physical product
    (including a physical distribution medium), accompanied by the
    Corresponding Source fixed on a durable physical medium
    customarily used for software interchange.

    b) Convey the object code in, or embodied in, a physical product
    (including a physical distribution medium), accompanied by a
    written offer, valid for at least three years and valid for as
    long as you offer spare parts or customer support for that product
    model, to give anyone who possesses the object code either (1) a
    copy of the Corresponding Source for all the software in the
    product that is covered by this License, on a durable physical
    medium customarily used for software interchange, for a price no
    more than your reasonable cost of physically performing this
    conveying of source, or (2) access to copy the
    Corresponding Source from a network server at no charge.

    c) Convey individual copies of the object code with a copy of the
    written offer to provide the Corresponding Source.  This
    alternative is allowed only occasionally and noncommercially, and
    only if you received the object code with such an offer, in accord
    with subsection 6b.

    d) Convey the object code by offering access from a designated
    place (gratis or for a charge), and offer equivalent access to the
    Corresponding Source in the same way through the same place at no
    further charge.  You need not require recipients to copy the
    Corresponding Source along with the object code.  If the place to
    copy the object code is a network server, the Corresponding Source
    may be on a different server (operated by you or a third party)
    that supports equivalent copying facilities, provided you maintain
    clear directions next to the object code saying where to find the
    Corresponding Source.  Regardless of what server hosts the
    Corresponding Source, you remain obligated to ensure that it is
    available for as long as needed to satisfy these requirements.

    e) Convey the object code using peer-to-peer transmission, provided
    you inform other peers where the object code and Corresponding
    Source of the work are being offered to the general public at no
    charge under subsection 6d.

  A separable portion of the object code, whose source code is excluded
from the Corresponding Source as a System Library, need not be
included in conveying the object code work.

  A "User Product" is either (1) a "consumer product", which means any
tangible personal property which is normally used for personal, family,
or household purposes, or (2) anything designed or sold for incorporation
into a dwelling.  In determining whether a product is a consumer product,
doubtful cases shall be resolved in favor of coverage.  For a particular
product received by a particular user, "normally used" refers to a
typical or common use of that class of product, regardless of the status
of the particular user or of the way in which the particular user
actually uses, or expects or is expected to use, the product.  A product
is a consumer product regardless of whether the product has substantial
commercial, industrial or non-consumer uses, unless such uses represent
the only significant mode of use of the product.

  "Installation Information" for a User Product means any methods,
procedures, authorization keys, or other information required to install
and execute modified versions of a covered work in that User Product from
a modified version of its Corresponding Source.  The information must
suffice to ensure that the continued functioning of the modified object
code is in no case prevented or interfered with solely because
modification has been made.

  If you convey an object code work under this section in, or with, or
specifically for use in, a User Product, and the conveying occurs as
part of a transaction in which the right of possession and use of the
User Product is transferred to the recipient in perpetuity or for a
fixed term (regardless of how the transaction is characterized), the
Corresponding Source conveyed under this section must be accompanied
by the Installation Information.  But this requirement does not apply
if neither you nor any third party retains the ability to install
modified object code on the User Product (for example, the work has
been installed in ROM).

  The requirement to provide Installation Information does not include a
requirement to continue to provide support service, warranty, or updates
for a work that has been modified or installed by the recipient, or for
the User Product in which it has been modified or installed.  Access to a
network may be denied when the modification itself materially and
adversely affects the operation of the network or violates the rules and
protocols for communication across the network.

  Corresponding Source conveyed, and Installation Information provided,
in accord with this section must be in a format that is publicly
documented (and with an implementation available to the public in
source code form), and must require no special password or key for
unpacking, reading or copying.

  7. Additional Terms.

  "Additional permissions" are terms that supplement the terms of this
License by making exceptions from one or more of its conditions.
Additional permissions that are applicable to the entire Program shall
be treated as though they were included in this License, to the extent
that they are valid under applicable law.  If additional permissions
apply only to part of the Program, that part may be used separately
under those permissions, but the entire Program remains governed by
this License without regard to the additional permissions.

  When you convey a copy of a covered work, you may at your option
remove any additional permissions from that copy, or from any part of
it.  (Additional permissions may be written to require their own
removal in certain cases when you modify the work.)  You may place
additional permissions on material, added by you to a covered work,
for which you have or can give appropriate copyright permission.

  Notwithstanding any other provision of this License, for material you
add to a covered work, you may (if authorized by the copyright holders of
that material) supplement the terms of this License with terms:

    a) Disclaiming warranty or limiting liability differently from the
    terms of sections 15 and 16 of this License; or

    b) Requiring preservation of specified reasonable legal notices or
    author attributions in that material or in the Appropriate Legal
    Notices displayed by works containing it; or

    c) Prohibiting misrepresentation of the origin of that material, or
    requiring that modified versions of such material be marked in
    reasonable ways as different from the original version; or

    d) Limiting the use for publicity purposes of names of licensors or
    authors of the material; or

    e) Declining to grant rights under trademark law for use of some
    trade names, trademarks, or service marks; or

    f) Requiring indemnification of licensors and authors of that
    material by anyone who conveys the material (or modified versions of
    it) with contractual assumptions of liability to the recipient, for
    any liability that these contractual assumptions directly impose on
    those licensors and authors.

  All other non-permissive additional terms are considered "further
restrictions" within the meaning of section 10.  If the Program as you
received it, or any part of it, contains a notice stating that it is
governed by this License along with a term that is a further
restriction, you may remove that term.  If a license document contains
a further restriction but permits relicensing or conveying under this
License, you may add to a covered work material governed by the terms
of that license document, provided that the further restriction does
not survive such relicensing or conveying.

  If you add terms to a covered work in accord with this section, you
must place, in the relevant source files, a statement of the
additional terms that apply to those files, or a notice indicating
where to find the applicable terms.

  Additional terms, permissive or non-permissive, may be stated in the
form of a separately written license, or stated as exceptions;
the above requirements apply either way.

  8. Termination.

  You may not propagate or modify a covered work except as expressly
provided under this License.  Any attempt otherwise to propagate or
modify it is void, and will automatically terminate your rights under
this License (including any patent licenses granted under the third
paragraph of section 11).

  However, if you cease all violation of this License, then your
license from a particular copyright holder is reinstated (a)
provisionally, unless and until the copyright holder explicitly and
finally terminates your license, and (b) permanently, if the copyright
holder fails to notify you of the violation by some reasonable means
prior to 60 days after the cessation.

  Moreover, your license from a particular copyright holder is
reinstated permanently if the copyright holder notifies you of the
violation by some reasonable means, this is the first time you have
received notice of violation of this License (for any work) from that
copyright holder, and you cure the violation prior to 30 days after
your receipt of the notice.

  Termination of your rights under this section does not terminate the
licenses of parties who have received copies or rights from you under
this License.  If your rights have been terminated and not permanently
reinstated, you do not qualify to receive new licenses for the same
material under section 10.

  9. Acceptance Not Required for Having Copies.

  You are not required to accept this License in order to receive or
run a copy of the Program.  Ancillary propagation of a covered work
occurring solely as a consequence of using peer-to-peer transmission
to receive a copy likewise does not require acceptance.  However,
nothing other than this License grants you permission to propagate or
modify any covered work.  These actions infringe copyright if you do
not accept this License.  Therefore, by modifying or propagating a
covered work, you indicate your acceptance of this License to do so.

  10. Automatic Licensing of Downstream Recipients.

  Each time you convey a covered work, the recipient automatically
receives a license from the original licensors, to run, modify and
propagate that work, subject to this License.  You are not responsible
for enforcing compliance by third parties with this License.

  An "entity transaction" is a transaction transferring control of an
organization, or substantially all assets of one, or subdividing an
organization, or merging organizations.  If propagation of a covered
work results from an entity transaction, each party to that
transaction who receives a copy of the work also receives whatever
licenses to the work the party's predecessor in interest had or could
give under the previous paragraph, plus a right to possession of the
Corresponding Source of the work from the predecessor in interest, if
the predecessor has it or can get it with reasonable efforts.

  You may not impose any further restrictions on the exercise of the
rights granted or affirmed under this License.  For example, you may
not impose a license fee, royalty, or other charge for exercise of
rights granted under this License, and you may not initiate litigation
(including a cross-claim or counterclaim in a lawsuit) alleging that
any patent claim is infringed by making, using, selling, offering for
sale, or importing the Program or any portion of it.

  11. Patents.

  A "contributor" is a copyright holder who authorizes use under this
License of the Program or a work on which the Program is based.  The
work thus licensed is called the contributor's "contributor version".

  A contributor's "essential patent claims" are all patent claims
owned or controlled by the contributor, whether already acquired or
hereafter acquired, that would be infringed by some manner, permitted
by this License, of making, using, or selling its contributor version,
but do not include claims that would be infringed only as a
consequence of further modification of the contributor version.  For
purposes of this definition, "control" includes the right to grant
patent sublicenses in a manner consistent with the requirements of
this License.

  Each contributor grants you a non-exclusive, worldwide, royalty-free
patent license under the contributor's essential patent claims, to
make, use, sell, offer for sale, import and otherwise run, modify and
propagate the contents of its contributor version.

  In the following three paragraphs, a "patent license" is any express
agreement or commitment, however denominated, not to enforce a patent
(such as an express permission to practice a patent or covenant not to
sue for patent infringement).  To "grant" such a patent license to a
party means to make such an agreement or commitment not to enforce a
patent against the party.

  If you convey a covered work, knowingly relying on a patent license,
and the Corresponding Source of the work is not available for anyone
to copy, free of charge and under the terms of this License, through a
publicly available network server or other readily accessible means,
then you must either (1) cause the Corresponding Source to be so
available, or (2) arrange to deprive yourself of the benefit of the
patent license for this particular work, or (3) arrange, in a manner
consistent with the requirements of this License, to extend the patent
license to downstream recipients.  "Knowingly relying" means you have
actual knowledge that, but for the patent license, your conveying the
covered work in a country, or your recipient's use of the covered work
in a country, would infringe one or more identifiable patents in that
country that you have reason to believe are valid.

  If, pursuant to or in connection with a single transaction or
arrangement, you convey, or propagate by procuring conveyance of, a
covered work, and grant a patent license to some of the parties
receiving the covered work authorizing them to use, propagate, modify
or convey a specific copy of the covered work, then the patent license
you grant is automatically extended to all recipients of the covered
work and works based on it.

  A patent license is "discriminatory" if it does not include within
the scope of its coverage, prohibits the exercise of, or is
conditioned on the non-exercise of one or more of the rights that are
specifically granted under this License.  You may not convey a covered
work if you are a party to an arrangement with a third party that is
in the business of distributing software, under which you make payment
to the third party based on the extent of your activity of conveying
the work, and under which the third party grants, to any of the
parties who would receive the covered work from you, a discriminatory
patent license (a) in connection with copies of the covered work
conveyed by you (or copies made from those copies), or (b) primarily
for and in connection with specific products or compilations that
contain the covered work, unless you entered into that arrangement,
or that patent license was granted, prior to 28 March 2007.

  Nothing in this License shall be construed as excluding or limiting
any implied license or other defenses to infringement that may
otherwise be available to you under applicable patent law.

  12. No Surrender of Others' Freedom.

  If conditions are imposed on you (whether by court order, agreement or
otherwise) that contradict the conditions of this License, they do not
excuse you from the conditions of this License.  If you cannot convey a
covered work so as to satisfy simultaneously your obligations under this
License and any other pertinent obligations, then as a consequence you may
not convey it at all.  For example, if you agree to terms that obligate you
to collect a royalty for further conveying from those to whom you convey
the Program, the only way you could satisfy both those terms and this
License would be to refrain entirely from conveying the Program.

  13. Use with the GNU Affero General Public License.

  Notwithstanding any other provision of this License, you have
permission to link or combine any covered work with a work licensed
under version 3 of the GNU Affero General Public License into a single
combined work, and to convey the resulting work.  The terms of this
License will continue to apply to the part which is the covered work,
but the special requirements of the GNU Affero General Public License,
section 13, concerning interaction through a network will apply to the
combination as such.

  14. Revised Versions of this License.

  The Free Software Foundation may publish revised and/or new versions of
the GNU General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

  Each version is given a distinguishing version number.  If the
Program specifies that a certain numbered version of the GNU General
Public License "or any later version" applies to it, you have the
option of following the terms and conditions either of that numbered
version or of any later version published by the Free Software
Foundation.  If the Program does not specify a version number of the
GNU General Public License, you may choose any version ever published
by the Free Software Foundation.

  If the Program specifies that a proxy can decide which future
versions of the GNU General Public License can be used, that proxy's
public statement of acceptance of a version permanently authorizes you
to choose that version for the Program.

  Later license versions may give you additional or different
permissions.  However, no additional obligations are imposed on any
author or copyright holder as a result of your choosing to follow a
later version.

  15. Disclaimer of Warranty.

  THERE IS NO WARRANTY FOR THE PROGRAM, TO THE EXTENT PERMITTED BY
APPLICABLE LAW.  EXCEPT WHEN OTHERWISE STATED IN WRITING THE COPYRIGHT
HOLDERS AND/OR OTHER PARTIES PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY
OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO,
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
ALL NECESSARY SERVICING, REPAIR OR CORRECTION.

  16. Limitation of Liability.

  IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES AND/OR CONVEYS
THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES, INCLUDING ANY
GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE
USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED TO LOSS OF
DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD
PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER PROGRAMS),
EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF
SUCH DAMAGES.

  17. Interpretation of Sections 15 and 16.

  If the disclaimer of warranty and limitation of liability provided
above cannot be given local legal effect according to their terms,
reviewing courts shall apply local law that most closely approximates
an absolute waiver of all civil liability in connection with the
Program, unless a warranty or assumption of liability accompanies a
copy of the Program in return for a fee.

  18. Additional terms

  In the light of Article 7 of the GPL license, the following additional 
terms apply:

  a) You are prohibited to make misrepresentations of the origin of that 
material, or to require that modified versions of such material be marked 
in reasonable ways as different from the original version;

  b) You are limited in the use for publicity purposes of names of 
licensors or authors of the material;

  c) You are declined any grant of rights under trademark law for use of 
the trade names, trademarks, or service marks of YOAST B.V.;

  d) You are required to indemnify licensors and authors of that material 
by anyone who conveys the material (or modified versions of it) with 
contractual assumptions of liability to the recipient, for any liability 
that these contractual assumptions directly impose on those licensors and 
authors.

END OF TERMS AND CONDITIONS
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         plugins/wordpress-seo-extended/packages/js/images/algolia-logo.svg                                  0000644                 00000012424 15122266557 0021403 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="100 100 525 160" width="150" height="40">
	<g fill="none" fill-rule="evenodd">
		<path fill="#5468FF"
			  d="M135.8 120.999h88.4c8.7 0 15.8 7.065 15.8 15.8v88.405c0 8.7-7.065 15.795-15.8 15.795h-88.4c-8.7 0-15.8-7.06-15.8-15.795v-88.445c0-8.695 7.06-15.76 15.8-15.76"/>
		<path fill="#FFF"
			  d="M192.505 147.788v-4.115a5.209 5.209 0 0 0-5.21-5.205H175.15a5.209 5.209 0 0 0-5.21 5.205v4.225c0 .47.435.8.91.69a37.966 37.966 0 0 1 10.57-1.49c3.465 0 6.895.47 10.21 1.38.44.11.875-.215.875-.69m-33.285 5.385l-2.075-2.075a5.206 5.206 0 0 0-7.365 0l-2.48 2.475a5.185 5.185 0 0 0 0 7.355l2.04 2.04c.33.325.805.25 1.095-.075a39.876 39.876 0 0 1 3.975-4.66 37.68 37.68 0 0 1 4.7-4c.364-.22.4-.73.11-1.06m22.164 13.065v17.8c0 .51.55.875 1.02.62l15.825-8.19c.36-.18.47-.62.29-.98-3.28-5.755-9.37-9.685-16.405-9.94-.365 0-.73.29-.73.69m0 42.88c-13.195 0-23.915-10.705-23.915-23.88 0-13.175 10.72-23.875 23.915-23.875 13.2 0 23.916 10.7 23.916 23.875s-10.68 23.88-23.916 23.88m0-57.8c-18.74 0-33.94 15.18-33.94 33.92 0 18.745 15.2 33.89 33.94 33.89s33.94-15.18 33.94-33.925c0-18.745-15.165-33.885-33.94-33.885"/>
		<path fill="#5468FF"
			  d="M359.214 216.177c-23.365.11-23.365-18.855-23.365-21.875l-.04-71.045 14.254-2.26v70.61c0 1.715 0 12.56 9.15 12.595v11.975zm-57.78-11.61c4.374 0 7.62-.255 9.88-.69v-14.485a29.196 29.196 0 0 0-3.43-.695 33.742 33.742 0 0 0-4.956-.365c-1.57 0-3.175.11-4.775.365-1.605.22-3.065.655-4.34 1.275-1.275.62-2.335 1.495-3.1 2.62-.8 1.13-1.165 1.785-1.165 3.495 0 3.345 1.165 5.28 3.28 6.55 2.115 1.275 4.995 1.93 8.606 1.93zm-1.24-51.685c4.7 0 8.674.585 11.884 1.75 3.206 1.165 5.796 2.8 7.69 4.875 1.935 2.11 3.245 4.915 4.046 7.9.84 2.985 1.24 6.26 1.24 9.86v36.62c-2.185.47-5.506 1.015-9.95 1.67-4.446.655-9.44.985-14.986.985-3.68 0-7.07-.365-10.095-1.055-3.065-.69-5.65-1.82-7.84-3.385-2.15-1.565-3.825-3.57-5.065-6.04-1.205-2.48-1.825-5.97-1.825-9.61 0-3.495.69-5.715 2.045-8.12 1.38-2.4 3.24-4.365 5.575-5.895 2.37-1.53 5.065-2.62 8.165-3.275 3.1-.655 6.345-.985 9.695-.985 1.57 0 3.21.11 4.96.29 1.715.185 3.575.515 5.545.985v-2.33c0-1.635-.185-3.2-.585-4.655a10.012 10.012 0 0 0-2.045-3.895c-.985-1.13-2.255-2.005-3.86-2.62-1.605-.62-3.65-1.095-6.09-1.095-3.28 0-6.27.4-9.005.875-2.735.47-4.995 1.02-6.71 1.635l-1.71-11.68c1.785-.62 4.445-1.24 7.875-1.855 3.425-.66 7.11-.95 11.045-.95zm281.51 51.285c4.375 0 7.615-.255 9.875-.695v-14.48c-.8-.22-1.93-.475-3.425-.695a33.813 33.813 0 0 0-4.96-.365c-1.565 0-3.17.11-4.775.365-1.6.22-3.06.655-4.335 1.275-1.28.62-2.335 1.495-3.1 2.62-.805 1.13-1.165 1.785-1.165 3.495 0 3.345 1.165 5.28 3.28 6.55 2.15 1.31 4.995 1.93 8.605 1.93zm-1.205-51.645c4.7 0 8.674.58 11.884 1.745 3.205 1.165 5.795 2.8 7.69 4.875 1.895 2.075 3.245 4.915 4.045 7.9.84 2.985 1.24 6.26 1.24 9.865v36.615c-2.185.47-5.505 1.015-9.95 1.675-4.445.655-9.44.98-14.985.98-3.68 0-7.07-.365-10.094-1.055-3.065-.69-5.65-1.82-7.84-3.385-2.15-1.565-3.825-3.57-5.065-6.04-1.205-2.475-1.825-5.97-1.825-9.61 0-3.495.695-5.715 2.045-8.12 1.38-2.4 3.24-4.365 5.575-5.895 2.37-1.525 5.065-2.62 8.165-3.275 3.1-.655 6.345-.98 9.7-.98 1.565 0 3.205.11 4.955.29s3.575.51 5.54.985v-2.33c0-1.64-.18-3.205-.58-4.66a9.977 9.977 0 0 0-2.045-3.895c-.985-1.13-2.255-2.005-3.86-2.62-1.606-.62-3.65-1.09-6.09-1.09-3.28 0-6.27.4-9.005.87-2.735.475-4.995 1.02-6.71 1.64l-1.71-11.685c1.785-.62 4.445-1.235 7.875-1.855 3.425-.62 7.105-.945 11.045-.945zm-42.8-6.77c4.774 0 8.68-3.86 8.68-8.63 0-4.765-3.866-8.625-8.68-8.625-4.81 0-8.675 3.86-8.675 8.625 0 4.77 3.9 8.63 8.675 8.63zm7.18 70.425h-14.326v-61.44l14.325-2.255v63.695zm-25.116 0c-23.365.11-23.365-18.855-23.365-21.875l-.04-71.045 14.255-2.26v70.61c0 1.715 0 12.56 9.15 12.595v11.975zm-46.335-31.445c0-6.155-1.35-11.285-3.974-14.85-2.625-3.605-6.305-5.385-11.01-5.385-4.7 0-8.386 1.78-11.006 5.385-2.625 3.6-3.904 8.695-3.904 14.85 0 6.225 1.315 10.405 3.94 14.01 2.625 3.64 6.305 5.425 11.01 5.425 4.7 0 8.385-1.82 11.01-5.425 2.624-3.64 3.934-7.785 3.934-14.01zm14.58-.035c0 4.805-.69 8.44-2.114 12.41-1.42 3.965-3.425 7.35-6.01 10.155-2.59 2.8-5.69 4.985-9.336 6.515-3.644 1.525-9.26 2.4-12.065 2.4-2.81-.035-8.385-.835-11.995-2.4-3.61-1.565-6.71-3.715-9.295-6.515-2.59-2.805-4.594-6.19-6.054-10.155-1.456-3.97-2.185-7.605-2.185-12.41s.654-9.43 2.114-13.36c1.46-3.93 3.5-7.28 6.125-10.08 2.625-2.805 5.76-4.955 9.33-6.48 3.61-1.53 7.585-2.255 11.885-2.255 4.305 0 8.275.76 11.92 2.255 3.65 1.525 6.786 3.675 9.336 6.48 2.584 2.8 4.59 6.15 6.05 10.08 1.53 3.93 2.295 8.555 2.295 13.36zm-107.284 0c0 5.965 1.31 12.59 3.935 15.355 2.625 2.77 6.014 4.15 10.175 4.15 2.26 0 4.41-.325 6.414-.945 2.005-.62 3.606-1.35 4.886-2.22v-35.34c-1.02-.22-5.286-1.095-9.41-1.2-5.175-.15-9.11 1.965-11.88 5.345-2.736 3.39-4.12 9.32-4.12 14.855zm39.625 28.095c0 9.72-2.48 16.815-7.476 21.33-4.99 4.51-12.61 6.77-22.89 6.77-3.755 0-11.555-.73-17.79-2.11l2.295-11.285c5.215 1.09 12.105 1.385 15.715 1.385 5.72 0 9.805-1.165 12.245-3.495 2.445-2.33 3.645-5.785 3.645-10.375v-2.33c-1.42.69-3.28 1.385-5.575 2.115-2.295.69-4.955 1.055-7.95 1.055-3.935 0-7.51-.62-10.75-1.86-3.245-1.235-6.055-3.055-8.35-5.46-2.295-2.4-4.12-5.42-5.395-9.025-1.275-3.605-1.935-10.045-1.935-14.775 0-4.44.695-10.01 2.046-13.725 1.384-3.71 3.35-6.915 6.014-9.57 2.626-2.655 5.835-4.695 9.59-6.19 3.755-1.49 8.16-2.435 12.935-2.435 4.635 0 8.9.58 13.055 1.275 4.155.69 7.69 1.415 10.57 2.215v56.49z"/>
	</g>
</svg>                                                                                                                                                                                                                                            plugins/wordpress-seo-extended/packages/js/images/connection-assistant.svg                          0000644                 00000042751 15122266557 0023211 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8"?>
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 752 247" stroke="#000" stroke-linecap="round"
	 stroke-linejoin="round" fill="#fff" fill-rule="evenodd" role="img" aria-hidden="true" focusable="false">
	<g stroke="none" class="B">
		<path
			d="M307.6714 13.3429c6.6273-6.5822 14.6953-12.7914 24.351-13.4694 5.0284-.8813 10.2037-.1243 15.0117 1.4464 3.5877 1.1469 6.294 3.8928 8.763 6.6047 3.3334-1.5819 7.0906-1.8362 10.7291-1.8023 3.3504-.1299 6.7968.8418 9.5201 2.8024 4.0453 3.9323 7.3223 8.8477 8.6782 14.3676.7119 3.808 1.0848 7.8873-.2203 11.6162.8249 2.3334 4.6103 2.7458 3.9606 5.6273-.7063 2.3729-2.6159 4.2091-4.3561 5.8984l-.6102.5424c-2.5312.6441-4.6047 2.2035-5.8872 4.4748-.322-1.921-2.6442-1.8645-3.9549-2.8419-5.9776-3.8137-10.7348-9.5201-13.3564-16.1192-1.2655-3.4351-2.7684-6.938-2.4746-10.6839-2.9831.661-6.0002 1.8588-8.068 4.1979-2.9775 2.8871-4.4521 6.8307-6.8194 10.1641-1.7459 2.3334-1.825 5.4522-3.5595 7.7856-1.1186 1.7006-3 2.5989-4.5877 3.7741 1.3786.3051 2.8984.3729 4.0736 1.243 1.113.9435.6723 2.8983-.6328 3.4238-2.2543.2881-4.5764-.2599-6.8025.3051-2.8249.8983-4.9945 3.0227-7.0115 5.0905-.9096 1.034-2.6102-.2203-2.4408-1.4181-.192-1.4068.9549-2.4125 1.6611-3.4803-2.7006.8983-5.5256 1.209-8.3166 1.5932-1.5085 1.243-2.8193 2.7628-4.0227 4.3053-1.5368-1.3447-3.0566-3.2205-5.3391-2.9097 1.2034 1.7006 2.8588 2.9888 4.4746 4.2713.2486 5.39 1.7741 10.5935 3.2713 15.7406.6159 2.2656 2.5199 3.808 4.3335 5.1357 3.5425 2.2204 7.4974 3.7403 10.9947 6.0228 1.8192 3.0283 3.3504 6.2544 4.4691 9.6161.5593 1.3729 1.0056 3.0453 2.5028 3.6837 4.6386 2.599 9.8083 4.0001 14.7124 6.0059 3.9549-2.0679 7.6951-4.5199 11.5313-6.7912.6159-.4633 1.5029-.7853 1.7176-1.6046 1.0566-3.8475 1.7402-7.7855 2.7459-11.6387.7401-2.6442 2.9605-4.4465 4.7063-6.4353 2.2657-2.6498 5.5652-4.4577 7.0002-7.7742.2543-5.8533.2939-11.7461-.6214-17.5542 1.2938-1.6328 2.5085-3.3278 3.7967-4.9606.0509 3.1018.5367 6.164.9831 9.2263.2598 3.6611.3502 7.3392.7119 10.989.0678.7458.2429 1.4972.6836 2.113 2.3334 3.0453 7.0793 3.7855 10.3111 1.7628-1.0001 3.2883-3.9324 5.3618-6.2262 7.7065-.0735.6553.2373 1.2712.3955 1.8983 1.096 2.5538 1.1638 5.7516-1.1639 7.65.9435 2.0509 1.6893 4.3221 1.1412 6.599-.209 1.4634-1.5367 2.2882-2.6045 3.1188.1751 2.0791 1.0395 4.26.1808 6.294-1.1131 1.8418-3.2657 2.5989-5.1753 3.3277-.4125 1.6611-.7685 3.4578-2.0396 4.7064-1.034 1.1808-2.7233 1.1526-4.1301 1.4916-1.13 2.2486-2.5368 4.6555-4.9606 5.6781-2.1244.548-4.2488-.339-6.3392-.6724-1.5255 1.599-3.1018 3.3844-5.3731 3.8589-2.3899.7967-4.7119-.5989-6.7798-1.6441-1.6385 1.2147-3.4238 2.599-5.5991 2.3334-2.3446.1469-3.9718-1.6328-5.5312-3.0735-2.1187 1.0791-4.5651 2.0001-6.9437 1.1977-2.2091-.5367-3.356-2.6554-4.6047-4.3504-2.8249-.0282-5.9776-.0282-8.2262-2.0057-1.695-1.322-2.3616-3.4238-3.034-5.3674-1.9153-.3051-4.1414-.3672-5.5256-1.9322-1.3503-1.6046-2.3108-3.5538-2.8306-5.5878-.565-2.3051.6385-4.4973 1.0283-6.7233-.9491-1.0848-2.4294-1.565-3.582-2.4182-3.1357-1.5763-3.5481-5.599-2.4238-8.5709-1.5651-1.1073-2.8362-2.5481-4.2205-3.8532 3.4634-.0226 7.0568-1.904 8.0285-5.4239.8475-2.2091.8023-4.6159.5989-6.938-.4294-4.5312-1.5085-8.9664-2.4351-13.4129-.4633-2.0735-3.0227-2.0113-4.4465-3.1357-2.2656-1.6441-4.5933-4.1187-4.1018-7.1471 1.1244-1.8814 3.1018-3.2656 3.469-5.5595-2.791-4.2769-1.8814-9.5652-.9265-14.2603 1.4407-5.2148 3.8306-10.328 7.7686-14.1303h0zm22.7803 19.3565c-2.6555 2.7797-4.39 6.2375-6.2827 9.5426 2.2713-.9039 3.9549-2.8079 5.6612-4.4916 3.6555-3.9662 7.7743-8.0737 9.006-13.5089-3.2544 2.3504-5.746 5.4747-8.3845 8.4579h0zm3.7402 8.3166c-1.0282 1.6215-2.3842 2.9888-3.4238 4.599 3.9549-2.1469 7.0567-6.2714 7.3731-10.8195-1.6102 1.8814-2.469 4.2487-3.9493 6.2205zm18.4187 7.6725c4.2995-2.5142 10.0737-3.1244 14.3224-.1356 1.4916.5876 2.13 2.8476.6046 3.7742-1.3955 1.1017-2.7403-.4238-4.0736-.7967-3.8645-1.0282-7.7856.5424-11.1077 2.4351-.8531-.4576-2.0452-1.017-1.9944-2.1639-.2147-1.4972 1.2317-2.339 2.2487-3.1131z"
			fill="#f17523"/>
		<path
			d="M349.7573 26.0043c2.0679-2.3391 5.0849-3.5369 8.0681-4.1979-.2939 3.7459 1.2091 7.2488 2.4746 10.6839 2.6215 6.5991 7.3787 12.3055 13.3563 16.1192 1.3108.9774 3.6329.9209 3.9549 2.8419-2.0847 2.8758-4.356 5.6216-6.3617 8.5539-1.8984.6441-3.8081 1.2656-5.7799 1.6554-1.4011-4.7064-7.7968-6.0228-11.4692-3.1922-.9662.8701-1.5933 2.034-2.1809 3.1753-1.4238-1.2995-3.0057-2.8589-5.1075-2.6668-2.9944.2091-5.3617 2.4408-7.0567 4.7403-.7966-2.8589-3.4295-5.1301-6.4521-5.0906-3.8929-.435-7.3844 2.7911-8.0455 6.503-1.8419.0791-3.7007.3164-5.5369.1074-3.0114-1.808-5.6838-4.147-8.3223-6.4465 1.2034-1.5425 2.5142-3.0623 4.0228-4.3052 2.791-.3842 5.6159-.695 8.3166-1.5933-.7062 1.0678-1.8532 2.0735-1.6611 3.4803-.1695 1.1978 1.5311 2.4521 2.4408 1.4181 2.017-2.0678 4.1866-4.1922 7.0115-5.0905 2.226-.565 4.5481-.017 6.8024-.3051 1.3052-.5255 1.7458-2.4803.6329-3.4238-1.1753-.8701-2.6951-.9379-4.0737-1.243 1.5877-1.1752 3.4691-2.0735 4.5878-3.7741 1.7345-2.3334 1.8136-5.4522 3.5594-7.7856 2.3673-3.3334 3.8419-7.277 6.8194-10.1641h0zm2.8532 22.6843c-1.017.774-2.4634 1.6158-2.2486 3.1131-.0509 1.1469 1.1413 1.7062 1.9943 2.1639 3.3222-1.8928 7.2432-3.4634 11.1078-2.4351 1.3333.3729 2.678 1.8983 4.0735.7966 1.5255-.9266.887-3.1865-.6045-3.7741-4.2488-2.9888-10.0229-2.3786-14.3225.1356zm-22.1588-15.9892c2.6384-2.9832 5.1301-6.1075 8.3844-8.4579-1.2317 5.4352-5.3504 9.5427-9.0059 13.5089-1.7063 1.6837-3.39 3.5877-5.6612 4.4916 1.8927-3.3051 3.6272-6.7629 6.2827-9.5426zm3.7402 8.3166c1.4803-1.9718 2.3391-4.3391 3.9493-6.2205-.3164 4.5482-3.4182 8.6726-7.3731 10.8195 1.0396-1.6102 2.3956-2.9775 3.4238-4.599zm49.3066 5.9607l.6101-.5424c1.6781-.5311 3.3391.9209 4.0905 2.339 2.6555 4.2092 3.39 9.441 2.3673 14.2717-1.0395 3.5085-2.6384 7.503-6.5086 8.616l-.0847.2656c-.4408-.6158-.6159-1.3673-.6837-2.1131-.3616-3.6498-.452-7.3279-.7119-10.989-.4464-3.0622-.9322-6.1245-.9831-9.2263.678-.8418 1.2995-1.7232 1.9041-2.6215zm-84.1099 8.3562c.5424-.5255 1.2035-1.6216 2.0735-.8927 1.4238 1.1243 3.9832 1.0622 4.4465 3.1357.9266 4.4464 2.0057 8.8816 2.4351 13.4128.2034 2.3221.2486 4.729-.5989 6.9381-1.4916-.243-3.0453-.5481-4.2035-1.582-4.4973-3.695-6.5991-9.7404-6.1189-15.4637.243-1.9323.5311-4.0962 1.9662-5.5482zm74.0192 5.0848l4.39-5.8589c.9153 5.8081.8757 11.7009.6214 17.5542-1.435 3.3165-4.7345 5.1244-7.0001 7.7742-1.7459 1.9888-3.9663 3.7911-4.7064 6.4353-1.0057 3.8532-1.6893 7.7912-2.7458 11.6387-.2147.8193-1.1018 1.1413-1.7176 1.6046-3.8363 2.2713-7.5765 4.7233-11.5314 6.7912-4.9041-2.0058-10.0738-3.4069-14.7123-6.0059-1.4972-.6384-1.9436-2.3108-2.5029-3.6837-1.1187-3.3617-2.6498-6.5878-4.4691-9.6161-3.4973-2.2826-7.4522-3.8024-10.9947-6.0228-1.8136-1.3277-3.7176-2.8701-4.3335-5.1357-1.4972-5.1471-3.0226-10.3506-3.2712-15.7406 3.1357 2.2034 5.5594 5.4408 9.1132 7.0171 1.8814.1074 3.7629-.0565 5.6443-.0791.1356 3.147 1.9435 6.3336 4.9663 7.4975 2.0847.7118 4.7232.9604 6.4917-.6272 2.322-1.7854 3.1639-4.7798 3.4351-7.5765 1.7684-2.0509 3.2995-4.7233 6.1471-5.3674 2.3503-.452 3.8249 1.808 5.1583 3.3391-.1695 2.4012.678 4.7742 2.1865 6.6273 2.8306 3.6555 9.6444 2.7289 11.2829-1.6328.8418-1.7402.661-3.7176.6836-5.5934 2.1809-.548 4.3617-1.13 6.4747-1.9209.6611-.2147.9606-.921 1.3899-1.4182h0zm-26.1194 21.4696c-2.7176.3729-5.2318-1.8644-7.9607-1.243 2.0113 1.9549 4.6781 3.4239 7.4014 4.0284 3.1582-.887 5.9267-3.0001 7.9041-5.5821-2.6893.0791-4.938 1.7967-7.3448 2.7967h0zm13.5766 1.4238c-6.7515 2.2204-13.2998 5.1753-20.3056 6.5256.678.8079 1.7402.9548 2.7458.7797 5.5143-.7797 10.6501-3.0284 15.8762-4.8307 1.6893-.6836 3.4803-1.4237 4.6499-2.8927-1.0057.0339-2.017.0847-2.9663.4181zm43.9053 73.3694c2.8419-.6723 6.0171-1.2317 8.746.1469 2.4973 1.3842 5.1471 2.5763 7.3223 4.4634 2.6385 2.3164 4.6555 5.2318 6.5143 8.1923 1.5706 2.6498 2.1469 5.729 2.3165 8.7686.305 1.5594.2994 3.1583.0847 4.7347-.5028 3.7515-.3616 7.8137-2.2487 11.2093-2.1808 2.4182-5.2431 3.7685-7.8025 5.729-2.7288 2.0114-6.4069 1.4577-9.2432-.0226-2.4972 2.5594-7.1979 3.9832-10.0455 1.2261-2.0904.2994-4.0961 1.4068-6.2487.9604-1.7176-.0847-3.0849-1.3277-4.1414-2.5706-1.0622-1.4182-.2317-3.181-.0282-4.729 1.1638-7.7065 2.0396-15.492 4.0961-23.029-4.9719.678-10.2376 1.0905-15.0852-.4576-1.7458-.6102-3.8193-1.2486-4.4295-3.2204-1.2373-2.0735-.3955-5.1358 2.0904-5.7516 3.9041-.2147 7.7461 1.8136 11.6558 1.0961 5.3165-2.599 10.6952-5.2262 16.4468-6.746zm-95.9068 18.9611c3.3447-1.1808 6.9663-2.4577 10.5257-1.4408 2.2713 1.3956 4.2656 3.277 5.8589 5.4126 3.486 4.7007 6.7177 9.5879 9.5088 14.7406.8022 1.4803 2.0396 3.147 1.2147 4.9041-.7458 2.0679-3.147 2.356-5.0284 2.6215-1.9379.339-3.3448-1.322-4.4464-2.6554-3.6273-4.8872-6.2318-10.6726-10.9213-14.6841 2.4634 4.2262 4.7007 8.5935 6.616 13.1021 1.0057 2.3843 1.2317 4.9889 2.0679 7.4183.6723 1.5424.3842 3.599-1.2712 4.373-2.3561.6102-4.8363 1.0961-7.2771.7797-1.2147-.1356-2.1752-1.0508-3.373-1.2203-1.5424.4237-2.9605 1.2373-4.5425 1.5367-2.5877.6328-5.277-.1299-7.8702.4972-2.8815.678-5.876.1017-8.5936-.9266-3.1695-1.2655-5.3447-4.1131-6.8476-7.0736-.6497-1.2656-1.1865-2.5877-1.565-3.9549-.5537-2.9267-.6724-5.9607-.3673-8.9156.2091-2.1582 2.2148-3.3221 3.8928-4.3052 7.3674-3.6272 14.4185-8.0624 22.4188-10.2093z"
			fill="#fccda3"/>
		<path
			d="M377.6113 51.4513c1.2825-2.2712 3.356-3.8306 5.8872-4.4747l-1.904 2.6216-3.7967 4.9606-4.39 5.8589c-.4294.4972-.7289 1.2034-1.3899 1.4181-2.113.791-4.2939 1.3729-6.4748 1.921l-.0734-2.0961c1.9718-.3899 3.8815-1.0114 5.7798-1.6555 2.0057-2.9322 4.277-5.6781 6.3618-8.5539zm-71.6519 4.4296c2.2825-.3108 3.8024 1.565 5.3391 2.9097 2.6385 2.2995 5.3109 4.6385 8.3223 6.4465 1.8362.209 3.695-.0283 5.5369-.1074l.0339 1.9605c-1.8814.0226-3.7628.1865-5.6442.0791-3.5538-1.5763-5.9776-4.8137-9.1133-7.0171-1.6159-1.2825-3.2713-2.5707-4.4747-4.2713zm33.696 7.8364c1.6949-2.2995 4.0623-4.5312 7.0567-4.7403 2.1017-.192 3.6837 1.3673 5.1075 2.6668l-.4294 2.7119c-1.3334-1.5311-2.808-3.791-5.1583-3.339-2.8476.644-4.3787 3.3164-6.1471 5.3673l-.4294-2.6667z"
			fill="#847972"/>
		<path
			d="M354.0004 58.4685c3.6725-2.8306 10.0681-1.5141 11.4693 3.1922l.0735 2.0961c-.0226 1.8758.1582 3.8532-.6836 5.5934-1.6385 4.3617-8.4523 5.2883-11.2829 1.6328-1.5085-1.8531-2.356-4.2261-2.1865-6.6273.0961-.9096.243-1.8136.4294-2.7119.5876-1.1413 1.2147-2.3052 2.1808-3.1753h0zm5.181 5.8872c-2.5651 1.2938-3.4521 5.3279-1.3673 7.424 2.6555 1.9096 5.8646-1.3617 5.4635-4.1414.5423-2.3278-2.0453-3.9606-4.0962-3.2826zm-34.0237.774c.661-3.712 4.1526-6.9381 8.0455-6.503 3.0226-.0395 5.6555 2.2317 6.4521 5.0905.1638.8814.3051 1.7741.4294 2.6668-.2712 2.7967-1.113 5.7911-3.4351 7.5765-1.7685 1.5876-4.4069 1.339-6.4917.6271-3.0227-1.1639-4.8307-4.3504-4.9663-7.4974l-.0339-1.9605h0zm7.6274 1.5255c-3.034 1.4859-2.8419 7.3674.9548 7.6443 3.4238.0056 4.9776-4.5087 3.3447-7.1415-1.0734-1.1413-2.9831-1.4012-4.2995-.5028zm27.7691-.4294c1.3165-.6159 1.7798 2.1526.3165 1.8475-.9323.1243-1.2148-1.6724-.3165-1.8475zm-25.8877 2.26c.7571-1.3447 2.5142.6101 1.339 1.3164-.8192.9492-1.7119-.565-1.339-1.3164z"/>
		<path
			d="M359.1814 64.3557c2.0509-.678 4.6385.9549 4.0962 3.2826.4011 2.7798-2.808 6.051-5.4635 4.1414-2.0848-2.0961-1.1978-6.1301 1.3673-7.424h0zm1.3729 1.8701c-.8983.1752-.6158 1.9718.3164 1.8475 1.4633.3051 1-2.4633-.3164-1.8475zm-27.7692.4293c1.3164-.8983 3.2261-.6384 4.2996.5029 1.6328 2.6328.0791 7.1471-3.3447 7.1415-3.7968-.2769-3.9889-6.1584-.9549-7.6444h0zm1.8814 1.8306c-.3729.7514.5198 2.2656 1.339 1.3164 1.1753-.7062-.5819-2.6611-1.339-1.3164zm337.4791 82.2963l79.1154-9.5257v4.7403l-58.2278 7.0962-81.8669 9.3788-105.4268 13.4694c-10.7687 1.6836-21.6222 2.7176-32.4361 4.0227l-21.2039 1.7232c-7.4466.4689-14.8763 1.4012-22.3454 1.2995.2147-1.5764.2204-3.1753-.0847-4.7346 6.26.0056 12.4919-.6159 18.735-1.0679 11.9834-1.0056 24.0233-1.6045 35.9276-3.4068 14.9553-1.2995 29.7241-4.1019 44.6059-5.9946 47.6455-6.3844 95.4322-11.6557 143.2077-17.0005zm-299.2124 21.6956l6.0736-1.1752c4.8477 1.5481 10.1133 1.1356 15.0852.4576-2.0565 7.537-2.9322 15.3225-4.0961 23.029-3.695.8136-7.424 1.4633-11.1303 2.226-.4407-1.6667-.7175-3.3786-1.2373-5.0227-1.9379-.2938-3.9945.7684-6.0115.8023-1.2712-.0339-3.3277.6723-3.9097-.8984-.6215-.8757.2599-1.887 1.017-2.2995 2.4633-1.2938 5.3109-1.4463 7.8928-2.4181l-1.4463-5.8533c-3.0736.3899-6.0793 1.5424-9.2037 1.3051-1.3729-.4915-1.5254-2.678-.2316-3.3278 2.582-1.2938 5.52-1.6045 8.2375-2.5594l-1.0396-4.2656zm-47.6851 7.1358l15.3338-2.0057c2.1639 7.7177 3.5086 15.6389 4.6838 23.56-6.9155 1.6894-13.9552 2.9154-21.0063 3.8928-.8362-2.4295-1.0622-5.034-2.0679-7.4183-1.9153-4.5086-4.1526-8.876-6.616-13.1021 4.6895 4.0114 7.294 9.7969 10.9213 14.6841 1.1016 1.3334 2.5085 2.9944 4.4464 2.6554 1.8814-.2655 4.2826-.5537 5.0284-2.6215.8248-1.7572-.4125-3.4239-1.2147-4.9041-2.791-5.1527-6.0228-10.0399-9.5088-14.7406zm-75.7028 21.0854c7.6951-.7062 15.4468-.4125 23.1419-1.1526 3.4013-.2994 6.8138-.5311 10.232-.4746.3785 1.3673.9152 2.6894 1.565 3.955-3.3617.1186-6.7234.3107-10.0625.7062-8.4295 1.1187-16.9552 1.0791-25.4244 1.7854-12.0456.9887-24.1533.5536-36.2045 1.4576l-1.0226.4238c-.7232-.356-1.5255-.3899-2.3108-.3277-10.1642.7627-20.3623.435-30.549.5932l-75.6859.0113c-27.8765-.1017-55.7531.2656-83.624-.3051-6.7007-.7119-13.4636-1.1356-20.0991-2.339v-2.9041l18.9635.6893 25.9895.113 140.654-.0622c8.8081-.0056 17.6333-.0791 26.4245-.5197 12.6388-1.1639 25.3679-.5481 38.0124-1.6498z"
			fill="#3e2723"/>
		<path
			d="M347.2884 81.8873c2.4069-1 4.6555-2.7176 7.3448-2.7967-1.9774 2.582-4.7459 4.6951-7.9041 5.5821-2.7233-.6045-5.39-2.0735-7.4014-4.0284 2.7289-.6215 5.2431 1.6159 7.9607 1.243zm13.5766 1.4238c.9492-.3333 1.9605-.3842 2.9662-.4181-1.1695 1.469-2.9606 2.2091-4.6499 2.8927-5.2261 1.8024-10.3619 4.051-15.8762 4.8307-1.0056.1751-2.0678.0282-2.7458-.7797 7.0059-1.3503 13.5541-4.3052 20.3057-6.5256z"
			fill="#dba378"/>
		<path
			d="M388.4534 83.2941c10.4071.7006 20.865 1.0565 31.1592 2.8927 14.006 2.5707 27.9386 5.5482 41.7978 8.8421 4.8363 1.0735 9.4692 2.8645 14.1586 4.4352 3.277 1.1356 6.633 2.0735 9.7574 3.6159 5.5369 2.678 11.684 4.3109 16.5259 8.2601 5.6104 4.2148 9.5201 10.3845 11.9438 16.9045 1.8589 6.4748 1.3674 13.4298-.2655 19.8933-.7006 4.1978-2.6837 8.1697-5.5933 11.2715-3.6838 3.5594-8.1076 6.2261-12.3733 9.0115-3.4464 2.3561-7.4296 3.6951-11.232 5.3561-11.9043 1.8024-23.9442 2.4012-35.9276 3.4069-6.2431.452-12.475 1.0735-18.735 1.0679-.1696-3.0397-.7459-6.1189-2.3165-8.7687-1.8588-2.9605-3.8758-5.8759-6.5143-8.1923-2.1752-1.8871-4.825-3.0792-7.3223-4.4634-2.7289-1.3786-5.9041-.8193-8.746-.1469-5.7516 1.5198-11.1303 4.147-16.4468 6.7459-3.9097.7176-7.7517-1.3107-11.6557-1.096-2.486.6158-3.3278 3.678-2.0905 5.7516.6102 1.9718 2.6837 2.6102 4.4295 3.2204-2.0452.2938-4.0623.7401-6.0736 1.1752.3672 1.4124.7119 2.8362 1.0396 4.2656-2.7176.9548-5.6556 1.2656-8.2376 2.5594-1.2938.6498-1.1413 2.8363.2317 3.3278 3.1244.2373 6.1301-.9153 9.2036-1.3051.5255 1.9379.9887 3.8928 1.4464 5.8533-2.582.9717-5.4295 1.1243-7.8929 2.4181-.7571.4125-1.6385 1.4238-1.017 2.2995.582 1.5707 2.6385.8645 3.9098.8983 2.0169-.0338 4.0735-1.096 6.0114-.8022.5198 1.6441.7967 3.356 1.2373 5.0227l11.1304-2.226c-.2035 1.548-1.034 3.3108.0282 4.7289 1.0565 1.243 2.4238 2.486 4.1414 2.5707 2.1526.4464 4.1583-.661 6.2488-.9604 2.8475 2.7571 7.5482 1.3333 10.0454-1.2261 2.8363 1.4803 6.5144 2.034 9.2432.0226 2.5594-1.9605 5.6217-3.3108 7.8025-5.729 1.8871-3.3955 1.7459-7.4578 2.2487-11.2094 7.4692.1018 14.8987-.8305 22.3453-1.2994.3503 7.7686 1.1526 15.5033 1.5537 23.2606-.0282 4.1413.7571 8.2262.7854 12.3619.1864 9.7178-.0735 19.4356-.0509 29.1478-65.1545-.0283-130.309.0056-195.4635-.017l.9492-16.6332c.113-5.0793.0508-10.1585.0734-15.2378-6.5877 1.7063-13.4862 2.034-20.2492 1.4973-9.8082-.6046-19.6164-3.3504-27.9274-8.667l1.0227-.4237c12.0512-.904 24.1589-.469 36.2045-1.4577 8.4691-.7062 16.9948-.6667 25.4245-1.7854 3.339-.3955 6.7007-.5875 10.0624-.7062 1.5029 2.9605 3.6781 5.8081 6.8477 7.0736 2.7175 1.0283 5.712 1.6046 8.5935.9267 2.5932-.6272 5.2826.1355 7.8702-.4972 1.582-.2995 3.0001-1.1131 4.5425-1.5368 1.1978.1695 2.1583 1.0848 3.373 1.2204 2.4408.3164 4.921-.1695 7.2771-.7797 1.6554-.7741 1.9435-2.8306 1.2712-4.373 7.0511-.9775 14.0908-2.2035 21.0063-3.8928-1.1752-7.9212-2.5198-15.8423-4.6838-23.56l-15.3338 2.0057c-1.5932-2.1357-3.5876-4.0171-5.8589-5.4126-3.5594-1.017-7.181.2599-10.5257 1.4407-8.0003 2.1469-15.0514 6.5821-22.4188 10.2093-1.678.9831-3.6837 2.147-3.8928 4.3053-.3051 2.9548-.1864 5.9888.3673 8.9155-3.4182-.0565-6.8308.1751-10.232.4746-7.6951.7401-15.4468.4463-23.1419 1.1525-12.6445 1.1018-25.3737.4859-38.0125 1.6498-1.8248-.1017-3.6554-.113-5.4804-.0565-4.2882-4.3504-7.034-10.0059-8.2827-15.9496-.4972-2.9041.3899-5.7629 1.0057-8.5765 1.0113-5.1697 3.712-9.8026 6.6273-14.1191 3.1809-4.6781 7.4918-8.3845 11.4241-12.3959 2.2374-2.6215 4.5707-5.2035 7.3901-7.2205 4.5481-3.599 9.2262-7.0285 13.9834-10.3562 10.0399-7.2602 20.6956-13.5937 31.0744-20.3453 4.712-3.2656 10.085-5.3448 15.153-7.972 5.034-2.3786 9.9269-5.1244 15.266-6.7855 2.2882-.7571 4.5255-1.678 6.6668-2.7967 1.1526.8531 2.6329 1.3334 3.5821 2.4181-.3899 2.2261-1.5933 4.4183-1.0283 6.7234.5198 2.034 1.4803 3.9832 2.8306 5.5878 1.3842 1.565 3.6103 1.6271 5.5256 1.9322.6723 1.9436 1.339 4.0453 3.034 5.3674 2.2487 1.9775 5.4013 1.9775 8.2262 2.0057 1.2487 1.695 2.3956 3.8137 4.6047 4.3504 2.3786.8023 4.825-.1186 6.9437-1.1977 1.5594 1.4407 3.1865 3.2204 5.5312 3.0735 2.1752.2655 3.9606-1.1187 5.5991-2.3334 2.0678 1.0452 4.3899 2.4407 6.7798 1.6441 2.2713-.4746 3.8476-2.2599 5.373-3.8589 2.0905.3334 4.2149 1.2204 6.3393.6724 2.4237-1.0227 3.8305-3.4295 4.9606-5.6782 1.4068-.339 3.0961-.3107 4.13-1.4915 1.2713-1.2487 1.6272-3.0453 2.0396-4.7064 1.9097-.7288 4.0623-1.4859 5.1753-3.3278.8588-2.0339-.0056-4.2148-.1808-6.2939 1.0678-.8306 2.3955-1.6555 2.6047-3.1188.5479-2.2769-.1978-4.5481-1.1413-6.5991 2.3277-1.8983 2.2599-5.0962 1.1638-7.6499z"
			fill="#bf360c"/>
	</g>
</svg>                       plugins/wordpress-seo-extended/packages/js/images/edd-logo.svg                                      0000644                 00000057547 15122266557 0020546 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8" standalone="no"?><!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd"><svg width="130" height="60" viewBox="0 0 339 157" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" xml:space="preserve" xmlns:serif="http://www.serif.com/" style="fill-rule:evenodd;clip-rule:evenodd;stroke-linejoin:round;stroke-miterlimit:1.41421;"><g><path d="M206.613,15.378c-9.499,-9.502 -22.628,-15.378 -37.125,-15.378c-14.501,0 -27.628,5.876 -37.13,15.378c-9.504,9.502 -15.38,22.631 -15.38,37.13c0,14.499 5.876,27.628 15.378,37.13c9.502,9.504 22.631,15.378 37.132,15.378c14.499,0 27.628,-5.879 37.127,-15.378c9.502,-9.504 15.378,-22.631 15.378,-37.13c0,-14.499 -5.878,-27.628 -15.38,-37.13Zm-1.732,72.524c-9.057,9.058 -21.572,14.661 -35.395,14.661c-13.823,0 -26.339,-5.603 -35.397,-14.661c-9.058,-9.058 -14.661,-21.572 -14.661,-35.394c0,-13.823 5.603,-26.337 14.661,-35.395c9.058,-9.058 21.572,-14.661 35.395,-14.661c13.823,0 26.337,5.603 35.395,14.661c9.058,9.058 14.66,21.572 14.66,35.395c0.002,13.822 -5.601,26.337 -14.658,35.394Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M214.656,52.07c-0.236,-24.746 -20.367,-44.734 -45.168,-44.734c-24.803,0 -44.936,19.99 -45.168,44.736l20.712,-20.712l6.946,6.946l-15.157,15.157l65.333,0l-15.157,-15.157l6.946,-6.946l20.713,20.71Zm-45.17,-7.762l-19.246,-20.196l12.79,0l0,-9.955c0,-2.584 2.906,-4.702 6.456,-4.702c3.55,0 6.456,2.116 6.456,4.702l0,9.955l12.79,0l-19.246,20.196Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M175.1,77.93c-1.026,-0.667 -2.326,-1.267 -3.894,-1.788c-1.199,-0.413 -2.186,-0.803 -2.956,-1.179c-0.764,-0.372 -1.328,-0.785 -1.689,-1.221c-0.37,-0.444 -0.543,-0.976 -0.543,-1.586c0,-0.488 0.147,-0.952 0.462,-1.391c0.309,-0.438 0.792,-0.803 1.448,-1.079c0.663,-0.28 1.507,-0.431 2.555,-0.436c0.844,0.004 1.615,0.072 2.312,0.193c0.689,0.125 1.3,0.276 1.823,0.449c0.53,0.179 0.958,0.348 1.295,0.507l1.166,-3.516c-0.705,-0.354 -1.588,-0.648 -2.66,-0.886c-0.91,-0.201 -1.952,-0.317 -3.135,-0.357l0,-3.638l-3.205,0l0,3.82c-0.514,0.083 -1.007,0.195 -1.473,0.33c-1.179,0.348 -2.192,0.831 -3.03,1.462c-0.831,0.628 -1.475,1.363 -1.917,2.205c-0.444,0.842 -0.663,1.77 -0.672,2.768c0.006,1.155 0.309,2.173 0.906,3.052c0.597,0.88 1.448,1.643 2.547,2.297c1.094,0.652 2.4,1.214 3.91,1.687c1.133,0.363 2.057,0.742 2.761,1.131c0.705,0.387 1.221,0.814 1.542,1.28c0.328,0.466 0.49,1.006 0.484,1.61c0,0.663 -0.195,1.238 -0.575,1.735c-0.376,0.49 -0.932,0.875 -1.663,1.148c-0.731,0.276 -1.623,0.411 -2.669,0.42c-0.851,-0.006 -1.676,-0.079 -2.481,-0.217c-0.799,-0.144 -1.549,-0.33 -2.24,-0.562c-0.698,-0.232 -1.308,-0.484 -1.838,-0.766l-1.127,3.662c0.51,0.28 1.146,0.532 1.925,0.764c0.779,0.234 1.636,0.42 2.575,0.562c0.932,0.138 1.89,0.208 2.864,0.217l0.171,-0.002l0,3.761l3.205,0l0,-4.063c0.407,-0.079 0.794,-0.173 1.162,-0.282c1.286,-0.387 2.354,-0.915 3.199,-1.591c0.851,-0.67 1.481,-1.446 1.899,-2.321c0.416,-0.875 0.624,-1.801 0.624,-2.787c0,-1.155 -0.247,-2.177 -0.757,-3.056c-0.515,-0.887 -1.281,-1.664 -2.311,-2.336Z" style="fill:#35495c;fill-rule:nonzero;"/></g><g><path d="M3.999,141.181c0,0.701 0.099,1.38 0.298,2.034c0.198,0.655 0.496,1.228 0.894,1.719c0.397,0.491 0.9,0.883 1.508,1.175c0.608,0.293 1.333,0.439 2.175,0.439c1.169,0 2.11,-0.251 2.824,-0.754c0.713,-0.502 1.245,-1.256 1.596,-2.262l3.788,0c-0.21,0.982 -0.573,1.859 -1.087,2.631c-0.515,0.772 -1.135,1.421 -1.859,1.946c-0.725,0.526 -1.538,0.924 -2.438,1.193c-0.901,0.268 -1.842,0.403 -2.824,0.403c-1.427,0 -2.689,-0.234 -3.788,-0.701c-1.1,-0.468 -2.029,-1.123 -2.789,-1.965c-0.76,-0.842 -1.333,-1.847 -1.719,-3.016c-0.385,-1.17 -0.578,-2.456 -0.578,-3.859c0,-1.286 0.205,-2.508 0.614,-3.665c0.409,-1.157 0.994,-2.175 1.754,-3.052c0.76,-0.877 1.678,-1.573 2.754,-2.087c1.075,-0.514 2.291,-0.772 3.648,-0.772c1.426,0 2.706,0.298 3.841,0.894c1.134,0.596 2.075,1.386 2.824,2.368c0.748,0.982 1.292,2.111 1.631,3.385c0.339,1.275 0.427,2.59 0.263,3.946l-13.33,0Zm9.33,-2.631c-0.047,-0.631 -0.182,-1.239 -0.404,-1.824c-0.223,-0.584 -0.526,-1.093 -0.912,-1.526c-0.386,-0.432 -0.854,-0.783 -1.403,-1.052c-0.55,-0.269 -1.163,-0.404 -1.841,-0.404c-0.702,0 -1.339,0.123 -1.912,0.369c-0.573,0.245 -1.064,0.585 -1.473,1.017c-0.409,0.433 -0.737,0.942 -0.982,1.526c-0.246,0.585 -0.38,1.216 -0.403,1.894l9.33,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M35.567,145.18c0,0.491 0.064,0.842 0.193,1.052c0.128,0.21 0.38,0.316 0.754,0.316c0.117,0 0.257,0 0.421,0c0.163,0 0.351,-0.023 0.561,-0.07l0,2.771c-0.14,0.047 -0.322,0.099 -0.544,0.158c-0.223,0.058 -0.45,0.111 -0.684,0.158c-0.234,0.047 -0.468,0.081 -0.702,0.105c-0.234,0.024 -0.433,0.035 -0.596,0.035c-0.819,0 -1.497,-0.164 -2.034,-0.491c-0.538,-0.327 -0.889,-0.9 -1.052,-1.719c-0.795,0.771 -1.771,1.333 -2.929,1.683c-1.157,0.351 -2.274,0.526 -3.35,0.526c-0.819,0 -1.602,-0.111 -2.35,-0.333c-0.749,-0.222 -1.409,-0.549 -1.982,-0.982c-0.573,-0.433 -1.029,-0.982 -1.368,-1.649c-0.339,-0.666 -0.509,-1.444 -0.509,-2.333c0,-1.122 0.204,-2.034 0.614,-2.736c0.409,-0.701 0.947,-1.251 1.613,-1.649c0.666,-0.397 1.414,-0.684 2.245,-0.859c0.83,-0.175 1.666,-0.31 2.508,-0.404c0.725,-0.14 1.415,-0.239 2.069,-0.298c0.654,-0.058 1.233,-0.158 1.736,-0.298c0.503,-0.14 0.9,-0.356 1.193,-0.649c0.292,-0.292 0.438,-0.73 0.438,-1.315c0,-0.514 -0.123,-0.935 -0.368,-1.263c-0.246,-0.327 -0.55,-0.579 -0.912,-0.754c-0.363,-0.175 -0.766,-0.292 -1.21,-0.351c-0.445,-0.058 -0.865,-0.088 -1.263,-0.088c-1.122,0 -2.047,0.234 -2.771,0.702c-0.725,0.468 -1.135,1.193 -1.228,2.175l-3.999,0c0.07,-1.169 0.351,-2.14 0.842,-2.911c0.491,-0.772 1.117,-1.391 1.877,-1.859c0.76,-0.467 1.619,-0.795 2.578,-0.982c0.958,-0.187 1.941,-0.281 2.946,-0.281c0.888,0 1.765,0.094 2.631,0.281c0.865,0.188 1.643,0.491 2.333,0.912c0.689,0.421 1.245,0.965 1.666,1.631c0.421,0.666 0.631,1.479 0.631,2.438l0,9.331l0.002,0Zm-3.998,-5.052c-0.608,0.398 -1.356,0.638 -2.245,0.719c-0.889,0.082 -1.777,0.205 -2.666,0.369c-0.421,0.07 -0.83,0.17 -1.228,0.298c-0.398,0.129 -0.749,0.304 -1.052,0.526c-0.304,0.223 -0.544,0.515 -0.719,0.877c-0.175,0.363 -0.263,0.801 -0.263,1.315c0,0.445 0.128,0.819 0.386,1.123c0.257,0.304 0.567,0.543 0.93,0.719c0.362,0.175 0.759,0.298 1.193,0.368c0.432,0.071 0.824,0.105 1.175,0.105c0.444,0 0.924,-0.058 1.438,-0.175c0.514,-0.117 1,-0.316 1.456,-0.596c0.456,-0.28 0.836,-0.636 1.14,-1.07c0.304,-0.432 0.456,-0.964 0.456,-1.596l0,-2.982l-0.001,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M42.898,143.39c0.117,1.17 0.561,1.988 1.333,2.456c0.772,0.468 1.695,0.701 2.771,0.701c0.374,0 0.801,-0.029 1.28,-0.088c0.479,-0.058 0.93,-0.169 1.351,-0.333c0.421,-0.163 0.766,-0.403 1.035,-0.719c0.268,-0.316 0.391,-0.731 0.368,-1.245c-0.024,-0.514 -0.21,-0.935 -0.561,-1.263c-0.351,-0.327 -0.801,-0.59 -1.35,-0.789c-0.55,-0.198 -1.175,-0.369 -1.877,-0.509c-0.702,-0.14 -1.415,-0.292 -2.14,-0.456c-0.749,-0.163 -1.468,-0.362 -2.157,-0.596c-0.69,-0.234 -1.31,-0.549 -1.859,-0.947c-0.55,-0.397 -0.988,-0.906 -1.315,-1.526c-0.328,-0.619 -0.491,-1.386 -0.491,-2.298c0,-0.982 0.239,-1.806 0.719,-2.473c0.479,-0.666 1.087,-1.204 1.824,-1.613c0.737,-0.409 1.555,-0.696 2.455,-0.859c0.9,-0.163 1.759,-0.245 2.578,-0.245c0.935,0 1.829,0.1 2.683,0.298c0.853,0.199 1.625,0.521 2.315,0.965c0.689,0.445 1.263,1.023 1.719,1.736c0.456,0.714 0.742,1.573 0.86,2.578l-4.174,0c-0.188,-0.958 -0.626,-1.601 -1.316,-1.929c-0.69,-0.327 -1.479,-0.491 -2.368,-0.491c-0.281,0 -0.614,0.024 -0.999,0.07c-0.386,0.047 -0.749,0.135 -1.087,0.263c-0.339,0.129 -0.626,0.316 -0.859,0.561c-0.234,0.246 -0.351,0.567 -0.351,0.965c0,0.491 0.169,0.889 0.509,1.193c0.339,0.304 0.783,0.556 1.333,0.754c0.549,0.199 1.175,0.368 1.876,0.508c0.702,0.14 1.426,0.293 2.175,0.456c0.724,0.164 1.438,0.363 2.14,0.596c0.701,0.234 1.327,0.55 1.877,0.947c0.549,0.397 0.994,0.9 1.333,1.508c0.339,0.609 0.508,1.356 0.508,2.245c0,1.076 -0.246,1.988 -0.737,2.736c-0.491,0.749 -1.128,1.357 -1.912,1.824c-0.784,0.468 -1.654,0.807 -2.613,1.017c-0.959,0.21 -1.906,0.316 -2.841,0.316c-1.146,0 -2.204,-0.129 -3.174,-0.386c-0.971,-0.257 -1.812,-0.649 -2.525,-1.175c-0.714,-0.526 -1.275,-1.18 -1.684,-1.964c-0.409,-0.784 -0.626,-1.713 -0.649,-2.789l3.997,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M56.086,131.079l4.385,0l4.735,13.539l0.07,0l4.595,-13.539l4.174,0l-7.05,19.116c-0.328,0.818 -0.649,1.601 -0.965,2.35c-0.316,0.748 -0.701,1.408 -1.157,1.982c-0.456,0.573 -1.023,1.029 -1.701,1.368c-0.678,0.339 -1.543,0.509 -2.596,0.509c-0.936,0 -1.859,-0.07 -2.771,-0.21l0,-3.368c0.327,0.047 0.643,0.1 0.947,0.158c0.304,0.058 0.619,0.088 0.947,0.088c0.467,0 0.853,-0.058 1.157,-0.175c0.304,-0.118 0.555,-0.287 0.754,-0.509c0.198,-0.223 0.368,-0.486 0.509,-0.79c0.14,-0.304 0.268,-0.655 0.386,-1.052l0.456,-1.403l-6.875,-18.064Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M90.18,149.213l0,-3.438l-0.07,0c-0.234,0.585 -0.579,1.123 -1.035,1.614c-0.456,0.491 -0.976,0.907 -1.561,1.245c-0.585,0.339 -1.21,0.602 -1.877,0.789c-0.666,0.187 -1.327,0.281 -1.982,0.281c-1.38,0 -2.578,-0.251 -3.595,-0.754c-1.017,-0.502 -1.865,-1.192 -2.543,-2.069c-0.679,-0.877 -1.181,-1.894 -1.508,-3.052c-0.328,-1.158 -0.491,-2.379 -0.491,-3.666c0,-1.286 0.163,-2.508 0.491,-3.665c0.327,-1.157 0.83,-2.175 1.508,-3.052c0.678,-0.877 1.526,-1.573 2.543,-2.087c1.017,-0.514 2.215,-0.772 3.595,-0.772c0.678,0 1.339,0.082 1.982,0.245c0.643,0.164 1.245,0.415 1.807,0.754c0.561,0.339 1.058,0.754 1.491,1.245c0.432,0.491 0.766,1.064 1,1.719l0.07,0l0,-10.382l2.21,0l0,25.044l-2.035,0l0,0.001Zm-12.119,-6.261c0.222,0.924 0.573,1.748 1.052,2.473c0.479,0.725 1.093,1.315 1.842,1.771c0.748,0.456 1.649,0.684 2.701,0.684c1.169,0 2.157,-0.228 2.964,-0.684c0.807,-0.456 1.461,-1.046 1.964,-1.771c0.503,-0.725 0.865,-1.549 1.087,-2.473c0.222,-0.924 0.333,-1.853 0.333,-2.789c0,-0.935 -0.111,-1.864 -0.333,-2.788c-0.223,-0.923 -0.585,-1.748 -1.087,-2.473c-0.503,-0.725 -1.157,-1.316 -1.964,-1.772c-0.807,-0.456 -1.795,-0.684 -2.964,-0.684c-1.052,0 -1.953,0.228 -2.701,0.684c-0.749,0.456 -1.363,1.047 -1.842,1.772c-0.48,0.725 -0.83,1.549 -1.052,2.473c-0.222,0.924 -0.333,1.853 -0.333,2.788c0,0.936 0.111,1.866 0.333,2.789Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M98.668,124.169l0,3.543l-2.21,0l0,-3.543l2.21,0Zm0,6.945l0,18.099l-2.21,0l0,-18.099l2.21,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M117.767,151.353c-0.293,1.076 -0.754,1.988 -1.386,2.736c-0.632,0.748 -1.456,1.321 -2.473,1.719c-1.017,0.397 -2.263,0.596 -3.736,0.596c-0.912,0 -1.801,-0.105 -2.666,-0.316c-0.865,-0.21 -1.643,-0.538 -2.332,-0.982c-0.69,-0.445 -1.263,-1.012 -1.719,-1.701c-0.456,-0.69 -0.719,-1.514 -0.789,-2.473l2.21,0c0.117,0.678 0.345,1.245 0.684,1.702c0.339,0.456 0.748,0.824 1.228,1.105c0.479,0.281 1.011,0.485 1.596,0.614c0.584,0.128 1.18,0.193 1.789,0.193c2.057,0 3.543,-0.585 4.455,-1.754c0.912,-1.17 1.368,-2.853 1.368,-5.051l0,-2.456l-0.07,0c-0.515,1.123 -1.269,2.023 -2.262,2.701c-0.994,0.678 -2.157,1.017 -3.49,1.017c-1.45,0 -2.689,-0.24 -3.718,-0.719c-1.029,-0.479 -1.877,-1.14 -2.543,-1.982c-0.666,-0.842 -1.152,-1.829 -1.456,-2.964c-0.304,-1.134 -0.456,-2.344 -0.456,-3.63c0,-1.239 0.181,-2.414 0.544,-3.525c0.362,-1.11 0.888,-2.081 1.578,-2.911c0.689,-0.83 1.543,-1.485 2.561,-1.964c1.017,-0.479 2.18,-0.719 3.49,-0.719c0.678,0 1.315,0.094 1.912,0.281c0.596,0.188 1.14,0.45 1.631,0.789c0.491,0.339 0.93,0.731 1.316,1.175c0.386,0.445 0.684,0.912 0.894,1.403l0.07,0l0,-3.122l2.21,0l0,16.626c-0.001,1.332 -0.148,2.536 -0.44,3.612Zm-5.068,-4.823c0.724,-0.409 1.333,-0.947 1.824,-1.613c0.491,-0.666 0.859,-1.432 1.105,-2.297c0.246,-0.865 0.368,-1.754 0.368,-2.666c0,-0.888 -0.105,-1.777 -0.316,-2.666c-0.21,-0.888 -0.55,-1.695 -1.017,-2.42c-0.468,-0.724 -1.07,-1.309 -1.806,-1.754c-0.737,-0.444 -1.631,-0.666 -2.683,-0.666c-1.052,0 -1.953,0.216 -2.701,0.649c-0.749,0.433 -1.368,1 -1.859,1.701c-0.491,0.701 -0.848,1.503 -1.07,2.403c-0.223,0.9 -0.333,1.818 -0.333,2.754c0,0.912 0.117,1.801 0.351,2.666c0.233,0.865 0.596,1.631 1.087,2.297c0.491,0.666 1.11,1.205 1.859,1.613c0.748,0.41 1.636,0.614 2.666,0.614c0.958,-0.001 1.8,-0.205 2.525,-0.615Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M124.66,124.169l0,3.543l-2.21,0l0,-3.543l2.21,0Zm0,6.945l0,18.099l-2.21,0l0,-18.099l2.21,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M136.094,131.114l0,1.859l-3.683,0l0,12.207c0,0.725 0.099,1.292 0.298,1.701c0.198,0.41 0.696,0.638 1.491,0.684c0.631,0 1.263,-0.035 1.894,-0.105l0,1.859c-0.328,0 -0.655,0.011 -0.982,0.035c-0.328,0.023 -0.655,0.035 -0.982,0.035c-1.473,0 -2.503,-0.286 -3.087,-0.86c-0.585,-0.573 -0.865,-1.631 -0.842,-3.174l0,-12.382l-3.157,0l0,-1.859l3.157,0l0,-5.437l2.21,0l0,5.437l3.683,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M139.742,133.92c0.374,-0.772 0.883,-1.403 1.526,-1.894c0.643,-0.491 1.397,-0.853 2.262,-1.087c0.865,-0.233 1.824,-0.351 2.877,-0.351c0.794,0 1.59,0.076 2.385,0.228c0.795,0.152 1.508,0.439 2.14,0.859c0.631,0.421 1.146,1.012 1.543,1.772c0.397,0.76 0.596,1.748 0.596,2.964l0,9.611c0,0.888 0.433,1.332 1.298,1.332c0.257,0 0.491,-0.047 0.701,-0.14l0,1.859c-0.257,0.047 -0.486,0.082 -0.684,0.105c-0.199,0.024 -0.45,0.035 -0.754,0.035c-0.561,0 -1.011,-0.075 -1.35,-0.228c-0.34,-0.152 -0.602,-0.369 -0.79,-0.649c-0.187,-0.281 -0.31,-0.614 -0.368,-1c-0.059,-0.386 -0.088,-0.812 -0.088,-1.281l-0.07,0c-0.398,0.585 -0.801,1.105 -1.21,1.561c-0.409,0.456 -0.865,0.837 -1.368,1.14c-0.503,0.304 -1.076,0.538 -1.719,0.701c-0.643,0.163 -1.409,0.246 -2.297,0.246c-0.842,0 -1.631,-0.1 -2.367,-0.298c-0.737,-0.198 -1.38,-0.514 -1.929,-0.947c-0.55,-0.432 -0.982,-0.982 -1.298,-1.649c-0.316,-0.666 -0.474,-1.455 -0.474,-2.367c0,-1.263 0.281,-2.251 0.842,-2.964c0.562,-0.713 1.303,-1.256 2.227,-1.631c0.924,-0.374 1.965,-0.637 3.122,-0.789c1.157,-0.152 2.333,-0.298 3.525,-0.438c0.467,-0.047 0.877,-0.105 1.228,-0.175c0.351,-0.07 0.643,-0.193 0.877,-0.368c0.233,-0.175 0.414,-0.415 0.543,-0.719c0.128,-0.304 0.193,-0.702 0.193,-1.193c0,-0.748 -0.123,-1.362 -0.369,-1.841c-0.246,-0.479 -0.585,-0.86 -1.017,-1.14c-0.433,-0.281 -0.936,-0.474 -1.508,-0.579c-0.573,-0.105 -1.188,-0.158 -1.842,-0.158c-1.403,0 -2.549,0.333 -3.437,1c-0.889,0.666 -1.356,1.736 -1.404,3.209l-2.21,0c0.071,-1.052 0.293,-1.964 0.668,-2.736Zm11.049,5.402c-0.141,0.258 -0.41,0.445 -0.807,0.561c-0.397,0.117 -0.748,0.199 -1.052,0.245c-0.936,0.164 -1.9,0.311 -2.894,0.439c-0.994,0.129 -1.9,0.322 -2.718,0.579c-0.819,0.258 -1.491,0.626 -2.017,1.105c-0.526,0.48 -0.789,1.163 -0.789,2.052c0,0.562 0.111,1.059 0.333,1.491c0.222,0.433 0.52,0.807 0.894,1.123c0.374,0.316 0.807,0.556 1.298,0.719c0.491,0.164 0.994,0.246 1.508,0.246c0.842,0 1.649,-0.128 2.42,-0.386c0.772,-0.257 1.444,-0.632 2.017,-1.123c0.572,-0.491 1.029,-1.087 1.368,-1.789c0.339,-0.702 0.509,-1.496 0.509,-2.385l0,-2.876l-0.07,0l0,-0.001Z" style="fill:#35495c;fill-rule:nonzero;"/><rect x="157.525" y="124.169" width="2.21" height="25.044" style="fill:#35495c;fill-rule:nonzero;"/><path d="M181.061,149.213l-3.788,0l0,-2.455l-0.07,0c-0.538,1.052 -1.321,1.806 -2.35,2.262c-1.029,0.456 -2.117,0.684 -3.262,0.684c-1.427,0 -2.672,-0.251 -3.736,-0.754c-1.064,-0.502 -1.947,-1.186 -2.648,-2.052c-0.702,-0.865 -1.228,-1.888 -1.578,-3.069c-0.351,-1.18 -0.526,-2.449 -0.526,-3.806c0,-1.636 0.222,-3.052 0.666,-4.244c0.444,-1.193 1.035,-2.175 1.772,-2.946c0.737,-0.772 1.578,-1.338 2.525,-1.701c0.947,-0.362 1.912,-0.544 2.894,-0.544c0.562,0 1.134,0.052 1.719,0.158c0.584,0.105 1.145,0.275 1.683,0.509c0.538,0.234 1.035,0.532 1.491,0.894c0.456,0.363 0.836,0.789 1.14,1.28l0.071,0l0,-9.26l3.998,0l0,25.044l-0.001,0Zm-13.96,-8.874c0,0.772 0.099,1.532 0.298,2.28c0.198,0.748 0.503,1.415 0.912,1.999c0.409,0.585 0.93,1.053 1.561,1.404c0.632,0.35 1.379,0.526 2.245,0.526c0.888,0 1.654,-0.187 2.298,-0.561c0.643,-0.374 1.169,-0.865 1.578,-1.473c0.409,-0.608 0.713,-1.292 0.912,-2.052c0.198,-0.76 0.298,-1.538 0.298,-2.333c0,-2.01 -0.45,-3.577 -1.351,-4.7c-0.901,-1.123 -2.122,-1.684 -3.665,-1.684c-0.936,0 -1.725,0.193 -2.368,0.579c-0.644,0.386 -1.169,0.889 -1.578,1.508c-0.409,0.62 -0.701,1.321 -0.877,2.105c-0.175,0.783 -0.263,1.584 -0.263,2.402Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M193.724,149.704c-1.45,0 -2.742,-0.24 -3.876,-0.719c-1.135,-0.479 -2.093,-1.14 -2.876,-1.982c-0.784,-0.842 -1.38,-1.847 -1.789,-3.017c-0.409,-1.169 -0.613,-2.455 -0.613,-3.859c0,-1.379 0.204,-2.653 0.613,-3.823c0.409,-1.169 1.006,-2.175 1.789,-3.017c0.783,-0.842 1.742,-1.502 2.876,-1.982c1.134,-0.479 2.426,-0.719 3.876,-0.719c1.45,0 2.742,0.24 3.876,0.719c1.134,0.48 2.093,1.14 2.877,1.982c0.783,0.842 1.379,1.848 1.789,3.017c0.409,1.17 0.614,2.444 0.614,3.823c0,1.404 -0.205,2.69 -0.614,3.859c-0.409,1.17 -1.006,2.175 -1.789,3.017c-0.784,0.842 -1.743,1.503 -2.877,1.982c-1.134,0.479 -2.427,0.719 -3.876,0.719Zm0,-3.157c0.888,0 1.66,-0.187 2.315,-0.561c0.655,-0.374 1.193,-0.865 1.614,-1.473c0.421,-0.608 0.731,-1.292 0.93,-2.052c0.198,-0.76 0.298,-1.538 0.298,-2.333c0,-0.771 -0.1,-1.543 -0.298,-2.315c-0.199,-0.772 -0.509,-1.456 -0.93,-2.052c-0.421,-0.596 -0.959,-1.081 -1.614,-1.456c-0.655,-0.374 -1.427,-0.561 -2.315,-0.561c-0.889,0 -1.66,0.188 -2.315,0.561c-0.655,0.374 -1.192,0.859 -1.613,1.456c-0.421,0.596 -0.731,1.28 -0.93,2.052c-0.199,0.772 -0.299,1.543 -0.299,2.315c0,0.795 0.1,1.573 0.299,2.333c0.198,0.76 0.509,1.444 0.93,2.052c0.421,0.608 0.958,1.099 1.613,1.473c0.654,0.375 1.426,0.561 2.315,0.561Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M204.562,131.079l4.244,0l3.543,13.539l0.071,0l3.402,-13.539l4.034,0l3.262,13.539l0.071,0l3.683,-13.539l4.069,0l-5.682,18.134l-4.104,0l-3.367,-13.469l-0.07,0l-3.332,13.469l-4.209,0l-5.615,-18.134Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M233.499,131.079l3.788,0l0,2.666l0.07,0.07c0.608,-1.005 1.403,-1.795 2.385,-2.368c0.982,-0.573 2.069,-0.859 3.262,-0.859c1.987,0 3.554,0.515 4.7,1.543c1.146,1.029 1.719,2.573 1.719,4.63l0,12.452l-3.998,0l0,-11.4c-0.047,-1.426 -0.351,-2.461 -0.912,-3.104c-0.562,-0.643 -1.438,-0.965 -2.631,-0.965c-0.678,0 -1.286,0.123 -1.824,0.369c-0.538,0.245 -0.994,0.585 -1.368,1.017c-0.374,0.433 -0.666,0.942 -0.877,1.526c-0.211,0.585 -0.316,1.205 -0.316,1.859l0,10.698l-3.998,0l0,-18.134Z" style="fill:#35495c;fill-rule:nonzero;"/><rect x="253.738" y="124.169" width="3.999" height="25.044" style="fill:#35495c;fill-rule:nonzero;"/><path d="M270.399,149.704c-1.45,0 -2.742,-0.24 -3.876,-0.719c-1.135,-0.479 -2.093,-1.14 -2.876,-1.982c-0.784,-0.842 -1.38,-1.847 -1.789,-3.017c-0.409,-1.169 -0.613,-2.455 -0.613,-3.859c0,-1.379 0.204,-2.653 0.613,-3.823c0.409,-1.169 1.006,-2.175 1.789,-3.017c0.783,-0.842 1.742,-1.502 2.876,-1.982c1.134,-0.479 2.426,-0.719 3.876,-0.719c1.45,0 2.742,0.24 3.876,0.719c1.134,0.48 2.093,1.14 2.877,1.982c0.783,0.842 1.379,1.848 1.789,3.017c0.409,1.17 0.614,2.444 0.614,3.823c0,1.404 -0.205,2.69 -0.614,3.859c-0.409,1.17 -1.006,2.175 -1.789,3.017c-0.784,0.842 -1.743,1.503 -2.877,1.982c-1.135,0.479 -2.427,0.719 -3.876,0.719Zm0,-3.157c0.888,0 1.66,-0.187 2.315,-0.561c0.655,-0.374 1.193,-0.865 1.614,-1.473c0.421,-0.608 0.731,-1.292 0.93,-2.052c0.198,-0.76 0.298,-1.538 0.298,-2.333c0,-0.771 -0.099,-1.543 -0.298,-2.315c-0.199,-0.772 -0.509,-1.456 -0.93,-2.052c-0.421,-0.596 -0.959,-1.081 -1.614,-1.456c-0.655,-0.374 -1.427,-0.561 -2.315,-0.561c-0.889,0 -1.66,0.188 -2.315,0.561c-0.655,0.374 -1.192,0.859 -1.613,1.456c-0.421,0.596 -0.731,1.28 -0.93,2.052c-0.199,0.772 -0.299,1.543 -0.299,2.315c0,0.795 0.1,1.573 0.299,2.333c0.198,0.76 0.509,1.444 0.93,2.052c0.421,0.608 0.958,1.099 1.613,1.473c0.654,0.375 1.426,0.561 2.315,0.561Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M298.073,145.18c0,0.491 0.064,0.842 0.193,1.052c0.128,0.21 0.38,0.316 0.754,0.316c0.117,0 0.257,0 0.421,0c0.163,0 0.35,-0.023 0.561,-0.07l0,2.771c-0.14,0.047 -0.321,0.099 -0.543,0.158c-0.223,0.058 -0.451,0.111 -0.684,0.158c-0.234,0.047 -0.468,0.081 -0.702,0.105c-0.234,0.024 -0.433,0.035 -0.596,0.035c-0.818,0 -1.497,-0.164 -2.034,-0.491c-0.538,-0.327 -0.889,-0.9 -1.052,-1.719c-0.795,0.771 -1.772,1.333 -2.929,1.683c-1.158,0.351 -2.274,0.526 -3.349,0.526c-0.819,0 -1.602,-0.111 -2.35,-0.333c-0.749,-0.222 -1.409,-0.549 -1.982,-0.982c-0.573,-0.433 -1.029,-0.982 -1.368,-1.649c-0.339,-0.666 -0.509,-1.444 -0.509,-2.333c0,-1.122 0.205,-2.034 0.614,-2.736c0.409,-0.701 0.947,-1.251 1.614,-1.649c0.666,-0.397 1.414,-0.684 2.244,-0.859c0.83,-0.175 1.666,-0.31 2.508,-0.404c0.725,-0.14 1.415,-0.239 2.069,-0.298c0.655,-0.058 1.233,-0.158 1.736,-0.298c0.503,-0.14 0.9,-0.356 1.193,-0.649c0.292,-0.292 0.439,-0.73 0.439,-1.315c0,-0.514 -0.123,-0.935 -0.369,-1.263c-0.246,-0.327 -0.55,-0.579 -0.912,-0.754c-0.363,-0.175 -0.766,-0.292 -1.21,-0.351c-0.445,-0.058 -0.865,-0.088 -1.263,-0.088c-1.123,0 -2.047,0.234 -2.771,0.702c-0.725,0.468 -1.135,1.193 -1.228,2.175l-3.999,0c0.071,-1.169 0.351,-2.14 0.842,-2.911c0.491,-0.772 1.116,-1.391 1.877,-1.859c0.759,-0.467 1.618,-0.795 2.578,-0.982c0.959,-0.187 1.941,-0.281 2.947,-0.281c0.888,0 1.765,0.094 2.631,0.281c0.865,0.188 1.643,0.491 2.333,0.912c0.689,0.421 1.245,0.965 1.666,1.631c0.421,0.666 0.631,1.479 0.631,2.438l0,9.331l-0.001,0Zm-3.998,-5.052c-0.609,0.398 -1.357,0.638 -2.245,0.719c-0.889,0.082 -1.777,0.205 -2.666,0.369c-0.421,0.07 -0.83,0.17 -1.228,0.298c-0.398,0.129 -0.749,0.304 -1.053,0.526c-0.304,0.223 -0.543,0.515 -0.719,0.877c-0.175,0.363 -0.263,0.801 -0.263,1.315c0,0.445 0.128,0.819 0.386,1.123c0.257,0.304 0.566,0.543 0.929,0.719c0.363,0.175 0.76,0.298 1.193,0.368c0.432,0.071 0.824,0.105 1.175,0.105c0.444,0 0.923,-0.058 1.438,-0.175c0.515,-0.117 1,-0.316 1.456,-0.596c0.456,-0.28 0.835,-0.636 1.14,-1.07c0.304,-0.432 0.456,-0.964 0.456,-1.596l0,-2.982l0.001,0Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M319.47,149.213l-3.788,0l0,-2.455l-0.07,0c-0.538,1.052 -1.321,1.806 -2.35,2.262c-1.029,0.456 -2.117,0.684 -3.262,0.684c-1.427,0 -2.672,-0.251 -3.736,-0.754c-1.064,-0.502 -1.947,-1.186 -2.648,-2.052c-0.702,-0.865 -1.228,-1.888 -1.578,-3.069c-0.351,-1.18 -0.526,-2.449 -0.526,-3.806c0,-1.636 0.222,-3.052 0.666,-4.244c0.444,-1.193 1.035,-2.175 1.772,-2.946c0.737,-0.772 1.578,-1.338 2.525,-1.701c0.947,-0.362 1.912,-0.544 2.894,-0.544c0.562,0 1.134,0.052 1.719,0.158c0.584,0.105 1.145,0.275 1.683,0.509c0.538,0.234 1.035,0.532 1.491,0.894c0.456,0.363 0.836,0.789 1.14,1.28l0.071,0l0,-9.26l3.998,0l0,25.044l-0.001,0Zm-13.96,-8.874c0,0.772 0.099,1.532 0.298,2.28c0.198,0.748 0.503,1.415 0.912,1.999c0.409,0.585 0.93,1.053 1.561,1.404c0.632,0.35 1.379,0.526 2.245,0.526c0.888,0 1.654,-0.187 2.298,-0.561c0.643,-0.374 1.169,-0.865 1.578,-1.473c0.409,-0.608 0.713,-1.292 0.912,-2.052c0.198,-0.76 0.298,-1.538 0.298,-2.333c0,-2.01 -0.45,-3.577 -1.351,-4.7c-0.901,-1.123 -2.122,-1.684 -3.665,-1.684c-0.936,0 -1.725,0.193 -2.368,0.579c-0.644,0.386 -1.169,0.889 -1.578,1.508c-0.409,0.62 -0.701,1.321 -0.877,2.105c-0.176,0.783 -0.263,1.584 -0.263,2.402Z" style="fill:#35495c;fill-rule:nonzero;"/><path d="M326.836,143.39c0.117,1.17 0.561,1.988 1.333,2.456c0.772,0.468 1.695,0.701 2.771,0.701c0.373,0 0.801,-0.029 1.28,-0.088c0.479,-0.058 0.93,-0.169 1.351,-0.333c0.421,-0.163 0.765,-0.403 1.034,-0.719c0.269,-0.316 0.392,-0.731 0.369,-1.245c-0.024,-0.514 -0.211,-0.935 -0.561,-1.263c-0.351,-0.327 -0.801,-0.59 -1.351,-0.789c-0.55,-0.198 -1.175,-0.369 -1.876,-0.509c-0.702,-0.14 -1.415,-0.292 -2.14,-0.456c-0.749,-0.163 -1.468,-0.362 -2.157,-0.596c-0.69,-0.234 -1.31,-0.549 -1.859,-0.947c-0.55,-0.397 -0.988,-0.906 -1.315,-1.526c-0.328,-0.619 -0.491,-1.386 -0.491,-2.298c0,-0.982 0.24,-1.806 0.719,-2.473c0.479,-0.667 1.087,-1.204 1.824,-1.613c0.737,-0.409 1.555,-0.696 2.455,-0.859c0.9,-0.163 1.76,-0.245 2.578,-0.245c0.935,0 1.83,0.1 2.684,0.298c0.853,0.199 1.625,0.521 2.315,0.965c0.689,0.445 1.263,1.023 1.719,1.736c0.456,0.714 0.742,1.573 0.86,2.578l-4.174,0c-0.187,-0.958 -0.626,-1.601 -1.315,-1.929c-0.69,-0.327 -1.479,-0.491 -2.367,-0.491c-0.281,0 -0.614,0.024 -1,0.07c-0.386,0.047 -0.749,0.135 -1.087,0.263c-0.339,0.129 -0.626,0.316 -0.86,0.561c-0.234,0.245 -0.35,0.567 -0.35,0.965c0,0.491 0.169,0.889 0.508,1.193c0.339,0.304 0.784,0.556 1.333,0.754c0.549,0.199 1.175,0.368 1.877,0.508c0.701,0.14 1.426,0.293 2.174,0.456c0.725,0.164 1.438,0.363 2.14,0.596c0.701,0.234 1.326,0.55 1.876,0.947c0.549,0.397 0.994,0.9 1.333,1.508c0.339,0.609 0.509,1.356 0.509,2.245c0,1.076 -0.246,1.988 -0.737,2.736c-0.492,0.749 -1.129,1.357 -1.912,1.824c-0.784,0.468 -1.655,0.807 -2.613,1.017c-0.959,0.21 -1.906,0.316 -2.841,0.316c-1.146,0 -2.204,-0.129 -3.174,-0.386c-0.971,-0.257 -1.813,-0.649 -2.526,-1.175c-0.714,-0.526 -1.275,-1.18 -1.683,-1.964c-0.41,-0.784 -0.626,-1.713 -0.649,-2.789l3.996,0Z" style="fill:#35495c;fill-rule:nonzero;"/></g></svg>                                                                                                                                                         plugins/wordpress-seo-extended/packages/js/images/elementor-logo.svg                                0000644                 00000013524 15122266557 0021767 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" width="170" height="70" viewBox="0 0 810 160" fill="none"><g clip-path="url(#clip0)"><path d="M505.95 73.9793C505.95 73.9793 499.077 75.6221 493.262 77.0086L484.423 79.0242C484.4 79.0242 484.365 79.0242 484.342 79.0242C484.342 76.6357 484.516 74.1191 485.086 71.8005C485.819 68.8294 487.424 65.3574 490.215 63.8311C493.25 62.1766 496.949 61.9785 500.159 63.2602C503.485 64.5768 505.008 67.7925 505.66 71.1247C505.846 72.0568 505.962 72.9889 506.055 73.9327L505.95 73.9793ZM529.001 79.0242C529.001 55.9199 514.475 45.9931 495.914 45.9931C474.933 45.9931 461.791 60.5454 461.791 79.1408C461.791 99.3556 472.979 112.521 497.065 112.521C510.091 112.521 517.464 110.214 526.233 105.822L522.895 90.6871C516.208 93.6931 509.986 95.534 501.682 95.534C492.576 95.534 487.389 92.0736 485.423 85.6072H528.42C528.768 83.8711 529.001 81.9021 529.001 79.0242Z" fill="#92003B"></path><path d="M312.998 73.9793C312.998 73.9793 306.124 75.6221 300.309 77.0086L291.471 79.0242C291.447 79.0242 291.412 79.0242 291.389 79.0242C291.389 76.6357 291.564 74.1191 292.133 71.8005C292.866 68.8294 294.471 65.3574 297.262 63.8311C300.298 62.1766 303.996 61.9785 307.206 63.2602C310.532 64.5768 312.056 67.7925 312.707 71.1247C312.893 72.0568 313.009 72.9889 313.102 73.9327L312.998 73.9793ZM336.048 79.0242C336.048 55.9199 321.522 45.9931 302.961 45.9931C281.981 45.9931 268.839 60.5454 268.839 79.1408C268.839 99.3556 280.027 112.521 304.112 112.521C317.138 112.521 324.511 110.214 333.28 105.822L329.943 90.6871C323.255 93.6931 317.033 95.534 308.729 95.534C299.623 95.534 294.436 92.0736 292.471 85.6072H335.467C335.816 83.8711 336.048 81.9021 336.048 79.0242Z" fill="#92003B"></path><path d="M259.814 29.0174H238.519V110.203H259.814V29.0174Z" fill="#92003B"></path><path d="M533.49 48.067H555.854L560.564 62.4213C563.507 55.3374 570.136 46.2261 581.894 46.2261C598.036 46.2261 606.793 54.4286 606.793 75.5638V110.214H584.429C584.429 102.991 584.441 95.7787 584.452 88.5549C584.452 85.246 584.394 81.937 584.441 78.6281C584.475 75.5755 584.696 72.418 583.068 69.68C581.963 67.8274 580.161 66.4642 578.219 65.5322C574.276 63.6447 569.996 63.6913 566.123 65.7186C565.17 66.2196 560.564 68.7129 560.564 69.8664V110.214H538.2V64.8331L533.49 48.067Z" fill="#92003B"></path><path d="M623.704 64.3554H613.446V48.067H623.704V37.8839L646.068 32.6059V48.067H668.549V64.3554H646.068V82.6012C646.068 89.7666 649.523 93.1105 654.71 93.1105C660.013 93.1105 663.013 92.4115 667.502 90.9201L670.154 107.779C664.048 110.436 656.442 111.706 648.708 111.706C632.45 111.706 623.692 103.969 623.692 88.9511V64.3554H623.704Z" fill="#92003B"></path><path d="M710.394 94.031C718.581 94.031 723.419 88.1355 723.419 78.6748C723.419 69.214 718.814 63.7729 710.743 63.7729C702.555 63.7729 697.834 69.2023 697.834 79.0126C697.834 88.2637 702.439 94.031 710.394 94.031ZM710.627 45.6436C731.607 45.6436 746.935 58.8095 746.935 79.2457C746.935 99.7984 731.607 112.16 710.394 112.16C689.297 112.16 674.306 99.4605 674.306 79.2457C674.306 58.8095 689.181 45.6436 710.627 45.6436Z" fill="#92003B"></path><path d="M441.754 47.8807C437.8 46.2496 433.322 45.6437 429.042 46.2612C426.867 46.5758 424.739 47.205 422.762 48.1837C417.342 50.8634 413.121 56.9803 410.853 62.4331C409.364 56.1531 405.026 50.5139 398.688 47.8924C394.734 46.2612 390.256 45.6554 385.976 46.2729C383.802 46.5875 381.673 47.2166 379.696 48.1953C374.288 50.8634 370.078 56.957 367.799 62.3981V62.002L363.24 48.0905H340.875L345.586 64.8565V110.226H367.799V69.6685C367.88 69.3655 368.869 68.8063 369.043 68.6781C371.648 66.8256 374.707 64.9148 377.963 64.6818C381.29 64.4371 384.569 66.1265 386.546 68.7713C386.756 69.0626 386.953 69.3539 387.139 69.6568C388.768 72.3949 388.547 75.5523 388.512 78.6049C388.477 81.9139 388.523 85.2228 388.523 88.5318C388.512 95.7555 388.5 102.968 388.5 110.191H410.865V75.5523C410.865 75.3776 410.865 75.2028 410.865 75.0397V69.6918C410.911 69.4005 411.935 68.8179 412.121 68.6898C414.726 66.8372 417.784 64.9264 421.041 64.6934C424.367 64.4487 427.647 66.1382 429.624 68.783C429.833 69.0743 430.031 69.3655 430.217 69.6685C431.845 72.4065 431.624 75.564 431.589 78.6166C431.554 81.9255 431.601 85.2345 431.601 88.5434C431.589 95.7671 431.578 102.979 431.578 110.203H453.942V75.5523C453.942 65.3925 452.488 52.3082 441.754 47.8807Z" fill="#92003B"></path><path d="M799.827 46.2261C788.069 46.2261 781.452 55.349 778.498 62.4213L773.787 48.067H751.423L756.133 64.8331V110.203H778.498V68.2818C781.684 67.7226 798.978 70.915 802.258 72.1151V46.3077C801.467 46.2611 800.664 46.2261 799.827 46.2261Z" fill="#92003B"></path><path d="M206.444 73.9793C206.444 73.9793 199.571 75.6221 193.756 77.0086L184.917 79.0242C184.894 79.0242 184.859 79.0242 184.836 79.0242C184.836 76.6357 185.01 74.1191 185.58 71.8005C186.313 68.8294 187.918 65.3574 190.709 63.8311C193.744 62.1766 197.443 61.9785 200.653 63.2602C203.979 64.5768 205.502 67.7925 206.154 71.1247C206.34 72.0568 206.456 72.9889 206.549 73.9327L206.444 73.9793ZM229.495 79.0242C229.495 55.9199 214.969 45.9931 196.408 45.9931C175.427 45.9931 162.285 60.5454 162.285 79.1408C162.285 99.3556 173.473 112.521 197.559 112.521C210.585 112.521 217.958 110.214 226.727 105.822L223.389 90.6871C216.702 93.6931 210.48 95.534 202.176 95.534C193.07 95.534 187.883 92.0736 185.917 85.6072H228.913C229.262 83.8711 229.495 81.9021 229.495 79.0242Z" fill="#92003B"></path><path d="M66.141 16.0495C30.8557 16.0495 2.25763 44.6998 2.25763 80.0495C2.25763 115.388 30.8557 144.05 66.141 144.05C101.426 144.05 130.024 115.399 130.024 80.0495C130.013 44.6998 101.415 16.0495 66.141 16.0495ZM50.1731 106.707H39.5316V53.3799H50.1731V106.707ZM92.7503 106.707H60.8145V96.0466H92.7503V106.707ZM92.7503 85.3741H60.8145V74.7133H92.7503V85.3741ZM92.7503 64.0408H60.8145V53.3799H92.7503V64.0408Z" fill="#92003B"></path></g><defs><clipPath id="clip0"><rect width="795.828" height="127.901" fill="white" transform="translate(2.25763 16.0495)"></rect></clipPath></defs></svg>                                                                                                                                                                            plugins/wordpress-seo-extended/packages/js/images/g2_logo_white_optm.svg                            0000644                 00000001503 15122266557 0022620 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1000 1000"><path fill="#fff" d="M500 0C223.9 0 0 223.9 0 500s223.9 500 500 500 500-223.9 500-500S776.1 0 500 0Zm87.2 412.4c0-21.9 4.3-40.2 13.1-54.4s24-27.1 45.9-38.2l10.1-4.9c17.8-9 22.4-16.7 22.4-26 0-11.1-9.5-19.1-25-19.1-18.3 0-32.2 9.5-41.8 28.9l-24.7-24.8c5.4-11.6 14.1-20.9 25.8-28.1a70.8 70.8 0 0 1 38.9-11.1c17.8 0 33.3 4.6 45.9 14.2s19.4 22.7 19.4 39.4c0 26.6-15 42.9-43.1 57.3l-15.7 8c-16.8 8.5-25.1 16-27.4 29.4h85.4v35.4H587.2Zm-82.1 373.3c-157.8 0-285.7-127.9-285.7-285.7s127.9-285.7 285.7-285.7a286.4 286.4 0 0 1 55.9 5.5l-55.9 116.9c-90 0-163.3 73.3-163.3 163.3s73.3 163.3 163.3 163.3a162.8 162.8 0 0 0 106.4-39.6l61.8 107.2a283.9 283.9 0 0 1-168.2 54.8ZM705 704.1l-70.7-122.5H492.9l70.7-122.4H705l70.7 122.4Z"/></svg>                                                                                                                                                                                             plugins/wordpress-seo-extended/packages/js/images/icon-arrow-down.svg                               0000644                 00000000406 15122266557 0022057 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg width="16" height="16" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" role="img" aria-hidden="true" focusable="false" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 14l-7 7m0 0l-7-7m7 7V3" /></svg>                                                                                                                                                                                                                                                          plugins/wordpress-seo-extended/packages/js/images/icon-facebook.svg                                 0000644                 00000000775 15122266557 0021542 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1600 1600" role="img" aria-hidden="true" focusable="false"><g fill="none" fill-rule="evenodd"><path fill="#1877f2" d="M1600 800a800 800 0 1 0-925 790v-559H472V800h203V624c0-201 119-311 302-311 88 0 179 15 179 15v197h-101c-99 0-130 62-130 125v150h222l-36 231H925v559a800 800 0 0 0 675-790"/><path fill="#fff" d="M1147 800H925V650c0-63 31-125 130-125h101V328s-91-15-179-15c-183 0-302 110-302 311v176H472v231h203v559a806 806 0 0 0 250 0v-559h186z"/></g></svg>   plugins/wordpress-seo-extended/packages/js/images/icon-twitter.svg                                  0000644                 00000001310 15122266557 0021455 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 400 400" role="img" aria-hidden="true" focusable="false"><circle cx="200" cy="200" r="200" fill="#1da1f2"/><path d="M163.4 305.5c88.7 0 137.2-73.5 137.2-137.2 0-2.1 0-4.2-.1-6.2 9.4-6.8 17.6-15.3 24.1-25-8.6 3.8-17.9 6.4-27.7 7.6 10-6 17.6-15.4 21.2-26.7-9.3 5.5-19.6 9.5-30.6 11.7-8.8-9.4-21.3-15.2-35.2-15.2-26.6 0-48.2 21.6-48.2 48.2 0 3.8.4 7.5 1.3 11-40.1-2-75.6-21.2-99.4-50.4-4.1 7.1-6.5 15.4-6.5 24.2 0 16.7 8.5 31.5 21.5 40.1-7.9-.2-15.3-2.4-21.8-6v.6c0 23.4 16.6 42.8 38.7 47.3-4 1.1-8.3 1.7-12.7 1.7-3.1 0-6.1-.3-9.1-.9 6.1 19.2 23.9 33.1 45 33.5-16.5 12.9-37.3 20.6-59.9 20.6-3.9 0-7.7-.2-11.5-.7 21.1 13.8 46.5 21.8 73.7 21.8" fill="#fff"/></svg>                                                                                                                                                                                                                                                                                                                        plugins/wordpress-seo-extended/packages/js/images/indexables_1_left_bubble_optm.svg                 0000644                 00000030602 15122266557 0024755 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 262.37 284.15"><path fill="#f0ecf0" d="M262.37 153A131.19 131.19 0 1 0 54.71 259.56a28.08 28.08 0 0 1 1.91-7.37c2-4.92-12.06-11.1-7.52-26.48 3.49-11.81 30.9-25.16 37.1-41.56 2.4-6.32 12-30.34 29.69-35.66 9-2.7 48.29-13.29 60.25-11.07s37.48 11 31.66 41.46c-10.26 53.79-44.8 59.12-46.28 94.64a27.46 27.46 0 0 0 .65 6.94A131.23 131.23 0 0 0 262.37 153Z"/><path fill="#006531" d="M161.52 273.52c1.44-35.53 36-40.85 46.28-94.64 5.82-30.51-19.7-39.25-31.66-41.46s-51.23 8.37-60.25 11.07c-17.72 5.32-27.29 29.34-29.69 35.66-6.2 16.4-33.61 29.75-37.1 41.56-4.54 15.38 9.53 21.56 7.52 26.48a28.08 28.08 0 0 0-1.91 7.37 131.71 131.71 0 0 0 12.68 8.05 184.18 184.18 0 0 1 7.18-34.05c-4.12-2.43-8.32-4-19.15-2.33 16.06-23.66 34.79-45.11 64.9-65.12 2.54 22.59 3.65 40.75 14.49 60-14.93-3.2-25.12 0-32 4.77a103.74 103.74 0 0 0-4.6 49.12 132 132 0 0 0 64 .49 27.23 27.23 0 0 1-.69-6.97Z"/><path fill="#89bc3b" d="M102.8 230.9c6.89-4.73 17.08-8 32-4.77-10.84-19.27-12-37.43-14.49-60-30.11 20-48.84 41.46-64.9 65.12 10.83-1.66 15-.1 19.15 2.33a184.18 184.18 0 0 0-7.18 34A130.42 130.42 0 0 0 98.21 280a103.65 103.65 0 0 1 4.59-49.1Z"/><path fill="#093f22" d="M61.29 184.66c-4.39 13.11-15 33.6-14.52 47.42 11.9-18 34.83-42.28 46.73-60.26 2.69-4.08 1.49-10.8-3.21-13.29-3.83-2-8.45-.15-11.7 1.92-8.3 5.28-14.21 14.98-17.3 24.21ZM133.9 243.31l18.32-60.92s.51 11.07 1.1 9.3 16.4-28.8 16.4-28.8l-28.14 81Z"/><path fill="#006531" d="M169.31 113.64c.71 13.17 6.24 22.36 5.36 25.68-33.67 6.42-49.16 29.46-54 28.88-6.67-3.85-5.76-14.39-5.54-17.71 3.55-6.65-.12-17.65 1.77-23.89 1.4-4.6 9.38-12.6 18.1-14.53s33.93-5.54 34.31 1.57Z"/><path fill="#b498c6" d="M122.32 166.11c6.09-9.75 11.1-30.31 14.62-33.9 3.33-3.39 22.1-12 26.12-13.51 1.56-1.22 2-2.55 2.09-6.65l.23-19.82-45.16 16.39 2.53 37.22c.67 5.09.35 10.08-.43 20.27Z"/><path fill="#8b6c9c" d="m166.15 92.89-.64 7.17c-.94 12.77-25.4 31.3-38.1 30.12-4.6-.42-5.78-10.38-7.39-15.06l-2.61-9.52Z"/><path fill="#b498c6" d="M157.71 192S128 247.74 126 254s-.23 13.67 4.49 15.24 16.51.6 24.27-7.47c5.9-6.14 28.74-39.14 29.45-39.85.06-13.3-7.49-22.14-26.5-29.92Z"/><ellipse cx="106.71" cy="192.91" fill="#fbb03b" stroke="#fc3" stroke-miterlimit="10" rx="8.8" ry="5.58" transform="rotate(-19.17 106.716 192.914)"/><path fill="#f19018" d="M92.72 206.13c4.25-1.18 9.78-1.68 17-1.26-6.39 3.89-15.24 15.48-15.51 17.65a87.16 87.16 0 0 1 11-1.88c-16 10.81-23.79 31.12-22.85 30.75-1.24.41-3.63-21.77 6.06-31.06a32.05 32.05 0 0 0-7.73 1.1c1.68-4.03 6.97-11.62 12.03-15.3Z"/><path fill="#fbb03b" stroke="#fc3" stroke-miterlimit="10" d="M92.72 206.13c4.25-1.18 9.78-1.68 17-1.26-6.39 3.89-15.24 15.48-15.51 17.65a87.16 87.16 0 0 1 11-1.88c-16 10.81-23.79 31.12-22.85 30.75-1.24.41-3.63-21.77 6.06-31.06a32.05 32.05 0 0 0-7.73 1.1c1.68-4.03 6.97-11.62 12.03-15.3Z"/><path fill="#006531" d="M183.45 223.15c-3.73-9.11-19.27-19.94-31.68-20.38 7.54-16.39 11.46-29.12 19.31-42.24 0 0-5.74-22.74-1-22.2 8.52 1 19-7.56 33.67-4.23 14.25 3.23 20.27 23.26 17 38-4.47 19.9-40.04 54.49-37.3 51.05Z"/><path fill="#8b6c9c" d="M80 254.38C77.79 252.17 71.81 244 65.6 244s-33 6.87-33 6.87l-.45 9.3s16.69 1.77 19.94 2.22 16-3.55 16-3.55Z"/><path fill="#8b6c9c" d="M72.14 260.36c-12.31-.11-23.45 6.28-26.55 5.39-4.1-1.33-4.21.82-6.64.59s-6.7-3.65-6.7-3.65l.12-12.3 39 7.09Z"/><path fill="#8b6c9c" d="M50.63 262.9c-4.43-2.21-15.73 4.28-17.74 2.71-4.64-3.65-8.27-6.21-9.3-8.42s2.47-5.91 1.59-16.1c-.44-5.07 4.31-5.32 6.09-1.33.93 2.1 1.33 13.51 1.33 13.51l7.63 2.1Z"/><path fill="#b498c6" d="M100.38 163.12c.89 4-22.61 60.35-31.91 76.3S40.9 257.36 40.9 257.36s-14.48-.79-9.17-22.05 24.47-27.82 31.55-37.42 20.27-29 20.27-33.45c0-3.1 15.95-5.31 16.83-1.32Z"/><path fill="#8b6c9c" d="M100.48 163.09a3.78 3.78 0 0 1 0 .95 9.1 9.1 0 0 1-.16.91c-.12.6-.29 1.19-.44 1.78-.33 1.17-.69 2.34-1.06 3.5-.76 2.32-1.56 4.62-2.38 6.92Q94 184 91.28 190.83q-5.4 13.59-11.26 27-2.92 6.71-6 13.34A63.91 63.91 0 0 1 66.55 244 35.8 35.8 0 0 1 55 253.26a38.85 38.85 0 0 1-14.07 4.2v-.2a41.07 41.07 0 0 0 6.94-2.09 39.27 39.27 0 0 0 6.44-3.17A35.56 35.56 0 0 0 65 242.71a26.19 26.19 0 0 0 2-2.9c.66-1 1.22-2.05 1.8-3.1q1.74-3.18 3.36-6.44c2.2-4.33 4.31-8.71 6.37-13.12q6.14-13.23 11.79-26.7 2.85-6.72 5.53-13.51c.88-2.27 1.74-4.55 2.56-6.84.4-1.14.79-2.29 1.15-3.46.17-.58.35-1.16.48-1.75a6.54 6.54 0 0 0 .18-.88 3.19 3.19 0 0 0 .06-.87Z"/><path fill="#8b6c9c" d="M98.61 166.22s10.4-10 11.73-14c1.78-5.31-.21-15.41 2.56-22.17 1.13-2.77 5-12.4 5.3-13.28s-3.13-2.36-4.76-1.77a17.06 17.06 0 0 0-6.26 5.61c-1.33 2.22-.68 10.15-4.37 11.34s-18.23 10.61-18.38 17.4.44 11.85 14.18 16.87Z"/><path fill="#8b6c9c" d="m100.38 163.12 8.42-13.74-13.74-19s4.35-12 5.09-16.39 3.84-11.14.56-18.17c-2.12-4.55-4.79-5-5.09-2.1s-1.16 13.72-2.62 17.12-8.08 14.62-8.08 14.62l-12.4 14.62Z"/><path fill="#b498c6" d="M138.58 5.48c-20.7-.38-38 19.23-39.7 21.63-1.34 1.94-5.06 12-6.58 19.34.42-1.33.87-2.62 1.33-3.85 13.19-6.56 37.64-4.79 70.11 2.89 1.75 6.74 1.36 12.38 1.15 20.18 0 0-15.4 8.58-24.28 9.94-7.5 1.16-11.68-1.37-14.5-3.62-4-3.18-7.61-9.22-13.15-9.67-5.81-.47-7.83 9.3-10.72 9-1.73-.18-7.25.7-9.63 2.76a34.28 34.28 0 0 0-.24 9.16c1.25 5.91 5.63 11.61 8.16 14.52s4.19 11.43 8.45 15.76c9.81 9.93 47-6.94 55.48-17.25 3.75-4.53 8.8-28 8.8-28s10.52-10.21 10.46-16.87c-.21-22.82-17.07-45.4-45.14-45.92Z"/><path fill="#fcee21" d="M92.41 51.14a78.25 78.25 0 0 1 .88-8.53C83.1 69 92.42 74.26 92.61 74.09a39.64 39.64 0 0 1 2.9-8.67s-3.36-7.58-3.1-14.28Z"/><path fill="#fbb03b" d="M102.24 71.33c2.89.29 4.91-9.48 10.72-9 5.54.45 9.14 6.49 13.15 9.67 2.82 2.25 7 4.78 14.5 3.62 8.88-1.36 24.28-9.94 24.28-9.94.21-7.8.6-13.44-1.15-20.18-32.47-7.69-56.92-9.5-70.11-2.9-.89 2-1.63 13.3-1.63 13.3-.3 5 .89 6.64 3.47 9.52a42 42 0 0 0-2.82 8.67c2.35-2.09 7.86-2.94 9.59-2.76Z"/><path fill="#89bc3b" d="M156 194.13c7.53-.89 21.71 6.87 26.81 12.62 14.42-12.32 25.39-37.37 27.91-45.19 6.64-20.6-11.52-28.35-11.52-28.35l2.44.66s18.38 5.76 14.84 31.9c-1.24 9.13-14.64 33.72-28.8 46.3.89 1.55 2.22 3.32 2.66 4.87l-6.65 6.43c-2.66-8.42-22.81-20.82-31.9-20.6Z"/><path fill="#b498c6" d="M86.65 168.87c-.89-1.32-17.89-25.57-18.34-28.67s4.69-7.7 6.13-7c9.64 4.61 24.21 17.67 26 22.1s-.71 11.59-.71 11.59ZM77.79 130.77s12.14-11.89 15.09-12 10.11 6 13.11 7.91c3.2 2 3.87 7.13 8.23 10.45.44 2.07-2.22 3.42-5.62 2.39s-7.34-6.81-7.34-6.81l-7.12-3.46-8.38 6.88Z"/><path fill="#b498c6" d="M74.24 133s11.93-9 13.4-9 12.38 7.59 13.42 9.22 1.88 8.59 3.75 14.86c.65 2.17-3.28 5.17-6.53 1.33s-4.77-10.32-4.77-10.32l-7.33-4.61-5.29 3.85Z"/><path fill="#8b6c9c" d="M84.58 124.87a4.75 4.75 0 0 1 3-1.15 7.83 7.83 0 0 1 3.22.82 21.82 21.82 0 0 1 2.69 1.76c.86.62 1.71 1.25 2.56 1.89a47.7 47.7 0 0 1 4.94 4.14 4.21 4.21 0 0 1 .55.71 5.87 5.87 0 0 1 .36.81c.18.53.33 1.05.46 1.57.26 1 .48 2.08.7 3.12.44 2.07.87 4.14 1.37 6.19.25 1 .53 2 .82 3a2.42 2.42 0 0 1 .12 1 2.36 2.36 0 0 1-.26.93 3.58 3.58 0 0 1-1.21 1.34 4 4 0 0 1-3.47.47 5.93 5.93 0 0 1-2.73-2.06 12.37 12.37 0 0 1-2.52-5.93 36.66 36.66 0 0 0 3.3 5.3 5 5 0 0 0 2.26 1.74 3 3 0 0 0 2.6-.35 1.93 1.93 0 0 0 1-2.16c-.3-1-.58-2.06-.83-3.09-.51-2.07-.94-4.15-1.38-6.22-.22-1-.44-2.07-.69-3.08-.13-.51-.27-1-.43-1.49a2.76 2.76 0 0 0-.7-1.17 49.75 49.75 0 0 0-4.81-4c-.84-.63-1.69-1.26-2.55-1.88s-1.75-1.16-2.57-1.8a6.81 6.81 0 0 0-2.78-1.11 4.77 4.77 0 0 0-3.02.7Z"/><path fill="#b498c6" d="M72.62 133.87s5-4.31 6.43-4.66c2.52-.63 12.51 5.9 13.36 7.77 1.31 2.91 2.2 7.57 3.87 11.4.9 2.06-2.84 2.15-5.49.53s-3.57-6.71-3.57-6.71c-2.51-1.89-4.82-2.52-6.54-3.69-.3 2.47-2.6 5.07-7 8.07Z"/><path fill="#8b6c9c" d="M79.05 129.21a5.63 5.63 0 0 1 2.7.27c.88.26 1.73.57 2.58.91a28.34 28.34 0 0 1 4.81 2.54c.77.48 1.53 1 2.27 1.52a12.53 12.53 0 0 1 2.14 1.84 2.76 2.76 0 0 1 .45.71c.09.21.17.44.24.66a12.73 12.73 0 0 1 .38 1.34c.21.9.37 1.8.52 2.68a38.28 38.28 0 0 0 1.15 5.2c.13.41.29.82.46 1.22a2.2 2.2 0 0 1 .2.84 1.33 1.33 0 0 1-.37 1 2.48 2.48 0 0 1-1.56.62 6 6 0 0 1-1.47-.08 8.62 8.62 0 0 1-2.69-.94 5.84 5.84 0 0 1-2.1-2 15.27 15.27 0 0 1-2-5.2l.19.31a9.76 9.76 0 0 0-2.57-1.41 12.13 12.13 0 0 1-3-1.25l.74-.19a14 14 0 0 1-2.58 2.75 8.41 8.41 0 0 0-2.18 2.84 8.13 8.13 0 0 1 1.78-3.21 14.17 14.17 0 0 0 2.08-2.85l.23-.52.5.33a5.89 5.89 0 0 0 1.27.66l1.46.5a10.54 10.54 0 0 1 2.83 1.55l.15.11v.2a14.16 14.16 0 0 0 1.83 4.84 4.89 4.89 0 0 0 1.75 1.69 7.75 7.75 0 0 0 2.38.83 5 5 0 0 0 1.22.06 1.54 1.54 0 0 0 .92-.31c.13-.13.14-.32 0-.72s-.35-.86-.49-1.3a36.1 36.1 0 0 1-1.18-5.34c-.15-.88-.31-1.76-.51-2.61a13 13 0 0 0-.35-1.26 3.72 3.72 0 0 0-.49-1 11.48 11.48 0 0 0-1.94-1.67c-.72-.52-1.46-1-2.22-1.48-1.5-1-3.1-1.75-4.62-2.68-.78-.44-1.57-.86-2.38-1.22a5.79 5.79 0 0 0-2.53-.78Z"/><path fill="#006531" d="M167.62 84.35c22.06 1.45 35.92 5.68 36.5-3.92S197.69 6.63 141.9.54C89-5.24 83.2 37.33 82.73 45s7.68 8.3 7.68 8.3c.22-4 2.07-7.82 3.22-10.71 15.18-4.88 64.17 1.48 67.52 6.22s.8 13 1.47 16.45 9.68 6.8 9.68 6.8Z"/><path fill="#093f22" d="M202.05 84.6c2.39-1.92-8.89-16.64-20.3-20.87l-6 7.31-.26 12.81c7.99 2.15 22.68 3.88 26.56.75Z"/><path fill="#b498c6" d="M180.18 72.06c3.68 2.46 4 8.69 2.45 12.56s-10.89 10.8-13.28 11.15a26.88 26.88 0 0 1-5.15-.2L171 71.7s5.51-2.1 9.18.36Z"/><path fill="#191a1a" d="M165.72 67.49s-13.58 37.88-20.45 41.87c1 1 7.06-1.52 9-3.51 3.55-2.44 16.85-34.8 16.85-34.8Z"/><path fill="#093f22" d="M136.35 21.41c-1.17-2.11 21.15-11.14 32.7-11.14 2.66 0 10.64 6.2 11.52 9.74-15.29-.23-43.05 3.5-44.22 1.4ZM141.17 32.51c-.92-3 31.73-6.14 43.24-3.93 6.28 1.2 8.82 13.26 4.14 11.81-12.21-3.78-46.78-5.97-47.38-7.88ZM99.94 23c2.07-.89.29-4.14-1.48-10.93a19.93 19.93 0 0 0-7.35 7.62s6.76 4.16 8.83 3.31ZM94.28 31.72c.18-1.29-5.85-5.57-6.48-4.62a17.56 17.56 0 0 0-2.94 8.78s9.23-2.88 9.42-4.16Z"/><path fill="#fcee21" d="M95.9 65.56c4.07.42 7.79 1.24 12.23-1 0 0-4-8.79-8.72-8s-3.51 9-3.51 9Z"/><path fill="#f15a24" d="M108.08 64.63c-5.34-9-13.34-9.9-12.12.92-.49.07-1.34-1.88-1.61-2.93-1.29-5.79 4.65-9.51 9.59-5.14 3.17 2.79 4.45 7.01 4.14 7.15Z"/><path fill="#f15a24" d="M106.93 52c1.27.52 2.88 4 1.65 3.5A18.34 18.34 0 0 0 92 57.38c.7-5.91 13-6.22 14.93-5.38Z"/><path fill="#8b6c9c" d="M121.12 71.07c.66 1.59 1.38 3.39-.41 4.53s-4.33-2-5.87-1.57-3.88 2.72-5.55 2.41c-3.52-.66-3.15-3.48-3.15-3.48s-1.61 5.55 4 5.62c3.81 0 4.23-2.58 7.44-2.27s3.46 2.53 4.89 1.61c2.88-1.82.31-6.69-1.35-6.85Z"/><path fill="#8b6c9c" d="M106.73 75.68s-2.65.74-1.19 2.49c-3.18 0-3.08-4.59-1.32-6-.75 2.66 2.51 3.51 2.51 3.51Z"/><ellipse cx="98.36" cy="62.94" fill="#f15a24" rx="3.53" ry="3.42" transform="rotate(-5.96 98.33 62.903)"/><path fill="#8b6c9c" d="M138.73 83.69c-8.66 7.53-14.4 12.67-22.62 13-9.3.32-7.2-7.2-7.2-7.2s.85-2.54 5.63-2.77 20.66-2.78 24.19-3.03Z"/><path fill="#4b3f51" d="M108.91 89.78c.34-.46 1.86-1.93 2.29-1 1.92 3.93 8.45 1.71 28.05-5.2-15 2.26-17.88.24-20.85.48s-3.62 1.67-4.72 1.45-3.43-1.68-4.73-1.1c-3.03 1.27-2.05 3.67-.04 5.37Z"/><path fill="#8b6c9c" d="M132.64 76.68c1.08 1.47 9.51 5.88 9.51 5.88a60.71 60.71 0 0 1-2.89 6.3c.41 0 1.39-5-.85-6.64-1.72-1.22-4.64-2.58-5.77-5.54Z"/><path fill="#b498c6" d="M65.82 248.62c-4.13-.44-11-7.38-15.72-6.64s-9.53 5.31-9.53 5.31-12.7-4.87-16.84-4-9.45 2.36-10.63 3.54-.59 5.91 2.22 5.76 6.05-1 9.3-.44 8.73 6.36 12.41 8c14.17 6.2 29.24.44 29.24.44s3.73-11.52-.45-11.97Z"/><path fill="#8b6c9c" d="M64.15 248c-2.62.11-5-1.72-6.28-2.59-3.86-2.56-4.62-2.62-19.34 1.45-5.79-2.69-4.72-1.58-7.88-2.74a18.46 18.46 0 0 1 8.14 1.76c-.64 0 13-5.16 16.5-3.92s6.23 5.3 8.86 6.04Z"/><path fill="#093f22" d="M90.49 53.46a87 87 0 0 1 3.14-10.86c19.79-3.78 64.81 2.3 67.52 6.22s1.23 13.54 1.18 16.09 8.72 6.72 8.72 6.72l4.57-.74s-8.51-2.28-8.66-6.71 1.56-16.78-3.22-18.69c-29.19-11.62-64.36-8.8-72.53-6.65-6.5 1.7-9.09 6.62-7.44 10.27s6.72 4.35 6.72 4.35ZM160.49 78.05l7.38 2.95-3.99 9.31-7.34-2.07 3.95-10.19z"/><path fill="#b498c6" d="M153.33 236.52c-10.63-4-31.44 3.41-44.53 5.89-29.24 5.53-37.33 5.55-44.09 6-5 .33-3.44 8.5-2.43 12.17 34.11 6.21 41.2 8.42 72 8.92 22.36-5.62 29.72-28.99 19.05-32.98Z"/><path fill="#8b6c9c" d="M131.58 242.4c.39-2 .09-3.41.48-5.52l2.21-.22a53.48 53.48 0 0 0-2.69 5.74Z"/><path fill="#fc3" d="M148.89 45c2 6.65 5.32 17.43 2.44 27.4-2.29 1-10 3.61-12.33 3.39.66-5.76.44-23.11-3.77-32.64 2.31.14 8.22 1.03 13.66 1.85Z"/><path fill="#fcee21" d="M145.22 62.31c-3.94 1.59-12.14 3.79-17.43 1.19 0 0-1.22-9.33 5.51-8.76s11.92 7.57 11.92 7.57Z"/><path fill="#f15a24" d="M127.69 63.52c-1.85-4-.92-8.41 4.11-9.48 4.24-.75 11.08 1.85 13.5 8.21-.29.22-7.6-8.06-14-6.15-4.44 1.33-3.13 7.35-3.61 7.42ZM122.88 45.74c-1.23.51.14 4 1.3 4.36s9.23-2.79 12.1-2.41 8.81 4.79 11.58 7c-.44-1.88-7.92-9.3-10.88-9.92s-12.87.45-14.1.97Z"/><ellipse cx="131.8" cy="60.56" fill="#f15a24" rx="3.76" ry="3.99" transform="rotate(-5.96 131.772 60.6)"/></svg>                                                                                                                              plugins/wordpress-seo-extended/packages/js/images/indexables_2_left_bubble_optm.svg                 0000644                 00000026045 15122266557 0024764 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg aria-hidden="true" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 289.98 291.69" class="yst-shrink-0 yst-h-28"><path fill="#006531" d="M178.09 168c6.2.88 11.73 57.74 10.9 64-1 7.61-20.47 21.56-35.83 16.83s-22.28-17.36-17.85-26.51 36.58-55.25 42.78-54.32z"></path><path fill="#f0ecf0" d="M256.47 217s6 6 13.08 13.82a131.19 131.19 0 10-181.07 40.45q3.55 2.25 7.23 4.27c.71-2.79 1.36-5 1.81-6.64 1.48-5.32 6.32-14.53 5.36-19.55s-4.13-7.68-3.54-20.08a114.5 114.5 0 013.54-25.72 14.88 14.88 0 00-7.09-1.77 113.17 113.17 0 0110.95-12c6.87-21.57 17-42 23.64-51.22 6.22-8.64 29.37-10.27 52.71-6.72s47.75 28.12 40.37 63-28.46 44.21-30.79 49.23-5.57 18.8-3.65 23.67c1.29 3.28 6.19 9.05 11.85 17.05a130.91 130.91 0 0048-28.94c-8.87-9.55-19.31-20.38-19.87-21.12 9.73-11.73 15.92-18.83 27.47-17.73z"></path><path fill="#006531" d="M150.14 131.06c-3.25-1.48-3.86-.12-7.4.48s-18.86-1.18-23.91.81c-9.9 3.91-14 11.82-16 20.27s.54 19.16 1.72 25.95-3.39 53.18-3.09 59.38 20.2 11.57 22.27.35 15.33-65.3 15.33-65.3 14.33-40.46 11.08-41.94z"></path><path fill="#b498c6" d="M49 177.58c26.5 32.94 70.46 51.81 71.35 58.6s-3.84 13.34-8.87 16.29-30.71-3.4-45.77-22.47c-15.55-19.67-28.56-37.13-28.56-37.13s5.68-8.18 11.85-15.29z"></path><path fill="#006531" d="M132.36 152.53c-4.21 2.95-21 27.42-26.52 32-4.18 3.48-14.36 15.36-15.25 23.63s6.39 19.91 16 25.4c5.24 3 23.88-5.3 25.65-13.27s4.32-70.72.12-67.76z"></path><path fill="#006531" d="M192.67 244.07c2.37-5 23.4-14.36 30.79-49.21s-17-59.44-40.37-63-46.49-1.86-52.71 6.74c-6.63 9.2-16.77 29.65-23.64 51.22 15.28-14.48 30.08-20.49 39.86-32.18-.3 1.55 8.49 23.45 11.81 28.36 4.15 6.13 14.16 17.89 20.38 31-10.34-1.92-21.49-1.6-31.9 2.41-2.66 6.3-2.07 16.93-.88 22.84-2.28 17.87-2.12 33.54.46 48.85 4.06.37 8.17.58 12.32.58a131.19 131.19 0 0042.08-6.9c-5.66-8-10.56-13.77-11.85-17-1.92-4.91 1.29-18.69 3.65-23.71zM112.56 231c-.27-7.26-.22-8.8.78-20.95-2.89-1.6-6.55-4.57-10.46-6.48a114.5 114.5 0 00-3.54 25.72c-.59 12.4 2.59 15.06 3.54 20.08S99 263.58 97.52 268.9c-.45 1.6-1.1 3.85-1.81 6.64a127.06 127.06 0 0013.85 6.58c-1.44-18.61-1.04-36.3 3-51.12z"></path><path fill="#b498c6" d="M269.55 230.83C262.43 223 256.47 217 256.47 217c-11.55-1.11-17.74 6-27.47 17.72.56.74 11 11.57 19.88 21.12a132.58 132.58 0 0020.67-25.01z"></path><path fill="#89bc3b" d="M95.79 201.78a14.88 14.88 0 017.09 1.77c1.1-4.56 2.4-9.17 3.86-13.73a113.74 113.74 0 00-10.95 11.96z"></path><path fill="#89bc3b" d="M112.56 231c-4 14.84-4.44 32.53-3 51.14a130.52 130.52 0 0036.91 9c-2.58-15.31-2.74-31-.46-48.85-1.19-5.91-1.78-16.54.88-22.84 10.41-4 21.56-4.33 31.9-2.41-6.22-13.12-16.23-24.88-20.38-31-3.32-4.91-12.11-26.81-11.81-28.36-9.78 11.69-24.58 17.7-39.86 32.18-1.46 4.56-2.76 9.17-3.86 13.73 3.91 1.91 7.57 4.88 10.46 6.48-1 12.11-1.05 13.65-.78 20.93z"></path><path fill="#b498c6" d="M35.85 151.6s2.66 3.18 4.43 5.55 4.66 16 8.68 20.43-7 12.22-7 12.22c-12-.92-23-13.34-21-28.7 0 0 3.25-4.73 3.84-6.5s-.18-5-.18-5a34.85 34.85 0 01-4.29-6.21c-1.21-2.65-.84-6.46.37-9.18s4.56-2.56 4.68-1.89c.31 1.74 2 8.86 1.86 8.6-.87-2.01 2.67 2.19 8.61 10.68z"></path><path fill="#b498c6" d="M32.13 155.37c-12 4.78-18.36 1.3-24.56 4.84s1.22 17.61 1.22 17.61 1.84 6.8 5.39 11.23 24.27 4.41 24.27 4.41l10.15-13.52z"></path><path fill="#b498c6" d="M28.47 184.32c-.59.89-5.5 6.45-8.45 7.45s-6.34-.66-6.34-.66c-3.64-1.42-4.81-3.6-3.07-6.2 0 0 10.29-.52 7.34.18s7.57-4.31 7.57-4.31c1.96.66 3.09 1.77 2.95 3.54z"></path><path fill="#8b6c9c" d="M25.52 180.78c7.47.85-3.24 16.33-11.84 10.33 5.39 1.3 7.32.21 10.71-3.07 2.5-2.49 3.06-2.79 2.42-5.24a2.59 2.59 0 00-1.29-2.02z"></path><path fill="#b498c6" d="M5.48 176.94c-2.11 2.55-1.17 5.19 4.13 8l8.39 1.15 8.57-5.31a8.67 8.67 0 00-.89-6.8l-10.34 3.25z"></path><path fill="#8b6c9c" d="M9.61 184.91c7.51.44 7.78 1.78 12.14-.95 5-3.14 5.69-4.17 3.88-10 4.06 5.07 1.3 7.9-3.35 10.83-3.09 1.94-6.79 2.92-12.67.12z"></path><path fill="#b498c6" d="M1.34 168.67c-3.06 1.58-.35 9.08 2.77 9l11.18-.41s9.12-2 10.34-3.25c1.81-1.8-.57-7.7-1.48-7.38l-10.93 3z"></path><path fill="#8b6c9c" d="M4.11 177.64c11.75.39 10.61 4.06 19.09-3.93 1.37-1.34-.15-7.07-1.05-7.11 1.23-.32 4.11 5.64 1.75 7.82-5.45 5.13-8.43 7.58-19.79 3.22z"></path><path fill="#b498c6" d="M4.59 159.81c-2.79.54-3.52 2-3.54 5.43s-.94 2.44 1.29 3.43 13.88.88 13.88.88l7.93-3c.73-2.37.66-4.59-1.18-6.5l-9.4-.89a69.28 69.28 0 00-8.98.65z"></path><path fill="#8b6c9c" d="M2.34 168.67c1.27.26 12.13.39 13.88.38-.45.09-.73.24 7.76-2.92-.91 1 .41-.43.05-2.9-.51-3.6-2.16-2.54-5.81-3.09-4.5-.9-9.43-1.88 4.8-.54.53.05 3 2.56 1.61 7.14-.11.36.83-.09-8.31 3.31-3.68.02-10.94.14-13.98-1.38z"></path><path fill="#006531" d="M197.4 137.56c4.72.88 31.31 19.79 39.83 35.92 3.33 6.31 6 21.38 8.9 27.58s13.59 18.31 14.48 19.79c-17.13 2.66-27.18 19.2-27.18 19.2-4-3.26-43.16-44.31-44-46.38-3.25-7.97 3.24-57 7.97-56.11z"></path><path fill="#006531" d="M195.4 110.09c-1.55 13 2 27.47 2 27.47s-44.31 21.26-50.8 20.08-3-24.33-3-24.33-.88-13.29 1.48-17.13 4.75-9.93 13-11.11 38.1-1.57 37.32 5.02z"></path><path fill="#b498c6" d="M147.3 154.39s11.48-21.63 15-24.29 18.72-8.83 22.56-10 3.84-5.31 5-8.56 1.48-29.79 1.48-29.79l-45.29 2 4.73 39.81c.36 12.52-4.85 19.83-3.48 30.83z"></path><path fill="#89bc3b" d="M195.45 138.74s27.35 19.2 39.05 36.26c2.65 3.85 4.84 20.79 7.2 25.81s10.05 15.07 10.05 15.07l5.31.29 3.55 4.73c-9.41 1.64-18.27 9.31-27.18 19.2l-6.49-5.62c5.46-7.09 11.49-13 18.9-15.94 0 0-7.68-12.11-9.75-16.54s-4.52-21.19-7.2-26.41c-9-17.6-39.06-34.73-39.06-34.73z"></path><path fill="#093f22" d="M192.38 197.7l21.02 21.79-2.66 5.31-15.36 5.61 11.88-10.33-14.88-22.38zm-71.73-50.53l1.55 13.29a79.66 79.66 0 01-7.31 11.54c-4.43 5.76-8.76 11.31-8.76 11.31L114 156l-.44 12z"></path><path fill="#8b6c9c" d="M191.38 81.73s1.91 5.52-1.07 17.09c-2.07 8-25.57 30.6-35 28.17-6-1.54-6-11.85-7.27-16.31L145 83.8z"></path><path fill="#b498c6" d="M172.22 6.3c-20-2.46-38.74 14.76-40.58 16.91-1.16 1.35-4.27 8.05-6.64 14.45 13.42-5 36.89-.82 67.51 9.88 1 6.7.06 12.11-.92 19.63 0 0-8.61 4.69-16 4.69s-16.29-2.46-18.45-5.23-8-9.63-13.77-10-11.37 5.87-13.89 6.59c-1.75.5-8 3-10.15 5.15v-.08a15.41 15.41 0 00-1.18 6.5c.8 8.65 6.53 18.51 8.69 21.59s2.72 9.45 6.41 14.07c3.13 3.91 15.33 3.46 20 2.1 7-2 29.38-9.37 32.77-14S198.23 70 198.23 70s12.4-8.23 13-14.69c2.12-22.13-11.91-45.67-39.01-49.01z"></path><path fill="#006531" d="M192.2 87.46c21.23 1.16 34.39 0 35.92-9.22s1.16-64.85-52.19-76.38c-50.56-10.93-60.49 29.65-61.72 37s7.59 8.91 7.59 8.91c-.61-3.09 1.55-2.73 3.19-10.14 15.18-3.18 61.91 7.92 64.68 12.84s-.54 12.68-.24 16.06 7.69 6.47 7.69 6.47z"></path><path fill="#093f22" d="M225.1 81.93c1.13-4.56-4.46-9.51-15.07-14.76l-5.22 5.23-4.92 15.37c12.11.23 23.63.05 25.21-5.84z"></path><path fill="#b498c6" d="M207 73.61c3.69 3.07 3.93 10.55 1.78 14.54s-14.78 8.49-17.21 7.57-4.92-3.88-4.92-3.88l10.45-18.62a12.33 12.33 0 014.49.11c.31 0 2.8-1 3.11-.91a6.32 6.32 0 012.3 1.19z"></path><path fill="#191a1a" d="M192.2 69s-19.69 37.82-22.44 39c.82.16 5.58-1.59 7.38-2.46s20-32.59 20-32.59z"></path><path fill="#093f22" d="M186.03 79.26l5.37 4.21-4.21 7.53-5.93-3.32 4.77-8.42z"></path><path fill="#fcee21" d="M122.11 47.5a54.43 54.43 0 012.89-9.84c-4.93 9.91-10.22 24.17-5.7 30.66a35.07 35.07 0 015.58-8.22s-3.7-6.1-2.77-12.6z"></path><path fill="#fbb03b" d="M129.49 65.25c2.72-.78 8.83-7.61 14.37-6.38s11.14 8 13.29 10.76 11.07 5.23 18.45 5.23 16-7.69 16-7.69c1-7.52 1.93-12.93.92-19.63-30.62-10.7-54.09-14.89-67.52-9.88a54.43 54.43 0 00-2.89 9.84c-.93 6.45 2.76 12.6 2.76 12.6a35.07 35.07 0 00-5.58 8.22v.08c2.2-2.15 6.93-2.21 10.2-3.15z"></path><path fill="#4b3f51" d="M164.1 84.58s-10-6.68-19.87-7.87c-1.84-.22-2.48.64-3.69.59-1.78-.07-2.9-.67-4.28-.74-4.79-.22-5.77 2.37-5.77 2.37a4.35 4.35 0 001.93 4.43c1.77 1 2.51-1.11 3.32 1.77 17.72-1.66 22.26-.59 28.36-.55z"></path><path fill="#8b6c9c" d="M135.67 85c0 2.77-5.87 11.68 5.81 12.3 12.65.7 19.82-9.23 22.52-12.67-9.76-.11-19.34-2.38-28.33.37z"></path><path fill="#f2f2f2" d="M136.67 81.48c-.95 1.52-.25 7.58 4.61 8.3 7.75 1 16.17-1.77 22-5.22-9.55-1.06-11.88-2.62-16.38-3.49-2.54-.48-9.09-1.42-10.23.41z"></path><path fill="#fcee21" d="M124.87 60.1a20.07 20.07 0 0012-.61s-2.15-7.92-6.76-7.61-5.24 8.22-5.24 8.22z"></path><path fill="#f15a24" d="M136.81 59.5c-3.13-10.48-10.72-10.06-11.88.6-1.93-.14-1.38-9.34 4.35-9.94 6.86-.67 8.23 9.15 7.53 9.34zm1.59-18.77c1.26.46.58 3.28-.69 3.38s-7.64-.52-9.45.31-5.08 4.17-6.25 6.08c.21-1.54 3.68-7.57 5-8.85s10.14-1.38 11.39-.92z"></path><path fill="#8b6c9c" stroke="#8b6c9c" stroke-miterlimit="10" d="M150.7 66.25c.48 1.6 1 3.41-.85 4.33s-4-2.41-5.53-2.1-4 2.24-5.61 1.77c-3.34-1-3.69-2.69-3.69-2.69s-1.12 4.2 4.3 4.84c3.68.43 5.35-2.06 8.42-1.45s2.09 2.8 3.57 2.06c2.97-1.47.98-6.44-.61-6.76z"></path><path fill="#8b6c9c" d="M154.79 69.68c2.27 1.94 3.94 10 11.63 10.35 1.28 2-.13 5.41-1.89 9.58a20.53 20.53 0 007.07-12.91c-11.05.3-13.26-5.8-16.81-7.02zm-18.22.44s-2.47-.4-2.47.87a.37.37 0 01-.16.3.38.38 0 01-.34.06c-2.4-.79-1.79-4.72-.07-5.86-1.72 4.02 3.04 4.63 3.04 4.63z"></path><path fill="#8b6c9c" d="M132.48 68.63c-2 2.71-2.16 7-2 10.24 0 0-1-1.86-2-1.79s-.71 5.3.79 9a31 31 0 01-6-15.73c2.67 1.07 6 .5 9.27-3.38z"></path><ellipse cx="139.18" cy="184.17" fill="#fbb03b" stroke="#fc3" stroke-miterlimit="10" rx="8.75" ry="5.25" transform="rotate(-9.22 139.183 184.166)"></ellipse><path fill="#f19018" d="M129.81 195.84l18.24-2.09a100.85 100.85 0 00-14.23 16.57l8-.9a78.29 78.29 0 00-8.82 18.82c-3.2 9.33-2.81 18.19-3 17.84a47.74 47.74 0 01-3.75-14.91 56 56 0 01.51-15.19c-3 .83-6 1.67-8.89 2.51a73.05 73.05 0 0111.94-22.65z"></path><path fill="#fbb03b" stroke="#fc3" stroke-miterlimit="10" d="M129.81 195.84l18.24-2.09a100.85 100.85 0 00-14.23 16.57l8-.9a78.29 78.29 0 00-8.82 18.82c-3.2 9.33-2.81 18.19-3 17.84a47.74 47.74 0 01-3.75-14.91 56 56 0 01.51-15.19c-3 .83-6 1.67-8.89 2.51a73.05 73.05 0 0111.94-22.65z"></path><path fill="#f15a24" d="M136 56.78c1 1.52-.76 3.87-2.65 3.87s-3.66-1.26-3.66-3.1a3.38 3.38 0 013.43-3.32c1.88 0 1.51.55 2.88 2.55z"></path><path fill="#8b6c9c" d="M25 161.28c-.08 6.93 1.26 20.08 15.09 24.42 0 0-8.67 1.28-11.66-1.38s-6.92-16.88-3.43-23.04z"></path><path fill="#093f22" d="M120.68 47.53l4.32-9.87c19.53-1.66 62.46 8.77 64.68 12.84s.07 13.6-.24 16.06 7.68 6.44 7.68 6.44l4.61.31s-8.3-3.07-8-7.38 3.2-16.07-1.23-18.4c-27.06-14.19-61.36-15-69.48-13.76-6.46 1-8.44 6.4-7.22 10.09.85 2.58 4.88 3.67 4.88 3.67zm47.77-26.05c-.92-2.15 18.91-8.27 30.13-7.68 6.17.32 13.26 10.33 5.91 9.45-14.77-1.77-35.11.38-36.04-1.77zM172 32.7c-.59-3 28.35-3 39.26.35 6 1.8 2.55 12.33-1.84 10.45C198 38.61 172.38 34.61 172 32.7zm-38.21-14.1L131 9.07l-6.15 6.15s8.02 4.61 8.94 3.38zm-7.07 8.61C127 26 120.57 21.06 120 22s-2.77 8.3-2.77 8.3 9.18-1.86 9.49-3.09z"></path><path fill="#fc3" d="M181.6 46.59c-.06 10.33-1.83 19.86-6 28.27A30.57 30.57 0 01163.36 73c3.15-11.43 5.89-22.37 5.61-29.76a90 90 0 0112.63 3.35z"></path><path fill="#fcee21" d="M172.83 62.87c-5.55.17-12.05 1.5-16.91-1.54 0 0 .7-8.22 7.15-7s9.76 8.54 9.76 8.54z"></path><path fill="#f15a24" d="M155 42.88c-1.24.38-.16 2.92.92 3.39s8.91 1.14 11.65 1.79 6.2 4.36 8.64 6.81c-.22-1.86-4.87-8.63-7.68-9.53s-12.29-2.83-13.53-2.46zm13.31 16.56a3.78 3.78 0 11-3.44-4.09 5 5 0 01.56.09 4.28 4.28 0 012.88 4z"></path><path fill="#f15a24" d="M155.82 61.32c-2.89-12.19 15.29-11.67 17.1 1.51-.51.23-2.75-6.29-8.94-8.06-7.49-2.22-7.59 6.6-8.16 6.55z"></path></svg>                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           plugins/wordpress-seo-extended/packages/js/images/jetpack-boost-integration-logo.svg                0000644                 00000042460 15122266557 0025064 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8"?>
<svg width="170px" height="31px" viewBox="0 0 170 31" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <title>Group</title>
    <g id="Jetpack-Boost-ad-in-Yoast-SEO" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
        <g id="Jetpack-Boost-ad-in-Yoast-SEO---Integrations-page---Active" transform="translate(-812.000000, -328.000000)" fill-rule="nonzero">
            <g id="Group" transform="translate(812.000000, 328.000000)">
                <path d="M125.929075,12.8823731 C126.393095,12.9496681 126.807918,13.0731435 127.174761,13.2527992 C127.548892,13.4323942 127.867146,13.6643432 128.128917,13.948707 C128.398582,14.2256004 128.603868,14.5586134 128.746597,14.947746 C128.888718,15.3294081 128.959779,15.7671899 128.959779,16.2610914 C128.959779,16.6128109 128.906939,16.983237 128.802474,17.3723696 C128.697401,17.7540318 128.521876,18.1281627 128.274682,18.4948839 C128.035384,18.8540739 127.71713,19.1870869 127.320526,19.4939229 C126.931819,19.8006981 126.445327,20.0514144 125.861659,20.2460111 C125.412216,20.3956635 124.914792,20.5004323 124.368779,20.5602568 C123.822159,20.620142 123.16743,20.6500847 122.403984,20.6500847 L118.520554,20.6500847 L118.520554,5.63096227 L122.269759,5.63096227 C122.613522,5.63096227 122.924488,5.63844488 123.201442,5.65341012 C123.485684,5.66089274 123.74381,5.67586405 123.97582,5.69831191 C124.215118,5.72075977 124.436196,5.75069632 124.637838,5.78810941 C124.840087,5.81804596 125.030796,5.85546513 125.210574,5.90036084 C125.681882,6.02757751 126.108245,6.20346764 126.490272,6.42794621 C126.878979,6.64495431 127.208773,6.90684597 127.477832,7.21368193 C127.747497,7.5205179 127.953391,7.86847182 128.095512,8.25760445 C128.244921,8.64673707 128.319626,9.06957792 128.319626,9.52606627 C128.319626,9.89272674 128.274682,10.2444463 128.184793,10.5812249 C128.1028,10.9179427 127.964323,11.2285443 127.769969,11.5129081 C127.582904,11.7972719 127.33571,12.0516931 127.028996,12.2761717 C126.72957,12.500711 126.362727,12.6877764 125.929075,12.8374288 L125.929075,12.8823731 Z M123.134025,12.1302849 C123.492973,12.1302849 123.803331,12.1152832 124.065709,12.0853406 C124.327479,12.0479275 124.563133,11.9955735 124.77267,11.9282177 C125.333867,11.7411523 125.74201,11.4492572 125.996492,11.0526541 C126.258262,10.656051 126.388843,10.1883266 126.388843,9.64954163 C126.388843,9.09575494 126.254618,8.6430322 125.984952,8.29131267 C125.715286,7.93959314 125.292567,7.67764075 124.716794,7.50551622 C124.431944,7.42321956 124.102757,7.36709991 123.728626,7.33715729 C123.354496,7.2997442 122.912948,7.28103765 122.403984,7.28103765 L120.349908,7.28103765 L120.349908,12.1302849 L123.134025,12.1302849 Z M120.349908,13.7803481 L120.349908,19.0000214 L122.561289,19.0000214 C123.085437,19.0000214 123.571321,18.9775493 124.020764,18.9326657 C124.4696,18.8802509 124.851019,18.8017199 125.165629,18.6969511 C125.51,18.5846511 125.801531,18.4424691 126.040829,18.2703446 C126.280734,18.0907496 126.471444,17.8962137 126.613565,17.6866761 C126.762974,17.469668 126.868047,17.2414238 126.928175,17.0019435 C126.994984,16.7624633 127.028996,16.5192782 127.028996,16.2723275 C127.028996,15.9879636 126.994984,15.7223064 126.928175,15.4753557 C126.860758,15.2208737 126.748398,14.9963952 126.591093,14.8018592 C126.441683,14.5997921 126.239434,14.4239019 125.984952,14.2742496 C125.737758,14.1245972 125.4274,14.0085923 125.053269,13.9262956 C124.828547,13.8738809 124.570421,13.8364678 124.27889,13.8139956 C123.994648,13.7915842 123.638738,13.7803481 123.212374,13.7803481 L120.349908,13.7803481 Z" id="Shape" fill="#000000"></path>
                <path d="M139.28968,15.3294081 C139.28968,14.6708527 139.207687,14.0871841 139.043094,13.578281 C138.885789,13.0619681 138.661067,12.6278912 138.369536,12.2761717 C138.084687,11.9169817 137.744568,11.6476195 137.347965,11.4679638 C136.951362,11.2808983 136.517102,11.1873656 136.045795,11.1873656 C135.574487,11.1873656 135.140228,11.2808983 134.743625,11.4679638 C134.347021,11.6476195 134.002651,11.9169817 133.71112,12.2761717 C133.426878,12.6278912 133.202156,13.0619681 133.037563,13.578281 C132.880258,14.0871841 132.801909,14.6708527 132.801909,15.3294081 C132.801909,15.9804324 132.880258,16.5641617 133.037563,17.0805353 C133.202156,17.5893777 133.430522,18.019689 133.722053,18.3714086 C134.014191,18.7231281 134.358561,18.9924902 134.755164,19.1796164 C135.151767,19.3592114 135.585419,19.4489786 136.056727,19.4489786 C136.528642,19.4489786 136.95865,19.3592114 137.347965,19.1796164 C137.744568,18.9924902 138.084687,18.7231281 138.369536,18.3714086 C138.661067,18.019689 138.885789,17.5893777 139.043094,17.0805353 C139.207687,16.5641617 139.28968,15.9804324 139.28968,15.3294081 Z M141.108102,15.3294081 C141.108102,16.1675586 140.988453,16.9308222 140.749155,17.6193203 C140.509857,18.3077577 140.169131,18.8989575 139.727583,19.3928589 C139.293324,19.8792899 138.76614,20.2571864 138.144815,20.5266093 C137.52349,20.7960322 136.828068,20.9306829 136.056727,20.9306829 C135.263521,20.9306829 134.552915,20.7960322 133.924302,20.5266093 C133.302977,20.2571864 132.771541,19.8792899 132.329994,19.3928589 C131.896342,18.8989575 131.563512,18.3077577 131.331502,17.6193203 C131.099492,16.9308222 130.983487,16.1675586 130.983487,15.3294081 C130.983487,14.4837872 131.103136,13.716758 131.342434,13.0282599 C131.581732,12.3398225 131.918814,11.7486227 132.352466,11.2547213 C132.794014,10.7608199 133.32545,10.3791577 133.946774,10.1097348 C134.575387,9.8403727 135.275061,9.70566127 136.045795,9.70566127 C136.831713,9.70566127 137.538674,9.8403727 138.167287,10.1097348 C138.7959,10.3791577 139.327336,10.7608199 139.760988,11.2547213 C140.195247,11.7486227 140.528078,12.3398225 140.760087,13.0282599 C140.992097,13.716758 141.108102,14.4837872 141.108102,15.3294081 L141.108102,15.3294081 Z" id="Shape" fill="#000000"></path>
                <path d="M151.202957,15.3294081 C151.202957,14.6708527 151.120356,14.0871841 150.955763,13.578281 C150.798458,13.0619681 150.574344,12.6278912 150.282206,12.2761717 C149.997964,11.9169817 149.657237,11.6476195 149.260634,11.4679638 C148.864031,11.2808983 148.430379,11.1873656 147.958464,11.1873656 C147.487156,11.1873656 147.052897,11.2808983 146.656294,11.4679638 C146.259691,11.6476195 145.915928,11.9169817 145.62379,12.2761717 C145.339547,12.6278912 145.114826,13.0619681 144.950233,13.578281 C144.792928,14.0871841 144.714579,14.6708527 144.714579,15.3294081 C144.714579,15.9804324 144.792928,16.5641617 144.950233,17.0805353 C145.114826,17.5893777 145.343192,18.019689 145.63533,18.3714086 C145.92686,18.7231281 146.271231,18.9924902 146.667834,19.1796164 C147.064437,19.3592114 147.498696,19.4489786 147.970004,19.4489786 C148.441312,19.4489786 148.871927,19.3592114 149.260634,19.1796164 C149.657237,18.9924902 149.997964,18.7231281 150.282206,18.3714086 C150.574344,18.019689 150.798458,17.5893777 150.955763,17.0805353 C151.120356,16.5641617 151.202957,15.9804324 151.202957,15.3294081 L151.202957,15.3294081 Z M153.021379,15.3294081 C153.021379,16.1675586 152.90173,16.9308222 152.661824,17.6193203 C152.422526,18.3077577 152.0818,18.8989575 151.640253,19.3928589 C151.206601,19.8792899 150.678809,20.2571864 150.057484,20.5266093 C149.436767,20.7960322 148.740738,20.9306829 147.970004,20.9306829 C147.176798,20.9306829 146.465584,20.7960322 145.836972,20.5266093 C145.216254,20.2571864 144.684818,19.8792899 144.243271,19.3928589 C143.809012,18.8989575 143.476181,18.3077577 143.244171,17.6193203 C143.012161,16.9308222 142.896157,16.1675586 142.896157,15.3294081 C142.896157,14.4837872 143.015806,13.716758 143.255104,13.0282599 C143.495009,12.3398225 143.831484,11.7486227 144.265743,11.2547213 C144.70729,10.7608199 145.238726,10.3791577 145.859444,10.1097348 C146.488057,9.8403727 147.18773,9.70566127 147.958464,9.70566127 C148.744382,9.70566127 149.451344,9.8403727 150.079957,10.1097348 C150.708569,10.3791577 151.240005,10.7608199 151.674265,11.2547213 C152.107916,11.7486227 152.441354,12.3398225 152.673364,13.0282599 C152.905374,13.716758 153.021379,14.4837872 153.021379,15.3294081 L153.021379,15.3294081 Z" id="Shape" fill="#000000"></path>
                <path d="M156.548899,12.8261927 C156.548899,13.1180878 156.612672,13.357568 156.739609,13.5446335 C156.867154,13.731699 157.035391,13.8851169 157.244929,14.0048267 C157.461754,14.1171267 157.708948,14.2106594 157.985902,14.2854856 C158.270145,14.3603118 158.562283,14.4276675 158.861101,14.4875528 C159.272888,14.5773199 159.662203,14.6783839 160.029046,14.7906231 C160.395281,14.895392 160.732363,15.0562804 161.039078,15.2732885 C161.353081,15.4902966 161.603919,15.7858965 161.790984,16.1600882 C161.985945,16.5267486 162.083122,17.0057091 162.083122,17.5968482 C162.083122,18.1356939 161.978049,18.6145937 161.768512,19.0336689 C161.558974,19.4452737 161.263799,19.7932276 160.881773,20.0775914 C160.500354,20.3544848 160.043622,20.5640224 159.512186,20.7062043 C158.981358,20.8558567 158.397689,20.9306829 157.761181,20.9306829 C157.371866,20.9306829 157.016563,20.9082715 156.694665,20.8633879 C156.380662,20.8259141 156.100064,20.77356 155.85287,20.7062043 C155.613572,20.6463191 155.407678,20.582729 155.235797,20.5153732 C155.063308,20.4480175 154.921187,20.3844274 154.808826,20.3245422 L154.808826,18.86531 C155.258269,19.044905 155.70346,19.1870869 156.145008,19.2918557 C156.586555,19.3966245 157.073047,19.4489786 157.603876,19.4489786 C158.022951,19.4489786 158.401333,19.4115655 158.737808,19.3367393 C159.07489,19.2619131 159.362777,19.1534394 159.602075,19.0111968 C159.849269,18.8615444 160.036334,18.6782445 160.163271,18.4611757 C160.298104,18.2366971 160.365521,17.9785103 160.365521,17.6866761 C160.365521,17.3873106 160.305393,17.1403598 160.185744,16.9458239 C160.066095,16.7512272 159.905145,16.5903388 159.703503,16.4631585 C159.501254,16.3359176 159.2656,16.2311488 158.995934,16.1488521 C158.734164,16.0664947 158.45721,15.9916685 158.165679,15.9243128 C157.783653,15.8420161 157.394338,15.7447178 156.997735,15.6324785 C156.609027,15.5202392 156.253117,15.3630556 155.931826,15.1610492 C155.609928,14.9589821 155.348158,14.6858543 155.145908,14.3416053 C154.951554,13.9898857 154.85377,13.5408679 154.85377,12.9946124 C154.85377,12.4333552 154.955199,11.9469243 155.156841,11.5353195 C155.35909,11.1237755 155.6324,10.7832313 155.97677,10.5138691 C156.320534,10.2444463 156.721388,10.0423791 157.177512,9.90772842 C157.634243,9.77301698 158.116484,9.70566127 158.625447,9.70566127 C159.217012,9.70566127 159.763024,9.76178091 160.2647,9.87402019 C160.773056,9.98625948 161.233432,10.1284414 161.645219,10.3005659 L161.645219,11.7822702 C161.196383,11.6101457 160.739652,11.4717294 160.275632,11.3669606 C159.819508,11.2547213 159.347593,11.194836 158.861101,11.1873656 C158.449922,11.1873656 158.098263,11.2285443 157.806125,11.3108409 C157.514594,11.3931376 157.274689,11.5091425 157.087624,11.6587949 C156.900558,11.8009768 156.762081,11.9731013 156.672192,12.1751685 C156.590199,12.3697651 156.548899,12.5867732 156.548899,12.8261927 L156.548899,12.8261927 Z" id="Path" fill="#000000"></path>
                <path d="M170,20.6725568 C169.849983,20.7323813 169.621618,20.788501 169.314903,20.8409157 C169.008189,20.8932698 168.641346,20.9232124 168.214982,20.9306829 C167.653786,20.9306829 167.167294,20.8596223 166.755507,20.7174404 C166.344328,20.582729 165.999957,20.388193 165.723003,20.1337111 C165.453337,19.8718194 165.251695,19.557513 165.116862,19.1908525 C164.982029,18.8241313 164.914613,18.4162921 164.914613,17.9672743 L164.914613,11.4791999 L162.860537,11.4791999 L162.860537,9.98625948 L164.914613,9.98625948 L164.914613,7.06779516 L166.654686,7.06779516 L166.654686,9.98625948 L169.842695,9.98625948 L169.842695,11.4791999 L166.654686,11.4791999 L166.654686,17.6979121 C166.654686,17.9448628 166.688091,18.173107 166.755507,18.3826446 C166.830819,18.5921823 166.942573,18.7754821 167.09259,18.9326657 C167.249895,19.0823181 167.447892,19.2020279 167.687191,19.2918557 C167.927096,19.3741524 168.214982,19.4153311 168.551457,19.4153311 C168.783467,19.4153311 169.027017,19.3966245 169.281499,19.3592114 C169.543269,19.3217983 169.782567,19.2731492 170,19.2132639 L170,20.6725568 Z" id="Path" fill="#000000"></path>
                <path d="M15.1750791,30.349065 C23.5559767,30.349065 30.3501583,23.5548835 30.3501583,15.1739859 C30.3501583,6.79302756 23.5559767,0 15.1750791,0 C6.79418153,0 0,6.79418153 0,15.1750791 C0,23.5559767 6.79418153,30.349065 15.1750791,30.349065 Z" id="Path" fill="#069E08"></path>
                <polygon id="Path" fill="#FFFFFF" points="15.9282606 12.6264335 15.9282606 27.3373717 23.5157698 12.6264335"></polygon>
                <polygon id="Path" fill="#FFFFFF" points="14.3928661 17.6945109 14.3928661 3.01162657 6.83450994 17.6945109"></polygon>
                <g transform="translate(37.936574, 5.358011)" fill="#000000">
                    <path d="M1.24513937,19.8363681 C0.810212146,19.1699777 0.405652693,18.5047413 0,17.8675647 C2.1429929,16.5640585 2.86671764,15.5223225 2.86671764,13.5524258 L2.86671764,2.17107095 L0.3472251,2.17107095 L0.3472251,6.07355432e-06 L5.70525398,6.07355432e-06 L5.70525398,12.9736769 C5.70525398,16.275261 4.75006609,18.1283024 1.24513937,19.8363681 Z" id="Path"></path>
                    <path d="M23.6899593,12.0758233 C23.6899593,13.1759262 24.4716258,13.2916882 24.9933441,13.2916882 C25.5144551,13.2916882 26.2675758,13.1174986 26.8463855,12.9444631 L26.8463855,14.971694 C26.0349587,15.2324317 25.1955935,15.4346811 24.0367593,15.4346811 C22.6468264,15.4346811 21.0252481,14.9132664 21.0252481,12.480322 L21.0252481,6.51439111 L19.5486456,6.51439111 L19.5486456,4.45788562 L21.0252481,4.45788562 L21.0252481,1.41819315 L23.6899593,1.41819315 L23.6899593,4.45903959 L27.0486349,4.45903959 L27.0486349,6.51548435 L23.6899593,6.51548435 L23.6899593,12.0758233 Z" id="Path"></path>
                    <path d="M29.249691,20.8499836 L29.249691,4.4298258 L31.7981544,4.4298258 L31.7981544,5.41422748 C32.8118306,4.63207515 33.9415117,4.13987431 35.3317483,4.13987431 C37.7350537,4.13987431 39.6464012,5.81988017 39.6464012,9.4394756 C39.6464012,13.0309505 37.5619574,15.4054673 34.1158227,15.4054673 C33.2758501,15.4054673 32.6095812,15.2897053 31.9141592,15.1447296 L31.9141592,20.8207698 L29.249691,20.8207698 L29.249691,20.8499836 Z M34.6357189,6.34129481 C33.8534451,6.34129481 32.8695293,6.71773371 31.9420976,7.52909983 L31.9420976,13.1186526 C32.5209073,13.2343538 33.1300848,13.320902 33.940297,13.320902 C35.8224914,13.320902 36.8944738,12.1330969 36.8944738,9.64287894 C36.8944738,7.35491029 36.1122,6.34129481 34.6357189,6.34129481 L34.6357189,6.34129481 Z" id="Shape"></path>
                    <path d="M50.1287486,15.1739434 L47.6385914,15.1739434 L47.6385914,13.9861384 L47.5802852,13.9861384 C46.711767,14.6525287 45.6397846,15.3762535 44.0472987,15.3762535 C42.6570621,15.3762535 41.151428,14.362638 41.151428,12.3061325 C41.151428,9.55523755 43.4964273,9.03382291 45.1472194,8.80229902 L47.4928261,8.48428771 L47.4928261,8.16627641 C47.4928261,6.71773371 46.9140163,6.2548074 45.5517181,6.2548074 C44.8854492,6.2548074 43.3221163,6.45705676 42.0478846,6.97847139 L41.8164822,4.83547849 C42.9753164,4.4298258 44.5671949,4.14102829 45.9003401,4.14102829 C48.5065023,4.14102829 50.1864474,5.18385756 50.1864474,8.28203836 L50.1864474,15.1739434 L50.1287486,15.1739434 Z M47.4642804,10.1642936 L45.2632243,10.5115187 C44.5969554,10.5980668 43.9021407,11.0037195 43.9021407,11.9881212 C43.9021407,12.8568217 44.394706,13.3490225 45.1180663,13.3490225 C45.9003401,13.3490225 46.7397053,12.8860355 47.463673,12.3646208 L47.463673,10.1642936 L47.4642804,10.1642936 Z" id="Shape"></path>
                    <path d="M61.1340291,14.8267183 C60.033501,15.2031572 59.0495852,15.4346811 57.8032919,15.4346811 C53.8068931,15.4346811 52.2137998,13.1467124 52.2137998,9.81706847 C52.2137998,6.31323499 54.4148559,4.14102829 57.9763882,4.14102829 C59.3077113,4.14102829 60.1191381,4.37249144 61.0168095,4.66244293 L61.0168095,6.92113704 C60.235143,6.63124629 59.1054619,6.31323499 58.0055412,6.31323499 C56.3839022,6.31323499 54.9936656,7.18187473 54.9936656,9.67209273 C54.9936656,12.4230484 56.3839022,13.2635676 58.1500918,13.2635676 C58.9900644,13.2635676 59.9168888,13.0894388 61.1619674,12.597238 L61.1619674,14.8267183 L61.1340291,14.8267183 Z" id="Path"></path>
                    <path d="M66.1732571,9.06412995 C66.4046595,8.80345299 66.5789705,8.54271531 69.937646,4.45903959 L73.4123265,4.45903959 L69.0679131,9.55633079 L73.8168252,15.2031572 L70.3421448,15.2031572 L66.2011954,10.105866 L66.2011954,15.2031572 L63.5379419,15.2031572 L63.5379419,-2.15775995e-15 L66.2024101,-2.15775995e-15 L66.2024101,9.06412995 L66.1732571,9.06412995 Z" id="Path"></path>
                    <path d="M17.5494745,14.8267183 C16.1594201,15.2615848 14.9716151,15.4346811 13.5815,15.4346811 C10.1642146,15.4346811 8.05043551,13.7265547 8.05043551,9.72942709 C8.05043551,6.80428186 9.84620331,4.13987431 13.2916092,4.13987431 C16.7088945,4.13987431 17.8966996,6.51439111 17.8966996,8.77308522 C17.8966996,9.52602375 17.838272,9.93167644 17.8101514,10.3654497 L10.9182464,10.3654497 C10.976674,12.710692 12.3083008,13.2613204 14.3063179,13.2613204 C15.4064816,13.2613204 16.3908832,13.0006434 17.5213539,12.59493 L17.5213539,14.8244711 L17.5494745,14.8244711 L17.5494745,14.8267183 Z M15.117684,8.60004966 C15.117684,7.2964827 14.6828176,6.16710527 13.2646426,6.16710527 C11.9329551,6.16710527 11.1216497,7.12229316 10.9474602,8.60004966 L15.117684,8.60004966 Z" id="Shape"></path>
                </g>
            </g>
        </g>
    </g>
</svg>                                                                                                                                                                                                                plugins/wordpress-seo-extended/packages/js/images/jetpack-boost-logo.svg                            0000644                 00000000412 15122266557 0022532 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" role="img" aria-hidden="true" focusable="false"><path fill="#069e08" d="M16,0C7.2,0,0,7.2,0,16s7.2,16,16,16c8.8,0,16-7.2,16-16S24.8,0,16,0z M15.2,18.7h-8l8-15.5V18.7z M16.8,28.8 V13.3h8L16.8,28.8z"/></svg>
                                                                                                                                                                                                                                                      plugins/wordpress-seo-extended/packages/js/images/jetpack-logo.svg                                  0000644                 00000016574 15122266557 0021426 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8" standalone="no"?>
<svg
   xmlns:dc="http://purl.org/dc/elements/1.1/"
   xmlns:cc="http://creativecommons.org/ns#"
   xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
   xmlns:svg="http://www.w3.org/2000/svg"
   xmlns="http://www.w3.org/2000/svg"
   xmlns:sodipodi="http://sodipodi.sourceforge.net/DTD/sodipodi-0.dtd"
   xmlns:inkscape="http://www.inkscape.org/namespaces/inkscape"
   version="1.1"
   width="153"
   height="108"
   id="svg8"
   viewBox="0 0 595.27557 841.88977"
   sodipodi:docname="Jetpack_Logo.cdr">
  <metadata
     id="metadata14">
    <rdf:RDF>
      <cc:Work
         rdf:about="">
        <dc:format>image/svg+xml</dc:format>
        <dc:type
           rdf:resource="http://purl.org/dc/dcmitype/StillImage" />
      </cc:Work>
    </rdf:RDF>
  </metadata>
  <defs
     id="defs12" />
  <sodipodi:namedview
     pagecolor="#ffffff"
     bordercolor="#666666"
     borderopacity="1"
     objecttolerance="10"
     gridtolerance="10"
     guidetolerance="10"
     inkscape:pageopacity="0"
     inkscape:pageshadow="2"
     inkscape:window-width="640"
     inkscape:window-height="480"
     id="namedview10" />
  <path
     d="m -81.1429,289.2353 c 26.6704,-1.36 53.7098,5.8403 76.0802,20.4506 27.8192,17.89 48.2397,46.8998 55.4701,79.1994 8.2,35.4807 0.5094,74.3199 -21.0305,103.7401 -19.5197,27.2404 -50.08,46.2895 -83.1898,51.6093 -26.43,4.3906 -54.2293,0.4005 -78.2198,-11.5898 -25.2096,-12.4302 -46.11,-33.3802 -58.4201,-58.6502 -13.5198,-27.3197 -16.6504,-59.5196 -8.9092,-88.9792 7.1699,-27.76 24.0593,-52.8602 46.9995,-70.0603 20.4194,-15.4902 45.5995,-24.6092 71.2196,-25.7199 z m -0.4198,25.3506 c -21.3301,41.5298 -42.7504,83.0199 -64.1302,124.5194 21.39,-0.0201 42.7899,0.0102 64.1798,-0.01 -0.0192,-41.5097 0.0907,-83.01 -0.0496,-124.5095 z m 13.0496,81.4997 c -0.0099,41.5094 0.0204,83.0293 -0.0099,124.5399 21.5501,-41.44 42.7606,-83.0506 64.2501,-124.5206 -21.4098,-0.0698 -42.8199,-0.0295 -64.2402,-0.0193 z"
     style="fill:#00be28;fill-rule:evenodd"
     id="path2" />
  <path
     d="m -81.5627,314.5859 c 0.1403,41.4995 0.0304,82.9998 0.0496,124.5095 -21.3899,0.0201 -42.7898,-0.0102 -64.1798,0.01 21.3798,-41.4995 42.8001,-82.9896 64.1302,-124.5194 z m 13.0496,81.4997 c 21.4203,-0.0102 42.8304,-0.0505 64.2402,0.0193 -21.4895,41.47 -42.7,83.0806 -64.2501,124.5206 0.0303,-41.5106 0,-83.0305 0.0099,-124.5399 z"
     style="fill:#ffffff;fill-rule:evenodd"
     id="path4" />
  <path
     d="m 120.6967,334.5554 c 15.1701,-0.0698 30.3406,-0.0102 45.5108,-0.0304 -0.0202,37.1603 0.009,74.3208 -0.0202,111.4801 -0.0898,10.5597 -1.4306,21.5008 -6.7011,30.8509 -6.7592,12.0991 -19.0187,19.6191 -31.0697,25.6793 -3.6895,-5.5102 -7.0197,-11.2405 -10.6798,-16.7601 7.7905,-4.8002 15.8301,-10.1996 20.3103,-18.4399 2.8995,-5.2903 3.8701,-11.3899 3.9194,-17.3605 -0.019,-32.2997 0,-64.6093 -0.009,-96.9191 -7.0609,-0.01 -14.1211,0 -21.1811,0 -0.0796,-6.1702 0.0303,-12.3404 -0.0796,-18.5003 z m 535.2063,0.0501 c 7.6068,-0.1601 15.2237,-0.0603 22.8305,-0.0603 -0.0907,25.501 0.1715,51.01 -0.1312,76.5099 10.7342,-12.8891 21.1053,-26.0802 31.8396,-38.9693 9.8163,0.0796 19.6225,-0.0303 29.4386,0.0595 -12.3384,14.2997 -24.5052,28.7405 -36.8436,43.0299 13.428,15.9804 26.9164,31.9102 40.3747,47.8806 -9.8971,-0.0102 -19.7838,0 -29.6807,-0.0102 -11.6524,-14.3399 -23.3249,-28.6707 -35.0076,-42.9905 0.0201,14.2906 -0.01,28.5698 0.0201,42.8604 -7.5764,0.2897 -15.173,0.0697 -22.7596,0.1202 -0.0808,-42.81 0.0805,-85.6199 -0.0808,-128.4302 z m -359.8866,11.81 c 7.6107,0.0502 15.2308,-0.1797 22.8407,0.1097 -0.1604,8.53 -0.0505,17.0601 -0.0607,25.5799 9.4802,0.0201 18.9698,-0.009 28.4599,0.0201 -0.0301,5.8705 -0.0301,11.74 -0.01,17.6097 -9.4893,0.0402 -18.9797,0 -28.4701,0.0201 0.0505,16.0903 -0.0697,32.1908 0.0604,48.28 -0.0805,4.0203 2.8704,7.8006 6.8604,8.5209 6.6696,1.3893 13.3594,-0.9303 19.8099,-2.4103 -0.0706,5.7696 -0.0504,11.5294 -0.011,17.3001 -8.6097,2.9296 -17.7792,4.2098 -26.8589,3.7599 -6.021,-0.3903 -12.3304,-2.2105 -16.5807,-6.7008 -4.4702,-4.6094 -6.0097,-11.27 -6.0199,-17.5099 -0.01,-17.0787 0.0102,-34.1598 -0.01,-51.2399 -4.1797,-0.009 -8.3597,0.011 -12.5303,-0.0201 0,-5.8102 -0.03,-11.619 0.0204,-17.4393 4.1706,0 8.3492,0 12.53,-0.01 -0.0303,-8.6298 0.0201,-17.2496 -0.0303,-25.8701 z m -93.4002,32.47 c 10.6807,-8.8304 25.6108,-10.9703 38.9106,-8.2099 9.3198,1.8198 17.9195,7.52 22.6398,15.84 6.37,10.7594 6.2005,23.79 4.8203,35.8197 -19.4598,0.0201 -38.9208,-0.0304 -58.3696,0.03 0.2897,6.4398 1.5599,13.4603 6.4497,18.1001 5.7101,5.4093 14.0394,6.2699 21.5492,6.3204 9.6509,0.229 19.0908,-2.3304 28.1109,-5.5406 -0.01,6.2802 0,12.5603 0,18.8393 -14.5406,4.5108 -30.0801,6.5002 -45.2109,4.3402 -9.9697,-1.499 -19.8898,-6.0591 -26.0498,-14.2592 -7.4191,-9.68 -9.3694,-22.32 -9.2492,-34.2104 0.1392,-13.8496 5.4689,-28.1594 16.399,-37.0696 z m 14.811,12.8702 c -4.1998,4.0796 -5.8303,9.9402 -6.6806,15.5494 11.8006,0.0808 23.6001,0.0102 35.3999,0.0403 -0.1798,-6.4698 -1.1997,-14.0403 -6.8593,-18.1199 -6.6504,-4.32 -16.3001,-3.0498 -21.86,2.5302 z m 170.0595,-11.5203 c 8.5691,-6.6606 19.1894,-10.8998 30.1499,-10.6696 10.4598,-0.1604 21.1002,4.4198 27.3693,12.9597 8.3007,11.1903 9.7606,25.9296 8.5612,39.4101 -1.2611,14.5094 -8.3211,29.2399 -21.2405,36.7395 -13.1304,7.9399 -29.3196,7.3698 -43.8401,4.3504 -0.0102,16.0299 -0.0102,32.0697 0,48.0994 -7.5603,0.0102 -15.1197,0 -22.6803,0.0102 0.0303,-46.4298 -0.0391,-92.8494 0.0405,-139.2693 7.2003,0.11 14.4094,0.0797 21.6097,0.009 0.0303,2.7805 0.0496,5.5701 0.0303,8.3605 z m 0.9896,17.9104 c -0.0493,15.7198 0.1202,31.4501 -0.0896,47.1691 5.9502,1.4598 12.1193,1.9806 18.2299,1.8706 6.9996,-0.1301 14.2795,-3.06 18.2798,-9.0102 5.0604,-7.3293 5.7997,-16.63 5.4901,-25.27 -0.3197,-6.5999 -1.2501,-13.68 -5.5196,-18.9797 -3.3605,-4.2503 -8.9898,-5.94 -14.2308,-5.6699 -8.3494,0.2795 -15.8998,4.6602 -22.1598,9.8901 z m 83.3802,-22.6006 c 11.4801,-3.9396 23.6296,-5.9995 35.7693,-5.9695 9.1706,0.1301 19.0806,1.8595 26.0104,8.3495 6.7805,6.37 8.9102,16.0701 9.11,25.0398 0.0403,20.0208 0.0102,40.0507 0.0102,60.0707 -7.0197,0.0402 -14.0403,0 -21.0501,0.0292 -0.2401,-3.4291 -0.1601,-6.8695 -0.1601,-10.2994 -8.3101,6.4698 -18.1103,11.8298 -28.9001,11.9095 -8.7698,0.5802 -17.7698,-3.7591 -22.3299,-11.3497 -4.6803,-7.8398 -4.9204,-17.9195 -1.6396,-26.3202 3.1992,-7.9195 10.9605,-12.8699 18.8901,-15.1702 10.5798,-3.0396 21.6692,-3.53 32.4893,-5.3399 -0.0394,-4.8192 0.1403,-10.1996 -2.9792,-14.2098 -2.9004,-3.7803 -8.0306,-4.5196 -12.4705,-4.6398 -10.5899,0.0596 -20.9602,2.6998 -30.9498,5.9897 -0.3804,-6.0492 -1.1602,-12.07 -1.8,-18.0899 z m 28.8797,48.0702 c -4.5902,0.6497 -9.0403,3.7298 -10.2198,8.3696 -1.8399,5.9805 0.6497,14.2602 7.5504,15.4 8.1295,0.9899 15.5597,-3.67 21.9892,-8.0696 -0.0899,-6.2501 -0.0292,-12.4898 -0.0394,-18.7297 -6.4205,1.0795 -12.8701,1.9392 -19.2804,3.0297 z m 75.6283,-43.9398 c 9.8567,-7.9005 22.9111,-10.5305 35.29,-10.0798 7.8791,0.1191 15.6172,1.8998 23.0825,4.3492 0.0204,6.4307 -0.01,12.86 0.0204,19.2804 -10.7547,-3.8296 -22.5581,-6.8096 -33.908,-4.1201 -8.0808,1.9097 -14.1744,8.9101 -15.8593,16.9004 -1.9066,8.8093 -1.715,18.3402 1.5134,26.8297 2.4415,6.5304 8.3129,11.5203 15.1228,12.99 11.5515,2.6201 23.4763,-0.4195 34.2811,-4.6003 -0.0201,6.29 -0.0102,12.5906 0,18.8909 -14.7597,5.2098 -31.0629,7.0792 -46.3068,2.9296 -8.9889,-2.3899 -17.312,-7.8701 -22.2352,-15.8703 -6.5293,-10.4003 -7.8086,-23.1795 -6.9993,-35.1902 0.9198,-12.2595 6.152,-24.6191 15.9984,-32.3095 z"
     style="fill:#000000;fill-rule:evenodd"
     id="path6" />
</svg>
                                                                                                                                    plugins/wordpress-seo-extended/packages/js/images/logo-g2-white.svg                                 0000644                 00000002146 15122266557 0021421 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8"?> <svg xmlns="http://www.w3.org/2000/svg" fill="none" height="30" viewBox="0 0 30 30" width="30"><g fill="#fff"><path d="m26.0539 8.87178h-4.4729c.1264-.70757.556-1.09926 1.4278-1.54149l.8213-.41697c1.4783-.75811 2.2617-1.60467 2.2617-3.00717 0-.88446-.3412-1.56677-1.0108-2.07217-.6571-.50541-1.4783-.74548-2.4007-.74548-.7455 0-1.4151.18953-2.0469.58122-.6065.37906-1.074.85919-1.352 1.47832l1.2888 1.30142c.5054-1.01081 1.2256-1.51622 2.1859-1.51622.8087 0 1.3141.41696 1.3141.99818 0 .49277-.2401.8971-1.1751 1.3646l-.5307.2527c-1.1498.58122-1.9458 1.23825-2.4007 1.99636-.4548.74548-.6949 1.70575-.6949 2.85552v.3159h6.7725v-1.84472zm-.6065 3.99272h-7.4169l-3.7021 6.4187h7.4169l3.7021 6.4187 3.7021-6.4187z"></path><path d="m14.9727 23.554c-4.7129 0-8.55402-3.8411-8.55402-8.5541 0-4.7129 3.84112-8.554 8.55402-8.554l2.9314-6.128069c-.9603-.189528-1.9459-.2906093-2.9314-.2906093-8.26341 0-14.9727 6.7092883-14.9727 14.9726783 0 8.2761 6.70929 14.9727 14.9727 14.9727 3.2978 0 6.3429-1.0613 8.8194-2.8682l-3.2473-5.6226c-1.4909 1.2888-3.4368 2.0722-5.5721 2.0722z"></path></g></svg>                                                                                                                                                                                                                                                                                                                                                                                                                           plugins/wordpress-seo-extended/packages/js/images/mastodon-logo.svg                                 0000644                 00000013071 15122266557 0021616 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" width="140" height="100%" viewBox="0 0 713.35878 175.8678">
	<path fill="#2b90d9" d="M160.55476 105.43125c-2.4125 12.40625-21.5975 25.9825-43.63375 28.61375-11.49125 1.3725-22.80375 2.63125-34.8675 2.07875-19.73-.90375-35.2975-4.71-35.2975-4.71 0 1.92125.11875 3.75.355 5.46 2.565 19.47 19.3075 20.6375 35.16625 21.18125 16.00625.5475 30.2575-3.9475 30.2575-3.9475l.65875 14.4725s-11.19625 6.01125-31.14 7.11625c-10.99875.605-24.65375-.27625-40.56-4.485C6.99851 162.08 1.06601 125.31.15851 88-.11899 76.9225.05226 66.47625.05226 57.74125c0-38.1525 24.99625-49.335 24.99625-49.335C37.65226 2.6175 59.27976.18375 81.76351 0h.5525c22.48375.18375 44.125 2.6175 56.72875 8.40625 0 0 24.99625 11.1825 24.99625 49.335 0 0 .3125 28.1475-3.48625 47.69"/>
	<path fill="#282c37" d="M34.65751 48.494c0-5.55375 4.5025-10.055 10.055-10.055 5.55375 0 10.055 4.50125 10.055 10.055 0 5.5525-4.50125 10.055-10.055 10.055-5.5525 0-10.055-4.5025-10.055-10.055m144.20725 12.20575v46.195h-18.30125v-44.8375c0-9.4525-3.9775-14.24875-11.9325-14.24875-8.79375 0-13.2025 5.69125-13.2025 16.94375V89.2935h-18.19375V64.75225c0-11.2525-4.40875-16.94375-13.2025-16.94375-7.955 0-11.9325 4.79625-11.9325 14.24875v44.8375H73.79851v-46.195c0-9.44125 2.40375-16.94375 7.2325-22.495 4.98-5.55 11.50125-8.395 19.595-8.395 9.36625 0 16.45875 3.59875 21.14625 10.79875l4.56 7.6425 4.55875-7.6425c4.68875-7.2 11.78-10.79875 21.1475-10.79875 8.09375 0 14.61375 2.845 19.59375 8.395 4.82875 5.55125 7.2325 13.05375 7.2325 22.495m63.048 22.963875c3.77625-3.99 5.595-9.015 5.595-15.075 0-6.06-1.81875-11.085-5.595-14.9275-3.63625-3.99125-8.25375-5.91125-13.84875-5.91125-5.59625 0-10.2125 1.92-13.84875 5.91125-3.6375 3.8425-5.45625 8.8675-5.45625 14.9275 0 6.06 1.81875 11.085 5.45625 15.075 3.63625 3.8425 8.2525 5.76375 13.84875 5.76375 5.595 0 10.2125-1.92125 13.84875-5.76375m5.595-52.025h18.04625v73.9h-18.04625v-8.72125c-5.455 7.2425-13.01 10.79-22.80125 10.79-9.3725 0-17.34625-3.695-24.06125-11.23375-6.57375-7.5375-9.93125-16.84875-9.93125-27.785 0-10.78875 3.3575-20.10125 9.93125-27.63875 6.715-7.5375 14.68875-11.38 24.06125-11.38 9.79125 0 17.34625 3.5475 22.80125 10.78875v-8.72zm78.76175 35.62c5.315 3.99 7.97375 9.60625 7.83375 16.7 0 7.53875-2.65875 13.45-8.11375 17.58875-5.45625 3.99125-12.03 6.06-20.00375 6.06-14.40875 0-24.20125-5.9125-29.3775-17.58875l15.66875-9.31c2.0975 6.35375 6.71375 9.60625 13.70875 9.60625 6.43375 0 9.6525-2.07 9.6525-6.35625 0-3.10375-4.1975-5.91125-12.73-8.1275-3.21875-.8875-5.87625-1.77375-7.97375-2.51375-2.9375-1.18125-5.455-2.5125-7.55375-4.1375-5.17625-3.99-7.83375-9.3125-7.83375-16.11 0-7.2425 2.5175-13.00625 7.55375-17.145 5.17625-4.28625 11.47-6.355 19.025-6.355 12.03 0 20.84375 5.1725 26.5775 15.66625l-15.38625 8.8675c-2.23875-5.02375-6.015-7.53625-11.19125-7.53625-5.45625 0-8.11375 2.06875-8.11375 6.05875 0 3.10375 4.19625 5.91125 12.73 8.12875 6.575 1.4775 11.75 3.695 15.5275 6.50375m57.357125-17.2925h-15.8075v30.7425c0 3.695 1.4 5.91125 4.0575 6.945 1.95875.74 5.875.8875 11.75.59125v17.29375c-12.16875 1.4775-20.9825.295-26.15875-3.69625-5.175-3.8425-7.69375-10.93625-7.69375-21.13375v-30.7425h-12.17v-18.3275h12.17v-14.9275l18.045-5.76375v20.69125h15.8075v18.3275zm57.49825 33.254375c3.6375-3.84375 5.455-8.72125 5.455-14.6325 0-5.91125-1.8175-10.78875-5.455-14.63125-3.6375-3.84375-8.11375-5.76375-13.57-5.76375-5.455 0-9.93125 1.92-13.56875 5.76375-3.4975 3.99-5.31625 8.8675-5.31625 14.63125 0 5.765 1.81875 10.6425 5.31625 14.6325 3.6375 3.8425 8.11375 5.76375 13.56875 5.76375 5.45625 0 9.9325-1.92125 13.57-5.76375m-39.86875 13.15375c-7.13375-7.5375-10.63125-16.70125-10.63125-27.78625 0-10.9375 3.4975-20.1 10.63125-27.6375 7.13375-7.5375 15.9475-11.38 26.29875-11.38 10.3525 0 19.165 3.8425 26.3 11.38 7.135 7.5375 10.77125 16.84875 10.77125 27.6375 0 10.9375-3.63625 20.24875-10.77125 27.78625-7.135 7.53875-15.8075 11.2325-26.3 11.2325-10.49125 0-19.165-3.69375-26.29875-11.2325M524.92126 83.663625c3.6375-3.99 5.455-9.015 5.455-15.075 0-6.06-1.8175-11.085-5.455-14.9275-3.63625-3.99125-8.25375-5.91125-13.84875-5.91125-5.59625 0-10.2125 1.92-13.98875 5.91125-3.63625 3.8425-5.45625 8.8675-5.45625 14.9275 0 6.06 1.82 11.085 5.45625 15.075 3.77625 3.8425 8.5325 5.76375 13.98875 5.76375 5.595 0 10.2125-1.92125 13.84875-5.76375m5.455-81.585h18.04625v103.46h-18.04625v-8.72125c-5.315 7.2425-12.87 10.79-22.66125 10.79-9.3725 0-17.485-3.695-24.2-11.23375-6.575-7.5375-9.9325-16.84875-9.9325-27.785 0-10.78875 3.3575-20.10125 9.9325-27.63875 6.715-7.5375 14.8275-11.38 24.2-11.38 9.79125 0 17.34625 3.5475 22.66125 10.78875v-38.28zm81.42 81.141875c3.63625-3.84375 5.455-8.72125 5.455-14.6325 0-5.91125-1.81875-10.78875-5.455-14.63125-3.6375-3.84375-8.11375-5.76375-13.57-5.76375-5.455 0-9.9325 1.92-13.56875 5.76375-3.49875 3.99-5.31625 8.8675-5.31625 14.63125 0 5.765 1.8175 10.6425 5.31625 14.6325 3.63625 3.8425 8.11375 5.76375 13.56875 5.76375 5.45625 0 9.9325-1.92125 13.57-5.76375m-39.86875 13.15375c-7.135-7.5375-10.63125-16.70125-10.63125-27.78625 0-10.9375 3.49625-20.1 10.63125-27.6375 7.135-7.5375 15.9475-11.38 26.29875-11.38 10.3525 0 19.165 3.8425 26.3 11.38 7.135 7.5375 10.77125 16.84875 10.77125 27.6375 0 10.9375-3.63625 20.24875-10.77125 27.78625-7.135 7.53875-15.8075 11.2325-26.3 11.2325-10.49125 0-19.16375-3.69375-26.29875-11.2325m141.43125-36.210375v45.37375h-18.04625v-43.00875c0-4.8775-1.25875-8.5725-3.77625-11.38-2.37875-2.5125-5.73625-3.84375-10.0725-3.84375-10.2125 0-15.3875 6.06-15.3875 18.3275v39.905h-18.04625v-73.89875h18.04625v8.27625c4.33625-6.94625 11.19-10.345 20.84375-10.345 7.69375 0 13.98875 2.66 18.885 8.12875 5.035 5.46875 7.55375 12.85875 7.55375 22.465"/>
</svg>
                                                                                                                                                                                                                                                                                                                                                                                                                                                                       plugins/wordpress-seo-extended/packages/js/images/motivated_bubble_woman_1_optim.svg                0000644                 00000016110 15122266557 0025171 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 277.7 317.3">
  <circle cx="146.5" cy="186.1" r="131.2" fill="#f0ecf0"/>
  <path fill="#54311e" d="M95 41.1c4.8-22 19-41.6 42.7-38.8 21 2.4 34.5 17.4 47.4 33.6 16.5 20.8 14.7 46.5 22.6 71.8.5 1.9 1.4 4.1 1 6-.6 2.4-.6 3.8-2.6 5.3-15.6 11.7-21.5 26-32.4 42.2-2.7 4-5.3 8-9.3 10.5-2.5 1.5-.3.5-2.7 2.2-14.8 10.2-41.3 11-56.6 8.4 5.9-26 6.3-38.3 6.3-38.3l-15 39.3c-8 4.8-20.8 1.4-28.8-3.4C80 155.6 89 112.6 90.2 82.6c.5-14.4 1.8-27.4 4.8-41.5Z"/>
  <path fill="#54311e" d="M93.9 43.6c-4 34.7-3.2 73-15 105.8 3-4.8 5.4-9.2 8.5-14 3.9-6 4.3-15 6.2-22 2-7.2-.3-16-.5-23.5-.6-17 .9-34 2.2-51"/>
  <path fill="#ce6d42" d="M21 236.3c7.5-7.6 35.2-26.2 43.7-33.2 11.1-9.3 52.2 8.7 37.4 21.9-32.5 29-93.9 58.6-99.6 43-2.8-7.4 13.4-26.8 18.4-31.8Z"/>
  <path fill="#77375a" d="M138 152c-11.1-2.1-35.6 6.3-43.7 14.4C86 174.6 81.6 189 59 206.8c6.7 6.3 15.9 28.4 19.8 36.5 16.2-9.8 29-22.6 40.2-32.4 1.5-1.4-5-22.8-4.2-24.4.5-1.1-1.1-5.7-1-7.2 0-4 24.1-20.5 15-17.7Z"/>
  <path fill="#f0ecf0" d="M262.1 248.1c-10.1-3.4-25-8.2-31.7-10.6a216.5 216.5 0 0 1-2.9 34.8c-.9 5.1.5 10.2 2.7 14.7a131.6 131.6 0 0 0 32-38.9Z"/>
  <path fill="#a52a6a" d="M227.5 272.3a216.5 216.5 0 0 0 2.9-34.8l-3.4-1.2a51.7 51.7 0 0 0 3.4-5c0-7.6-.1-15.3.2-24 .5-18 1.3-22.5-17-41.6-22.8-24-91-23.3-111.5 5.6-17.8 25.2-7.7 40.4-9.9 58C90.1 246 65 286 112.6 295.5c1.9 4.5 7.5 13 12.2 19.9A131.3 131.3 0 0 0 230.2 287a24.5 24.5 0 0 1-2.7-14.7Z"/>
  <path fill="#f4a269" d="M256.7 192.5c-7.3 9.2-20.2 29.5-26.3 38.8v6.2l31.7 10.6a130.3 130.3 0 0 0 13.7-39.5c-7.9-6.3-16-12.9-19.1-16.1Z"/>
  <path fill="#f4a269" d="M230.4 231.3a51.7 51.7 0 0 1-3.4 5l3.4 1.2v-6.2Z"/>
  <path fill="#a52a6a" d="M248.8 243.2c-36.8-3.1-61.4-25.4-95.7-33.9 9-20.8 16.4-62 45-58.4 52.2 6.6 47 33.3 74.9 54.8-2.9 9-17.5 31.4-24.2 37.5Z"/>
  <path fill="#f4a269" d="M204.4 160c-8.8 25.2-10 40.6-21.1 56.5-12 0-76.7.7-79.8-6.4-.3-.7 6-18.4 13.4-30.3 9.2-14.9 25.4-21 27-25.4 1.7-4.7-6.5-26-5.3-28.7.7-1.4-3.7.9 44.8-38.9 2.8 7.6 6 42.2 6 55.5 0 17.7 3.1 13.4 15 17.7Z"/>
  <path fill="#ce6d42" d="M185.5 105.9c-12 11-29.5 23-46.1 25.5 1.5 6.3 4.6 16 4.7 20.8 21-1.9 41.3-46.2 41.4-46.3Z"/>
  <path fill="#54311e" d="M163.5 2.5c27 4.9 45.3 49 44.8 72.9-1 52.3 34.6 49 34.3 94.3-35.3-32.2-45.6-3-59.3 25-7.8-2.9-18.4-5-22-6.4-.6-25.4 7.1-31.8 8.7-43.9-7.2 12.5-11 15.4-15.7 29 .3-31.8 16-39 26.7-66.6v-1.3c.4-8.2 1-13.8 1.5-22.5s5.2-12.5 5.6-23.6c.6-17.6-19.9-40.5-24.6-57Z"/>
  <path fill="#7c2050" d="M190.6 257.2c2.5 5.3.4 11.5-1.8 16.9a27.4 27.4 0 0 1-6.9 11.2c-3 2.5-6.8 3.7-10.6 4.4a65.3 65.3 0 0 1-27.1-.8c19-1.8 37.8-10.5 46.4-31.7Z"/>
  <path fill="#f4a269" d="M98 72.3c-3.3 4.1-3.4 9.9-3 18.7.6 11.2 13.6 31.2 16.4 39.3 4.2 12.1 15.3 12.1 28.4 6.3 10.9-4.9 36.1-18.3 41-29.7A74.3 74.3 0 0 0 187 81s9.4-25.8 6.2-47.1-55.7-21.8-55.7-21.8S87 23.6 98 72.3Z"/>
  <path fill="#d17154" d="M110.2 109.8c.8-2.6 7.5-.3 27-1.9-7.5 3.2-11.5 9.2-17 10-4.5.6-6 .7-8.3-.9-3.3-2.3-3.9-5-1.8-7.3Z"/>
  <path fill="#b24736" d="M140.8 106.7c-10 1.6-16.7-1.6-21.9-1.4-1.7 0-2.5 1.8-3.7 1.5s-2.8-1.9-4.1-2c-1 0-2.8.1-3.5 2.8-.8 1.5 1 3 2.2 2.9-.1-.6 2-1 3.3-.4.8.4 1.8 2.6 4.7-.2 3.7 1 13 1 23-3.2Z"/>
  <path fill="#54311e" d="M151.6 5c46.8-13.5 65.1 42.1 55.6 76.3-2.7 1.3-12.1-9.3-19-3.9-20.4-11.8-18.4-31-21-37.5-12-14.2-59-27.4-68.2-1.4-2-18.3 24.8-51 52.6-33.5Z"/>
  <path d="M102.8 50.2c-4 .8-6.7 5.5-6.5 10.5.9-1.3 3.6-7.1 6.4-7.4 2.5-.2 4.9 1.6 6.6 2 1 .2 1.3-1.2.6-3-2.6-2.4-4.4-2.6-7-2.1Z"/>
  <path fill="#ce6d42" d="M186.8 86.6c11-14 16-6.7 11.5.8-.3-10.5-5-4.4-9.8-1 5.8 3.4.9 8.1-.7 7s-4.8-.8-3.4 2c-1.8 0-1.2-1.6-1.3-2.8 0-1.8 1.6-.7 4-1.6 3.4-1.2 0-4-.3-4.4Z"/>
  <path fill="#f4a269" d="M183.5 82.6c2.4-5.6 8.2-12.7 16.8-9.8 4.3 1.5 7.2 6.2-.5 19.2-3.5 6-8 15.4-15.1 15-5.1-.4-5.6-7-4-15Z"/>
  <path fill="#ce6d42" d="M187 85.1c8.6-15.6 16.6-10 13.2-1.9-2-10.3-7.4-2.6-11.6 1.5 6.3 2.4 2.2 7.8.5 7s-4.9 0-3 2.4c-1.8.3-1.5-1.3-1.7-2.5-.4-1.7 1.4-1 3.6-2.2 3.1-1.7-.7-4-1-4.3Z"/>
  <path fill="#fff" d="M104.6 65.4c-3.4-.8-5 .6-6.5 3.7.3 3 1 6 3.7 6.8s10.4.3 13.5-1c-4.2-7-4.4-8-10.7-9.5Z"/>
  <path d="M97.6 71c.1 2.4 2.6 5.1 5 5s3.7-3 3.6-5.5a4 4 0 0 0-4-3.8h-.4a4.1 4.1 0 0 0-4.2 4Z"/>
  <path fill="#fff" d="M140.6 66.3c-3.5 1.2-5.2 5.1-6.4 8.6 1.3 4 17.6 3 21.7-3-6.6-3.4-9.8-7.6-15.3-5.6Z"/>
  <path d="M134.5 72.2c-1.6 4 1.9 5.2 4.6 5a5.1 5.1 0 0 0 4.6-5.4c-.2-2.7-2.2-5.4-4.9-5.2s-2.8 2-4.3 5.6Z"/>
  <path fill="#ce6d42" d="M144.5 60c7.3.5 10 9.9 11.2 12-9.9-8.3-17.7-5.1-21.5-.2-.2-5.5 4-12 10.3-11.7Z"/>
  <path d="M159.8 70.8c-8.9 0-6.5-8-16.7-6.4-6 1.1-9.7 7.6-9.3 11.3.5.2 1.8-10 12-8.2 4.5.7 6.7 6 10.7 5 .9-.2 3.5-1.3 3.3-1.7Z"/>
  <path fill="#ce6d42" d="M108.1 59.2c-9.4-4.4-10.4 9.2-10.1 10 1.4-4.5 10.5-4.5 16.4 4.1-.2-3.8-3.2-12.6-6.3-14Z"/>
  <path d="M115.1 74.8c-2-5.7-4.4-10.6-9.7-11.5a7.2 7.2 0 0 0-8.1 5.3c-.4 1.4-2.8.3-3.4.3.4.9 3 3.5 4 2.8 1.7-1 2.6-4.7 4.5-4.7a17.2 17.2 0 0 1 7.4 2c3.5 1.9 4.7 6 5.3 5.8Z"/>
  <path fill="#ce6d42" d="M115.3 72c-4 6-13.5 23.7-10.1 27.8 1.8 2.4 13.2 0 19.9-2.8-5.3-.7-14.1 1.5-15.9 0-3.3-5.6 7.5-21.1 9.4-26.7 1.6-4.5.5-9.4-6.2-13.9 8.7 7.4 5.7 11 2.9 15.7Z"/>
  <path fill="#54311e" d="M148.3 48.7c-21.5-2.5-33.3-17.5-15.9-33.4 2.4 13.5 0 25.4 16 33.4Z"/>
  <path fill="#54311e" d="M127.7 21c-19.9 4.7-16.4 27.6-6.7 41-5.7-12.7.3-30 5.7-39.5"/>
  <path fill="#f4a269" d="M39.8 236c-5.6 35.6-27.5 41.4-35.4 37-6.4-3.4-4.1-8.3-3.5-14.8.7-7.4-1.2-11.5 0-21 1.3-11.3 27.3-62 37.3-80.4 7.8-14 25.3-4.8 20.3 2.8-20.4 40.6-14 46.3-18.7 76.3Z"/>
  <path fill="#f4a269" d="M54.6 162.1c6.3.9 4.7 1.8 13.3-4.4 5.7-4.2 1.3-3.4 6.4-8.4 8.3-8.2 7.4-21.2 19.6-24.1 7.5-1.9 19 5 23.5-4.2-10.5-1.5-14.5-2.9-21.9-4.7a6.3 6.3 0 0 0-3.5.2c-5.1 2-7.6 2.8-10.2 4.3-3.5-.1-6.3-.5-9.5 3-9.5 10.6-25.2 19.7-32.2 30.2-6.7 10 4.5 7.4 14.5 8.1Z"/>
  <path fill="#f4a269" d="M67.7 138.5a35 35 0 0 1 10.3 6.9c2.9 4.1 6 8.8 5.8 10.3-.4 4.3-3.6 5.2-6 5.6-2.4.4-8.4 2.6-9.4 1.6-3.5-3.7-3.2-5-3.4-7.5-.3-5 3.3-4.4 4.5-2.7-2.8-3.2-2-2.5-2.6-6.9-.5-4.2-3.5-6.9.8-7.3Z"/>
  <path fill="#f4a269" d="M79 127.3c3.7 2 5.2 8 7.3 11.5 1.7 2.7 5.3 9.8 5.2 13.1-.2 4-12 5-14.4 4.2a87.6 87.6 0 0 1-7.2-13.9C67 135 76.4 132 80.5 127.1a5.5 5.5 0 0 0 1 .5"/>
  <path fill="#f4a269" d="M84.8 152.3c3.4-.6 13.6-1.2 12-6.5-1-3.5-4-8.3-6.2-11.4-2.5-3.5-3.6-10.3-7.7-8.3-13.8 6.8-4.7 27.8 1.9 26.2Z"/>
  <path fill="#ce6d42" d="M87.5 129.2c-.6-2.4-1.5-3.6.5-6 0 4-.5 6 1.6 9.6Zm4 22.9c-2-8.5-15.8-16.4-10.5-23.2-1.6 5.2 3 7.7 5.9 12 2.3 3.3 4.8 7.9 4.6 11.2Zm-7.8 4.3c.5-3-1.7-8.3-4.3-12.4a9 9 0 0 0-7-4.6c3.2 1.3 3.5 2.2 5.2 4.8 2 3 5.4 7.9 6 12.2Zm-7.5 3.6c-2-.6-5.1-1-6.6-2.7-2-2.1-2.5-9-2.5-11.8.3 2 1.2 6.5 2.6 8.2s5.1 4.3 6.5 6.2Z"/>
  <path fill="#ce6d42" d="M69.1 155.7c-1 1.2-1 5.2.6 6.1-.3-1.4-.5-3.9 1-4.7m-3.3-8.1c.8 6-6.4 12.5-17.9 10.4 3.4 2 7.8 4.4 11.3 3.1a14.8 14.8 0 0 0 7.8-7.3c.6-1.4-1-3.7-1.2-6.2Zm49-24.3a3.1 3.1 0 0 0 1.7-2.3c-.1-.5-.4-.9-1.8-1.1-4.2-.7-10.5-1.4-14-3.3 5.7 1.8 15.8 2.7 16.6 3a1.2 1.2 0 0 1 .7 1.4c-2 6.3-13.2 3.8-21.4 2.4 4.8.4 12 2.2 18.2-.1Zm22.3-17.7c2-.7 3.2-2.5 2.5-5.8 1.6.7 4.5 5.8 4.8 7.8-2.4-2.7-4.6-1.7-6-2"/>
  <path d="M144.3 56.5c-3-.3-10 2.4-12.7 2.6-1.7 0-2 1.6-.6 3 5.4.8 10-3 13.5-3 3.7 0 9.4.6 14.1 3.3-2.1-3.9-10.3-5.7-14.3-6Z"/>
</svg>
                                                                                                                                                                                                                                                                                                                                                                                                                                                        plugins/wordpress-seo-extended/packages/js/images/semrush-logo.svg                                  0000644                 00000007766 15122266557 0021476 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg width="170" height="27" viewBox="0 0 200 27" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M138.086 10.4127C138.086 7.13658 136.073 4.76172 132.217 4.76172H119.751V21.7382H123.85V15.9412H128.7L133.453 21.7365H138.072V21.3622L133.453 15.8438C136.335 15.2497 138.086 13.1333 138.086 10.4127ZM131.757 12.5123H123.85V8.27785H131.757C133.219 8.27785 134.158 9.0566 134.158 10.401C134.157 11.7671 133.25 12.5123 131.757 12.5123Z" fill="#171A22"/>
<path d="M200 4.76172H196.167V11.1982H185.836V4.76172H181.689V21.7382H185.836V15.0533H196.167V21.7382H200V4.76172Z" fill="#171A22"/>
<path d="M108.78 4.76172L104.222 19.0461H103.972L99.388 4.76172H92.0889V21.7382H95.9927V7.79281H96.2361L100.794 21.7382H107.083L111.667 7.79281H111.908V21.7382H115.935V4.76172H108.78Z" fill="#171A22"/>
<path d="M64.1764 11.7084C62.728 11.5574 60.0275 11.3107 58.5791 11.1613C57.1542 11.0153 56.3302 10.5923 56.3302 9.65079C56.3302 8.74785 57.2046 7.98588 60.7325 7.98588C63.8458 7.98588 66.7191 8.65218 69.2266 9.86227V6.06417C66.7191 4.86751 63.9633 4.28345 60.521 4.28345C55.6924 4.28345 52.3491 6.3025 52.3491 9.71625C52.3491 12.6064 54.3111 14.184 58.2468 14.6137C59.6717 14.7697 62.0936 14.9829 63.7837 15.1222C65.6248 15.2749 66.172 15.8372 66.172 16.6763C66.172 17.8277 64.8813 18.5427 61.6119 18.5427C58.2838 18.5427 54.9153 17.4551 52.5186 15.9328V19.8417C54.4437 21.129 57.7769 22.2216 61.4894 22.2216C66.7644 22.2216 70.1497 20.1874 70.1497 16.5454C70.1497 13.8047 68.3421 12.1415 64.1764 11.7084Z" fill="#171A22"/>
<path d="M72.8335 4.76172V21.7382H88.6687V18.269H76.7373V14.8251H88.4656V11.3828H76.7373V8.22918H88.6687V4.76172H72.8335Z" fill="#171A22"/>
<path d="M173.218 11.7082C171.77 11.5572 169.07 11.3104 167.621 11.1611C166.196 11.015 165.372 10.5921 165.372 9.65055C165.372 8.7476 166.247 7.98563 169.774 7.98563C172.888 7.98563 175.761 8.65194 178.269 9.86202V6.06393C175.759 4.86895 173.005 4.2832 169.563 4.2832C164.734 4.2832 161.391 6.30225 161.391 9.71601C161.391 12.6061 163.353 14.1838 167.289 14.6134C168.714 14.7695 171.136 14.9827 172.826 15.122C174.667 15.2747 175.214 15.8369 175.214 16.6761C175.214 17.8274 173.923 18.5424 170.654 18.5424C167.326 18.5424 163.957 17.4549 161.561 15.9326V19.8415C163.486 21.1288 166.819 22.2214 170.531 22.2214C175.806 22.2214 179.192 20.1872 179.192 16.5452C179.192 13.8045 177.386 12.1412 173.218 11.7082Z" fill="#171A22"/>
<path d="M154.503 4.76172V13.4522C154.503 16.7451 152.503 18.5527 149.495 18.5527C146.471 18.5527 144.487 16.772 144.487 13.4018V4.76172H140.412V13.0158C140.412 19.1804 144.264 22.2232 149.554 22.2232C154.644 22.2232 158.575 19.2995 158.575 13.2172V4.76172H154.503Z" fill="#171A22"/>
<path d="M38.3066 13.1515C38.3066 13.9789 37.892 14.1065 36.8414 14.1065C35.7269 14.1065 35.5356 13.9152 35.4081 13.0877C35.1848 10.9546 33.7515 9.13859 31.333 8.94894C30.5694 8.88516 30.3781 8.59816 30.3781 7.64318C30.3781 6.75198 30.5056 6.33743 31.2055 6.33743C35.4081 6.33743 38.3066 9.71258 38.3066 13.1515ZM44.4208 13.1515C44.4208 6.75198 40.0906 0 30.0911 0H10.2144C9.81329 0 9.56489 0.209793 9.56489 0.575672C9.56489 0.775396 9.71426 0.954979 9.85021 1.06071C10.5786 1.63303 11.6393 2.26241 13.0626 2.97403C14.4422 3.66383 15.5079 4.11195 16.5904 4.55167C17.0369 4.73293 17.2148 4.93098 17.2148 5.19616C17.2148 5.5419 16.9714 5.76176 16.4679 5.76176H0.691478C0.224898 5.76176 0 6.06218 0 6.36596C0 6.62443 0.0906307 6.83087 0.310494 7.05912C1.59107 8.39508 3.6269 10.0063 6.60261 11.8676C9.21076 13.4989 12.4332 15.2595 14.9893 16.4847C15.419 16.6911 15.5683 16.9295 15.5583 17.1762C15.5465 17.4615 15.3216 17.7032 14.8248 17.7032H7.59283C7.18164 17.7032 6.95338 17.9197 6.95338 18.252C6.95338 18.4366 7.10276 18.6716 7.29745 18.8495C8.94222 20.3415 11.5722 21.9729 15.0766 23.4649C19.7508 25.4555 24.5089 26.6521 29.8527 26.6521C39.9799 26.6521 44.4208 19.0744 44.4208 13.1515ZM31.2055 22.2263C26.2376 22.2263 22.0988 18.1815 22.0988 13.1515C22.0988 8.18361 26.2376 4.20426 31.2055 4.20426C36.3009 4.20426 40.2803 8.18529 40.2803 13.1515C40.2803 18.1815 36.3009 22.2263 31.2055 22.2263Z" fill="#FF642D"/>
</svg>          plugins/wordpress-seo-extended/packages/js/images/ssp-logo.svg                                      0000644                 00000074161 15122266557 0020606 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8"?>
<!-- Generated by Pixelmator Pro 3.1.1 -->
<svg width="280" height="50" viewBox="0 0 280 50" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <path id="Path" fill="#2a2a2a" stroke="none" d="M 107.36734 25 C 107.36734 37.90543 96.905426 48.367348 84 48.367348 C 71.094574 48.367348 60.632652 37.90543 60.632652 25 C 60.632652 12.09457 71.094574 1.632652 84 1.632652 C 96.905426 1.632652 107.36734 12.09457 107.36734 25 Z"/>
    <path id="path1" fill="#2a2a2a" fill-rule="evenodd" stroke="none" d="M 84 50 C 97.807144 50 109 38.807144 109 25 C 109 11.192875 97.807144 0 84 0 C 70.192879 0 59 11.192875 59 25 C 59 38.807144 70.192879 50 84 50 Z M 84 48.979591 C 97.243469 48.979591 107.979599 38.243469 107.979599 25 C 107.979599 11.756432 97.243469 1.020409 84 1.020409 C 70.756432 1.020409 60.020409 11.756432 60.020409 25 C 60.020409 38.243469 70.756432 48.979591 84 48.979591 Z"/>
    <path id="path2" fill="#ffffff" fill-rule="evenodd" stroke="none" d="M 72.282471 16.767059 C 72.576691 16.588104 72.956955 16.356773 73.387756 16.42857 C 73.623169 16.467796 73.873627 16.657875 74.162415 16.876999 C 74.62468 17.227814 75.185089 17.653061 75.938782 17.653061 C 76.808289 17.653061 77.26619 17.447247 77.531647 17.327919 C 77.640045 17.279202 77.716339 17.2449 77.775513 17.2449 C 77.979599 17.2449 78.285721 17.551022 77.877548 17.857143 C 77.810303 17.90757 77.726456 17.977406 77.626404 18.060673 C 77.119308 18.482878 76.196823 19.250917 74.918365 19.591837 C 74.67543 19.656633 74.45047 19.711123 74.242691 19.761471 C 73.141464 20.028288 72.522842 20.178164 72.265305 21.12245 C 71.959183 22.244898 72.163269 26.734694 75.938782 26.734694 C 77.502655 26.734694 78.733871 26.121838 80.176331 25.404083 C 82.216324 24.388775 84.678772 23.163265 89.102036 23.163265 C 96.653061 23.163265 98.18367 29.591837 98.18367 32.857143 C 98.18367 36.122448 94.510208 40.612244 88.18367 40.612244 C 83.733261 40.612244 82.217758 39.113266 81.431229 38.335102 C 81.323273 38.228367 81.228973 38.135101 81.142853 38.061226 C 81.653061 39.387756 84.204086 42.224487 89.102036 42.142857 C 89.034088 42.516937 87.346939 43.265305 81.142853 43.265305 C 73.387756 43.265305 67.367348 36.32653 67.367348 30.102041 C 67.367348 25.671429 68.504753 23.102039 69.160347 21.621019 C 69.425797 21.021429 69.612244 20.600206 69.612244 20.306122 C 69.612244 19.355593 68.549713 18.405062 67.579391 17.537003 C 67.508064 17.473183 67.437225 17.409817 67.367348 17.346939 C 66.346939 16.42857 66.14286 14.693878 66.244896 14.387756 C 66.346939 14.081635 66.755104 13.979591 66.85714 14.591835 C 66.959183 15.204082 67.979591 15.816326 68.693878 15.816326 C 68.98645 15.816326 69.142059 15.782082 69.293938 15.748672 C 69.512878 15.700508 69.72406 15.654041 70.32653 15.714287 C 70.836304 15.765266 71.452957 16.410797 71.809486 16.784061 C 71.866554 16.843777 71.916962 16.896549 71.959183 16.938774 C 72.049911 16.908531 72.158554 16.842449 72.282471 16.767059 Z M 81.755096 35.306122 C 82.149597 35.306122 82.469391 34.986328 82.469391 34.591835 C 82.469391 34.19735 82.149597 33.877548 81.755096 33.877548 C 81.360611 33.877548 81.040817 34.19735 81.040817 34.591835 C 81.040817 34.986328 81.360611 35.306122 81.755096 35.306122 Z"/>
    <path id="path3" fill="#cf3529" stroke="none" d="M 78.18367 12.448978 C 77.775513 12.766426 77.713882 13.572571 78.133331 13.984653 C 78.553329 14.460693 79.102036 14.452061 79.668976 14.035 C 83.489792 11.224491 93.591843 11.32653 97.979599 13.984653 C 98.476532 14.285713 99.070404 14.358795 99.450821 13.971615 C 99.450821 13.971615 99.450821 13.971615 99.482452 13.939346 C 99.894485 13.519897 99.918365 12.763142 99.404892 12.404797 C 94.204086 8.775509 82.775513 8.877552 78.18367 12.448978 Z"/>
    <path id="path4" fill="#cf3529" stroke="none" d="M 80.63266 16.122448 C 80.020401 16.42857 80.020401 17.142857 80.326538 17.653061 C 80.63266 18.163265 81.276932 18.173552 81.857147 17.857143 C 86.174896 15.928654 91.316742 15.710407 95.734695 17.959183 C 96.228363 18.210409 97.061218 18.265306 97.36734 17.653061 C 97.659592 17.068653 97.673462 16.530613 96.959183 16.122448 C 92.583466 13.622063 85.619598 13.331001 80.63266 16.122448 Z"/>
    <path id="path5" fill="#cf3529" stroke="none" d="M 82.877548 20.102041 C 82.265305 20.408163 82.163269 21.071428 82.469391 21.530613 C 82.945511 22.244898 83.591843 22.025309 84.14183 21.822653 C 87.877548 20.445713 89.918365 20.510204 93.489792 21.772243 C 94.213669 22.027958 94.714279 22.040817 95.122452 21.530613 C 95.428574 21.020409 95.366943 20.331713 94.612244 20 C 90.227753 18.072918 86.551025 18.061224 82.877548 20.102041 Z"/>
    <path id="Seriously-Simple" fill="#000000" fill-rule="evenodd" stroke="none" d="M 208.695313 24.455078 L 208.695313 13.416016 L 210.199219 13.416016 L 210.296875 14.167969 C 210.638199 13.858635 211.010681 13.629803 211.416016 13.480469 C 211.82135 13.331135 212.300797 13.255859 212.855469 13.255859 C 213.943466 13.255859 214.792389 13.552528 215.400391 14.144531 C 216.008392 14.736534 216.3125 15.714737 216.3125 17.080078 C 216.3125 18.44541 215.996521 19.467196 215.367188 20.144531 C 214.737854 20.821869 213.880966 21.160156 212.792969 21.160156 C 211.875626 21.160156 211.090118 20.946199 210.439453 20.519531 L 210.439453 24.455078 L 208.695313 24.455078 Z M 175.224609 23.816406 C 174.648605 23.816406 174.157288 23.741133 173.751953 23.591797 L 173.751953 22.296875 C 173.933289 22.360876 174.101196 22.402494 174.255859 22.423828 C 174.410538 22.445162 174.568512 22.455078 174.728516 22.455078 C 175.059189 22.455078 175.324677 22.403543 175.527344 22.296875 C 175.730011 22.190208 175.90979 22.022303 176.064453 21.792969 C 176.219131 21.563635 176.387039 21.266966 176.568359 20.904297 L 173.191406 13.416016 L 175 13.416016 L 177.431641 19.048828 L 179.896484 13.416016 L 181.6875 13.416016 L 178.599609 20.568359 C 178.332932 21.208363 178.048141 21.770523 177.744141 22.255859 C 177.44014 22.741194 177.087494 23.123056 176.6875 23.400391 C 176.287491 23.677725 175.800613 23.816406 175.224609 23.816406 Z M 151.576172 21.160156 C 150.210831 21.160156 149.210831 20.819389 148.576172 20.136719 C 147.941498 19.454048 147.623047 18.47637 147.623047 17.207031 C 147.623047 15.937691 147.943985 14.961967 148.583984 14.279297 C 149.223984 13.596626 150.221497 13.255859 151.576172 13.255859 C 152.941513 13.255859 153.943985 13.596626 154.583984 14.279297 C 155.223984 14.961967 155.544922 15.937691 155.544922 17.207031 C 155.544922 18.47637 155.226471 19.454048 154.591797 20.136719 C 153.957138 20.819389 152.952179 21.160156 151.576172 21.160156 Z M 159.015625 21.160156 C 158.108948 21.160156 157.413071 20.909538 156.927734 20.408203 C 156.442398 19.906866 156.199219 19.169888 156.199219 18.199219 L 156.199219 13.416016 L 157.943359 13.416016 L 157.943359 18.199219 C 157.943359 18.764553 158.082047 19.180597 158.359375 19.447266 C 158.636703 19.713934 159.064621 19.847656 159.640625 19.847656 C 160.067291 19.847656 160.461548 19.767424 160.824219 19.607422 C 161.18689 19.447422 161.5009 19.23542 161.767578 18.96875 L 161.767578 13.416016 L 163.496094 13.416016 L 163.496094 21 L 161.912109 21 L 161.832031 20.216797 C 161.490692 20.483465 161.084579 20.707336 160.615234 20.888672 C 160.145905 21.070005 159.612961 21.160156 159.015625 21.160156 Z M 167.095703 21.160156 C 166.573044 21.160156 166.07428 21.123495 165.599609 21.048828 C 165.124939 20.974161 164.73317 20.882059 164.423828 20.775391 L 164.423828 19.304688 C 164.786499 19.443354 165.185715 19.54838 165.623047 19.623047 C 166.060394 19.697714 166.476425 19.736328 166.871094 19.736328 C 167.46843 19.736328 167.896347 19.682838 168.152344 19.576172 C 168.408356 19.469505 168.535156 19.260508 168.535156 18.951172 C 168.535156 18.727173 168.459351 18.55183 168.304688 18.423828 C 168.150024 18.295826 167.919235 18.178936 167.615234 18.072266 C 167.311234 17.965599 166.924423 17.836834 166.455078 17.6875 C 166.00708 17.538166 165.617783 17.375217 165.287109 17.199219 C 164.956451 17.023216 164.700867 16.796867 164.519531 16.519531 C 164.338196 16.242195 164.248047 15.879642 164.248047 15.431641 C 164.248047 14.738304 164.498657 14.202885 165 13.824219 C 165.501343 13.445549 166.306671 13.255859 167.416016 13.255859 C 167.853348 13.255859 168.269394 13.28756 168.664063 13.351563 C 169.058731 13.415562 169.394547 13.495796 169.671875 13.591797 L 169.671875 15.048828 C 169.373215 14.931496 169.04483 14.83939 168.6875 14.775391 C 168.330154 14.711388 167.996826 14.679688 167.6875 14.679688 C 167.132828 14.679688 166.714294 14.728218 166.431641 14.824219 C 166.148972 14.920219 166.007813 15.112389 166.007813 15.400391 C 166.007813 15.699059 166.153931 15.908581 166.447266 16.03125 C 166.740601 16.153915 167.212616 16.321823 167.863281 16.535156 C 168.460617 16.716492 168.93512 16.908661 169.287109 17.111328 C 169.639114 17.313995 169.89415 17.557175 170.048828 17.839844 C 170.203491 18.122513 170.279297 18.487545 170.279297 18.935547 C 170.279297 19.682217 170.008835 20.23942 169.464844 20.607422 C 168.920837 20.975424 168.130386 21.160156 167.095703 21.160156 Z M 172.919922 21.160156 C 172.301254 21.160156 171.831726 21.007126 171.511719 20.703125 C 171.191727 20.399122 171.03125 19.895411 171.03125 19.191406 L 171.03125 10.216797 L 172.775391 10.216797 L 172.775391 19.03125 C 172.775391 19.329918 172.828873 19.533958 172.935547 19.640625 C 173.042206 19.747293 173.202682 19.800781 173.416016 19.800781 C 173.672012 19.800781 173.917679 19.767124 174.152344 19.703125 L 174.152344 20.96875 C 173.949677 21.043417 173.755035 21.092472 173.568359 21.119141 C 173.381699 21.145807 173.165253 21.160156 172.919922 21.160156 Z M 187.960938 21.160156 C 186.627594 21.160156 185.55481 20.989771 184.744141 20.648438 L 184.744141 19.111328 C 185.202805 19.281996 185.696609 19.420677 186.224609 19.527344 C 186.752609 19.63401 187.278107 19.6875 187.800781 19.6875 C 188.622116 19.6875 189.237778 19.592392 189.648438 19.400391 C 190.059113 19.208389 190.263672 18.824047 190.263672 18.248047 C 190.263672 17.896046 190.178482 17.613724 190.007813 17.400391 C 189.837143 17.187054 189.554825 17.002323 189.160156 16.847656 C 188.765488 16.692989 188.220688 16.530041 187.527344 16.359375 C 186.417999 16.071373 185.639404 15.706341 185.191406 15.263672 C 184.743408 14.821003 184.519531 14.210312 184.519531 13.431641 C 184.519531 12.524971 184.850388 11.816689 185.511719 11.304688 C 186.173065 10.792683 187.160645 10.535156 188.472656 10.535156 C 189.069992 10.535156 189.632156 10.578728 190.160156 10.664063 C 190.688156 10.749393 191.10173 10.844505 191.400391 10.951172 L 191.400391 12.488281 C 190.589722 12.178947 189.694229 12.023438 188.712891 12.023438 C 187.944885 12.023438 187.346588 12.125458 186.919922 12.328125 C 186.493256 12.530792 186.279297 12.898304 186.279297 13.431641 C 186.279297 13.740974 186.354568 13.991596 186.503906 14.183594 C 186.653244 14.375595 186.911301 14.541019 187.279297 14.679688 C 187.647293 14.818356 188.157928 14.973862 188.808594 15.144531 C 189.65126 15.357864 190.303574 15.610962 190.767578 15.904297 C 191.231583 16.197632 191.557465 16.538399 191.744141 16.927734 C 191.930801 17.31707 192.023438 17.757378 192.023438 18.248047 C 192.023438 19.154716 191.687622 19.86743 191.015625 20.384766 C 190.343613 20.902102 189.326263 21.160156 187.960938 21.160156 Z M 218.935547 21.160156 C 218.316879 21.160156 217.847351 21.007126 217.527344 20.703125 C 217.207336 20.399122 217.048828 19.895411 217.048828 19.191406 L 217.048828 10.216797 L 218.792969 10.216797 L 218.792969 19.03125 C 218.792969 19.329918 218.844513 19.533958 218.951172 19.640625 C 219.057846 19.747293 219.218292 19.800781 219.431641 19.800781 C 219.687637 19.800781 219.933304 19.767124 220.167969 19.703125 L 220.167969 20.96875 C 219.965302 21.043417 219.770645 21.092472 219.583984 21.119141 C 219.397324 21.145807 219.180878 21.160156 218.935547 21.160156 Z M 224.408203 21.160156 C 223.085526 21.160156 222.061279 20.834263 221.335938 20.183594 C 220.610596 19.532925 220.248047 18.535406 220.248047 17.191406 C 220.248047 15.975399 220.557114 15.016502 221.175781 14.3125 C 221.794449 13.608498 222.733505 13.255859 223.992188 13.255859 C 225.144196 13.255859 226.010468 13.557487 226.591797 14.160156 C 227.173141 14.762825 227.464844 15.538944 227.464844 16.488281 L 227.464844 17.896484 L 221.912109 17.896484 C 221.997437 18.632488 222.264877 19.138681 222.712891 19.416016 C 223.160889 19.69335 223.842438 19.832031 224.759766 19.832031 C 225.143768 19.832031 225.538025 19.79537 225.943359 19.720703 C 226.348694 19.646036 226.701324 19.548975 227 19.431641 L 227 20.712891 C 226.658661 20.862225 226.266876 20.974161 225.824219 21.048828 C 225.381546 21.123495 224.909531 21.160156 224.408203 21.160156 Z M 139.560547 21 L 139.560547 13.416016 L 141.160156 13.416016 L 141.255859 14.296875 C 141.61853 14.062206 142.051392 13.848251 142.552734 13.65625 C 143.054062 13.464252 143.548401 13.330524 144.039063 13.255859 L 144.039063 14.226563 L 144.136719 13.416016 L 146.871094 13.416016 L 146.871094 21 L 145.128906 21 L 145.128906 14.744141 L 143.976563 14.744141 L 143.994141 14.59375 C 143.707718 14.646713 143.402161 14.719975 143.072266 14.816406 C 142.725586 14.91774 142.397232 15.029675 142.087891 15.152344 C 141.778564 15.275009 141.518021 15.399342 141.304688 15.527344 L 141.304688 21 L 139.560547 21 Z M 193.144531 21 L 193.144531 14.744141 L 191.992188 14.744141 L 192.152344 13.416016 L 194.888672 13.416016 L 194.888672 21 L 193.144531 21 Z M 195.960938 21 L 195.960938 13.416016 L 197.560547 13.416016 L 197.640625 14.216797 C 198.013962 13.918129 198.423096 13.682384 198.871094 13.511719 C 199.319107 13.341049 199.800507 13.255859 200.3125 13.255859 C 200.856506 13.255859 201.279984 13.343529 201.583984 13.519531 C 201.887985 13.69553 202.119293 13.938709 202.279297 14.248047 C 202.631302 13.960045 203.03743 13.721825 203.496094 13.535156 C 203.954758 13.348488 204.504532 13.255859 205.144531 13.255859 C 206.019211 13.255859 206.664078 13.479733 207.080078 13.927734 C 207.496078 14.375736 207.703125 15.064182 207.703125 15.992188 L 207.703125 21 L 205.992188 21 L 205.992188 16.167969 C 205.992188 15.581299 205.868393 15.170216 205.623047 14.935547 C 205.377716 14.700878 204.9785 14.583984 204.423828 14.583984 C 204.061157 14.583984 203.717911 14.647388 203.392578 14.775391 C 203.067245 14.903389 202.787399 15.095558 202.552734 15.351563 C 202.584732 15.468895 202.604568 15.592701 202.615234 15.720703 C 202.6259 15.848701 202.632813 15.987385 202.632813 16.136719 L 202.632813 21 L 201.015625 21 L 201.015625 16.183594 C 201.015625 15.628925 200.920517 15.224754 200.728516 14.96875 C 200.536514 14.712749 200.173965 14.583984 199.640625 14.583984 C 199.267288 14.583984 198.91217 14.671658 198.576172 14.847656 C 198.240173 15.023659 197.938553 15.235134 197.671875 15.480469 L 197.671875 21 L 195.960938 21 Z M 212.34375 19.847656 C 213.815765 19.847656 214.552734 18.920458 214.552734 17.064453 C 214.552734 16.168449 214.387314 15.531013 214.056641 15.152344 C 213.725967 14.773674 213.171249 14.583984 212.392578 14.583984 C 211.613907 14.583984 210.962112 14.851429 210.439453 15.384766 L 210.439453 19.175781 C 210.684784 19.378448 210.959656 19.541395 211.263672 19.664063 C 211.567673 19.78673 211.92775 19.847656 212.34375 19.847656 Z M 151.576172 19.6875 C 152.376175 19.6875 152.943298 19.492851 153.279297 19.103516 C 153.615295 18.71418 153.783203 18.081703 153.783203 17.207031 C 153.783203 16.332359 153.615295 15.704315 153.279297 15.320313 C 152.943298 14.93631 152.376175 14.744141 151.576172 14.744141 C 150.786835 14.744141 150.22467 14.93631 149.888672 15.320313 C 149.552673 15.704315 149.384766 16.332359 149.384766 17.207031 C 149.384766 18.081703 149.552673 18.71418 149.888672 19.103516 C 150.22467 19.492851 150.786835 19.6875 151.576172 19.6875 Z M 221.912109 16.759766 L 225.927734 16.759766 L 225.927734 16.232422 C 225.927734 15.709755 225.784088 15.301147 225.496094 15.007813 C 225.208084 14.714478 224.721741 14.568359 224.039063 14.568359 C 223.228394 14.568359 222.671188 14.743702 222.367188 15.095703 C 222.063187 15.447704 221.912109 16.00243 221.912109 16.759766 Z M 144.984375 11.832031 L 144.984375 10.392578 L 146.888672 10.392578 L 146.888672 11.832031 L 144.984375 11.832031 Z M 193 11.832031 L 193 10.392578 L 194.904297 10.392578 L 194.904297 11.832031 L 193 11.832031 Z M 135.768005 21.16 C 134.445328 21.16 133.421341 20.834669 132.695999 20.184 C 131.970673 19.533331 131.608002 18.536007 131.608002 17.192001 C 131.608002 15.975994 131.917328 15.016003 132.536011 14.312 C 133.154678 13.607998 134.093338 13.256001 135.352005 13.256001 C 136.504013 13.256001 137.370667 13.557331 137.952011 14.16 C 138.53334 14.762669 138.824005 15.538662 138.824005 16.487999 L 138.824005 17.896 L 133.272003 17.896 C 133.35733 18.632004 133.624008 19.138666 134.072006 19.416 C 134.520004 19.693335 135.202667 19.832001 136.12001 19.832001 C 136.504013 19.832001 136.898666 19.794666 137.304001 19.719999 C 137.709335 19.645332 138.06134 19.549334 138.360001 19.431999 L 138.360001 20.712 C 138.018661 20.861334 137.626678 20.973333 137.184006 21.048 C 136.741333 21.122667 136.269333 21.16 135.768005 21.16 Z M 133.272003 16.759998 L 137.28801 16.759998 L 137.28801 16.231998 C 137.28801 15.709332 137.144012 15.301334 136.856003 15.007999 C 136.568008 14.714664 136.082672 14.568001 135.400009 14.568001 C 134.58934 14.568001 134.032013 14.743999 133.727997 15.096001 C 133.423996 15.448002 133.272003 16.002663 133.272003 16.759998 Z M 127.127998 21.16 C 125.794662 21.16 124.722672 20.989334 123.912003 20.648001 L 123.912003 19.112 C 124.370667 19.282667 124.863998 19.421333 125.391998 19.528 C 125.920006 19.634666 126.445328 19.688 126.968002 19.688 C 127.789337 19.688 128.405334 19.592001 128.81601 19.4 C 129.226669 19.207998 129.432007 18.824001 129.432007 18.248001 C 129.432007 17.896 129.346664 17.613335 129.17601 17.400002 C 129.005341 17.186665 128.722672 17.002666 128.328003 16.848 C 127.933334 16.693333 127.389336 16.530666 126.695999 16.360001 C 125.586662 16.071999 124.807999 15.706669 124.360001 15.264 C 123.911995 14.821331 123.688004 14.21067 123.688004 13.431999 C 123.688004 12.52533 124.018661 11.816002 124.68 11.304001 C 125.341339 10.791996 126.327995 10.535999 127.639999 10.535999 C 128.237335 10.535999 128.800003 10.578667 129.328003 10.664001 C 129.856003 10.749332 130.269333 10.845333 130.568008 10.952 L 130.568008 12.487999 C 129.757339 12.178665 128.861343 12.023998 127.879997 12.023998 C 127.112 12.023998 126.514671 12.125332 126.087997 12.327999 C 125.661331 12.530666 125.447998 12.898663 125.447998 13.431999 C 125.447998 13.741333 125.522667 13.992001 125.671997 14.183998 C 125.821335 14.375999 126.080002 14.541332 126.447998 14.68 C 126.816002 14.818668 127.325333 14.973331 127.975998 15.144001 C 128.81868 15.357334 129.472 15.610664 129.936005 15.903999 C 130.400009 16.197334 130.725342 16.538666 130.912003 16.928001 C 131.098679 17.317337 131.192001 17.757332 131.192001 18.248001 C 131.192001 19.154671 130.856003 19.866665 130.184006 20.384001 C 129.511993 20.901337 128.493347 21.16 127.127998 21.16 Z"/>
    <path id="Podcasting" fill="#000000" fill-rule="evenodd" stroke="none" d="M 206.400391 38.816406 C 205.835068 38.816406 205.268463 38.782227 204.703125 38.712891 C 204.137787 38.643555 203.65834 38.539059 203.263672 38.400391 L 203.263672 37.056641 C 203.690338 37.195309 204.181671 37.302284 204.736328 37.376953 C 205.291 37.451622 205.81897 37.488281 206.320313 37.488281 C 207.088303 37.488281 207.647995 37.44471 208 37.359375 C 208.352005 37.27404 208.527344 37.081871 208.527344 36.783203 C 208.527344 36.5592 208.442154 36.403168 208.271484 36.3125 C 208.100815 36.221832 207.776871 36.175781 207.296875 36.175781 L 205.248047 36.175781 C 203.87204 36.175781 203.183594 35.691376 203.183594 34.720703 C 203.183594 34.411369 203.273743 34.122135 203.455078 33.855469 C 203.636414 33.588799 203.925644 33.386715 204.320313 33.248047 C 203.882965 33.034714 203.559555 32.740524 203.351563 32.367188 C 203.143555 31.993853 203.039063 31.546106 203.039063 31.023438 C 203.039063 30.052767 203.330765 29.349443 203.912109 28.912109 C 204.493439 28.474773 205.36467 28.255859 206.527344 28.255859 C 206.772675 28.255859 207.030197 28.275166 207.296875 28.3125 C 207.563538 28.349833 207.770584 28.384016 207.919922 28.416016 L 210.703125 28.416016 L 210.65625 29.552734 L 209.376953 29.552734 C 209.57962 29.723402 209.72821 29.935406 209.824219 30.191406 C 209.920212 30.447409 209.96875 30.72477 209.96875 31.023438 C 209.96875 31.834108 209.723099 32.471542 209.232422 32.935547 C 208.74176 33.399548 207.999817 33.632813 207.007813 33.632813 C 206.837143 33.632813 206.679169 33.623421 206.535156 33.607422 C 206.391159 33.591423 206.235123 33.574066 206.064453 33.552734 C 205.744461 33.584732 205.455215 33.669926 205.199219 33.808594 C 204.943222 33.947258 204.816406 34.132519 204.816406 34.367188 C 204.816406 34.527187 204.879822 34.643555 205.007813 34.712891 C 205.135818 34.782227 205.349777 34.816406 205.648438 34.816406 L 207.775391 34.816406 C 208.53273 34.816406 209.116669 34.979355 209.527344 35.304688 C 209.938019 35.630024 210.144531 36.102036 210.144531 36.720703 C 210.144531 37.467373 209.821121 38.002789 209.175781 38.328125 C 208.530457 38.653458 207.605743 38.816406 206.400391 38.816406 Z M 182.960938 36.160156 C 182.438263 36.160156 181.939514 36.123497 181.464844 36.048828 C 180.990173 35.974159 180.596451 35.882057 180.287109 35.775391 L 180.287109 34.304688 C 180.64978 34.443356 181.050934 34.548378 181.488281 34.623047 C 181.925613 34.697716 182.34166 34.736328 182.736328 34.736328 C 183.333679 34.736328 183.759613 34.682838 184.015625 34.576172 C 184.271622 34.469505 184.400391 34.260506 184.400391 33.951172 C 184.400391 33.727173 184.322632 33.551826 184.167969 33.423828 C 184.013306 33.29583 183.78447 33.178932 183.480469 33.072266 C 183.176468 32.965599 182.789642 32.836834 182.320313 32.6875 C 181.872299 32.538166 181.483017 32.375221 181.152344 32.199219 C 180.821686 32.023216 180.566101 31.796865 180.384766 31.519531 C 180.20343 31.242195 180.111328 30.879644 180.111328 30.431641 C 180.111328 29.738304 180.361954 29.202888 180.863281 28.824219 C 181.364624 28.445551 182.169952 28.255859 183.279297 28.255859 C 183.716629 28.255859 184.132675 28.287563 184.527344 28.351563 C 184.922012 28.415564 185.257828 28.495798 185.535156 28.591797 L 185.535156 30.048828 C 185.236496 29.931494 184.910065 29.839392 184.552734 29.775391 C 184.195404 29.711391 183.862061 29.679688 183.552734 29.679688 C 182.998062 29.679688 182.579544 29.728218 182.296875 29.824219 C 182.014206 29.920219 181.871094 30.112391 181.871094 30.400391 C 181.871094 30.699059 182.019165 30.908583 182.3125 31.03125 C 182.605835 31.153917 183.07785 31.321823 183.728516 31.535156 C 184.325851 31.716492 184.800354 31.908661 185.152344 32.111328 C 185.504349 32.313995 185.757446 32.557175 185.912109 32.839844 C 186.066772 33.122513 186.144531 33.487541 186.144531 33.935547 C 186.144531 34.682217 185.872131 35.239418 185.328125 35.607422 C 184.784119 35.975426 183.995605 36.160156 182.960938 36.160156 Z M 189.552734 36.160156 C 188.742065 36.160156 188.136337 35.943722 187.736328 35.511719 C 187.336334 35.079716 187.136719 34.49081 187.136719 33.744141 L 187.136719 29.775391 L 186.015625 29.775391 L 186.015625 28.416016 L 187.136719 28.416016 L 187.136719 26.671875 L 188.863281 26.144531 L 188.863281 28.416016 L 190.880859 28.416016 L 190.767578 29.775391 L 188.863281 29.775391 L 188.863281 33.648438 C 188.863281 34.085773 188.965302 34.392357 189.167969 34.568359 C 189.370636 34.744362 189.701477 34.832031 190.160156 34.832031 C 190.448151 34.832031 190.752258 34.778542 191.072266 34.671875 L 191.072266 35.904297 C 190.65625 36.074966 190.15007 36.160156 189.552734 36.160156 Z M 192.224609 36 L 192.224609 29.744141 L 191.072266 29.744141 L 191.232422 28.416016 L 193.96875 28.416016 L 193.96875 36 L 192.224609 36 Z M 195.039063 36 L 195.039063 28.416016 L 196.640625 28.416016 L 196.720703 29.216797 C 197.062042 28.950129 197.468689 28.721825 197.943359 28.535156 C 198.41803 28.34849 198.943527 28.255859 199.519531 28.255859 C 200.50087 28.255859 201.216064 28.48469 201.664063 28.943359 C 202.112061 29.402029 202.335938 30.117222 202.335938 31.087891 L 202.335938 36 L 200.607422 36 L 200.607422 31.167969 C 200.607422 30.581301 200.488052 30.170214 200.248047 29.935547 C 200.008041 29.700878 199.557831 29.583984 198.896484 29.583984 C 198.501816 29.583984 198.115005 29.664219 197.736328 29.824219 C 197.357666 29.984221 197.039215 30.191265 196.783203 30.447266 L 196.783203 36 L 195.039063 36 Z M 206.527344 32.527344 C 207.220688 32.527344 207.704559 32.410446 207.976563 32.175781 C 208.248566 31.941113 208.384766 31.541899 208.384766 30.976563 C 208.384766 30.411226 208.248566 30.000143 207.976563 29.744141 C 207.704559 29.48814 207.220688 29.359375 206.527344 29.359375 C 205.876678 29.359375 205.402176 29.483181 205.103516 29.728516 C 204.80484 29.97385 204.65625 30.389893 204.65625 30.976563 C 204.65625 31.520565 204.797409 31.914822 205.080078 32.160156 C 205.362747 32.405491 205.844681 32.527344 206.527344 32.527344 Z M 192.080078 26.832031 L 192.080078 25.392578 L 193.984375 25.392578 L 193.984375 26.832031 L 192.080078 26.832031 Z M 175.007996 36.16 C 174.528 36.16 174.082672 36.074669 173.671997 35.903999 C 173.261337 35.733334 172.933334 35.47467 172.688004 35.127998 C 172.442673 34.781334 172.320007 34.352001 172.320007 33.84 C 172.320007 33.114662 172.567993 32.53067 173.063995 32.087997 C 173.559998 31.64533 174.325333 31.424 175.360001 31.424 L 177.792007 31.424 L 177.792007 31.087999 C 177.792007 30.725332 177.736008 30.437334 177.623993 30.224001 C 177.511993 30.010666 177.309341 29.856001 177.016006 29.76 C 176.722672 29.664 176.293335 29.615999 175.727997 29.615999 C 174.832001 29.615999 173.994675 29.749332 173.216003 30.016001 L 173.216003 28.719999 C 173.557343 28.581333 173.973328 28.469334 174.464005 28.384001 C 174.954666 28.298666 175.482666 28.256001 176.048004 28.256001 C 177.157333 28.256001 178.00267 28.479998 178.584 28.927999 C 179.165329 29.376001 179.455994 30.106661 179.455994 31.120001 L 179.455994 36 L 177.968002 36 L 177.856003 35.232002 C 177.546661 35.53067 177.165329 35.759998 176.712006 35.919998 C 176.258667 36.080002 175.690674 36.16 175.007996 36.16 Z M 175.472 34.911999 C 175.994675 34.911999 176.450668 34.824001 176.839996 34.647999 C 177.22934 34.472 177.546661 34.234669 177.792007 33.936001 L 177.792007 32.608002 L 175.391998 32.608002 C 174.880005 32.608002 174.509338 32.706665 174.279999 32.903999 C 174.050659 33.101334 173.936005 33.402664 173.936005 33.807999 C 173.936005 34.202667 174.069336 34.485332 174.335999 34.655998 C 174.602661 34.826668 174.981339 34.911999 175.472 34.911999 Z M 169.744003 36.16 C 168.41066 36.16 167.402664 35.821335 166.720001 35.144001 C 166.037323 34.466663 165.695999 33.488007 165.695999 32.208 C 165.695999 30.87466 166.071991 29.882669 166.824005 29.232 C 167.576004 28.581329 168.581329 28.256001 169.839996 28.256001 C 170.362671 28.256001 170.802658 28.293333 171.160004 28.368 C 171.517334 28.442667 171.861328 28.559999 172.192001 28.719999 L 172.192001 30.016001 C 171.637329 29.738665 170.965332 29.6 170.175995 29.6 C 169.301331 29.6 168.629333 29.794664 168.160004 30.184 C 167.690659 30.573336 167.455994 31.247995 167.455994 32.208 C 167.455994 33.11467 167.664001 33.778664 168.080002 34.200001 C 168.496002 34.621334 169.18399 34.832001 170.143997 34.832001 C 170.901337 34.832001 171.589325 34.688004 172.207993 34.400002 L 172.207993 35.728001 C 171.856003 35.866669 171.479996 35.973335 171.080002 36.048 C 170.679993 36.122665 170.234665 36.16 169.744003 36.16 Z M 160.960007 36.16 C 160.277328 36.16 159.674667 36.042667 159.151993 35.807999 C 158.629333 35.573334 158.223999 35.184002 157.936005 34.639999 C 157.647995 34.095997 157.503998 33.365337 157.503998 32.447998 C 157.503998 31.541328 157.658661 30.776003 157.968002 30.152 C 158.277328 29.527996 158.706665 29.056002 159.255997 28.736 C 159.805344 28.415998 160.426666 28.256001 161.119995 28.256001 C 161.578674 28.256001 161.986664 28.309332 162.343994 28.416 C 162.70134 28.522667 163.045334 28.688 163.376007 28.912001 L 163.376007 25.216 L 165.119995 25.216 L 165.119995 36 L 163.632004 36 L 163.520004 35.248001 C 163.178665 35.557335 162.805328 35.786667 162.399994 35.936001 C 161.994659 36.085335 161.514664 36.16 160.960007 36.16 Z M 161.423996 34.848 C 162.202667 34.848 162.853333 34.581337 163.376007 34.048 L 163.376007 30.256001 C 162.874664 29.807999 162.245331 29.584 161.488007 29.584 C 160.751999 29.584 160.197342 29.82933 159.824005 30.32 C 159.450668 30.810669 159.264008 31.52533 159.264008 32.464001 C 159.264008 33.37067 159.437332 33.994667 159.783997 34.335999 C 160.130661 34.677334 160.677322 34.848 161.423996 34.848 Z M 153.119995 36.16 C 151.754654 36.16 150.754669 35.818672 150.119995 35.136002 C 149.485336 34.453331 149.167999 33.477341 149.167999 32.208 C 149.167999 30.93866 149.487991 29.962669 150.128006 29.280001 C 150.768005 28.59733 151.76532 28.256001 153.119995 28.256001 C 154.485336 28.256001 155.487991 28.59733 156.128006 29.280001 C 156.768005 29.962669 157.087997 30.93866 157.087997 32.208 C 157.087997 33.477341 156.770676 34.453331 156.136002 35.136002 C 155.501328 35.818672 154.496002 36.16 153.119995 36.16 Z M 153.119995 34.688 C 153.919998 34.688 154.487991 34.493336 154.824005 34.104 C 155.160004 33.714664 155.328003 33.082672 155.328003 32.208 C 155.328003 31.333328 155.160004 30.704002 154.824005 30.32 C 154.487991 29.935999 153.919998 29.743999 153.119995 29.743999 C 152.330658 29.743999 151.768005 29.935999 151.432007 30.32 C 151.095993 30.704002 150.927994 31.333328 150.927994 32.208 C 150.927994 33.082672 151.095993 33.714664 151.432007 34.104 C 151.768005 34.493336 152.330658 34.688 153.119995 34.688 Z M 141.007996 36 L 141.007996 25.695999 L 145.600006 25.695999 C 146.389343 25.695999 147.039993 25.845331 147.552002 26.143999 C 148.063995 26.442669 148.442673 26.850664 148.688004 27.368 C 148.933334 27.885336 149.056 28.474663 149.056 29.136 C 149.056 30.160006 148.738663 30.981331 148.104004 31.6 C 147.46933 32.21867 146.613342 32.528 145.535995 32.528 L 142.751999 32.528 L 142.751999 36 Z M 142.751999 31.152 L 145.216003 31.152 C 145.909332 31.152 146.431992 30.970669 146.783997 30.608 C 147.136002 30.245331 147.311996 29.754669 147.311996 29.136 C 147.311996 28.474663 147.143997 27.962669 146.807999 27.6 C 146.472 27.237331 145.968002 27.056 145.296005 27.056 L 142.751999 27.056 Z"/>
</svg>
                                                                                                                                                                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/packages/js/images/star-rating-half.svg                              0000644                 00000001602 15122266557 0022174 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8"?> <svg xmlns="http://www.w3.org/2000/svg" id="Layer_1" data-name="Layer 1" viewBox="0 0 500 475.53"><defs><style>.cls-1{fill:#fbbf24;}.cls-2{fill:#f3f4f6;}</style></defs><polygon class="cls-1" points="250 392.04 98.15 471.87 127.15 302.78 4.3 183.03 174.07 158.36 250 4.52 325.93 158.36 495.7 183.03 372.85 302.78 401.85 471.87 250 392.04"></polygon><path class="cls-1" d="M250,33.51l73.67,149.27.93,1.88,2.08.3L491.4,208.9,372.21,325.09l-1.51,1.47.36,2.07L399.2,492.69,251.86,415.23l-1.86-1-1.86,1L100.8,492.69l28.14-164.06.36-2.07-1.51-1.47L8.6,208.9,173.32,185l2.08-.3.93-1.88L250,33.51m0-9L172.75,181,0,206.11,125,328,95.49,500,250,418.77,404.51,500,375,328,500,206.11,327.25,181,250,24.47Z" transform="translate(0 -24.47)"></path><polygon class="cls-2" points="500 181.64 327.25 156.53 250 0 250 394.3 404.51 475.53 375 303.48 500 181.64"></polygon></svg>                                                                                                                               plugins/wordpress-seo-extended/packages/js/images/star-rating-star.svg                              0000644                 00000000542 15122266557 0022235 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?xml version="1.0" encoding="UTF-8"?> <svg xmlns="http://www.w3.org/2000/svg" id="Layer_1" data-name="Layer 1" viewBox="0 0 500 475.53"><defs><style>.cls-1{fill:#fbbf24;}</style></defs><polygon class="cls-1" points="250 0 327.25 156.53 500 181.64 375 303.48 404.51 475.53 250 394.3 95.49 475.53 125 303.48 0 181.64 172.75 156.53 250 0"></polygon></svg>                                                                                                                                                               plugins/wordpress-seo-extended/packages/js/images/succes_marieke_bubble_optm.svg                    0000644                 00000016451 15122266557 0024375 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 360"><circle cx="226" cy="211" r="149" fill="#f0ecf0"/><path d="M173.53,189.38s-35.47-5.3-41.78-11c-9.39-24.93-29.61-48-35.47-66.21-.71-2.24,3.72-11.39,3.53-15.41s-5.34-11.64-5.23-14-.09-15.27-.09-15.27l-4.75-.72s-5.13,6.07-3.56,9.87c-1.73-4.19,4.3,7.93.5,9.35,0,0-6-5.94-11.76-8.27s-19.57-3.65-19.57-3.65L43.19,73l-4.42.6L31,69.7l-2.85,5.12,7.53,5.29L40.86,92,58.05,102.2l10.2,10.56,9.86,3.56s26.49,79.67,45,92c17,11.33,37.23,15.92,37.23,15.92Z" fill="#fbd2a6"/><path d="M270.52,345.13c2.76-14.59,15.94-35.73,30.24-54.58,16.22-21.39,14-79.66-33.19-91.46-17.3-4.32-52.25-1-59.85-3.41C186.54,189,170,187,168,190.17c-5,10.51-7.73,27.81-5.51,36.26,1.18,4.73,3.54,5.91,20.49,13.4-5.12,15-16.35,26.3-22.86,37s7.88,27.2,7.1,33.51c-.48,3.8-4.26,21.13-7.18,34.25a149.47,149.47,0,0,0,110.3,8.66A25.66,25.66,0,0,1,270.52,345.13Z" fill="#a4286a"/><path d="M206.76,66.43c-5,14.4-1.42,25.67-3.93,40.74-10,60.34-24.08,43.92-31.44,93.6,7.24-14.19,14.32-15.82,20.63-23.11-.83,3.09-10.25,13.75-8.05,34.81,9.85-8.51,6.35-8.75,11.86-8.54.36,3.25,3.53,3.22-3.59,10.53,2.52.69,17.42-14.32,20.16-12.66s0,5.72-6,7.76c2.15,2.2,30.47-3.87,43.81-14.71,4.93-4,10-13.16,13.38-18.2,7.17-10.62,12.38-24.77,17.71-36.6,8.94-19.87,15.09-39.34,16.11-61.31C297.94,68.3,294,60.3,293,49.88,290.43,22.08,225.37,12.62,206.76,66.43Z" fill="#9a5815"/><path d="M277.74,179.06c.62-.79,1.24-1.59,1.84-2.39C278.73,179.26,278.06,180.4,277.74,179.06Z" fill="#efb17c"/><path d="M216.1,206.72c3.69-5.42,8.28-3.35,15.57-8.28,3.76-3.06,1.57-9.46,1.77-11.82,18.25,4.56,37.38-1.18,49.07-16,.62,5.16-2.77,22.27-.2,27,4.73,8.67,13.4,18.92,13.4,18.92-35.47-2.76-63.45,39-89.86,44.54C211.37,232.34,203.49,225.24,216.1,206.72Z" fill="#fbd2a6"/><path d="M235.21,167.9l53.21-25.23s-3.65,24-6.5,32.72C217.87,238.05,235.45,168.06,235.21,167.9Z" fill="#f6b488"/><path d="M277.68,178.82v0Z" fill="#efb17c"/><path d="M226.86,50.64C215,59.31,206.37,93.21,204,95.57c-19.46,19.47-3.59,41.39-3.94,51.24-.2,5.52-4.14,25.42,5.72,29.36,22.22,8.89,60-3.48,67.19-12.61,13.28-16.75,40.89-94.78,17.74-108.19C282.79,50.79,247.93,35.19,226.86,50.64Z" fill="#fbd2a6"/><path d="M243.69,143.66c-10.7-6.16-8.56-6.73-19.76-12.71-3.86-2.07-3.94.64-6.32,0-2.91-.79-1.39-2.74-5.37-3.48-6.52-1.21-3.67,3.63-3.15,6,1.32,6.15-8.17,17.3,3.26,21.42C225,159.44,233.73,145.48,243.69,143.66Z" fill="#e5766c"/><path d="M240.68,143.9c-11.49-5.53-11.65-8.17-24.64-11.69-8.6-2.32-5.53,1-5.69,4.42-.2,4.16-1.26,9.87,4.9,12.66C224.25,153.38,233.41,143.27,240.68,143.9Z" fill="#fff"/><path d="M241.38,103c-.16,1.26-.06,4.9,5.46,8.25,11.43-4.73,16.36-2.56,17-3.33,1.48-1.76-2-8.87-7.88-9.85C250.38,97.13,241.82,99.31,241.38,103Z" fill="#fff"/><path d="M263.53,108.19c-4.32-4.33-6.85-6.24-12.26-8.21-2.77-1-6.18.18-8.65,1.67a3.65,3.65,0,0,0-1.24,1.23l-.12,0a3.73,3.73,0,0,1,1-1.52,12.53,12.53,0,0,1,11.93-3c4.73,1,9.43,4.63,9.42,9.82Z" fill="#000001"/><circle cx="254.13" cy="104.05" r="4.19" fill="#000001"/><path d="M225.26,99.22c-.29,1-6.6,3.45-10.92,1.48-1.15-3.24-5-6.43-5.25-6.71-.5-2.86,5.55-8,10.06-6.3A10.21,10.21,0,0,1,225.26,99.22Z" fill="#fff"/><path d="M209.29,94.21c-.19-2.34,1.84-4.1,3.65-5.2,7-3.87,13.18,3,12.43,10h-.12c-.14-4-2.38-8.44-6.47-9.11a3.19,3.19,0,0,0-2.42.31c-1.37.85-2.38,2-3.89,2.56-1,.45-1.92.42-3,1.4l-.22,0Z" fill="#000001"/><circle cx="219.55" cy="95.28" r="4" fill="#000001"/><path d="M218.66,120.27a27.32,27.32,0,0,0,4.54,3.45c-2.29-.72-4.28-.69-6.32-2.27-2.53-2-3.39-5.16-.73-7.72,10.24-9.82,12.56-13.82,14.77-24.42-1,12.37-6,17.77-10.63,23.18C217.76,115.46,215.61,117.55,218.66,120.27Z" fill="#efb17c"/><path d="M231.22,69.91c-.67-3.41-8.78-2.83-11.06-1.93-3.48,1.39-6.08,5.22-7.13,8.53,2.9-4.3,6.74-8.12,12.46-6,1.16.42,3.18,2.35,4.48,1.85S231,70.16,231.22,69.91Z" fill="#a57c52"/><path d="M263.38,78.47c-2.75-1.66-12.24-5.08-12.18.82,2.56.24,5-.19,7.64.95C270.06,85,271.61,97.85,271.69,98.1c.2-.53.1,1.26.23.7C271.9,99,272.87,86.68,263.38,78.47Z" fill="#a57c52"/><path d="M53.43,250.73c6.29,0-.6-.17,7.34,0,1.89.05-2.38-.7,0-.69,4.54-4.2,12.48-.74,20.6-2.45,4.55.35,3.93,1.35,5.59,4.19,4.89,8.38,4.78,14.21,14,19.56,16.42,8.38,66,12.92,88.49,18.86,5.52.83,42.64-20.15,61-23.75,6.51,10.74,11.46,28.68,8.39,34.93-6.54,13.3-57.07,25.4-75.91,25.15C156.47,326.18,94,294,92.2,293c-.94-.57.7-.7-7.68,0s-10.15.72-17.47-1.4c-3-.87-4.61-1.33-6.33-3.54-2,.22-3.39.2-4.78-1-3.15-2.74-4.84-6.61-2.73-10.06,0,0-.08,0-.12,0-3.35-2.48-6.54-7.69-3.08-11.72,1-1.18,6.06-1.94,7.77-2.28-1.58-.29-6.37.19-7.49-.72C47.23,259.78,45.33,250.73,53.43,250.73Z" fill="#fbd2a6"/><path d="M303.22,237.52c-9.87-11.88-41.59,8.19-47.8,12.34s-14.89,17.95-14.89,17.95c6,9.43,8.36,31,5.65,46.34l30.51-3s18-15.62,22.59-28.7,6.3-42.54,6.3-42.54" fill="#a4286a"/><path d="M278.63,31.67c-6.08,0-22.91,4.07-22.93,12.91,0,11,47.9,38.38,16.14,85.85,10.21-.79,10.79-8.12,14.92-14.93-3.66,77-49.38,93.58-40.51,142.25,7.68-25.81,20.3-11.62,38.13-33.84,3.45,4.88,9,18.28-9.46,33.78,50-31.26,57.31-56.6,51.92-95C319.93,113.53,348.7,42,278.63,31.67Z" fill="#cb9833"/><path d="M283.64,126.83c-2.42,9.67-8,15.76-1.48,16.46A21.26,21.26,0,0,0,302,132.6c5.17-8.52,3.93-16.44-2.46-18S286.06,117.16,283.64,126.83Z" fill="#fbd2a6"/><path d="M38,73.45c1.92,2,4.25,9.21,6.32,10.91,2.25,1.85,5.71,2.12,8.1,4.45,3.66-2,6-8.72,10-9.31C59.83,80.81,58,83,55.49,84.38c-1.42.8-3,1.31-4.38,2.25-2.16-1.46-4.27-1.77-6.26-3.38C42.33,81.23,39.54,75.25,38,73.45Z" fill="#efb17c"/><path d="M39,74.4c4.83,1.1,12.52,6.44,15.89,10C51.67,83.06,40.16,78.25,39,74.4Z" fill="#efb17c"/><path d="M39.62,72.9c6.71-.79,18,1.54,23.29,5.9-3.85-.2-5.42-1.48-9-2.94C49.83,74.17,45.08,73.83,39.62,72.9Z" fill="#efb17c"/><path d="M86.05,87.48C82.33,86.16,75.53,86.35,72.83,91c2-1.16,1.84-2.11,4.18-1.72-3.81-4.15,8.16-.74,11.6-.24" fill="#efb17c"/><path d="M85.83,102.19c.56-3.29-8-7.81-10.58-9.17C69,89.73,63.09,94.38,55.92,88.49c5.94,6.1,14.23,2.5,19.55,5.76,3.06,1.88,8.65,6.09,9.35,9.38C84.59,103.23,86.11,102.19,85.83,102.19Z" fill="#efb17c"/><circle cx="38.13" cy="30.03" r="3.14" fill="#b89ac8"/><circle cx="60.26" cy="39.96" r="3.14" fill="#e31e0c"/><circle cx="50.29" cy="25.63" r="3.14" fill="#3baa45"/><circle cx="22.19" cy="19.21" r="3.14" fill="#2ca9e1"/><circle cx="22.19" cy="30.03" r="3.14" fill="#e31e0c"/><circle cx="26.86" cy="8.28" r="3.14" fill="#3baa45"/><circle cx="49.32" cy="39.99" r="3.14" fill="#e31e0c"/><circle cx="63.86" cy="59.52" r="3.14" fill="#f8ad39"/><circle cx="50.88" cy="50.72" r="3.14" fill="#3baa45"/><circle cx="63.47" cy="76.17" r="3.14" fill="#e31e0c"/><circle cx="38.34" cy="14.83" r="3.14" fill="#2ca9e1"/><circle cx="44.44" cy="5.92" r="3.14" fill="#f8ad39"/><circle cx="57.42" cy="10.24" r="3.14" fill="#e31e0c"/><circle cx="66.81" cy="12.4" r="3.14" fill="#2ca9e1"/><circle cx="77.95" cy="5.14" r="3.14" fill="#b89ac8"/><circle cx="77.95" cy="30.34" r="3.14" fill="#e31e0c"/><circle cx="80.97" cy="16.55" r="3.14" fill="#f8ad39"/><circle cx="62.96" cy="27.27" r="3.14" fill="#3baa45"/><circle cx="75.36" cy="48.67" r="3.14" fill="#2ca9e1"/><circle cx="76.11" cy="65.31" r="3.14" fill="#3baa45"/><path d="M78.58,178.43C54.36,167.26,32,198.93,5,198.93c19.56,20.49,63.53,1.52,69,15.5C75.48,200.42,78.11,183.53,78.58,178.43Z" fill="#71b026"/><path d="M67.75,251.08c0-4.65,10.13-72.65,10.13-72.65h2.8l-9.09,72.3Z" fill="#074a67"/><ellipse cx="255.38" cy="103.18" rx="1.84" ry="1.77" fill="#fff"/><ellipse cx="221.24" cy="94.75" rx="1.84" ry="1.77" fill="#fff"/></svg>                                                                                                                                                                                                                       plugins/wordpress-seo-extended/packages/js/images/tec-logo.svg                                      0000644                 00000035626 15122266557 0020557 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg width="280" height="50" viewBox="0 0 280 119" fill="none" xmlns="http://www.w3.org/2000/svg">
<path fill-rule="evenodd" clip-rule="evenodd" d="M143.062 49.1982C140.605 50.1949 138.331 50.7325 136.678 50.6435C134.537 50.5242 133.623 49.5798 133.62 49.2246C133.618 48.9718 134.416 48.1796 136.881 48.0004C137.267 47.9729 137.675 47.9553 138.095 47.9553C139.779 47.9553 141.653 48.2324 143.062 49.1982M147.657 70.9116C138.649 73.3963 129.13 71.1045 123.402 65.0681C119.349 60.7968 117.99 55.4513 119.673 50.4027C123.058 40.245 139.281 34.1481 149.649 34.1481C149.761 34.1481 149.872 34.1486 149.982 34.1503C153.845 34.1998 156.521 35.1337 156.965 36.5888C157.213 37.403 156.625 38.8284 155.392 40.4006C153.217 43.175 149.911 45.6894 146.548 47.5337C144.416 45.1947 140.996 44.1304 136.623 44.4454C131.754 44.7989 130.038 47.1616 130.055 49.2549C130.075 51.5373 132.288 53.9698 136.48 54.2029C139.018 54.3414 142.051 53.5597 145.096 52.2272C145.543 53.9808 145.405 56.2814 144.673 59.0245C144.42 59.975 144.985 60.9523 145.936 61.2058C146.886 61.4592 147.863 60.8941 148.117 59.9425C149.111 56.2154 149.185 53.0892 148.342 50.6055C152.318 48.3742 155.993 45.4102 158.197 42.6005C160.265 39.9635 160.997 37.591 160.374 35.5482C159.922 34.0651 158.857 32.8887 157.252 32.0542C157.374 18.6609 159.748 9.40965 161.886 9.39701H161.899C162.213 9.39701 162.638 9.46737 163.158 9.98466C164.895 11.7108 166.964 17.3773 166.964 34.5719C166.964 51.5379 165.512 65.9828 147.657 70.9116M130.651 31.3406C130.649 31.2961 130.655 31.2521 130.649 31.2071C130.641 31.1427 130.634 31.0746 130.625 31.0086C130.369 27.3513 130.808 24.4692 131.923 22.9234C132.477 22.1559 133.172 21.7448 134.111 21.6293C135.419 21.4716 136.483 21.8091 137.377 22.6677C139.108 24.3317 140.118 27.9296 140.022 31.8656C137.054 32.5979 134.015 33.6407 131.128 34.9831C130.901 33.7055 130.744 32.4885 130.651 31.3406M146.005 25.1305C146.662 24.4164 147.499 24.069 148.567 24.069C150.265 24.069 151.002 24.6242 151.41 25.042C152.385 26.0408 152.911 27.978 152.983 30.7992C152.07 30.6727 151.085 30.5991 150.028 30.5859C149.907 30.5842 149.787 30.5837 149.666 30.5837C148.057 30.5837 146.303 30.7233 144.473 30.9894C144.394 28.4041 144.934 26.2975 146.005 25.1305M165.672 7.45649C164.602 6.39334 163.297 5.83208 161.896 5.83208H161.864C157.168 5.86011 155.548 12.8668 154.737 18.1348C154.5 19.6806 154.316 21.2583 154.173 22.7854C154.103 22.7073 154.035 22.6265 153.961 22.5518C152.636 21.1934 150.82 20.5046 148.567 20.5046C146.504 20.5046 144.71 21.2698 143.38 22.7189C143.001 23.1323 142.675 23.6111 142.38 24.1245C141.792 22.5781 140.971 21.178 139.847 20.0978C138.179 18.4943 136.046 17.8 133.676 18.0913C131.766 18.3261 130.16 19.2749 129.032 20.8372C129.028 20.8443 129.024 20.852 129.019 20.8581C127.066 11.1171 123.353 -0.645256 116.874 0.0275989C115.279 0.195263 113.871 1.05337 112.8 2.50793C107.802 9.30245 110.147 29.5068 112.32 40.3736C112.49 41.2207 113.234 41.8073 114.066 41.8073C114.182 41.8073 114.3 41.7958 114.418 41.7721C115.383 41.5792 116.009 40.6402 115.816 39.6749C112.177 21.4793 112.989 8.26733 115.672 4.6205C116.302 3.76349 116.873 3.61176 117.247 3.57273C120.823 3.20057 125.229 15.6968 127.089 31.4687C127.238 33.4219 127.534 35.2508 127.819 36.6724C127.82 36.679 127.823 36.685 127.825 36.6916C122.469 39.7788 118.049 44.0001 116.291 49.2758C114.178 55.6135 115.828 62.2645 120.816 67.522C125.687 72.6542 132.873 75.4605 140.416 75.4605C143.13 75.46 145.891 75.0966 148.606 74.3473C168.852 68.7589 170.529 52.3437 170.529 34.5719C170.529 19.3667 168.985 10.751 165.672 7.45649" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M16.2216 96.9075H0.343574C0.153921 96.9075 0 97.0614 0 97.251V99.9749C0 100.165 0.153921 100.318 0.343574 100.318H6.40697V117.896C6.40697 118.085 6.56144 118.24 6.75054 118.24H9.84546C10.0351 118.24 10.189 118.085 10.189 117.896V100.318H16.2216C16.4113 100.318 16.5652 100.165 16.5652 99.9749V97.251C16.5652 97.0614 16.4113 96.9075 16.2216 96.9075" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M27.2181 102.231C25.4612 102.231 23.6224 102.964 22.2685 104.197V97.2508C22.2685 97.0617 22.1145 96.9073 21.9249 96.9073H19.1395C18.9498 96.9073 18.7959 97.0617 18.7959 97.2508V117.896C18.7959 118.086 18.9498 118.24 19.1395 118.24H21.9249C22.1145 118.24 22.2685 118.086 22.2685 117.896V107.332C23.0084 106.365 24.4563 105.395 26.0417 105.395C28.0356 105.395 28.886 106.255 28.886 108.27V117.896C28.886 118.086 29.0399 118.24 29.2296 118.24H32.0155C32.2046 118.24 32.3591 118.086 32.3591 117.896V107.341C32.3591 103.999 30.5813 102.231 27.2181 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M42.795 105.209C45.4127 105.209 46.7744 107.09 46.9772 108.978H38.599C38.8865 107.09 40.2849 105.209 42.795 105.209M42.7949 102.231C38.329 102.231 34.9614 105.746 34.9614 110.406C34.9614 115.237 38.2845 118.611 43.0428 118.611C45.5737 118.611 47.7325 117.806 49.2876 116.285C49.4091 116.166 49.4262 115.979 49.3272 115.84L48.0271 114.014C47.9678 113.93 47.8754 113.878 47.7737 113.87C47.6731 113.865 47.5714 113.902 47.5011 113.973C46.5044 114.997 44.9031 115.633 43.3215 115.633C40.3201 115.633 38.9265 113.521 38.6335 111.678H50.0688C50.2584 111.678 50.4124 111.524 50.4124 111.335V110.653C50.4124 105.694 47.2801 102.231 42.7949 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M72.5495 96.9075H58.6828C58.4937 96.9075 58.3392 97.0614 58.3392 97.251V117.896C58.3392 118.085 58.4937 118.24 58.6828 118.24H72.5495C72.7391 118.24 72.893 118.085 72.893 117.896V115.172C72.893 114.983 72.7391 114.829 72.5495 114.829H62.1213V109.078H72.3323C72.522 109.078 72.6759 108.924 72.6759 108.735V106.011C72.6759 105.821 72.522 105.667 72.3323 105.667H62.1213V100.318H72.5495C72.7391 100.318 72.893 100.165 72.893 99.9749V97.251C72.893 97.0614 72.7391 96.9075 72.5495 96.9075" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M89.791 102.603H86.7884C86.6472 102.603 86.5218 102.687 86.469 102.819L82.1147 113.749L77.7604 102.819C77.7082 102.687 77.5823 102.603 77.441 102.603H74.4698C74.3549 102.603 74.2488 102.66 74.1845 102.755C74.1207 102.851 74.1086 102.971 74.1526 103.077L80.3117 118.027C80.365 118.156 80.4892 118.239 80.6288 118.239H83.6314C83.771 118.239 83.8953 118.156 83.9486 118.027L90.1082 103.077C90.1522 102.971 90.1401 102.851 90.0763 102.755C90.012 102.66 89.9059 102.603 89.791 102.603" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M98.3718 105.209C100.99 105.209 102.351 107.09 102.554 108.978H94.1758C94.4633 107.09 95.8618 105.209 98.3718 105.209M98.3718 102.231C93.9059 102.231 90.5383 105.746 90.5383 110.406C90.5383 115.237 93.8614 118.611 98.6197 118.611C101.151 118.611 103.309 117.806 104.865 116.285C104.986 116.166 105.003 115.979 104.904 115.84L103.604 114.014C103.545 113.93 103.452 113.878 103.351 113.87C103.251 113.865 103.148 113.902 103.078 113.973C102.081 114.997 100.48 115.633 98.8984 115.633C95.8964 115.633 94.5029 113.521 94.2099 111.678H105.646C105.835 111.678 105.989 111.524 105.989 111.335V110.653C105.989 105.694 102.857 102.231 98.3718 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M116.983 102.231C114.507 102.231 112.765 103.527 112.033 104.195V102.946C112.033 102.757 111.879 102.603 111.689 102.603H108.904C108.714 102.603 108.56 102.757 108.56 102.946V117.896C108.56 118.086 108.714 118.24 108.904 118.24H111.689C111.879 118.24 112.033 118.086 112.033 117.896V107.332C112.773 106.365 114.221 105.395 115.806 105.395C117.773 105.395 118.651 106.3 118.651 108.332V117.896C118.651 118.086 118.804 118.24 118.994 118.24H121.78C121.969 118.24 122.124 118.086 122.124 117.896V107.404C122.124 104.02 120.346 102.231 116.983 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M133.156 115.093C133.117 114.978 133.019 114.891 132.902 114.867C132.792 114.844 132.657 114.882 132.575 114.973C132.369 115.202 131.887 115.448 131.314 115.448C130.283 115.448 130.203 114.313 130.203 113.965V105.735H132.893C133.082 105.735 133.236 105.581 133.236 105.392V102.946C133.236 102.757 133.082 102.603 132.893 102.603H130.203V98.8606C130.203 98.671 130.049 98.5171 129.859 98.5171H127.074C126.884 98.5171 126.73 98.671 126.73 98.8606V102.603H124.598C124.408 102.603 124.254 102.757 124.254 102.946V105.392C124.254 105.581 124.408 105.735 124.598 105.735H126.73V114.584C126.73 117.181 128.116 118.611 130.633 118.611C132.002 118.611 133.03 118.266 133.779 117.558C133.876 117.465 133.911 117.325 133.868 117.198L133.156 115.093Z" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M142.473 108.616C140.599 108.179 139.272 107.791 139.272 106.784C139.272 105.793 140.315 105.178 141.993 105.178C143.67 105.178 145.303 105.846 146.154 106.879C146.219 106.958 146.315 107.004 146.419 107.004C146.421 107.004 146.438 107.003 146.44 107.003C146.549 106.997 146.651 106.938 146.709 106.845L147.947 104.894C148.034 104.756 148.012 104.574 147.893 104.461C146.359 103.003 144.309 102.231 141.962 102.231C137.813 102.231 135.923 104.704 135.923 107.001C135.923 110.486 139.127 111.228 141.701 111.824C143.535 112.238 145.054 112.681 145.054 113.903C145.054 115.026 143.979 115.696 142.179 115.696C140.183 115.696 138.312 114.646 137.418 113.671C137.353 113.6 137.261 113.559 137.165 113.559C137.155 113.559 137.147 113.56 137.137 113.561C137.031 113.57 136.934 113.628 136.877 113.716L135.546 115.759C135.458 115.893 135.476 116.073 135.589 116.187C137.143 117.772 139.379 118.611 142.055 118.611C145.93 118.611 148.434 116.678 148.434 113.687C148.434 109.966 145.13 109.218 142.473 108.616" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M167.004 100.009C169.166 100.009 171.189 101.097 172.283 102.849C172.376 102.998 172.578 103.054 172.736 102.972L175.367 101.579C175.453 101.533 175.516 101.453 175.539 101.36C175.563 101.265 175.546 101.165 175.492 101.085C173.493 98.087 170.638 96.567 167.004 96.567C160.773 96.567 156.074 101.305 156.074 107.589C156.074 113.873 160.773 118.611 167.004 118.611C170.6 118.611 173.454 117.092 175.491 114.096C175.545 114.015 175.563 113.914 175.539 113.82C175.516 113.726 175.454 113.646 175.367 113.599L172.736 112.206C172.578 112.125 172.376 112.18 172.283 112.329C171.188 114.081 169.166 115.169 167.004 115.169C163.114 115.169 160.5 112.123 160.5 107.589C160.5 102.984 163.053 100.009 167.004 100.009" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M187.429 112.195V114.497C186.683 115.422 185.379 115.974 183.935 115.974C182.114 115.974 180.843 114.893 180.843 113.347C180.843 111.799 182.114 110.718 183.935 110.718C185.379 110.718 186.683 111.27 187.429 112.195M184.554 102.231C182.047 102.231 179.904 103.087 178.185 104.775C178.07 104.887 178.05 105.068 178.135 105.203L179.342 107.123C179.398 107.211 179.49 107.269 179.595 107.281C179.699 107.295 179.804 107.255 179.876 107.182C181.14 105.917 182.529 105.302 184.12 105.302C186.13 105.302 187.429 106.333 187.429 107.93V109.579C186.28 108.599 184.712 108.081 182.883 108.081C180.213 108.081 177.339 109.719 177.339 113.315C177.339 116.755 180.195 118.611 182.883 118.611C184.664 118.611 186.232 118.077 187.429 117.065V117.896C187.429 118.086 187.583 118.24 187.773 118.24H190.558C190.748 118.24 190.902 118.086 190.902 117.896V107.806C190.902 104.263 188.588 102.231 184.554 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M198.111 96.9075H195.326C195.136 96.9075 194.982 97.0614 194.982 97.251V117.896C194.982 118.085 195.136 118.24 195.326 118.24H198.111C198.301 118.24 198.455 118.085 198.455 117.896V97.251C198.455 97.0614 198.301 96.9075 198.111 96.9075" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M209.409 105.209C212.027 105.209 213.388 107.09 213.591 108.978H205.213C205.501 107.09 206.899 105.209 209.409 105.209M209.409 102.231C204.943 102.231 201.576 105.746 201.576 110.406C201.576 115.237 204.899 118.611 209.657 118.611C212.188 118.611 214.347 117.806 215.902 116.285C216.023 116.166 216.04 115.979 215.941 115.84L214.641 114.014C214.582 113.93 214.489 113.878 214.388 113.87C214.292 113.865 214.185 113.902 214.115 113.973C213.118 114.997 211.517 115.633 209.936 115.633C206.934 115.633 205.541 113.521 205.248 111.678H216.683C216.873 111.678 217.026 111.524 217.026 111.335V110.653C217.026 105.694 213.894 102.231 209.409 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M228.019 102.231C225.543 102.231 223.802 103.527 223.07 104.195V102.946C223.07 102.757 222.916 102.603 222.726 102.603H219.941C219.751 102.603 219.597 102.757 219.597 102.946V117.896C219.597 118.086 219.751 118.24 219.941 118.24H222.726C222.916 118.24 223.07 118.086 223.07 117.896V107.332C223.81 106.365 225.258 105.395 226.843 105.395C228.81 105.395 229.687 106.3 229.687 108.332V117.896C229.687 118.086 229.841 118.24 230.031 118.24H232.817C233.006 118.24 233.16 118.086 233.16 117.896V107.404C233.16 104.02 231.383 102.231 228.019 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M247.307 107.352V113.522C246.532 114.656 244.972 115.448 243.503 115.448C241.006 115.448 239.328 113.434 239.328 110.437C239.328 107.421 241.006 105.395 243.503 105.395C244.935 105.395 246.531 106.217 247.307 107.352M250.436 96.9075H247.651C247.461 96.9075 247.307 97.0614 247.307 97.251V104.186C246.069 102.924 244.461 102.231 242.761 102.231C238.556 102.231 235.731 105.529 235.731 110.437C235.731 115.326 238.556 118.611 242.761 118.611C244.435 118.611 246.082 117.909 247.307 116.681V117.896C247.307 118.085 247.461 118.24 247.651 118.24H250.436C250.626 118.24 250.78 118.085 250.78 117.896V97.251C250.78 97.0614 250.626 96.9075 250.436 96.9075" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M264.084 112.195V114.497C263.338 115.422 262.034 115.974 260.589 115.974C258.769 115.974 257.498 114.893 257.498 113.347C257.498 111.799 258.769 110.718 260.589 110.718C262.034 110.718 263.338 111.27 264.084 112.195M261.209 102.231C258.702 102.231 256.559 103.087 254.839 104.775C254.725 104.887 254.704 105.068 254.79 105.203L255.997 107.123C256.053 107.211 256.144 107.269 256.249 107.281C256.353 107.295 256.458 107.255 256.53 107.182C257.795 105.917 259.183 105.302 260.775 105.302C262.785 105.302 264.084 106.333 264.084 107.93V109.579C262.935 108.599 261.366 108.081 259.537 108.081C256.867 108.081 253.994 109.719 253.994 113.315C253.994 116.755 256.85 118.611 259.537 118.611C261.318 118.611 262.887 118.077 264.084 117.065V117.896C264.084 118.086 264.238 118.24 264.428 118.24H267.213C267.403 118.24 267.557 118.086 267.557 117.896V107.806C267.557 104.263 265.243 102.231 261.209 102.231" fill="#0F1031"/>
<path fill-rule="evenodd" clip-rule="evenodd" d="M279.656 102.262C278.042 102.262 276.394 103.01 275.11 104.319V102.946C275.11 102.757 274.955 102.603 274.766 102.603H271.98C271.791 102.603 271.637 102.757 271.637 102.946V117.896C271.637 118.085 271.791 118.239 271.98 118.239H274.766C274.955 118.239 275.11 118.085 275.11 117.896V107.592C275.755 106.586 277.459 105.673 278.697 105.673C279.037 105.673 279.326 105.701 279.582 105.758C279.682 105.779 279.79 105.755 279.872 105.69C279.953 105.624 280 105.527 280 105.423V102.606C280 102.416 279.846 102.262 279.656 102.262" fill="#0F1031"/>
</svg>
                                                                                                          plugins/wordpress-seo-extended/packages/js/images/wincher-logo.svg                                  0000644                 00000010773 15122266557 0021437 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 566 122" width="140" height="100%"><g fill="none" fill-rule="evenodd"><g transform="translate(183.8 37)" fill-rule="nonzero"><circle fill="#FFA23A" cx="375.8" cy="58.2" r="5.6"></circle><path d="M31.805 62.112a2.4 2.4 0 01-2.282 1.656h-7.337a2.4 2.4 0 01-2.277-1.641L.323 3.365A2.4 2.4 0 012.6.206h8.895a2.4 2.4 0 012.298 1.708l12.37 41.04 13.974-41.12A2.4 2.4 0 0142.409.206h4.7a2.4 2.4 0 012.268 1.616l14.228 41.131 12.2-41.03A2.4 2.4 0 0178.107.205H87a2.4 2.4 0 012.278 3.156L69.77 62.124a2.4 2.4 0 01-2.278 1.644h-7.335a2.4 2.4 0 01-2.282-1.656L44.84 22.139 31.805 62.112zm74.538-45.22a2.4 2.4 0 012.4 2.4v42.076a2.4 2.4 0 01-2.4 2.4H98.95a2.4 2.4 0 01-2.4-2.4V19.292a2.4 2.4 0 012.4-2.4h7.393zM102.555 0c3.095 0 5.605 2.496 5.605 5.575s-2.51 5.574-5.605 5.574c-3.095 0-5.605-2.495-5.605-5.574 0-3.08 2.51-5.575 5.605-5.575zm29.076 61.368a2.4 2.4 0 01-2.4 2.4h-7.392a2.4 2.4 0 01-2.4-2.4V19.292a2.4 2.4 0 012.4-2.4h7.133a2.4 2.4 0 012.4 2.4v4.937c1.268-2.35 3.228-4.243 5.88-5.677 2.651-1.433 5.62-2.15 8.906-2.15 5.13 0 9.483 1.692 13.057 5.075s5.361 8.2 5.361 14.45v25.44a2.4 2.4 0 01-2.4 2.4h-7.392a2.4 2.4 0 01-2.4-2.4V39.196c0-3.67-.894-6.523-2.68-8.558-1.788-2.036-4.209-3.054-7.264-3.054-3.228 0-5.837.304-7.826 2.512-1.989 2.207-2.983 5.86-2.983 9.358v21.915zM198.736 64c-7.131 0-13.018-2.242-17.66-6.725-4.584-4.427-6.876-10.172-6.876-17.233s2.32-12.806 6.962-17.233C185.86 18.27 191.83 16 199.076 16c4.754 0 9.07 1.12 12.947 3.363 3.38 1.953 5.996 4.589 7.849 7.905.643 1.151.272 1.78-.517 2.208l-6.947 3.774c-1.003.545-1.63.246-2.293-.654-2.74-3.722-6.363-5.584-10.87-5.584-3.848 0-6.99 1.261-9.423 3.783-2.49 2.466-3.736 5.548-3.736 9.247 0 3.81 1.217 6.95 3.65 9.415 2.491 2.466 5.576 3.699 9.255 3.699 2.434 0 4.754-.645 6.962-1.934 1.73-1.01 3.094-2.243 4.094-3.7.47-.685 1.14-1.03 2.077-.484l7.195 4.187c.738.43.968 1.038.538 1.742-1.954 3.2-4.679 5.784-8.173 7.755C207.806 62.907 203.49 64 198.736 64zm44.522-2.632a2.4 2.4 0 01-2.4 2.4h-7.393a2.4 2.4 0 01-2.4-2.4V2.4a2.4 2.4 0 012.4-2.4h7.047a2.4 2.4 0 012.4 2.4v21.399c1.268-2.179 3.257-3.957 5.966-5.333 2.71-1.376 5.679-2.064 8.907-2.064 5.073 0 9.396 1.692 12.97 5.075 3.575 3.383 5.362 8.2 5.362 14.45v25.44a2.4 2.4 0 01-2.4 2.4h-7.393a2.4 2.4 0 01-2.4-2.4V39.196c0-3.67-.879-6.523-2.637-8.558-1.758-2.036-4.165-3.054-7.22-3.054-3.229 0-5.837 1.104-7.826 3.312-1.989 2.207-2.983 5.06-2.983 8.558v21.915zM311.177 64c-8.045 0-14.233-2.326-18.564-6.977-4.275-4.596-6.413-10.256-6.413-16.981 0-7.005 2.222-12.75 6.666-17.233C297.31 18.27 303.076 16 310.164 16c6.694 0 12.207 2.074 16.539 6.22 4.331 4.148 6.497 9.752 6.497 16.813 0 1.087-.085 2.375-.254 3.866-.104.92-.483 1.346-1.483 1.346h-33.197c1.688 6.501 6.16 9.752 13.417 9.752 4.807 0 9.295-1.247 13.464-3.74.787-.471 1.621-.471 2.059.299l3.324 5.854c.565.997.43 1.514-.38 2.05-5.582 3.693-11.907 5.54-18.973 5.54zM321.2 36c-.554-3.1-1.827-5.54-3.82-7.325-1.992-1.783-4.483-2.675-7.472-2.675-2.934 0-5.439.892-7.515 2.675-2.076 1.784-3.473 4.226-4.193 7.325h23zm34.954 25.368a2.4 2.4 0 01-2.4 2.4h-7.393a2.4 2.4 0 01-2.4-2.4V19.292a2.4 2.4 0 012.4-2.4h6.874a2.4 2.4 0 012.4 2.4v5.472c2.825-5.046 7.148-7.57 12.97-7.57 1.491 0 2.982.212 4.472.635.86.243 1.193.871 1.11 1.796l-.778 8.712c-.097 1.091-.567 1.348-1.636 1.091-1.487-.357-2.86-.536-4.119-.536-3.574 0-6.384 1.19-8.43 3.57-2.047 2.38-3.07 5.92-3.07 10.622v18.284z" fill="#37343B"></path></g><path d="M154.949.162a3.423 3.423 0 012.235 4.308l-37.107 115.146a3.446 3.446 0 01-3.285 2.384H94.376c-.356 0-.71-.055-1.049-.162a3.423 3.423 0 01-2.235-4.308L128.199 2.384A3.446 3.446 0 01131.484 0h24.485c.356 0-1.36.055-1.02.162zm-70.7 14.231c.355 0 .707.055 1.046.162a3.423 3.423 0 012.24 4.305L55.247 119.613A3.446 3.446 0 0151.962 122H29.545c-.355 0-.708-.055-1.047-.162a3.423 3.423 0 01-2.239-4.305L58.545 16.78a3.446 3.446 0 013.286-2.387h22.417z" fill="#FF8F3B"></path><path d="M107.849 65.53l14.936 45.68-2.708 8.406a3.446 3.446 0 01-3.285 2.384l-11.788-47.64 2.845-8.83zm-64.877-.151l14.976 45.805-2.7 8.429a3.425 3.425 0 01-.157.4l-15.13-45.236 3.011-9.398z" fill="#FF7F3B"></path><path d="M84.376 14.393c1.501 0 2.83.966 3.285 2.387l32.287 100.753a3.423 3.423 0 01-2.24 4.305 3.468 3.468 0 01-1.046.162H94.245a3.446 3.446 0 01-3.286-2.387L58.673 18.86a3.423 3.423 0 012.24-4.305 3.468 3.468 0 011.045-.162zM26.454 37.011c1.497 0 2.823.96 3.282 2.376l25.363 78.135a3.423 3.423 0 01-2.224 4.313c-.342.11-.699.165-1.058.165H29.408a3.446 3.446 0 01-3.282-2.376L.763 41.49a3.423 3.423 0 012.224-4.314 3.485 3.485 0 011.058-.165z" fill="#FFA23A"></path></g></svg>     plugins/wordpress-seo-extended/packages/js/images/woocommerce-seo-logo.svg                          0000644                 00000013005 15122266557 0023072 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg width="153" height="58" role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 107.51 40.69"><path fill="#7c579f" fill-rule="evenodd" d="M16.75 23.78h23.13c1.46 0 2.65 1.19 2.65 2.65v8.82c0 1.46-1.18 2.65-2.65 2.65h-8.3l1.14 2.79-5.01-2.79H16.75c-1.46 0-2.65-1.18-2.65-2.65v-8.82c0-1.46 1.19-2.65 2.65-2.65Z"/><path fill="#fff" d="M15.53 26.18c.17-.22.41-.34.73-.36.59-.04.93.24 1.02.83.36 2.42.75 4.48 1.17 6.17L21 27.98c.23-.44.52-.67.87-.69.51-.04.83.29.96.97.29 1.54.66 2.86 1.1 3.97.3-2.95.82-5.08 1.53-6.4a.92.92 0 0 1 .77-.52c.27-.02.52.06.73.23.22.17.34.39.35.67.01.2-.02.39-.11.55-.45.84-.83 2.24-1.12 4.2-.29 1.89-.4 3.37-.32 4.43.02.29-.02.55-.13.76-.14.26-.35.4-.62.42-.31.02-.61-.12-.92-.43-1.08-1.11-1.94-2.76-2.57-4.94-.75 1.49-1.31 2.6-1.68 3.35-.68 1.31-1.27 1.99-1.75 2.02-.32.02-.59-.24-.81-.8-.6-1.52-1.24-4.46-1.92-8.82-.04-.3.02-.57.18-.78ZM40.51 28c-.42-.73-1.04-1.18-1.86-1.35-.22-.05-.43-.07-.63-.07-1.11 0-2.02.58-2.73 1.74-.6.99-.9 2.08-.9 3.27 0 .9.19 1.66.56 2.3.42.73 1.04 1.18 1.86 1.35.22.05.43.07.63.07 1.13 0 2.03-.58 2.73-1.74.6-1 .9-2.1.9-3.29 0-.9-.19-1.66-.56-2.28Zm-1.47 3.22c-.16.76-.45 1.34-.89 1.72-.34.31-.65.43-.94.37-.28-.06-.51-.31-.68-.75-.14-.36-.2-.72-.2-1.06a3.847 3.847 0 0 1 .7-2.23c.39-.57.79-.81 1.21-.72.28.06.51.31.68.75.14.36.2.72.2 1.06 0 .3-.03.58-.09.85ZM33.22 28c-.42-.73-1.04-1.18-1.86-1.35-.22-.05-.43-.07-.63-.07-1.11 0-2.02.58-2.73 1.74-.6.99-.9 2.08-.9 3.27 0 .9.19 1.66.56 2.3.42.73 1.04 1.18 1.86 1.35.22.05.43.07.63.07 1.12 0 2.03-.58 2.73-1.74.6-1 .9-2.1.9-3.29 0-.9-.19-1.66-.56-2.28Zm-1.47 3.22c-.16.76-.45 1.34-.89 1.72-.34.31-.65.43-.94.37-.28-.06-.51-.31-.68-.75-.14-.36-.2-.72-.2-1.06a3.847 3.847 0 0 1 .7-2.23c.39-.57.79-.81 1.21-.72.28.06.51.31.68.75.14.36.2.72.2 1.06 0 .3-.03.58-.09.85Z"/><path d="M45.47 27.9c-.78.77-1.16 1.74-1.16 2.92 0 1.27.38 2.3 1.15 3.08.77.78 1.77 1.17 3.01 1.17.36 0 .76-.06 1.21-.18v-1.88c-.4.12-.76.18-1.06.18-.62 0-1.11-.21-1.48-.62-.37-.41-.55-.97-.55-1.69 0-.66.18-1.2.55-1.63.37-.43.83-.64 1.38-.64.36 0 .75.06 1.16.17V26.9c-.39-.1-.81-.15-1.27-.15-1.19 0-2.17.38-2.94 1.15Zm8.06-1.15c-1.07 0-1.91.36-2.51 1.07-.6.71-.9 1.71-.9 2.98 0 1.38.3 2.44.9 3.17.6.73 1.46 1.1 2.6 1.1s1.95-.37 2.54-1.1c.59-.73.89-1.76.89-3.09s-.3-2.35-.91-3.06c-.6-.71-1.48-1.07-2.62-1.07Zm.92 6.04c-.21.33-.52.49-.92.49-.37 0-.65-.16-.85-.49-.2-.33-.29-.98-.29-1.96 0-1.52.39-2.28 1.16-2.28.82 0 1.23.77 1.23 2.32 0 .96-.11 1.6-.32 1.93Zm8.36-5.82-.42 1.78c-.11.45-.21.92-.3 1.4l-.23 1.23c-.22-1.23-.52-2.7-.91-4.4h-2.7l-1.02 7.91h2.02l.55-5.45 1.38 5.45h1.45l1.32-5.44.57 5.44h2.12l-1.07-7.91h-2.76Zm9.68 0-.42 1.78c-.11.45-.21.92-.3 1.4l-.23 1.23c-.22-1.23-.52-2.7-.91-4.4h-2.7l-1.02 7.91h2.02l.55-5.45 1.38 5.45h1.45l1.32-5.44.57 5.44h2.12l-1.07-7.91h-2.76Zm6.57 4.78h1.9v-1.64h-1.9v-1.46h2.19v-1.68h-4.32v7.91h4.34v-1.69h-2.2v-1.44Zm8.22-1.25c.22-.36.33-.73.33-1.11 0-.75-.29-1.34-.87-1.77-.58-.43-1.38-.65-2.38-.65h-2.49v7.91h2.14v-3.6h.04l1.73 3.6h2.26l-1.71-3.57c.42-.18.75-.45.97-.81ZM84 30.37v-1.88c.51.01.87.1 1.09.25.22.16.33.42.33.79 0 .54-.47.82-1.42.84Zm4.88-2.47c-.78.77-1.16 1.74-1.16 2.92 0 1.27.38 2.3 1.15 3.08.77.78 1.77 1.17 3.01 1.17.36 0 .76-.06 1.21-.18v-1.88c-.4.12-.76.18-1.06.18-.62 0-1.11-.21-1.48-.62-.37-.41-.55-.97-.55-1.69 0-.66.18-1.2.55-1.63.37-.43.83-.64 1.38-.64.36 0 .75.06 1.16.17V26.9c-.39-.1-.81-.15-1.27-.15-1.18 0-2.17.38-2.94 1.15Zm6.94 5.29v-1.44h1.9v-1.64h-1.9v-1.46h2.19v-1.68h-4.32v7.91h4.34v-1.69h-2.2Z"/><g fill="#a71d69"><path d="M21.21 6.8c-4.96 0-6.94 3.33-6.94 7.03s1.95 7.18 6.94 7.18 6.93-3.48 6.92-6.95c-.01-4.22-2.15-7.26-6.92-7.26Zm-3 7.24c.12-4.96 3.93-4.95 5.2-3.18 1.16 1.62 1.4 7.15-2.2 7.11-1.66-.02-2.96-1.14-3-3.94ZM61.43 17.58V10.4H64V7.28h-2.57V3.51h-3.97v3.77h-2.01v3.12h2.01v6.77c0 3.85 2.68 6.14 5.52 6.61l1.18-3.19c-1.65-.21-2.72-1.44-2.74-3.02ZM41.42 16.44v-5.4c0-.2-.01-.38-.03-.56-.34-4.36-6.54-4.35-11.34-2.68l1.39 2.79c3.12-1.48 4.99-1.1 5.65-.37.02.02.03.04.05.06.33.45.25 1.16.25 1.72-4.08 0-8.43.54-8.43 5.02 0 3.4 4.27 5.59 8.72 2.35l.66 1.6h3.82c-.34-1.87-.75-3.48-.75-4.53Zm-3.99-.03c-3.14 3.51-6.02.21-3.07-1.24.87-.3 2.04-.31 3.07-.31v1.55ZM47.51 10.75c0-1.33 2.13-1.95 5.49-.42l1.16-2.82c-4.52-1.37-10.69-1.49-10.73 3.24-.02 2.27 1.43 3.48 3.53 4.26 1.45.54 3.54.82 3.54 1.97 0 1.51-3.25 1.74-6.2-.29l-1.19 3.06c4.03 2 11.52 2.06 11.48-2.96-.04-4.97-7.06-4.12-7.06-6.04ZM12.6 0 6.85 15.97 4.1 7.35H0l4.58 11.78c.39 1 .39 2.12 0 3.12-.51 1.31-1.37 2.44-3.44 2.72v3.49c4.05 0 6.23-2.49 8.19-7.94L16.98 0H12.6Z"/></g><g fill="#a71d69"><path d="M76.58 17.69c0 1.05-.38 1.87-1.14 2.46s-1.8.89-3.14.89-2.42-.21-3.27-.62V18.6c.54.25 1.11.45 1.71.6s1.17.22 1.69.22c.76 0 1.33-.14 1.69-.43s.54-.68.54-1.17c0-.44-.17-.81-.5-1.12-.33-.31-1.02-.67-2.06-1.09-1.07-.43-1.83-.93-2.27-1.49-.44-.56-.66-1.23-.66-2.01 0-.98.35-1.76 1.05-2.32.7-.56 1.64-.85 2.81-.85s2.25.25 3.37.74l-.61 1.57c-1.05-.44-1.98-.66-2.81-.66-.62 0-1.1.14-1.42.41-.32.27-.48.63-.48 1.08 0 .31.06.57.19.79s.34.42.64.62c.3.19.83.45 1.6.77.86.36 1.5.7 1.9 1.01.4.31.7.66.89 1.06.19.39.28.85.28 1.39ZM85.36 20.89h-6.67V9.12h6.67v1.63h-4.74v3.21h4.44v1.61h-4.44v3.7h4.74v1.63ZM98.21 14.98c0 1.92-.48 3.41-1.44 4.47-.96 1.06-2.32 1.59-4.07 1.59s-3.14-.53-4.1-1.58c-.95-1.06-1.43-2.55-1.43-4.5s.48-3.44 1.44-4.48c.96-1.04 2.33-1.56 4.11-1.56s3.11.53 4.06 1.59c.96 1.06 1.43 2.55 1.43 4.47Zm-9 0c0 1.45.29 2.55.88 3.3.59.75 1.45 1.12 2.6 1.12s2.01-.37 2.59-1.11c.58-.74.87-1.84.87-3.31s-.29-2.54-.87-3.29c-.58-.75-1.44-1.12-2.58-1.12s-2.03.37-2.62 1.12c-.59.75-.88 1.84-.88 3.29Z"/></g><path fill="#77b22b" d="m101.16 21.06 4.62-12.17h1.73l-4.61 12.17h-1.74z"/></svg>                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           plugins/wordpress-seo-extended/packages/js/images/wordproof-logo.svg                                0000644                 00000006301 15122266557 0022011 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg viewBox="0 0 331 100" fill="none" xmlns="http://www.w3.org/2000/svg" width="140" height="100%"><rect width="100" height="100" rx="2" fill="#2000FF"></rect><path d="M81.9995 24.6154C83.3174 24.6154 84.2751 25.868 83.9295 27.1398L70.8032 75.4475C70.5665 76.3185 69.7758 76.9231 68.8732 76.9231H57.4891L50.7693 49.6795H61.3784L64.0335 62.108L74.5165 24.6154H81.9995Z" fill="white"></path><path opacity="0.8" d="M45.2764 24.6154L34.8392 62.0208L26.7866 24.6154H17.9375C16.6391 24.6154 15.6849 25.8333 15.9956 27.094L27.9026 75.4017C28.1228 76.2952 28.9243 76.9231 29.8445 76.9231H39.6469C40.5495 76.9231 41.3403 76.3184 41.5769 75.4474L54.699 27.1397C55.0444 25.8678 54.0868 24.6154 52.7689 24.6154H45.2764Z" fill="white"></path><path d="M141.36 35.28L136.8 54.04L132.16 35.28H125.24L132.92 64H139.36L144.08 45.64L148.88 64H155.32L162.96 35.28H156.2L151.6 54.04L147 35.28H141.36ZM183.786 53.48C183.786 46.92 179.426 42.48 173.226 42.48C166.986 42.48 162.666 46.92 162.666 53.48C162.666 60.04 166.986 64.44 173.226 64.44C179.426 64.44 183.786 60.04 183.786 53.48ZM169.026 53.48C169.026 49.8 170.666 47.72 173.226 47.72C175.786 47.72 177.426 49.8 177.426 53.48C177.426 57.16 175.786 59.24 173.226 59.24C170.666 59.24 169.026 57.16 169.026 53.48ZM200.804 42.88C200.404 42.84 200.004 42.8 199.404 42.8C196.964 42.8 194.924 44 193.964 45.68V42.92H187.564V64H193.964V53.64C193.964 50.6 196.164 48.88 198.924 48.88C199.644 48.88 200.124 48.92 200.804 49.04V42.88ZM211.723 64.44C214.243 64.44 216.243 63.4 217.443 61.92V64H223.843V35.28H217.443V45.04C216.243 43.52 214.243 42.48 211.723 42.48C205.923 42.48 202.643 47.44 202.643 53.48C202.643 59.48 205.923 64.44 211.723 64.44ZM217.483 52.92V54.04C217.483 57.2 215.763 59 213.363 59C210.563 59 209.083 56.72 209.083 53.48C209.083 50.24 210.563 47.96 213.363 47.96C215.763 47.96 217.483 49.72 217.483 52.92ZM235.67 54H241.11C247.87 54 252.07 50.76 252.07 44.64C252.07 38.48 247.87 35.28 241.11 35.28H228.99V64H235.67V54ZM235.67 48.4V40.92H240.75C243.91 40.92 245.51 42.28 245.51 44.64C245.51 47 243.91 48.4 240.75 48.4H235.67ZM268.424 42.88C268.024 42.84 267.624 42.8 267.024 42.8C264.584 42.8 262.544 44 261.584 45.68V42.92H255.184V64H261.584V53.64C261.584 50.6 263.784 48.88 266.544 48.88C267.264 48.88 267.744 48.92 268.424 49.04V42.88ZM291.226 53.48C291.226 46.92 286.866 42.48 280.666 42.48C274.426 42.48 270.106 46.92 270.106 53.48C270.106 60.04 274.426 64.44 280.666 64.44C286.866 64.44 291.226 60.04 291.226 53.48ZM276.466 53.48C276.466 49.8 278.106 47.72 280.666 47.72C283.226 47.72 284.866 49.8 284.866 53.48C284.866 57.16 283.226 59.24 280.666 59.24C278.106 59.24 276.466 57.16 276.466 53.48ZM315.042 53.48C315.042 46.92 310.682 42.48 304.482 42.48C298.242 42.48 293.922 46.92 293.922 53.48C293.922 60.04 298.242 64.44 304.482 64.44C310.682 64.44 315.042 60.04 315.042 53.48ZM300.282 53.48C300.282 49.8 301.922 47.72 304.482 47.72C307.042 47.72 308.682 49.8 308.682 53.48C308.682 57.16 307.042 59.24 304.482 59.24C301.922 59.24 300.282 57.16 300.282 53.48ZM329.94 35.04C329.06 34.96 328.46 34.92 327.58 34.92C322.06 34.92 319.62 37.44 319.62 41.92V42.92H316.9V48.2H319.62V64H326.02V48.2H329.9V42.92H326.02V42.28C326.02 40.44 326.86 40 328.5 40C329.1 40 329.46 40 329.94 40.04V35.04Z" fill="#2000FF"></path></svg>                                                                                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/packages/js/images/wp-recipe-maker-logo.svg                          0000644                 00000024241 15122266557 0022763 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg width="185" role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1488.21 189.15"><rect width="23.81" height="26.01" x="113.83" y="17.03" fill="#3a5160" rx="4" ry="4" transform="rotate(-10 125.733 30.062)"/><rect width="23.81" height="26.01" x="119.56" y="49.62" fill="#3a5160" rx="4" ry="4" transform="rotate(-10 131.463 62.64)"/><path fill="#010101" d="M1120.63 107.35c-2.45 8.59-4.88 17.07-7.33 25.62-3.61-.29-7.03-.55-10.43-.83-1.86-.15-3.71-.42-5.57-.45-1.3-.02-1.94-.36-2.25-1.74-3.51-15.67-7.15-31.32-10.65-47-2.81-12.58-5.48-25.18-8.27-37.77-.26-1.17-.89-2.26-1.69-3.33.53 34.76 1.77 69.45 3.62 104.35-8.55-.78-16.89-1.54-25.37-2.31-1.24-42.47-3.33-84.87 1.77-127.36 3.42.26 6.78.51 10.13.77 8.19.65 16.38 1.33 24.58 1.92 1.11.08 1.54.43 1.82 1.46 4.04 14.57 7.19 29.33 9.99 44.17 1.23 6.51 2.43 13.02 3.64 19.53.03.18.1.35.27.93 7.73-24.53 13.76-49.07 18.93-73.99 2.23.17 4.42.31 6.61.5 8.44.73 16.87 1.47 25.31 2.21 1.05.09 2.11.21 3.17.25.97.03 1.37.34 1.39 1.41.21 8.53.65 17.05.74 25.58.26 23.65.55 47.3.48 70.95-.03 11.06-.84 22.13-1.31 33.19-.03.73-.14 1.45-.22 2.28-8.85-.83-17.5-1.63-26.35-2.46 3.22-33.66 3.51-67.33 4.22-100.99-.27-.06-.54-.13-.81-.19-.52 1.33-1.18 2.62-1.53 3.99-1.39 5.46-2.64 10.95-4.02 16.42-3.6 14.26-7.23 28.51-10.87 42.88ZM265.8 87.29c-2.62 11.14-5.09 22.19-7.86 33.17-1.9 7.51-4.21 14.92-6.35 22.37-.08.28-.24.53-.4.89-3.54-.27-7.05-.53-10.56-.8-2.11-.16-4.22-.4-6.33-.49-1.11-.05-1.61-.3-1.85-1.54-2.74-14.44-5.65-28.85-8.35-43.29-2.31-12.35-4.52-24.72-6.57-37.11-1.61-9.76-2.95-19.56-4.35-29.35-.85-5.96-1.59-11.93-2.36-17.9-.06-.48 0-.98 0-1.63 8.26.69 16.46 1.38 24.63 2.06 3.21 29.25 6.41 58.4 9.61 87.54.24.02.47.03.71.05 3.2-12.67 5.75-25.49 8.2-38.32 2.46-12.87 4.49-25.83 6.73-38.86 6.2.55 12.56 1.12 19.02 1.7 2.28 25.19 5.32 50.17 8.09 75.17.25 0 .5.02.76.03 7.1-29.92 13.76-59.93 17.67-90.72 8.43.7 16.82 1.39 25.32 2.09-.87 4.78-1.66 9.33-2.54 13.86-4.56 23.44-9.8 46.74-16.01 69.81-3.96 14.69-8.3 29.28-12.41 43.93-.38 1.34-.9 1.62-2.31 1.47-5.21-.56-10.44-.94-15.66-1.32-.85-.06-1.19-.3-1.4-1.15-3.78-14.91-7.62-29.81-11.4-44.72-.93-3.65-1.66-7.36-2.49-11.04-.06-.28-.23-.54-.44-1.02-.39 1.82-.74 3.41-1.11 5.13ZM583.7 120.05c4.39 6.04 8.71 11.99 13.09 18.03-6.24 4.56-12.38 9.05-18.32 13.39-14.44-19.46-28.78-38.78-43.12-58.11-.16.08-.32.15-.48.23-.58 17.99-1.16 35.99-1.74 54.13-8.88-.97-17.46-1.9-25.78-2.81 0-3.14-.13-6.08.02-9 1.29-25.57 1.83-51.15 1.74-76.75-.05-13.67.09-27.34.09-41.01 0-1.21.45-1.72 1.52-2.18 10.29-4.43 21.01-7.33 32.16-8.52 7.54-.81 15.09-1.09 22.51 1.03 14.48 4.15 23.19 13.68 26.52 28.08 2.34 10.1 1.28 19.78-4.3 28.79-3.46 5.59-7.73 10.49-12.74 14.71-4.37 3.68-8.98 7.08-13.47 10.6l22.3 29.38m-15.29-80.78c-2.59-6.62-7.63-9.99-15.03-10.21-5.83-.18-11.4 1.07-16.93 2.68-1.11.32-1.49.79-1.49 1.95-.02 14.86-.1 29.71-.15 44.57 0 .47.05.94.09 1.68 2.3-1.2 4.41-2.23 6.46-3.37 9.2-5.15 17.52-11.34 23.96-19.78 4-5.25 5.34-10.95 3.1-17.51ZM378.34 8.76c9.83-1.09 19.28-.47 28.16 3.98 13.54 6.79 21.67 17.63 24.2 32.39 2.72 15.86-2.85 28.92-14.5 39.69-8.92 8.25-19.54 13.61-31 17.5-1.8.61-3.61 1.25-5.46 1.69-1.04.25-1.22.73-1.2 1.66.16 9.93.28 19.86.41 29.78.05 3.69.1 7.38.14 11.08 0 .34-.05.67-.08 1.15-8.68-.82-17.28-1.63-25.92-2.44-.1-2.25-.23-4.39-.29-6.52-.63-24.35-.69-48.69-.14-73.04.28-12.47.64-24.94.99-37.4.12-4.34.34-8.67.49-13.01.03-.78.23-1.22 1.08-1.51 7.48-2.51 15.12-4.24 23.12-5m11.61 68.82c1.39-.99 2.84-1.91 4.15-2.99 9.26-7.68 13.24-18.64 9.7-30.14-1.47-4.78-4.18-8.72-8.41-11.53-4.72-3.12-9.85-2.69-15.01-1.49-.33.08-.66.89-.68 1.38-.18 4.03-.31 8.07-.4 12.1-.29 12.36-.56 24.72-.83 37.08 0 .24.04.49.07.83 4.01-1.24 7.74-2.94 11.42-5.25ZM893.36 78.78c.89 5.83.84 11.57.28 17.28-.97 9.85-3.39 19.35-7.54 28.38-2.81 6.12-6.3 11.76-11.28 16.45-4.45 4.2-9.67 5.49-15.58 4.64-2.72-.39-5.42-.91-8.27-1.4.43 14.95.87 29.85 1.3 45.02-7.34-.73-14.77-1.46-22.39-2.21-1.11-46.35-2-92.64 1.68-139.13 6.73.77 13.3 1.53 20.11 2.31-.2 5.13-.4 10.22-.61 15.52 1.57-1.65 2.9-3.25 4.43-4.63 4.64-4.2 10.02-6.6 16.45-6.23 6.49.37 11.49 3.39 15.26 8.48 3.18 4.29 4.97 9.17 5.9 14.38.06.34.16.68.26 1.15m-33.25-3.25c-3.68 3.79-5.8 8.45-7.53 13.3-.66 1.85-1.25 3.82-1.3 5.75-.24 9.12-.29 18.24-.4 27.37 0 .34.02.72.18 1.01 2.28 4.2 6.74 4.88 10.26 1.62 3.54-3.28 5.38-7.54 6.89-11.93 3.1-9.04 4.05-18.33 2.74-27.8-.47-3.4-1.34-6.73-3.24-9.7-1.19-1.85-2.59-2.36-4.63-1.49-1 .43-1.87 1.14-2.96 1.87Z"/><path fill="#020202" d="M1307.39 122.54c-4.09-4.4-8.1-8.72-12.12-13.04-.13.05-.27.11-.4.16.26 12.07.51 24.13.77 36.33-7.46-.69-14.9-1.39-22.48-2.09-1.58-46.55-.61-92.99 1.71-139.57 7.62.58 15.04 1.15 22.22 1.7-.99 29.54-1.98 58.88-2.97 88.23.22.05.45.11.67.16 11.62-14.1 21.84-29.09 29.48-46.04 5.39 4.5 10.64 8.88 15.99 13.35-6.1 13.98-13.95 26.91-22.87 39.14 4.19 5.22 8.5 10.29 12.47 15.62 3.97 5.32 7.6 10.9 11.43 16.43-5.06 4.7-9.91 9.2-14.91 13.85-5.74-8.57-12.06-16.56-18.99-24.21ZM1242.27 129.14l-1.25 17.55c-2.51-.14-4.87-.27-7.22-.42-2.72-.18-5.43-.34-8.14-.61-.43-.04-1.04-.55-1.18-.96-.51-1.55-.85-3.16-1.27-4.82-2.52 1.06-4.86 2.17-7.29 3.04-5.96 2.15-12 2.34-18.07.46-8.06-2.5-13.54-9.59-14.3-17.91-1.04-11.52 4.12-19.78 13.51-25.88 5.95-3.87 12.56-5.98 19.6-6.82l4.69-.55c.08-4.79.21-9.47-.99-14.05-.26-1-.76-1.97-1.32-2.84-2.02-3.17-5.77-3.61-8.52-1.02-3.26 3.07-4.74 7.11-6.04 11.23-.34 1.09-.41 2.55-1.16 3.13-.65.5-2.1.02-3.19-.07-4.71-.41-9.42-.83-14.45-1.28.73-2.67 1.34-5.26 2.15-7.79 2.55-7.94 6.21-15.24 12.46-21.07 9.36-8.75 24.76-8.96 33.63-.29 4.33 4.24 6.71 9.6 7.92 15.38.89 4.22 1.58 8.55 1.71 12.86.43 14.22-.32 28.42-1.29 42.75m-30.89-1.07c2.98-.63 5.97-1.23 8.94-1.94.33-.08.69-.78.71-1.21.14-3.53.21-7.07.28-10.6.03-1.48 0-2.95 0-4.55-2.22.45-4.3.79-6.32 1.31-4.43 1.15-8.39 3.07-10.92 7.1-2.63 4.19-.79 8.59 4.04 9.62.97.21 1.99.18 3.27.26ZM1371.79 57.82c8.98-6.55 18.49-7.72 28.52-3.01 4.13 1.94 6.94 5.4 9.16 9.29 3.27 5.71 5.06 11.94 5.77 18.39.77 6.97 1 14 1.5 21.01.07.96-.17 1.34-1.2 1.59-11.84 2.91-23.83 5.07-35.89 6.86-.93.14-1.85.33-2.92.52.63 5.04 1.1 10 4.13 14.24 1.56 2.19 3.14 2.71 5.76 1.99 3.43-.94 5.86-3.28 8-5.88 1.96-2.39 3.66-4.99 5.6-7.67 4.79 2.81 9.66 5.66 14.63 8.56-5.15 8.94-10.94 17.18-21.11 20.95-14 5.18-27.62.46-34.85-13.06-3.12-5.83-4.61-12.13-4.89-18.64-.56-13.37 1.19-26.44 6.29-38.94 2.54-6.23 6.12-11.79 11.5-16.2m13.65 38.82c3.16-.74 6.31-1.47 9.73-2.27-.41-4.68-.62-9.31-1.28-13.87-.37-2.61-1.2-5.23-2.29-7.64-1.23-2.71-3.63-3.09-5.88-1.06a15.806 15.806 0 0 0-3.3 4.2c-3.7 6.95-4.77 14.55-5.43 22.39 2.83-.6 5.51-1.17 8.45-1.76ZM662.46 58.62c5.16 5.37 7.48 11.97 9.02 18.88 1.99 8.95 2.28 18.05 2.26 27.28-13.11 3.37-26.44 5.61-39.9 7.64.55 4.32.98 8.51 2.94 12.35 2.14 4.2 4.64 5.91 10.18 2.42 3.65-2.3 6.08-5.77 8.4-9.29.6-.91 1.2-1.82 1.85-2.82 4.93 2.89 9.77 5.72 14.72 8.63-5.15 9.05-11.04 17.34-21.36 21.04-13.9 4.98-27.46.44-34.72-13.4-3.78-7.21-4.95-14.99-4.88-23.01.1-11.04 1.74-21.83 5.6-32.22 2.44-6.59 5.86-12.58 11.2-17.34 5.32-4.74 11.59-6.87 18.7-6.72 6.15.13 11.54 2.1 15.98 6.58m-27.88 35.11c-.16 1.51-.33 3.01-.52 4.72 6.16-1.36 12.07-2.67 18.31-4.04-.54-5.21-.9-10.32-1.65-15.36-.35-2.3-1.28-4.59-2.32-6.71-.96-1.97-2.75-2.41-4.57-1.16-1.41.96-2.73 2.26-3.62 3.7-3.51 5.69-4.81 12.08-5.64 18.86Z"/><path fill="#010101" d="M926.11 142.76c-6.82-4.43-10.73-10.77-13.02-18.22-2.25-7.3-2.43-14.78-1.91-22.33.5-7.21 1.61-14.31 3.71-21.24 2.46-8.09 5.94-15.64 12.2-21.65 8.54-8.2 22.19-9.77 31.91-3.54 4.52 2.89 7.31 7.22 9.39 12.01 2.69 6.18 4 12.71 4.56 19.38.48 5.81.69 11.63 1.02 17.53-13.29 3.43-26.63 5.68-40.17 7.71.69 5.07 1.11 10.09 4.18 14.37 1.51 2.1 3.16 2.63 5.65 1.93 4.12-1.17 6.78-4.18 9.25-7.35 1.53-1.96 2.87-4.07 4.38-6.24 4.84 2.84 9.67 5.68 14.63 8.59-3.06 5.33-6.32 10.41-10.89 14.59-6.25 5.71-13.56 8.6-22.11 8.33-4.57-.14-8.82-1.35-12.78-3.86m11.79-63.33c-2.35 6.08-3.34 12.41-3.81 19.02l18.22-4.04c-.44-4.8-.64-9.44-1.35-14-.43-2.75-1.43-5.47-2.58-8.02-.94-2.08-2.67-2.37-4.69-1.2-.85.49-1.73 1.14-2.25 1.94-1.27 1.96-2.34 4.04-3.55 6.3Z"/><path fill="#020202" d="M714.55 110.73c.03 3.69.1 7.24 1.51 10.56.51 1.21 1.25 2.4 2.14 3.38 1.73 1.91 3.74 2.13 5.98.81 3.13-1.84 5.16-4.73 7.06-7.66 2.39-3.67 4.59-7.47 6.94-11.33 4.68 3.34 9.4 6.71 14.16 10.12-3.09 7.86-6.75 15.32-12.56 21.6-10.5 11.33-24.8 10.15-34.82 3.47-6.67-4.44-10.21-10.98-11.89-18.58-1.53-6.9-1.26-13.85-.65-20.83.72-8.31 2.2-16.45 4.89-24.37 2.26-6.66 5.24-12.93 10.19-18.13 7.32-7.69 18.53-9.76 27.48-4.89 5.24 2.85 8.71 7.36 11.2 12.59 3.44 7.25 5.07 14.87 4.04 22.9-.24 1.89-.75 3.75-1.13 5.62-6.29-.59-12.34-1.16-18.19-1.71 0-2.32.05-4.5-.01-6.68-.1-3.57-.73-7.03-2.51-10.19-2.07-3.66-5.08-3.78-7.55-.35-2.46 3.42-3.31 7.42-4.13 11.41-1.5 7.29-1.91 14.7-2.13 22.27ZM1478.68 74.89c-5.02-1.18-8.22 1.08-10.32 5.16-2.82 5.49-4.74 11.24-4.7 17.53.08 11.92.08 23.84.02 35.76-.02 4.38-.32 8.76-.5 13.35-7.53-.61-14.93-1.2-22.48-1.81.75-28.07 1.59-55.97-2.83-83.89 2.83.27 5.42.51 8.01.76 1.75.17 3.5.43 5.26.52.99.05 1.4.46 1.65 1.34.79 2.72 1.64 5.43 2.5 8.23 1.62-2.77 3.14-5.46 4.74-8.1 1.99-3.27 4.39-6.23 7.39-8.64 4.54-3.64 9.55-3.86 15.17-.59 1.81 1.05 3.4 2.47 5.04 3.79.32.26.64.88.54 1.23-1.53 5.58-3.13 11.14-4.74 16.82-1.61-.5-3.13-.96-4.77-1.46Z"/><path fill="#010101" d="M775.99 52.99c.82-.19 1.63-.32 2.43-.25 6.42.59 12.84 1.21 19.25 1.83.09 0 .17.09.31.17v91.03c-7.34-.7-14.59-1.38-22.01-2.09 0-30.22 0-60.38.01-90.69Z"/><path fill="#020202" d="M798.68 26.96c-.47 5.38-.93 10.63-1.41 16.04-7.27-.88-14.35-1.73-21.71-2.62.63-7.04 1.26-14.05 1.91-21.28 7.31.83 14.48 1.65 21.74 2.48-.19 1.88-.36 3.56-.53 5.38Z"/><path fill="#607983" d="m8.856 20.455 97.89-17.26c6.795-1.199 13.282 3.344 14.48 10.139l20.88 118.413c1.198 6.795-3.344 13.283-10.14 14.48l-97.89 17.261a7.097 7.097 0 0 1-8.213-5.75L3.105 28.667a7.097 7.097 0 0 1 5.75-8.213Z"/><path fill="#3a5160" d="M21.37 18.25 8.86 20.46c-3.84.68-6.43 4.37-5.75 8.21l22.76 129.08c.68 3.84 4.37 6.43 8.21 5.75l12.51-2.21L21.37 18.25Z"/><path fill="#d2e0e4" d="M89.24 39.48c-1.9.34-3.17 2.15-2.84 4.05l3.83 21.7c.45 2.57-1.29 5.02-3.85 5.47s-4.99-1.26-5.44-3.83l-3.83-21.7c-.34-1.9-2.15-3.17-4.05-2.84s-3.17 2.15-2.84 4.05l3.83 21.7c.45 2.57-1.29 5.02-3.85 5.47s-4.99-1.26-5.44-3.83l-3.83-21.7c-.34-1.9-2.15-3.17-4.05-2.84s-3.17 2.15-2.84 4.05l4.46 25.28c1.49 8.45 8.11 14.69 16.08 16.14l6.09 34.55c.66 3.75 4.24 6.26 8 5.6s6.26-4.24 5.6-8l-6.09-34.55c7-4.09 11.08-12.21 9.59-20.66l-4.46-25.28a3.507 3.507 0 0 0-4.05-2.84Z"/></svg>                                                                                                                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/packages/js/images/x-logo.svg                                        0000644                 00000000630 15122266557 0020236 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg viewBox="0 0 1200 1227" fill="current" xmlns="http://www.w3.org/2000/svg">
<path d="M714.163 519.284L1160.89 0H1055.03L667.137 450.887L357.328 0H0L468.492 681.821L0 1226.37H105.866L515.491 750.218L842.672 1226.37H1200L714.137 519.284H714.163ZM569.165 687.828L521.697 619.934L144.011 79.6944H306.615L611.412 515.685L658.88 583.579L1055.08 1150.3H892.476L569.165 687.854V687.828Z" fill="current"/>
</svg>
                                                                                                        plugins/wordpress-seo-extended/packages/js/images/Yoast_icon_kader.svg                              0000644                 00000002471 15122266557 0022313 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 425 456.27" role="img" aria-hidden="true" focusable="false"><path d="M73 405.26a66.79 66.79 0 0 1-6.54-1.7 64.75 64.75 0 0 1-6.28-2.31c-1-.42-2-.89-3-1.37-1.49-.72-3-1.56-4.77-2.56-1.5-.88-2.71-1.64-3.83-2.39-.9-.61-1.8-1.26-2.68-1.92a70.154 70.154 0 0 1-5.08-4.19 69.21 69.21 0 0 1-8.4-9.17c-.92-1.2-1.68-2.25-2.35-3.24a70.747 70.747 0 0 1-3.44-5.64 68.29 68.29 0 0 1-8.29-32.55V142.13a68.26 68.26 0 0 1 8.29-32.55c1-1.92 2.21-3.82 3.44-5.64s2.55-3.58 4-5.27a69.26 69.26 0 0 1 14.49-13.25C50.37 84.19 52.27 83 54.2 82A67.59 67.59 0 0 1 73 75.09a68.75 68.75 0 0 1 13.75-1.39h169.66L263 55.39H86.75A86.84 86.84 0 0 0 0 142.13v196.09A86.84 86.84 0 0 0 86.75 425h11.32v-18.35H86.75A68.75 68.75 0 0 1 73 405.26zM368.55 60.85l-1.41-.53-6.41 17.18 1.41.53a68.06 68.06 0 0 1 8.66 4c1.93 1 3.82 2.2 5.65 3.43A69.19 69.19 0 0 1 391 98.67c1.4 1.68 2.72 3.46 3.95 5.27s2.39 3.72 3.44 5.64a68.29 68.29 0 0 1 8.29 32.55v264.52H233.55l-.44.76c-3.07 5.37-6.26 10.48-9.49 15.19L222 425h203V142.13a87.2 87.2 0 0 0-56.45-81.28z"/><path d="M119.8 408.28v46c28.49-1.12 50.73-10.6 69.61-29.58 19.45-19.55 36.17-50 52.61-96L363.94 1.9H305l-98.25 272.89-48.86-153h-54l71.7 184.18a75.67 75.67 0 0 1 0 55.12c-7.3 18.68-20.25 40.66-55.79 47.19z" stroke="#000" stroke-miterlimit="10" stroke-width="3.81"/></svg>                                                                                                                                                                                                       plugins/wordpress-seo-extended/packages/js/images/yoast_loading_spinner.svg                         0000644                 00000002212 15122266557 0023421 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg role="img" aria-hidden="true" focusable="false" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 229.95 98"><defs><style>.cls-1{fill:#e7c9da;}.cls-2{fill:#a4296a;}</style><symbol id="Spinner" data-name="Spinner" viewBox="0 0 98 98"><path class="cls-1" d="M49,0A49,49,0,1,0,98,49,49,49,0,0,0,49,0Zm0,89A40,40,0,1,1,89,49,40,40,0,0,1,49,89Z"/><path class="cls-2" d="M91.44,73.5A49,49,0,0,1,49,98V89A40,40,0,0,0,83.65,69Z"/></symbol></defs><use width="98" height="98" xlink:href="#Spinner"/><path class="cls-1" d="M221.45,137.3a24,24,0,1,0,24,24,24,24,0,0,0-24-24Zm0,43.59A19.59,19.59,0,1,1,241,161.3a19.59,19.59,0,0,1-19.59,19.59Z" transform="translate(-65.23 -112.3)"/><path class="cls-2" d="M242.23,173.3a24,24,0,0,1-20.78,12v-4.41a19.6,19.6,0,0,0,17-9.79Z" transform="translate(-65.23 -112.3)"/><path class="cls-1" d="M283.18,149.3a12,12,0,0,0-.11,24h.11a12,12,0,0,0,.1-24Zm0,21.8h-.09a9.8,9.8,0,0,1,.09-19.6h.08a9.8,9.8,0,0,1-.08,19.6Z" transform="translate(-65.23 -112.3)"/><path class="cls-2" d="M289.27,151a12,12,0,0,1,5.91,10.45l-2.21,0a9.76,9.76,0,0,0-4.82-8.53Z" transform="translate(-65.23 -112.3)"/></svg>                                                                                                                                                                                                                                                                                                                                                                                      plugins/wordpress-seo-extended/packages/js/images/Yoast_SEO_Icon.svg                                0000644                 00000002033 15122266557 0021605 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" role="img" aria-hidden="true" viewBox="0 0 500 500" focusable="false"><path d="M80 0h340a80 80 0 0 1 80 80v420H80a80 80 0 0 1-80-80V80A80 80 0 0 1 80 0z" fill="#a4286a"/><path d="M437.61 2L155.89 500H500V80a80 80 0 0 0-62.39-78z" fill="#6c2548"/><path d="M74.4 337.3v34.9c21.6-.9 38.5-8 52.8-22.5s27.4-38 39.9-72.9l92.6-248h-44.8L140.3 236l-37-116.2h-41l54.4 139.8a57.54 57.54 0 0 1 0 41.8c-5.5 14.2-15.4 30.9-42.3 35.9z" fill="#fff"/><circle cx="368.33" cy="124.68" r="97.34" transform="rotate(-45 368.335 124.68)" fill="#9fda4f"/><path d="M416.2 39.93l-95.74 169.51A97.34 97.34 0 1 0 416.2 39.93z" fill="#77b227"/><path d="M294.78 254.75l-.15-.08-.13-.07a63.6 63.6 0 0 0-62.56 110.76h.13a63.6 63.6 0 0 0 62.71-110.67z" fill="#fec228"/><path d="M294.5 254.59l-62.56 110.76a63.6 63.6 0 1 0 62.56-110.76z" fill="#f49a00"/><path d="M222.31 450.07A38.16 38.16 0 0 0 203 416.83a38.18 38.18 0 1 0 19.41 33.27z" fill="#ff4e47"/><path d="M202.9 416.8l-37.54 66.48a38.17 38.17 0 0 0 37.54-66.48z" fill="#ed261f"/></svg>                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     plugins/wordpress-seo-extended/packages/js/images/yoast-seo-simple-logo.svg                         0000644                 00000002443 15122266557 0023205 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 425 456.27" fill="currentColor" stroke="none" role="img" aria-hidden="true" focusable="false"><path d="M73 405.26a66.79 66.79 0 0 1-6.54-1.7 64.75 64.75 0 0 1-6.28-2.31c-1-.42-2-.89-3-1.37-1.49-.72-3-1.56-4.77-2.56-1.5-.88-2.71-1.64-3.83-2.39-.9-.61-1.8-1.26-2.68-1.92a70.154 70.154 0 0 1-5.08-4.19 69.21 69.21 0 0 1-8.4-9.17c-.92-1.2-1.68-2.25-2.35-3.24a70.747 70.747 0 0 1-3.44-5.64 68.29 68.29 0 0 1-8.29-32.55V142.13a68.26 68.26 0 0 1 8.29-32.55c1-1.92 2.21-3.82 3.44-5.64s2.55-3.58 4-5.27a69.26 69.26 0 0 1 14.49-13.25C50.37 84.19 52.27 83 54.2 82A67.59 67.59 0 0 1 73 75.09a68.75 68.75 0 0 1 13.75-1.39h169.66L263 55.39H86.75A86.84 86.84 0 0 0 0 142.13v196.09A86.84 86.84 0 0 0 86.75 425h11.32v-18.35H86.75A68.75 68.75 0 0 1 73 405.26zM368.55 60.85l-1.41-.53-6.41 17.18 1.41.53a68.06 68.06 0 0 1 8.66 4c1.93 1 3.82 2.2 5.65 3.43A69.19 69.19 0 0 1 391 98.67c1.4 1.68 2.72 3.46 3.95 5.27s2.39 3.72 3.44 5.64a68.29 68.29 0 0 1 8.29 32.55v264.52H233.55l-.44.76c-3.07 5.37-6.26 10.48-9.49 15.19L222 425h203V142.13a87.2 87.2 0 0 0-56.45-81.28z"/><path d="M119.8 408.28v46c28.49-1.12 50.73-10.6 69.61-29.58 19.45-19.55 36.17-50 52.61-96L363.94 1.9H305l-98.25 272.89-48.86-153h-54l71.7 184.18a75.67 75.67 0 0 1 0 55.12c-7.3 18.68-20.25 40.66-55.79 47.19z"/></svg>
                                                                                                                                                                                                                             plugins/wordpress-seo-extended/readme.txt                                                           0000644                 00000061230 15122266557 0014652 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       === Yoast SEO ===
Contributors: yoast, joostdevalk, tdevalk
Donate link: https://yoa.st/1up
License: GPLv3
License URI: http://www.gnu.org/licenses/gpl.html
Tags: SEO, XML sitemap, Content analysis, Readability, Schema
Tested up to: 6.5
Stable tag: 22.5
Requires PHP: 7.2.5

Improve your WordPress SEO: Write better content and have a fully optimized WordPress site using the Yoast SEO plugin.

== Description ==

## YOAST SEO: THE #1 WORDPRESS SEO PLUGIN

Supercharge your website's visibility and attract organic traffic with Yoast SEO, the WordPress SEO plugin trusted by millions worldwide.

With those millions of users, we've definitely helped someone like you! Users of our plugin range from owners of small-town bakeries and local physical stores to some of the world's largest and most influential organizations. And we've done this since 2008!

Yoast SEO Free provides the essentials to kickstart your SEO, and the [Yoast SEO Premium](https://yoa.st/1v8) plugin and its extensions unlock extra tools and functionality to take your SEO to the next level.

### SEO FOR EVERYONE, MADE BY SEO EXPERTS

SEO is the most consistent and cost-effective website traffic source, but it can be a challenging and complex maze. Luckily, you don't have to do this alone. We're here to help!

Yoast's mission is **SEO for everyone**. Whether you're a beginner or a seasoned expert, Yoast SEO equips you with all the essentials to navigate the complexities of SEO, turning those challenges into opportunities for growth and success.

### STAY ON TOP OF SEO WITH NEW FEATURES

Creating engaging and optimized SEO titles and meta descriptions has never been easier, faster, and more precise! With the AI title & meta description generator, you'll save much time. The feature is available for [Yoast SEO Premium](https://yoa.st/1v8) users.

### WHY YOAST SEO?

Are you agonizing over the intricate details of search engine optimization? Are you losing sleep trying to crack the code of SEO algorithms? Is the ever-evolving landscape of SEO leaving you overwhelmed and behind the curve?

These are the everyday struggles website owners face when trying to master the art of SEO and gain that coveted top spot in search results.

With Yoast SEO, you can overcome these challenges and unlock extraordinary gains:

* **Beat the competition:** Leave your rivals in the dust with our powerful SEO tools. Outperform industry competitors and enjoy the traffic and recognition you deserve.

* **Streamline your SEO strategy:** Bid farewell to the complexities of SEO. Yoast SEO handles the technical aspects, such as canonical URLs and meta tags, automating essential optimizations behind the scenes. Unleash your focus on creating captivating content that connects with your audience.

* **Climb those rankings:** Use advanced features like XML sitemaps, title and meta description templating, and Schema.org structured data integration. Empower search engines to fully understand your website, helping you reach the top of search results.

* **Optimize content mastery:** Yoast SEO offers comprehensive analysis tools that help elevate your content's SEO and readability. Get powerful insights and actionable recommendations to craft helpful content that resonates with readers and search engines.

### LET'S GO OVER KEY FEATURES IN YOAST SEO

Yoast SEO helps you do WordPress SEO in many ways. Let's take a look!

#### QUICK AND EASY SETUP TO GET YOU STARTED

Setting up Yoast SEO is smooth, hassle-free, and doesn't require any advanced knowledge! Our step-by-step configuration walks you through the essentials, helping you get up and running quickly.

You'll be prompted to fill in details about your site during the setup. This enables Yoast SEO to translate all this information into structured data, which helps search engines better understand your content and website!

Are you already using another SEO plugin? Transitioning from other plugins to Yoast SEO is a breeze. We've designed it to be seamless, ensuring you don't lose critical data.

#### TAKING CARE OF YOUR TECHNICAL SEO NEEDS

Although technical optimization is a big part of SEO, it's not easy. We understand that only some people who work on SEO are experts or have the time to dive into the technical aspects of a website. That's why Yoast SEO handles much of the technical optimization, freeing up your time to work on other parts of your website, like writing helpful content.

* **Get automated technical SEO improvements**, like optimized meta tags, right out of the box.

* **Add canonical URLs** to tell search engines which content they should show when you have pages with similar content.

* Get **advanced XML sitemaps**, making it effortless for search engines to understand your site structure and index your web pages effectively.

* **Get best-in-class Schema.org structured data integration**, significantly increasing your chances of getting visually rich search results that attract more users.

* **Take complete control over your site's breadcrumbs**, allowing visitors and search engines to navigate your website seamlessly.

* **Significantly improves your website's loading times**, courtesy of our innovative data management techniques tailored for WordPress.

* [Advanced] Yoast SEO comes with crawl settings that **optimize how search engines crawl your site** and reduce its carbon footprint. This lowers your site's environmental impact and contributes to a sustainable web.

#### WRITE AWESOME CONTENT THAT USERS AND SEARCH ENGINES LOVE

Unlock the full potential of your content with Yoast SEO's **state-of-the-art content analysis**. Crafting compelling content becomes easier with features designed to help you increase your website's performance.

* **Leverage detailed SEO analysis** that guides you toward creating SEO-friendly content, allowing you to target the right keywords and boost your visibility in search results.

* **Drive engagement and enhance readability** with the integrated readability analysis. Ensure your content is clear, concise, and effortlessly readable by humans and search engines alike.

* Preview your content as it appears in SERPs, even on mobile devices. This helps you to **fine-tune your meta titles and descriptions to maximize click-through rates**.

* Enable your HowTo content to be displayed in search results by using the **innovative Schema structured data blocks** for the WordPress block editor

* **A dedicated breadcrumbs block** ensures your users always know their location within your website.

* **Embrace inclusivity in your content creation process with the inclusive language analysis**. This optional feature analyzes your text and provides suggestions to make your content more considerate of different audiences. By using inclusive language, you’ll ensure your content resonates with diverse groups of people.

* **Seamlessly do keyword research** right within the plugin with the Semrush integration. Find out what related keywords people are searching for, so you can optimize your content to reach more audiences.

* **Track your rankings in Yoast SEO** with the Wincher integration. Yoast SEO and Wincher show how your content and keywords rank in Google.

* Are you using Elementor to build your website? No worries because **Yoast SEO integrates with Elementor**. Take advantage of all Yoast SEO's benefits within your favorite website builder!

* **Yoast SEO Premium even has advanced AI features**, helping you write titles and meta descriptions at the touch of a button thanks to generative AI. The artificial intelligence tool is free to use once you have Yoast SEO Premium.

#### KEEP YOUR SITE IN SHAPE FOR SEO

Yoast SEO empowers you to keep your website in perfect shape, regardless of your role or expertise:

* Fine-tune your website's engine, allowing you to focus on creating engaging and valuable content. With Yoast SEO, **technical optimization becomes effortless**, allowing you to prioritize what truly matters.

* **Structure your website easily** using Yoast SEO's cornerstone content features, enabling search engines to understand and index your most important pages effectively.

* **Translate valuable content into structured data**, allowing search engines to fully comprehend your website's meaning and context.

* Yoast SEO includes a **powerful front-end SEO inspector** that lets you preview and fine-tune the SEO settings directly on your front end. With this intuitive tool, you can easily optimize elements such as meta titles, meta descriptions, URL slugs, robots meta tags, and structured data by seeing their appearance in real-time.

* **Yoast SEO has a regular 2-week update cycle**, ensuring you always stay up-to-date with the latest development and updates from search engines.

#### POWERFUL INTEGRATION WITH OTHER TOOLS

Yoast SEO seamlessly integrates with various themes, plugins, and tools to enhance the user experience and improve your WordPress SEO workflow.

* Utilize the full potential of the [Advanced Custom Fields](https://wordpress.org/plugins/advanced-custom-fields/) plugin when combined with the [ACF Content Analysis for Yoast SEO](https://wordpress.org/plugins/acf-content-analysis-for-yoast-seo/) plugin, and reap the benefits of Yoast SEO's powerful analysis.

* Seamlessly integrate Yoast SEO with the [Elementor](https://wordpress.org/plugins/elementor/) website builder, empowering you to optimize your stunning designs effectively.

* Elevate your site search quality with [Algolia](https://wordpress.org/plugins/wp-search-with-algolia/) integration, ensuring users can effortlessly find the content they want.

* Combine Yoast SEO with Semrush, a leading online marketing tool. Access comprehensive keyword data directly within Yoast SEO, empowering you to supercharge your SEO strategy with data-driven insights.

* Connect your Yoast SEO plugin with Wincher, a powerful SEO tracking tool. Monitor your keyword rankings, and track your website's visibility in search results.

#### TRUST THE EXPERTS

Yoast is powered by expert developers, testers, software architects, and SEO consultants. They work constantly to stay at the cutting edge of WordPress SEO and to improve the plugin with every release.

#### WE'LL ONBOARD YOU AND HELP YOU DO SEO

We're here to help you succeed with your SEO goals because empowering our users is at the core of our philosophy!

**Onboarding experience:** After you install the plugin and run the first-time setup wizard, we offer an intuitive onboarding experience that guides you through the essential settings and features. This ensures you're making the most out of your new plugin, step by step.

**Courses in Yoast SEO academy:** If you're looking for a structured learning path, our [Yoast SEO academy](https://yoa.st/3ri) offers free and paid online courses. These lessons cover everything from the basics of WordPress to advanced SEO techniques. Signing up for Yoast SEO Premium also gives you access to all the courses at no extra charge.

**SEO blog:** Our regularly updated SEO blog is a treasure trove of information, filled with practical tips, insights, and guides on improving your SEO.

**Newsletter:** Stay in the loop with our newsletter. It's more than just news; it's a timely update on important changes and trends in the SEO world. We'll also provide you with tips and guides on how to improve your SEO.

**Webinars:** Our regular webinars provide live and interactive learning experiences. They cover various topics and are designed to help you get started with SEO, no matter your skill level.

#### EXPAND YOUR SEO KNOWLEDGE WITH ONLINE COURSES

Investing in [Yoast SEO Premium](https://yoa.st/1v8) gives you FULL access to all courses, from SEO copywriting to ecommerce SEO. These courses equip you with more advanced knowledge, skills, and tips to help your website grow and succeed. Get two great products for a single price with Yoast SEO Premium and the Yoast SEO academy!

### SERIOUS ABOUT GETTING HIGHER RANKINGS AND MORE TRAFFIC?

Not only do you get many additional benefits by upgrading to [Yoast SEO Premium](https://yoa.st/1v8), but you'll also get 24/7 personalized support that takes away your worry.

* Optimize your content and diversify your keywords using synonyms and related keyphrases. Expand your reach and deliver more natural, engaging content to your audience. You can optimize up to 5 keyphrases with Yoast SEO Premium.

* Optimizes your articles for different word forms, singular and plural variations, different verb forms, synonyms, and related keyphrases, thanks to the semantic understanding technology in Yoast SEO Premium.

* Seamlessly handle URL changes or page deletions with our redirect manager. Automatically create redirects to prevent "404: page not found" errors and retain valuable traffic and backlinks.

* Get internal linking suggestions in real-time. The WordPress SEO plugin enhances your article's depth and authority by recommending related posts to link to.

* Gain precise control over your page's appearance on social networks like Facebook and X with social media previews. Customize your social presence and entice users to engage with your content effectively.

* Simplify your SEO workflows and get guidance to work on time-consuming SEO tasks, like finding unlinked content, with the Yoast SEO workouts.

* Get actionable feedback on using inclusive vocabulary to enhance your writing and make it accessible to a broader audience. (Note: This feature is currently available in English and beta.)

* Take advantage of the IndexNow integration, which instantly pings search engines like Microsoft Bing every time you publish or update content to ensure timely indexing.

* Easily create optimized SEO titles and meta descriptions at the touch of a button, thanks to generative AI. Crafting optimized and engaging SEO titles and meta descriptions has never been faster and easier.

* Get 24/7 personalized e-mail support, so no matter when a question or issue arises, you're never left in the dark. Whether it's a technical hurdle or simply a need for guidance, our dedicated support team is always available to assist.

* Avoid your content being used to train AI bots: Effortlessly safeguard your intellectual property, uphold data privacy, and maintain control over content by blocking AI bots from scraping it. These AI web crawlers include OpenAI's GPTBot, Common Crawl's CCBot and Google-Extended, used to train Google Gemini.

### EXTEND YOUR WORDPRESS SEO

Take your WordPress SEO to new heights with these powerful Yoast SEO add-ons:

* [Yoast Local SEO](https://yoa.st/1uu): Optimize your website for a local audience, drive foot traffic to your stores, and establish a strong presence in local SERPs.

* [Yoast Video SEO](https://yoa.st/1uw): Improves the performance of videos and ensures that Google fully understands its content. This helps you rank your videos higher in video search results.

* [Yoast News SEO](https://yoa.st/1uv): Amplify your visibility and performance in Google News, allowing your news website to reach a broader audience.

* [Yoast WooCommerce SEO](https://yoa.st/3rh): Enhance your online shop's discoverability with extra tools designed specifically for online stores, helping you drive more targeted traffic and dominate the search results for your products. It has generative AI tools to help you write great titles and meta descriptions for your products! Also, use WooCommerce SEO to easily import and export global identifiers like GTIN8, UPC, and ISBN for your products.

### BUG REPORTS

Find a bug in Yoast SEO? We welcome your bug reports! Please report bugs in the WordPress SEO [repository on GitHub](https://github.com/Yoast/wordpress-seo). Note that GitHub is not a support forum but an efficient platform for addressing and resolving issues efficiently.

### THERE'S MORE TO LEARN ABOUT SEO

For a comprehensive resource on search engine optimization and Yoast SEO, please explore our [website](https://yoast.com), which is rich with insights and inspiration. Seek guidance in our extensively curated help center, designed to help you provide the knowledge to optimize your website effectively.

Take your SEO skills to new heights with "[WordPress SEO - The Definitive Guide](https://yoa.st/1v6)" by Yoast - a must-read for WordPress SEO enthusiasts. Discover more exceptional plugins and solutions by Team Yoast, tailored to enhance your digital presence and achieve unparalleled success.

== Installation ==
Starting with Yoast SEO consists of just two steps: installing and setting up the plugin. Yoast SEO is designed to work with your site’s specific needs, so don’t forget to go through the Yoast SEO first-time configuration as explained in the ‘after activation’ step!

### INSTALL YOAST SEO FROM WITHIN WORDPRESS

1. Visit the plugins page within your dashboard and select ‘Add New’;
1. Search for ‘Yoast SEO’;
1. Activate Yoast SEO from your Plugins page;
1. Go to ‘after activation’ below.

### INSTALL YOAST SEO MANUALLY

1. Upload the ‘wordpress-seo’ folder to the /wp-content/plugins/ directory;
1. Activate the Yoast SEO plugin through the ‘Plugins’ menu in WordPress;
1. Go to ‘after activation’ below.

### AFTER ACTIVATION

1. You should see (a notice to start) the Yoast SEO first-time configuration;
1. Go through this configuration and set up the plugin for your site;
1. You’re done!

== Frequently Asked Questions ==

= How do the XML Sitemaps in the Yoast SEO plugin work? =

Having an XML sitemap can be beneficial for SEO, as Google can retrieve essential pages of a website very fast, even if the internal linking of a site isn’t flawless.
The sitemap index and individual sitemaps are updated automatically as you add or remove content and will include the post types you want search engines to index. Post Types marked as noindex will not appear in the sitemap. [Learn more about XML Sitemaps](https://yoa.st/3qt).

= How can I add my website to Google Search Console? =

It is straightforward to add your website to Google Search Console.
1. Create a Google Search Console account and login into your account.
2. Click ‘Add a property’ under the search drop-down.
3. Enter your website URL in the box and click ‘Continue’.
4. Click the arrow next to ‘HTML tag’ to expand the option.
5. Copy the meta tag.
6. Log in to your WordPress website.
7. Click on ‘SEO’ in the dashboard.
8. Click on ‘General’.
9. Click on the ‘Webmaster Tools’ tab.
10. Paste the code in the Google field and click ‘Save Changes’.
11. Go back to Google Search Console and click ‘Verify’.

If you want more details steps, please visit [our article on our help center](https://yoa.st/3qu).

= How do I implement Yoast SEO breadcrumbs? =

The steps below are a temporary solution as manual edits made to theme files may be overwritten with future theme updates. Please contact the theme developer for a permanent solution. We’ve written an article about the [importance of breadcrumbs for SEO](https://yoa.st/3qv).

To implement the [breadcrumbs](https://yoa.st/3qw) function in Yoast SEO, you will have to edit your theme. We recommend that prior to any editing of the theme files, a backup is taken. Your host provider can help you take a backup.
Copy the following code into your theme where you want the breadcrumbs to be. If you are not sure, you will need to experiment with placement:

<code>
<?php
if ( function_exists( 'yoast_breadcrumb' ) ) {
    yoast_breadcrumb( '<p id="breadcrumbs">','</p>' );
}
?>
</code>

Common places where you could place your breadcrumbs are inside your `single.php` and/or `page.php` file just above the page’s title. Another option that makes it really easy in some themes is by just pasting the code in `header.php` at the very end.

In most non-WooTheme themes, this code snippet should not be added to your `functions.php` file.
Alternatively, you can manually add the breadcrumb shortcode to individual posts or pages: `[wpseo_breadcrumb]`

If you need more details or a step by step guide, read our [Implementation guide for Yoast SEO breadcrumbs](https://yoa.st/3qx).

= How do I noindex URLS? =

Yoast SEO provides multiple options for setting a URL or group of URLs to noindex. [Read more about how to do this in this guide](https://yoa.st/3qy/).

= Google shows the wrong description, how do I fix this? =

If you’ve crafted nice meta descriptions for your blog posts, nothing is more annoying than Google showing another description for your site completely in the search result snippet.

Possible causes could be:
1. wrong description in code
2. Google cache is outdated
3. Search term manipulation
4. Google ignored the meta description

You can [read more here on how to solve the issue with the wrong description](https://yoa.st/3qz).

= How often is Yoast SEO updated? =

Yoast SEO is updated every two weeks. If you want to know why, please read [this post on why we release every two weeks](https://yoa.st/3q-)!

= How do I get support? =

As our free plugin is used by millions of people worldwide, we cannot offer you all one on one support. If you have trouble with the Yoast SEO for WordPress plugin, you can get help on the support forums here at [wordpress.org](https://wordpress.org/support/plugin/wordpress-seo/) or by checking out our help center at [yoast.com/help/](https://yoa.st/3r1).

The plugins you buy at Yoast are called ‘premium plugins’ (even if Premium isn’t in its name) and include a complete year of free updates and premium support. This means you can contact our support team if you have any questions about that plugin.

[Read more on how to get support](https://yoa.st/3r2)

= What happens to my data if I enable usage tracking? =

[This page on yoast.com explains what data we collect to improve Yoast SEO](https://yoa.st/4w7). We only collect data when you explicitly opt in. Read more about how we handle your data in [our Privacy Policy](https://yoa.st/4w8).

= I have a different question than listed here =

Your question has most likely been answered on our help center: [yoast.com/help/](https://yoa.st/1va).

== Screenshots ==

1. The modern interface makes Yoast SEO easy to work with.
2. Easily manage how your posts and pages appear in SERPs.
3. Yoast SEO Premium has extra crawl optimization options.
4. Yoast SEO integrates with tools like Semrush and Wincher.
5. The famous SEO and readability analyses in Yoast SEO.
6. See what your post looks like in Google.
7. The First-time configuration helps you get started quickly.
8. The inclusive language analysis in Yoast SEO.

== Changelog ==

= 22.5 =

Release date: 2024-04-16

Yoast SEO 22.5 brings general maintenance updates across the plugin, improved guidance around best practice for taxonomies and a more effortless user experience overall. For more details about the release check out [the release post for 22.5](https://yoa.st/release-16-4-24).

#### Enhancements

* Prevents image formats other than JPG, GIF, PNG and WEBP to be used for the `twitter:image` meta tag.
* Recalibrates the recommended length of the text on taxonomy pages for the _text length_ assessment.

#### Bugfixes

* Fixes an incompatibility with `ES_WP_Query` library used by the WordPress VIP Enterprise Search.

#### Other

* Renames all user-facing `Twitter` references to `X`, including the X profile's `sameAs` schema output URL.
* Adds an "AI for SEO" course card to the Academy page.
* Removes the link to the now-retired Google Mobile-Friendly Test from the admin bar menu.
* Renames the `leiCode` fields to `LEI code` in the plugin's settings.
* Resolves a typo in the list of German transition words. Props to [magnusmartin85](https://github.com/magnusmartin85).
* Deprecates unused method `is_image_url_valid`.

= 22.4 =

Release date: 2024-04-03

Yoast SEO 22.4 brings optimizations and bug fixes. Our customers who use Elementor can wave goodbye to that unnecessary redirect when saving a post for the first time. Find more about Yoast SEO 22.4 in [our release post](https://yoa.st/release-3-4-24)!

#### Enhancements

* Removes a duplicated database query on author pages with internal images.
* Removes two unnecessary database queries on pages where external images are of SEO interest, like author pages with Gravatar author images.

#### Bugfixes

* Fixes a bug where an unneeded redirect would be created when a post was first published in Elementor.

#### Other

* Deprecates the slug of the `_dependencies` property of the private `$plugins` property in the `WPSEO_Plugin_Availability` class.
* Fixes the width of the `Page type` dropdown field in the `Settings` page.
* Replaces `.substr()` with `.substring()` in JS files since it's deprecated.
* Stops showing deprecation warnings when retrieving data for paid add-ons goes wrong.
* Stops showing notifications about incompatibility with the WooCommerce's new beta product editor, after WooCommerce is deactivated.
* Stops showing a notification prompting to install Yoast Woo SEO, after WooCommerce is deactivated.

= Earlier versions =
For the changelog of earlier versions, please refer to [the changelog on yoast.com](https://yoa.st/yoast-seo-changelog).
                                                                                                                                                                                                                                                                                                                                                                        plugins/wordpress-seo-extended/src/actions/addon-installation/addon-activate-action.php             0000644                 00000004522 15122266557 0025537 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Addon_Installation;

use WPSEO_Addon_Manager;
use Yoast\WP\SEO\Exceptions\Addon_Installation\Addon_Activation_Error_Exception;
use Yoast\WP\SEO\Exceptions\Addon_Installation\User_Cannot_Activate_Plugins_Exception;
use Yoast\WP\SEO\Helpers\Require_File_Helper;

/**
 * Represents the endpoint for activating a specific Yoast Plugin on WordPress.
 */
class Addon_Activate_Action {

	/**
	 * The addon manager.
	 *
	 * @var WPSEO_Addon_Manager
	 */
	protected $addon_manager;

	/**
	 * The require file helper.
	 *
	 * @var Require_File_Helper
	 */
	protected $require_file_helper;

	/**
	 * Addon_Activate_Action constructor.
	 *
	 * @param WPSEO_Addon_Manager $addon_manager       The addon manager.
	 * @param Require_File_Helper $require_file_helper A file helper.
	 */
	public function __construct(
		WPSEO_Addon_Manager $addon_manager,
		Require_File_Helper $require_file_helper
	) {
		$this->addon_manager       = $addon_manager;
		$this->require_file_helper = $require_file_helper;
	}

	/**
	 * Activates the plugin based on the given plugin file.
	 *
	 * @param string $plugin_slug The plugin slug to get download url for.
	 *
	 * @return bool True when activation is successful.
	 *
	 * @throws Addon_Activation_Error_Exception       Exception when the activation encounters an error.
	 * @throws User_Cannot_Activate_Plugins_Exception Exception when the user is not allowed to activate.
	 */
	public function activate_addon( $plugin_slug ) {
		if ( ! \current_user_can( 'activate_plugins' ) ) {
			throw new User_Cannot_Activate_Plugins_Exception();
		}

		if ( $this->addon_manager->is_installed( $plugin_slug ) ) {
			return true;
		}

		$this->load_wordpress_classes();

		$plugin_file       = $this->addon_manager->get_plugin_file( $plugin_slug );
		$activation_result = \activate_plugin( $plugin_file );

		if ( $activation_result !== null && \is_wp_error( $activation_result ) ) {
			throw new Addon_Activation_Error_Exception( $activation_result->get_error_message() );
		}

		return true;
	}

	/**
	 * Requires the files needed from WordPress itself.
	 *
	 * @codeCoverageIgnore Only loads a WordPress file.
	 *
	 * @return void
	 */
	protected function load_wordpress_classes() {
		if ( ! \function_exists( 'get_plugins' ) ) {
			$this->require_file_helper->require_file_once( \ABSPATH . 'wp-admin/includes/plugin.php' );
		}
	}
}
                                                                                                                                                                              plugins/wordpress-seo-extended/src/actions/addon-installation/addon-install-action.php              0000644                 00000007562 15122266557 0025414 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Addon_Installation;

use Plugin_Upgrader;
use WP_Error;
use WPSEO_Addon_Manager;
use Yoast\WP\SEO\Exceptions\Addon_Installation\Addon_Already_Installed_Exception;
use Yoast\WP\SEO\Exceptions\Addon_Installation\Addon_Installation_Error_Exception;
use Yoast\WP\SEO\Exceptions\Addon_Installation\User_Cannot_Install_Plugins_Exception;
use Yoast\WP\SEO\Helpers\Require_File_Helper;

/**
 * Represents the endpoint for downloading and installing a zip-file from MyYoast.
 */
class Addon_Install_Action {

	/**
	 * The addon manager.
	 *
	 * @var WPSEO_Addon_Manager
	 */
	protected $addon_manager;

	/**
	 * The require file helper.
	 *
	 * @var Require_File_Helper
	 */
	protected $require_file_helper;

	/**
	 * Addon_Activate_Action constructor.
	 *
	 * @param WPSEO_Addon_Manager $addon_manager       The addon manager.
	 * @param Require_File_Helper $require_file_helper A helper that can require files.
	 */
	public function __construct(
		WPSEO_Addon_Manager $addon_manager,
		Require_File_Helper $require_file_helper
	) {
		$this->addon_manager       = $addon_manager;
		$this->require_file_helper = $require_file_helper;
	}

	/**
	 * Installs the plugin based on the given slug.
	 *
	 * @param string $plugin_slug  The plugin slug to install.
	 * @param string $download_url The plugin download URL.
	 *
	 * @return bool True when install is successful.
	 *
	 * @throws Addon_Already_Installed_Exception  When the addon is already installed.
	 * @throws Addon_Installation_Error_Exception When the installation encounters an error.
	 * @throws User_Cannot_Install_Plugins_Exception        When the user does not have the permissions to install plugins.
	 */
	public function install_addon( $plugin_slug, $download_url ) {
		if ( ! \current_user_can( 'install_plugins' ) ) {
			throw new User_Cannot_Install_Plugins_Exception( $plugin_slug );
		}

		if ( $this->is_installed( $plugin_slug ) ) {
			throw new Addon_Already_Installed_Exception( $plugin_slug );
		}

		$this->load_wordpress_classes();

		$install_result = $this->install( $download_url );
		if ( \is_wp_error( $install_result ) ) {
			throw new Addon_Installation_Error_Exception( $install_result->get_error_message() );
		}

		return $install_result;
	}

	/**
	 * Requires the files needed from WordPress itself.
	 *
	 * @codeCoverageIgnore
	 *
	 * @return void
	 */
	protected function load_wordpress_classes() {
		if ( ! \class_exists( 'WP_Upgrader' ) ) {
			$this->require_file_helper->require_file_once( \ABSPATH . 'wp-admin/includes/class-wp-upgrader.php' );
		}

		if ( ! \class_exists( 'Plugin_Upgrader' ) ) {
			$this->require_file_helper->require_file_once( \ABSPATH . 'wp-admin/includes/class-plugin-upgrader.php' );
		}

		if ( ! \class_exists( 'WP_Upgrader_Skin' ) ) {
			$this->require_file_helper->require_file_once( \ABSPATH . 'wp-admin/includes/class-wp-upgrader-skin.php' );
		}

		if ( ! \function_exists( 'get_plugin_data' ) ) {
			$this->require_file_helper->require_file_once( \ABSPATH . 'wp-admin/includes/plugin.php' );
		}

		if ( ! \function_exists( 'request_filesystem_credentials' ) ) {
			$this->require_file_helper->require_file_once( \ABSPATH . 'wp-admin/includes/file.php' );
		}
	}

	/**
	 * Checks is a plugin is installed.
	 *
	 * @param string $plugin_slug The plugin to check.
	 *
	 * @return bool True when plugin is installed.
	 */
	protected function is_installed( $plugin_slug ) {
		return $this->addon_manager->get_plugin_file( $plugin_slug ) !== false;
	}

	/**
	 * Runs the installation by using the WordPress installation routine.
	 *
	 * @codeCoverageIgnore Contains WordPress specific logic.
	 *
	 * @param string $plugin_download The url to the download.
	 *
	 * @return bool|WP_Error True when success, WP_Error when something went wrong.
	 */
	protected function install( $plugin_download ) {
		$plugin_upgrader = new Plugin_Upgrader();

		return $plugin_upgrader->install( $plugin_download );
	}
}
                                                                                                                                              plugins/wordpress-seo-extended/src/actions/alert-dismissal-action.php                               0000644                 00000012576 15122266557 0022175 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions;

use Yoast\WP\SEO\Helpers\User_Helper;

/**
 * Class Alert_Dismissal_Action.
 */
class Alert_Dismissal_Action {

	public const USER_META_KEY = '_yoast_alerts_dismissed';

	/**
	 * Holds the user helper instance.
	 *
	 * @var User_Helper
	 */
	protected $user;

	/**
	 * Constructs Alert_Dismissal_Action.
	 *
	 * @param User_Helper $user User helper.
	 */
	public function __construct( User_Helper $user ) {
		$this->user = $user;
	}

	/**
	 * Dismisses an alert.
	 *
	 * @param string $alert_identifier Alert identifier.
	 *
	 * @return bool Whether the dismiss was successful or not.
	 */
	public function dismiss( $alert_identifier ) {
		$user_id = $this->user->get_current_user_id();
		if ( $user_id === 0 ) {
			return false;
		}

		if ( $this->is_allowed( $alert_identifier ) === false ) {
			return false;
		}

		$dismissed_alerts = $this->get_dismissed_alerts( $user_id );
		if ( $dismissed_alerts === false ) {
			return false;
		}

		if ( \array_key_exists( $alert_identifier, $dismissed_alerts ) === true ) {
			// The alert is already dismissed.
			return true;
		}

		// Add this alert to the dismissed alerts.
		$dismissed_alerts[ $alert_identifier ] = true;

		// Save.
		return $this->user->update_meta( $user_id, static::USER_META_KEY, $dismissed_alerts ) !== false;
	}

	/**
	 * Resets an alert.
	 *
	 * @param string $alert_identifier Alert identifier.
	 *
	 * @return bool Whether the reset was successful or not.
	 */
	public function reset( $alert_identifier ) {
		$user_id = $this->user->get_current_user_id();
		if ( $user_id === 0 ) {
			return false;
		}

		if ( $this->is_allowed( $alert_identifier ) === false ) {
			return false;
		}

		$dismissed_alerts = $this->get_dismissed_alerts( $user_id );
		if ( $dismissed_alerts === false ) {
			return false;
		}

		$amount_of_dismissed_alerts = \count( $dismissed_alerts );
		if ( $amount_of_dismissed_alerts === 0 ) {
			// No alerts: nothing to reset.
			return true;
		}

		if ( \array_key_exists( $alert_identifier, $dismissed_alerts ) === false ) {
			// Alert not found: nothing to reset.
			return true;
		}

		if ( $amount_of_dismissed_alerts === 1 ) {
			// The 1 remaining dismissed alert is the alert to reset: delete the alerts user meta row.
			return $this->user->delete_meta( $user_id, static::USER_META_KEY, $dismissed_alerts );
		}

		// Remove this alert from the dismissed alerts.
		unset( $dismissed_alerts[ $alert_identifier ] );

		// Save.
		return $this->user->update_meta( $user_id, static::USER_META_KEY, $dismissed_alerts ) !== false;
	}

	/**
	 * Returns if an alert is dismissed or not.
	 *
	 * @param string $alert_identifier Alert identifier.
	 *
	 * @return bool Whether the alert has been dismissed.
	 */
	public function is_dismissed( $alert_identifier ) {
		$user_id = $this->user->get_current_user_id();
		if ( $user_id === 0 ) {
			return false;
		}

		if ( $this->is_allowed( $alert_identifier ) === false ) {
			return false;
		}

		$dismissed_alerts = $this->get_dismissed_alerts( $user_id );
		if ( $dismissed_alerts === false ) {
			return false;
		}

		return \array_key_exists( $alert_identifier, $dismissed_alerts );
	}

	/**
	 * Returns an object with all alerts dismissed by current user.
	 *
	 * @return array|false An array with the keys of all Alerts that have been dismissed
	 *                     by the current user or `false`.
	 */
	public function all_dismissed() {
		$user_id = $this->user->get_current_user_id();
		if ( $user_id === 0 ) {
			return false;
		}

		$dismissed_alerts = $this->get_dismissed_alerts( $user_id );
		if ( $dismissed_alerts === false ) {
			return false;
		}

		return $dismissed_alerts;
	}

	/**
	 * Returns if an alert is allowed or not.
	 *
	 * @param string $alert_identifier Alert identifier.
	 *
	 * @return bool Whether the alert is allowed.
	 */
	public function is_allowed( $alert_identifier ) {
		return \in_array( $alert_identifier, $this->get_allowed_dismissable_alerts(), true );
	}

	/**
	 * Retrieves the dismissed alerts.
	 *
	 * @param int $user_id User ID.
	 *
	 * @return string[]|false The dismissed alerts. False for an invalid $user_id.
	 */
	protected function get_dismissed_alerts( $user_id ) {
		$dismissed_alerts = $this->user->get_meta( $user_id, static::USER_META_KEY, true );
		if ( $dismissed_alerts === false ) {
			// Invalid user ID.
			return false;
		}

		if ( $dismissed_alerts === '' ) {
			/*
			 * When no database row exists yet, an empty string is returned because of the `single` parameter.
			 * We do want a single result returned, but the default should be an empty array instead.
			 */
			return [];
		}

		return $dismissed_alerts;
	}

	/**
	 * Retrieves the allowed dismissable alerts.
	 *
	 * @return string[] The allowed dismissable alerts.
	 */
	protected function get_allowed_dismissable_alerts() {
		/**
		 * Filter: 'wpseo_allowed_dismissable_alerts' - List of allowed dismissable alerts.
		 *
		 * @param string[] $allowed_dismissable_alerts Allowed dismissable alerts list.
		 */
		$allowed_dismissable_alerts = \apply_filters( 'wpseo_allowed_dismissable_alerts', [] );

		if ( \is_array( $allowed_dismissable_alerts ) === false ) {
			return [];
		}

		// Only allow strings.
		$allowed_dismissable_alerts = \array_filter( $allowed_dismissable_alerts, 'is_string' );

		// Filter unique and reorder indices.
		$allowed_dismissable_alerts = \array_values( \array_unique( $allowed_dismissable_alerts ) );

		return $allowed_dismissable_alerts;
	}
}
                                                                                                                                  plugins/wordpress-seo-extended/src/actions/configuration/first-time-configuration-action.php        0000644                 00000021147 15122266557 0026671 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Configuration;

use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Social_Profiles_Helper;

/**
 * Class First_Time_Configuration_Action.
 */
class First_Time_Configuration_Action {

	/**
	 * The fields for the site representation payload.
	 */
	public const SITE_REPRESENTATION_FIELDS = [
		'company_or_person',
		'company_name',
		'website_name',
		'company_logo',
		'company_logo_id',
		'person_logo',
		'person_logo_id',
		'company_or_person_user_id',
		'description',
	];

	/**
	 * The Options_Helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * The Social_Profiles_Helper instance.
	 *
	 * @var Social_Profiles_Helper
	 */
	protected $social_profiles_helper;

	/**
	 * First_Time_Configuration_Action constructor.
	 *
	 * @param Options_Helper         $options_helper         The WPSEO options helper.
	 * @param Social_Profiles_Helper $social_profiles_helper The social profiles helper.
	 */
	public function __construct( Options_Helper $options_helper, Social_Profiles_Helper $social_profiles_helper ) {
		$this->options_helper         = $options_helper;
		$this->social_profiles_helper = $social_profiles_helper;
	}

	/**
	 * Stores the values for the site representation.
	 *
	 * @param array $params The values to store.
	 *
	 * @return object The response object.
	 */
	public function set_site_representation( $params ) {
		$failures   = [];
		$old_values = $this->get_old_values( self::SITE_REPRESENTATION_FIELDS );

		foreach ( self::SITE_REPRESENTATION_FIELDS as $field_name ) {
			if ( isset( $params[ $field_name ] ) ) {
				$result = $this->options_helper->set( $field_name, $params[ $field_name ] );

				if ( ! $result ) {
					$failures[] = $field_name;
				}
			}
		}

		// Delete cached logos in the db.
		$this->options_helper->set( 'company_logo_meta', false );
		$this->options_helper->set( 'person_logo_meta', false );

		/**
		 * Action: 'wpseo_post_update_site_representation' - Allows for Hiive event tracking.
		 *
		 * @param array $params     The new values of the options.
		 * @param array $old_values The old values of the options.
		 * @param array $failures   The options that failed to be saved.
		 *
		 * @internal
		 */
		\do_action( 'wpseo_ftc_post_update_site_representation', $params, $old_values, $failures );

		if ( \count( $failures ) === 0 ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}

		return (object) [
			'success'  => false,
			'status'   => 500,
			'error'    => 'Could not save some options in the database',
			'failures' => $failures,
		];
	}

	/**
	 * Stores the values for the social profiles.
	 *
	 * @param array $params The values to store.
	 *
	 * @return object The response object.
	 */
	public function set_social_profiles( $params ) {
		$old_values = $this->get_old_values( \array_keys( $this->social_profiles_helper->get_organization_social_profile_fields() ) );
		$failures   = $this->social_profiles_helper->set_organization_social_profiles( $params );

		/**
		 * Action: 'wpseo_post_update_social_profiles' - Allows for Hiive event tracking.
		 *
		 * @param array $params     The new values of the options.
		 * @param array $old_values The old values of the options.
		 * @param array $failures   The options that failed to be saved.
		 *
		 * @internal
		 */
		\do_action( 'wpseo_ftc_post_update_social_profiles', $params, $old_values, $failures );

		if ( empty( $failures ) ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}

		return (object) [
			'success'  => false,
			'status'   => 200,
			'error'    => 'Could not save some options in the database',
			'failures' => $failures,
		];
	}

	/**
	 * Stores the values for the social profiles.
	 *
	 * @param array $params The values to store.
	 *
	 * @return object The response object.
	 */
	public function set_person_social_profiles( $params ) {
		$social_profiles = \array_filter(
			$params,
			static function ( $key ) {
				return $key !== 'user_id';
			},
			\ARRAY_FILTER_USE_KEY
		);

		$failures = $this->social_profiles_helper->set_person_social_profiles( $params['user_id'], $social_profiles );

		if ( \count( $failures ) === 0 ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}

		return (object) [
			'success'  => false,
			'status'   => 200,
			'error'    => 'Could not save some options in the database',
			'failures' => $failures,
		];
	}

	/**
	 * Gets the values for the social profiles.
	 *
	 * @param int $user_id The person ID.
	 *
	 * @return object The response object.
	 */
	public function get_person_social_profiles( $user_id ) {

		return (object) [
			'success'         => true,
			'status'          => 200,
			'social_profiles' => $this->social_profiles_helper->get_person_social_profiles( $user_id ),
		];
	}

	/**
	 * Stores the values to enable/disable tracking.
	 *
	 * @param array $params The values to store.
	 *
	 * @return object The response object.
	 */
	public function set_enable_tracking( $params ) {
		$success      = true;
		$option_value = $this->options_helper->get( 'tracking' );

		if ( $option_value !== $params['tracking'] ) {
			$this->options_helper->set( 'toggled_tracking', true );
			$success = $this->options_helper->set( 'tracking', $params['tracking'] );
		}

		/**
		 * Action: 'wpseo_post_update_enable_tracking' - Allows for Hiive event tracking.
		 *
		 * @param array $new_value The new value.
		 * @param array $old_value The old value.
		 * @param bool  $failure   Whether the option failed to be stored.
		 *
		 * @internal
		 */
		// $success is negated to be aligned with the other two actions which pass $failures.
		\do_action( 'wpseo_ftc_post_update_enable_tracking', $params['tracking'], $option_value, ! $success );

		if ( $success ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}

		return (object) [
			'success' => false,
			'status'  => 500,
			'error'   => 'Could not save the option in the database',
		];
	}

	/**
	 * Checks if the current user has the capability a specific user.
	 *
	 * @param int $user_id The id of the user to be edited.
	 *
	 * @return object The response object.
	 */
	public function check_capability( $user_id ) {
		if ( $this->can_edit_profile( $user_id ) ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}

		return (object) [
			'success' => false,
			'status'  => 403,
		];
	}

	/**
	 * Stores the first time configuration state.
	 *
	 * @param array $params The values to store.
	 *
	 * @return object The response object.
	 */
	public function save_configuration_state( $params ) {
		// If the finishedSteps param is not present in the REST request, it's a malformed request.
		if ( ! isset( $params['finishedSteps'] ) ) {
			return (object) [
				'success' => false,
				'status'  => 400,
				'error'   => 'Bad request',
			];
		}

		// Sanitize input.
		$finished_steps = \array_map( '\sanitize_text_field', \wp_unslash( $params['finishedSteps'] ) );

		$success = $this->options_helper->set( 'configuration_finished_steps', $finished_steps );

		if ( ! $success ) {
			return (object) [
				'success' => false,
				'status'  => 500,
				'error'   => 'Could not save the option in the database',
			];
		}

		// If all the five steps of the configuration have been completed, set first_time_install option to false.
		if ( \count( $params['finishedSteps'] ) === 3 ) {
			$this->options_helper->set( 'first_time_install', false );
		}

		return (object) [
			'success' => true,
			'status'  => 200,
		];
	}

	/**
	 * Gets the first time configuration state.
	 *
	 * @return object The response object.
	 */
	public function get_configuration_state() {
		$configuration_option = $this->options_helper->get( 'configuration_finished_steps' );

		if ( ! \is_null( $configuration_option ) ) {
			return (object) [
				'success' => true,
				'status'  => 200,
				'data'    => $configuration_option,
			];
		}

		return (object) [
			'success' => false,
			'status'  => 500,
			'error'   => 'Could not get data from the database',
		];
	}

	/**
	 * Checks if the current user has the capability to edit a specific user.
	 *
	 * @param int $person_id The id of the person to edit.
	 *
	 * @return bool
	 */
	private function can_edit_profile( $person_id ) {
		return \current_user_can( 'edit_user', $person_id );
	}

	/**
	 * Gets the old values for the given fields.
	 *
	 * @param array $fields_names The fields to get the old values for.
	 *
	 * @return array The old values.
	 */
	private function get_old_values( array $fields_names ): array {
		$old_values = [];

		foreach ( $fields_names as $field_name ) {
			$old_values[ $field_name ] = $this->options_helper->get( $field_name );
		}

		return $old_values;
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                         plugins/wordpress-seo-extended/src/actions/importing/abstract-aioseo-importing-action.php           0000644                 00000015315 15122266557 0026170 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Importing;

use Exception;
use Yoast\WP\SEO\Helpers\Aioseo_Helper;
use Yoast\WP\SEO\Helpers\Import_Cursor_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Sanitization_Helper;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Replacevar_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Provider_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Transformer_Service;

/**
 * Importing action interface.
 */
abstract class Abstract_Aioseo_Importing_Action implements Importing_Action_Interface {

	/**
	 * The plugin the class deals with.
	 *
	 * @var string
	 */
	public const PLUGIN = null;

	/**
	 * The type the class deals with.
	 *
	 * @var string
	 */
	public const TYPE = null;

	/**
	 * The AIOSEO helper.
	 *
	 * @var Aioseo_Helper
	 */
	protected $aioseo_helper;

	/**
	 * The import cursor helper.
	 *
	 * @var Import_Cursor_Helper
	 */
	protected $import_cursor;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	protected $options;

	/**
	 * The sanitization helper.
	 *
	 * @var Sanitization_Helper
	 */
	protected $sanitization;

	/**
	 * The replacevar handler.
	 *
	 * @var Aioseo_Replacevar_Service
	 */
	protected $replacevar_handler;

	/**
	 * The robots provider service.
	 *
	 * @var Aioseo_Robots_Provider_Service
	 */
	protected $robots_provider;

	/**
	 * The robots transformer service.
	 *
	 * @var Aioseo_Robots_Transformer_Service
	 */
	protected $robots_transformer;

	/**
	 * Abstract_Aioseo_Importing_Action constructor.
	 *
	 * @param Import_Cursor_Helper              $import_cursor      The import cursor helper.
	 * @param Options_Helper                    $options            The options helper.
	 * @param Sanitization_Helper               $sanitization       The sanitization helper.
	 * @param Aioseo_Replacevar_Service         $replacevar_handler The replacevar handler.
	 * @param Aioseo_Robots_Provider_Service    $robots_provider    The robots provider service.
	 * @param Aioseo_Robots_Transformer_Service $robots_transformer The robots transfomer service.
	 */
	public function __construct(
		Import_Cursor_Helper $import_cursor,
		Options_Helper $options,
		Sanitization_Helper $sanitization,
		Aioseo_Replacevar_Service $replacevar_handler,
		Aioseo_Robots_Provider_Service $robots_provider,
		Aioseo_Robots_Transformer_Service $robots_transformer
	) {
		$this->import_cursor      = $import_cursor;
		$this->options            = $options;
		$this->sanitization       = $sanitization;
		$this->replacevar_handler = $replacevar_handler;
		$this->robots_provider    = $robots_provider;
		$this->robots_transformer = $robots_transformer;
	}

	/**
	 * Sets the AIOSEO helper.
	 *
	 * @required
	 *
	 * @param Aioseo_Helper $aioseo_helper The AIOSEO helper.
	 *
	 * @return void
	 */
	public function set_aioseo_helper( Aioseo_Helper $aioseo_helper ) {
		$this->aioseo_helper = $aioseo_helper;
	}

	/**
	 * The name of the plugin we import from.
	 *
	 * @return string The plugin we import from.
	 *
	 * @throws Exception If the PLUGIN constant is not set in the child class.
	 */
	public function get_plugin() {
		$class  = static::class;
		$plugin = $class::PLUGIN;

		if ( $plugin === null ) {
			throw new Exception( 'Importing action without explicit plugin' );
		}

		return $plugin;
	}

	/**
	 * The data type we import from the plugin.
	 *
	 * @return string The data type we import from the plugin.
	 *
	 * @throws Exception If the TYPE constant is not set in the child class.
	 */
	public function get_type() {
		$class = static::class;
		$type  = $class::TYPE;

		if ( $type === null ) {
			throw new Exception( 'Importing action without explicit type' );
		}

		return $type;
	}

	/**
	 * Can the current action import the data from plugin $plugin of type $type?
	 *
	 * @param string|null $plugin The plugin to import from.
	 * @param string|null $type   The type of data to import.
	 *
	 * @return bool True if this action can handle the combination of Plugin and Type.
	 *
	 * @throws Exception If the TYPE constant is not set in the child class.
	 */
	public function is_compatible_with( $plugin = null, $type = null ) {
		if ( empty( $plugin ) && empty( $type ) ) {
			return true;
		}

		if ( $plugin === $this->get_plugin() && empty( $type ) ) {
			return true;
		}

		if ( empty( $plugin ) && $type === $this->get_type() ) {
			return true;
		}

		if ( $plugin === $this->get_plugin() && $type === $this->get_type() ) {
			return true;
		}

		return false;
	}

	/**
	 * Gets the completed id (to be used as a key for the importing_completed option).
	 *
	 * @return string The completed id.
	 */
	public function get_completed_id() {
		return $this->get_cursor_id();
	}

	/**
	 * Returns the stored state of completedness.
	 *
	 * @return int The stored state of completedness.
	 */
	public function get_completed() {
		$completed_id          = $this->get_completed_id();
		$importers_completions = $this->options->get( 'importing_completed', [] );

		return ( isset( $importers_completions[ $completed_id ] ) ) ? $importers_completions[ $completed_id ] : false;
	}

	/**
	 * Stores the current state of completedness.
	 *
	 * @param bool $completed Whether the importer is completed.
	 *
	 * @return void
	 */
	public function set_completed( $completed ) {
		$completed_id                  = $this->get_completed_id();
		$current_importers_completions = $this->options->get( 'importing_completed', [] );

		$current_importers_completions[ $completed_id ] = $completed;
		$this->options->set( 'importing_completed', $current_importers_completions );
	}

	/**
	 * Returns whether the importing action is enabled.
	 *
	 * @return bool True by default unless a child class overrides it.
	 */
	public function is_enabled() {
		return true;
	}

	/**
	 * Gets the cursor id.
	 *
	 * @return string The cursor id.
	 */
	protected function get_cursor_id() {
		return $this->get_plugin() . '_' . $this->get_type();
	}

	/**
	 * Minimally transforms data to be imported.
	 *
	 * @param string $meta_data The meta data to be imported.
	 *
	 * @return string The transformed meta data.
	 */
	public function simple_import( $meta_data ) {
		// Transform the replace vars into Yoast replace vars.
		$transformed_data = $this->replacevar_handler->transform( $meta_data );

		return $this->sanitization->sanitize_text_field( \html_entity_decode( $transformed_data ) );
	}

	/**
	 * Transforms URL to be imported.
	 *
	 * @param string $meta_data The meta data to be imported.
	 *
	 * @return string The transformed URL.
	 */
	public function url_import( $meta_data ) {
		// We put null as the allowed protocols here, to have the WP default allowed protocols, see https://developer.wordpress.org/reference/functions/wp_allowed_protocols.
		return $this->sanitization->sanitize_url( $meta_data, null );
	}
}
                                                                                                                                                                                                                                                                                                                   wordpress-seo-extended/src/actions/importing/aioseo/abstract-aioseo-settings-importing-action.php   0000644                 00000024610 15122266557 0031224 0                                                                                                    ustar 00                                                                                plugins                                                                                                                                                                <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

use Exception;
use Yoast\WP\SEO\Actions\Importing\Abstract_Aioseo_Importing_Action;
use Yoast\WP\SEO\Helpers\Import_Helper;

/**
 * Abstract class for importing AIOSEO settings.
 */
abstract class Abstract_Aioseo_Settings_Importing_Action extends Abstract_Aioseo_Importing_Action {

	/**
	 * The plugin the class deals with.
	 *
	 * @var string
	 */
	public const PLUGIN = null;

	/**
	 * The type the class deals with.
	 *
	 * @var string
	 */
	public const TYPE = null;

	/**
	 * The option_name of the AIOSEO option that contains the settings.
	 */
	public const SOURCE_OPTION_NAME = null;

	/**
	 * The map of aioseo_options to yoast settings.
	 *
	 * @var array
	 */
	protected $aioseo_options_to_yoast_map = [];

	/**
	 * The tab of the aioseo settings we're working with, eg. taxonomies, posttypes.
	 *
	 * @var string
	 */
	protected $settings_tab = '';

	/**
	 * Additional mapping between AiOSEO replace vars and Yoast replace vars.
	 *
	 * @var array
	 *
	 * @see https://yoast.com/help/list-available-snippet-variables-yoast-seo/
	 */
	protected $replace_vars_edited_map = [];

	/**
	 * The import helper.
	 *
	 * @var Import_Helper
	 */
	protected $import_helper;

	/**
	 * Builds the mapping that ties AOISEO option keys with Yoast ones and their data transformation method.
	 *
	 * @return void
	 */
	abstract protected function build_mapping();

	/**
	 * Sets the import helper.
	 *
	 * @required
	 *
	 * @param Import_Helper $import_helper The import helper.
	 *
	 * @return void
	 */
	public function set_import_helper( Import_Helper $import_helper ) {
		$this->import_helper = $import_helper;
	}

	/**
	 * Retrieves the source option_name.
	 *
	 * @return string The source option_name.
	 *
	 * @throws Exception If the SOURCE_OPTION_NAME constant is not set in the child class.
	 */
	public function get_source_option_name() {
		$source_option_name = static::SOURCE_OPTION_NAME;

		if ( empty( $source_option_name ) ) {
			throw new Exception( 'Importing settings action without explicit source option_name' );
		}

		return $source_option_name;
	}

	/**
	 * Returns the total number of unimported objects.
	 *
	 * @return int The total number of unimported objects.
	 */
	public function get_total_unindexed() {
		return $this->get_unindexed_count();
	}

	/**
	 * Returns the limited number of unimported objects.
	 *
	 * @param int $limit The maximum number of unimported objects to be returned.
	 *
	 * @return int The limited number of unindexed posts.
	 */
	public function get_limited_unindexed_count( $limit ) {
		return $this->get_unindexed_count( $limit );
	}

	/**
	 * Returns the number of unimported objects (limited if limit is applied).
	 *
	 * @param int|null $limit The maximum number of unimported objects to be returned.
	 *
	 * @return int The number of unindexed posts.
	 */
	protected function get_unindexed_count( $limit = null ) {
		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = null;
		}

		$settings_to_create = $this->query( $limit );

		$number_of_settings_to_create = \count( $settings_to_create );
		$completed                    = $number_of_settings_to_create === 0;
		$this->set_completed( $completed );

		return $number_of_settings_to_create;
	}

	/**
	 * Imports AIOSEO settings.
	 *
	 * @return array|false An array of the AIOSEO settings that were imported or false if aioseo data was not found.
	 */
	public function index() {
		$limit            = $this->get_limit();
		$aioseo_settings  = $this->query( $limit );
		$created_settings = [];

		$completed = \count( $aioseo_settings ) === 0;
		$this->set_completed( $completed );

		// Prepare the setting keys mapping.
		$this->build_mapping();

		// Prepare the replacement var mapping.
		foreach ( $this->replace_vars_edited_map as $aioseo_var => $yoast_var ) {
			$this->replacevar_handler->compose_map( $aioseo_var, $yoast_var );
		}

		$last_imported_setting = '';
		try {
			foreach ( $aioseo_settings as $setting => $setting_value ) {
				// Map and import the values of the setting we're working with (eg. post, book-category, etc.) to the respective Yoast option.
				$this->map( $setting_value, $setting );

				// Save the type of the settings that were just imported, so that we can allow chunked imports.
				$last_imported_setting = $setting;

				$created_settings[] = $setting;
			}
		}
		finally {
			$cursor_id = $this->get_cursor_id();
			$this->import_cursor->set_cursor( $cursor_id, $last_imported_setting );
		}

		return $created_settings;
	}

	/**
	 * Checks if the settings tab subsetting is set in the AIOSEO option.
	 *
	 * @param string $aioseo_settings The AIOSEO option.
	 *
	 * @return bool Whether the settings are set.
	 */
	public function isset_settings_tab( $aioseo_settings ) {
		return isset( $aioseo_settings['searchAppearance'][ $this->settings_tab ] );
	}

	/**
	 * Queries the database and retrieves unimported AiOSEO settings (in chunks if a limit is applied).
	 *
	 * @param int|null $limit The maximum number of unimported objects to be returned.
	 *
	 * @return array The (maybe chunked) unimported AiOSEO settings to import.
	 */
	protected function query( $limit = null ) {
		$aioseo_settings = \json_decode( \get_option( $this->get_source_option_name(), '' ), true );

		if ( empty( $aioseo_settings ) ) {
			return [];
		}

		// We specifically want the setttings of the tab we're working with, eg. postTypes, taxonomies, etc.
		$settings_values = $aioseo_settings['searchAppearance'][ $this->settings_tab ];
		if ( ! \is_array( $settings_values ) ) {
			return [];
		}

		$flattened_settings = $this->import_helper->flatten_settings( $settings_values );

		return $this->get_unimported_chunk( $flattened_settings, $limit );
	}

	/**
	 * Retrieves (a chunk of, if limit is applied) the unimported AIOSEO settings.
	 * To apply a chunk, we manipulate the cursor to the keys of the AIOSEO settings.
	 *
	 * @param array $importable_data All of the available AIOSEO settings.
	 * @param int   $limit           The maximum number of unimported objects to be returned.
	 *
	 * @return array The (chunk of, if limit is applied)) unimported AIOSEO settings.
	 */
	protected function get_unimported_chunk( $importable_data, $limit ) {
		\ksort( $importable_data );

		$cursor_id = $this->get_cursor_id();
		$cursor    = $this->import_cursor->get_cursor( $cursor_id, '' );

		/**
		 * Filter 'wpseo_aioseo_<identifier>_import_cursor' - Allow filtering the value of the aioseo settings import cursor.
		 *
		 * @param int $import_cursor The value of the aioseo posttype default settings import cursor.
		 */
		$cursor = \apply_filters( 'wpseo_aioseo_' . $this->get_type() . '_import_cursor', $cursor );

		if ( $cursor === '' ) {
			return \array_slice( $importable_data, 0, $limit, true );
		}

		// Let's find the position of the cursor in the alphabetically sorted importable data, so we can return only the unimported data.
		$keys = \array_flip( \array_keys( $importable_data ) );
		// If the stored cursor now no longer exists in the data, we have no choice but to start over.
		$position = ( isset( $keys[ $cursor ] ) ) ? ( $keys[ $cursor ] + 1 ) : 0;

		return \array_slice( $importable_data, $position, $limit, true );
	}

	/**
	 * Returns the number of objects that will be imported in a single importing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_aioseo_<identifier>_indexation_limit' - Allow filtering the number of settings imported during each importing pass.
		 *
		 * @param int $max_posts The maximum number of posts indexed.
		 */
		$limit = \apply_filters( 'wpseo_aioseo_' . $this->get_type() . '_indexation_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}

	/**
	 * Maps/imports AIOSEO settings into the respective Yoast settings.
	 *
	 * @param string|array $setting_value The value of the AIOSEO setting at hand.
	 * @param string       $setting       The setting at hand, eg. post or movie-category, separator etc.
	 *
	 * @return void
	 */
	protected function map( $setting_value, $setting ) {
		$aioseo_options_to_yoast_map = $this->aioseo_options_to_yoast_map;

		if ( isset( $aioseo_options_to_yoast_map[ $setting ] ) ) {
			$this->import_single_setting( $setting, $setting_value, $aioseo_options_to_yoast_map[ $setting ] );
		}
	}

	/**
	 * Imports a single setting in the db after transforming it to adhere to Yoast conventions.
	 *
	 * @param string $setting         The name of the setting.
	 * @param string $setting_value   The values of the setting.
	 * @param array  $setting_mapping The mapping of the setting to Yoast formats.
	 *
	 * @return void
	 */
	protected function import_single_setting( $setting, $setting_value, $setting_mapping ) {
		$yoast_key = $setting_mapping['yoast_name'];

		// Check if we're supposed to save the setting.
		if ( $this->options->get_default( 'wpseo_titles', $yoast_key ) !== null ) {
			// Then, do any needed data transfomation before actually saving the incoming data.
			$transformed_data = \call_user_func( [ $this, $setting_mapping['transform_method'] ], $setting_value, $setting_mapping );

			$this->options->set( $yoast_key, $transformed_data );
		}
	}

	/**
	 * Minimally transforms boolean data to be imported.
	 *
	 * @param bool $meta_data The boolean meta data to be imported.
	 *
	 * @return bool The transformed boolean meta data.
	 */
	public function simple_boolean_import( $meta_data ) {
		return $meta_data;
	}

	/**
	 * Imports the noindex setting, taking into consideration whether they defer to global defaults.
	 *
	 * @param bool  $noindex The noindex of the type, without taking into consideration whether the type defers to global defaults.
	 * @param array $mapping The mapping of the setting we're working with.
	 *
	 * @return bool The noindex setting.
	 */
	public function import_noindex( $noindex, $mapping ) {
		return $this->robots_transformer->transform_robot_setting( 'noindex', $noindex, $mapping );
	}

	/**
	 * Returns a setting map of the robot setting for one subset of post types/taxonomies/archives.
	 * For custom archives, it returns an empty array because AIOSEO excludes some custom archives from this option structure, eg. WooCommerce's products and we don't want to raise a false alarm.
	 *
	 * @return array The setting map of the robot setting for one subset of post types/taxonomies/archives or an empty array.
	 */
	public function pluck_robot_setting_from_mapping() {
		return [];
	}
}
                                                                                                                        plugins/wordpress-seo-extended/src/actions/importing/aioseo/aioseo-cleanup-action.php               0000644                 00000011022 15122266557 0025254 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

use wpdb;
use Yoast\WP\SEO\Actions\Importing\Abstract_Aioseo_Importing_Action;
use Yoast\WP\SEO\Helpers\Options_Helper;

/**
 * Importing action for cleaning up AIOSEO data.
 */
class Aioseo_Cleanup_Action extends Abstract_Aioseo_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'cleanup';

	/**
	 * The AIOSEO meta_keys to be cleaned up.
	 *
	 * @var array
	 */
	protected $aioseo_postmeta_keys = [
		'_aioseo_title',
		'_aioseo_description',
		'_aioseo_og_title',
		'_aioseo_og_description',
		'_aioseo_twitter_title',
		'_aioseo_twitter_description',
	];

	/**
	 * The WordPress database instance.
	 *
	 * @var wpdb
	 */
	protected $wpdb;

	/**
	 * Class constructor.
	 *
	 * @param wpdb           $wpdb    The WordPress database instance.
	 * @param Options_Helper $options The options helper.
	 */
	public function __construct(
		wpdb $wpdb,
		Options_Helper $options
	) {
		$this->wpdb    = $wpdb;
		$this->options = $options;
	}

	/**
	 * Retrieves the postmeta along with the db prefix.
	 *
	 * @return string The postmeta table name along with the db prefix.
	 */
	protected function get_postmeta_table() {
		return $this->wpdb->prefix . 'postmeta';
	}

	/**
	 * Just checks if the cleanup has been completed in the past.
	 *
	 * @return int The total number of unimported objects.
	 */
	public function get_total_unindexed() {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			return 0;
		}

		return ( ! $this->get_completed() ) ? 1 : 0;
	}

	/**
	 * Just checks if the cleanup has been completed in the past.
	 *
	 * @param int $limit The maximum number of unimported objects to be returned.
	 *
	 * @return int|false The limited number of unindexed posts. False if the query fails.
	 */
	public function get_limited_unindexed_count( $limit ) {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			return 0;
		}

		return ( ! $this->get_completed() ) ? 1 : 0;
	}

	/**
	 * Cleans up AIOSEO data.
	 *
	 * @return Indexable[]|false An array of created indexables or false if aioseo data was not found.
	 */
	public function index() {
		if ( $this->get_completed() ) {
			return [];
		}

		// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared -- Reason: There is no unescaped user input.
		$meta_data                  = $this->wpdb->query( $this->cleanup_postmeta_query() );
		$aioseo_table_truncate_done = $this->wpdb->query( $this->truncate_query() );
		// phpcs:enable WordPress.DB.PreparedSQL.NotPrepared

		if ( $meta_data === false && $aioseo_table_truncate_done === false ) {
			return false;
		}

		$this->set_completed( true );

		return [
			'metadata_cleanup'   => $meta_data,
			'indexables_cleanup' => $aioseo_table_truncate_done,
		];
	}

	/**
	 * Creates a DELETE query string for deleting AIOSEO postmeta data.
	 *
	 * @return string The query to use for importing or counting the number of items to import.
	 */
	public function cleanup_postmeta_query() {
		$table               = $this->get_postmeta_table();
		$meta_keys_to_delete = $this->aioseo_postmeta_keys;

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Reason: There is no unescaped user input.
		return $this->wpdb->prepare(
			"DELETE FROM {$table} WHERE meta_key IN (" . \implode( ', ', \array_fill( 0, \count( $meta_keys_to_delete ), '%s' ) ) . ')',
			$meta_keys_to_delete
		);
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Creates a TRUNCATE query string for emptying the AIOSEO indexable table, if it exists.
	 *
	 * @return string The query to use for importing or counting the number of items to import.
	 */
	public function truncate_query() {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			// If the table doesn't exist, we need a string that will amount to a quick query that doesn't return false when ran.
			return 'SELECT 1';
		}

		$table = $this->aioseo_helper->get_table();

		return "TRUNCATE TABLE {$table}";
	}

	/**
	 * Used nowhere. Exists to comply with the interface.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_aioseo_cleanup_limit' - Allow filtering the number of posts indexed during each indexing pass.
		 *
		 * @param int $max_posts The maximum number of posts cleaned up.
		 */
		$limit = \apply_filters( 'wpseo_aioseo_cleanup_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              src/actions/importing/aioseo/aioseo-custom-archive-settings-importing-action.php                    0000644                 00000007366 15122266557 0032363 0                                                                                                    ustar 00                                                                                plugins/wordpress-seo-extended                                                                                                                                         <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

use Yoast\WP\SEO\Helpers\Import_Cursor_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Post_Type_Helper;
use Yoast\WP\SEO\Helpers\Sanitization_Helper;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Replacevar_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Provider_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Transformer_Service;

/**
 * Importing action for AIOSEO custom archive settings data.
 *
 * @phpcs:disable Yoast.NamingConventions.ObjectNameDepth.MaxExceeded
 */
class Aioseo_Custom_Archive_Settings_Importing_Action extends Abstract_Aioseo_Settings_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'custom_archive_settings';

	/**
	 * The option_name of the AIOSEO option that contains the settings.
	 */
	public const SOURCE_OPTION_NAME = 'aioseo_options_dynamic';

	/**
	 * The map of aioseo_options to yoast settings.
	 *
	 * @var array
	 */
	protected $aioseo_options_to_yoast_map = [];

	/**
	 * The tab of the aioseo settings we're working with.
	 *
	 * @var string
	 */
	protected $settings_tab = 'archives';

	/**
	 * The post type helper.
	 *
	 * @var Post_Type_Helper
	 */
	protected $post_type;

	/**
	 * Aioseo_Custom_Archive_Settings_Importing_Action constructor.
	 *
	 * @param Import_Cursor_Helper              $import_cursor      The import cursor helper.
	 * @param Options_Helper                    $options            The options helper.
	 * @param Sanitization_Helper               $sanitization       The sanitization helper.
	 * @param Post_Type_Helper                  $post_type          The post type helper.
	 * @param Aioseo_Replacevar_Service         $replacevar_handler The replacevar handler.
	 * @param Aioseo_Robots_Provider_Service    $robots_provider    The robots provider service.
	 * @param Aioseo_Robots_Transformer_Service $robots_transformer The robots transfomer service.
	 */
	public function __construct(
		Import_Cursor_Helper $import_cursor,
		Options_Helper $options,
		Sanitization_Helper $sanitization,
		Post_Type_Helper $post_type,
		Aioseo_Replacevar_Service $replacevar_handler,
		Aioseo_Robots_Provider_Service $robots_provider,
		Aioseo_Robots_Transformer_Service $robots_transformer
	) {
		parent::__construct( $import_cursor, $options, $sanitization, $replacevar_handler, $robots_provider, $robots_transformer );

		$this->post_type = $post_type;
	}

	/**
	 * Builds the mapping that ties AOISEO option keys with Yoast ones and their data transformation method.
	 *
	 * @return void
	 */
	protected function build_mapping() {
		$post_type_objects = \get_post_types( [ 'public' => true ], 'objects' );

		foreach ( $post_type_objects as $pt ) {
			// Use all the custom post types that have archives.
			if ( ! $pt->_builtin && $this->post_type->has_archive( $pt ) ) {
				$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/title' ]                       = [
					'yoast_name'       => 'title-ptarchive-' . $pt->name,
					'transform_method' => 'simple_import',
				];
				$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/metaDescription' ]             = [
					'yoast_name'       => 'metadesc-ptarchive-' . $pt->name,
					'transform_method' => 'simple_import',
				];
				$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/advanced/robotsMeta/noindex' ] = [
					'yoast_name'       => 'noindex-ptarchive-' . $pt->name,
					'transform_method' => 'import_noindex',
					'type'             => 'archives',
					'subtype'          => $pt->name,
					'option_name'      => 'aioseo_options_dynamic',
				];
			}
		}
	}
}
                                                                                                                                                                                                                                                                          src/actions/importing/aioseo/aioseo-default-archive-settings-importing-action.php                   0000644                 00000005610 15122266557 0032463 0                                                                                                    ustar 00                                                                                plugins/wordpress-seo-extended                                                                                                                                         <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

/**
 * Importing action for AIOSEO default archive settings data.
 *
 * @phpcs:disable Yoast.NamingConventions.ObjectNameDepth.MaxExceeded
 */
class Aioseo_Default_Archive_Settings_Importing_Action extends Abstract_Aioseo_Settings_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'default_archive_settings';

	/**
	 * The option_name of the AIOSEO option that contains the settings.
	 */
	public const SOURCE_OPTION_NAME = 'aioseo_options';

	/**
	 * The map of aioseo_options to yoast settings.
	 *
	 * @var array
	 */
	protected $aioseo_options_to_yoast_map = [];

	/**
	 * The tab of the aioseo settings we're working with.
	 *
	 * @var string
	 */
	protected $settings_tab = 'archives';

	/**
	 * Builds the mapping that ties AOISEO option keys with Yoast ones and their data transformation method.
	 *
	 * @return void
	 */
	protected function build_mapping() {
		$this->aioseo_options_to_yoast_map = [
			'/author/title'                       => [
				'yoast_name'       => 'title-author-wpseo',
				'transform_method' => 'simple_import',
			],
			'/author/metaDescription'             => [
				'yoast_name'       => 'metadesc-author-wpseo',
				'transform_method' => 'simple_import',
			],
			'/date/title'                         => [
				'yoast_name'       => 'title-archive-wpseo',
				'transform_method' => 'simple_import',
			],
			'/date/metaDescription'               => [
				'yoast_name'       => 'metadesc-archive-wpseo',
				'transform_method' => 'simple_import',
			],
			'/search/title'                       => [
				'yoast_name'       => 'title-search-wpseo',
				'transform_method' => 'simple_import',
			],
			'/author/advanced/robotsMeta/noindex' => [
				'yoast_name'       => 'noindex-author-wpseo',
				'transform_method' => 'import_noindex',
				'type'             => 'archives',
				'subtype'          => 'author',
				'option_name'      => 'aioseo_options',
			],
			'/date/advanced/robotsMeta/noindex'   => [
				'yoast_name'       => 'noindex-archive-wpseo',
				'transform_method' => 'import_noindex',
				'type'             => 'archives',
				'subtype'          => 'date',
				'option_name'      => 'aioseo_options',
			],
		];
	}

	/**
	 * Returns a setting map of the robot setting for author archives.
	 *
	 * @return array The setting map of the robot setting for author archives.
	 */
	public function pluck_robot_setting_from_mapping() {
		$this->build_mapping();

		foreach ( $this->aioseo_options_to_yoast_map as $setting ) {
			// Return the first archive setting map.
			if ( $setting['transform_method'] === 'import_noindex' && isset( $setting['subtype'] ) && $setting['subtype'] === 'author' ) {
				return $setting;
			}
		}

		return [];
	}
}
                                                                                                                        wordpress-seo-extended/src/actions/importing/aioseo/aioseo-general-settings-importing-action.php    0000644                 00000013752 15122266557 0031043 0                                                                                                    ustar 00                                                                                plugins                                                                                                                                                                <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

use Yoast\WP\SEO\Helpers\Image_Helper;
use Yoast\WP\SEO\Helpers\Import_Cursor_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Sanitization_Helper;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Replacevar_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Provider_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Transformer_Service;

/**
 * Importing action for AIOSEO general settings.
 */
class Aioseo_General_Settings_Importing_Action extends Abstract_Aioseo_Settings_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'general_settings';

	/**
	 * The option_name of the AIOSEO option that contains the settings.
	 */
	public const SOURCE_OPTION_NAME = 'aioseo_options';

	/**
	 * The map of aioseo_options to yoast settings.
	 *
	 * @var array
	 */
	protected $aioseo_options_to_yoast_map = [];

	/**
	 * The tab of the aioseo settings we're working with.
	 *
	 * @var string
	 */
	protected $settings_tab = 'global';

	/**
	 * The image helper.
	 *
	 * @var Image_Helper
	 */
	protected $image;

	/**
	 * Aioseo_General_Settings_Importing_Action constructor.
	 *
	 * @param Import_Cursor_Helper              $import_cursor      The import cursor helper.
	 * @param Options_Helper                    $options            The options helper.
	 * @param Sanitization_Helper               $sanitization       The sanitization helper.
	 * @param Image_Helper                      $image              The image helper.
	 * @param Aioseo_Replacevar_Service         $replacevar_handler The replacevar handler.
	 * @param Aioseo_Robots_Provider_Service    $robots_provider    The robots provider service.
	 * @param Aioseo_Robots_Transformer_Service $robots_transformer The robots transfomer service.
	 */
	public function __construct(
		Import_Cursor_Helper $import_cursor,
		Options_Helper $options,
		Sanitization_Helper $sanitization,
		Image_Helper $image,
		Aioseo_Replacevar_Service $replacevar_handler,
		Aioseo_Robots_Provider_Service $robots_provider,
		Aioseo_Robots_Transformer_Service $robots_transformer
	) {
		parent::__construct( $import_cursor, $options, $sanitization, $replacevar_handler, $robots_provider, $robots_transformer );

		$this->image = $image;
	}

	/**
	 * Builds the mapping that ties AOISEO option keys with Yoast ones and their data transformation method.
	 *
	 * @return void
	 */
	protected function build_mapping() {
		$this->aioseo_options_to_yoast_map = [
			'/separator'               => [
				'yoast_name'       => 'separator',
				'transform_method' => 'transform_separator',
			],
			'/siteTitle'               => [
				'yoast_name'       => 'title-home-wpseo',
				'transform_method' => 'simple_import',
			],
			'/metaDescription'         => [
				'yoast_name'       => 'metadesc-home-wpseo',
				'transform_method' => 'simple_import',
			],
			'/schema/siteRepresents'   => [
				'yoast_name'       => 'company_or_person',
				'transform_method' => 'transform_site_represents',
			],
			'/schema/person'           => [
				'yoast_name'       => 'company_or_person_user_id',
				'transform_method' => 'simple_import',
			],
			'/schema/organizationName' => [
				'yoast_name'       => 'company_name',
				'transform_method' => 'simple_import',
			],
			'/schema/organizationLogo' => [
				'yoast_name'       => 'company_logo',
				'transform_method' => 'import_company_logo',
			],
			'/schema/personLogo'       => [
				'yoast_name'       => 'person_logo',
				'transform_method' => 'import_person_logo',
			],
		];
	}

	/**
	 * Imports the organization logo while also accounting for the id of the log to be saved in the separate Yoast option.
	 *
	 * @param string $logo_url The company logo url coming from AIOSEO settings.
	 *
	 * @return string The transformed company logo url.
	 */
	public function import_company_logo( $logo_url ) {
		$logo_id = $this->image->get_attachment_by_url( $logo_url );
		$this->options->set( 'company_logo_id', $logo_id );

		$this->options->set( 'company_logo_meta', false );
		$logo_meta = $this->image->get_attachment_meta_from_settings( 'company_logo' );
		$this->options->set( 'company_logo_meta', $logo_meta );

		return $this->url_import( $logo_url );
	}

	/**
	 * Imports the person logo while also accounting for the id of the log to be saved in the separate Yoast option.
	 *
	 * @param string $logo_url The person logo url coming from AIOSEO settings.
	 *
	 * @return string The transformed person logo url.
	 */
	public function import_person_logo( $logo_url ) {
		$logo_id = $this->image->get_attachment_by_url( $logo_url );
		$this->options->set( 'person_logo_id', $logo_id );

		$this->options->set( 'person_logo_meta', false );
		$logo_meta = $this->image->get_attachment_meta_from_settings( 'person_logo' );
		$this->options->set( 'person_logo_meta', $logo_meta );

		return $this->url_import( $logo_url );
	}

	/**
	 * Transforms the site represents setting.
	 *
	 * @param string $site_represents The site represents setting.
	 *
	 * @return string The transformed site represents setting.
	 */
	public function transform_site_represents( $site_represents ) {
		switch ( $site_represents ) {
			case 'person':
				return 'person';

			case 'organization':
			default:
				return 'company';
		}
	}

	/**
	 * Transforms the separator setting.
	 *
	 * @param string $separator The separator setting.
	 *
	 * @return string The transformed separator.
	 */
	public function transform_separator( $separator ) {
		switch ( $separator ) {
			case '&#45;':
				return 'sc-dash';

			case '&ndash;':
				return 'sc-ndash';

			case '&mdash;':
				return 'sc-mdash';

			case '&raquo;':
				return 'sc-raquo';

			case '&laquo;':
				return 'sc-laquo';

			case '&gt;':
				return 'sc-gt';

			case '&bull;':
				return 'sc-bull';

			case '&#124;':
				return 'sc-pipe';

			default:
				return 'sc-dash';
		}
	}
}
                      plugins/wordpress-seo-extended/src/actions/importing/aioseo/aioseo-posts-importing-action.php       0000644                 00000054677 15122266557 0027032 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

use wpdb;
use Yoast\WP\SEO\Actions\Importing\Abstract_Aioseo_Importing_Action;
use Yoast\WP\SEO\Helpers\Image_Helper;
use Yoast\WP\SEO\Helpers\Import_Cursor_Helper;
use Yoast\WP\SEO\Helpers\Indexable_Helper;
use Yoast\WP\SEO\Helpers\Indexable_To_Postmeta_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Sanitization_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Replacevar_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Provider_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Transformer_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Social_Images_Provider_Service;

/**
 * Importing action for AIOSEO post data.
 */
class Aioseo_Posts_Importing_Action extends Abstract_Aioseo_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'posts';

	/**
	 * The map of aioseo to yoast meta.
	 *
	 * @var array
	 */
	protected $aioseo_to_yoast_map = [
		'title'               => [
			'yoast_name'       => 'title',
			'transform_method' => 'simple_import_post',
		],
		'description'         => [
			'yoast_name'       => 'description',
			'transform_method' => 'simple_import_post',
		],
		'og_title'            => [
			'yoast_name'       => 'open_graph_title',
			'transform_method' => 'simple_import_post',
		],
		'og_description'      => [
			'yoast_name'       => 'open_graph_description',
			'transform_method' => 'simple_import_post',
		],
		'twitter_title'       => [
			'yoast_name'       => 'twitter_title',
			'transform_method' => 'simple_import_post',
			'twitter_import'   => true,
		],
		'twitter_description' => [
			'yoast_name'       => 'twitter_description',
			'transform_method' => 'simple_import_post',
			'twitter_import'   => true,
		],
		'canonical_url'       => [
			'yoast_name'       => 'canonical',
			'transform_method' => 'url_import_post',
		],
		'keyphrases'          => [
			'yoast_name'       => 'primary_focus_keyword',
			'transform_method' => 'keyphrase_import',
		],
		'og_image_url'        => [
			'yoast_name'                   => 'open_graph_image',
			'social_image_import'          => true,
			'social_setting_prefix_aioseo' => 'og_',
			'social_setting_prefix_yoast'  => 'open_graph_',
			'transform_method'             => 'social_image_url_import',
		],
		'twitter_image_url'   => [
			'yoast_name'                   => 'twitter_image',
			'social_image_import'          => true,
			'social_setting_prefix_aioseo' => 'twitter_',
			'social_setting_prefix_yoast'  => 'twitter_',
			'transform_method'             => 'social_image_url_import',
		],
		'robots_noindex'      => [
			'yoast_name'       => 'is_robots_noindex',
			'transform_method' => 'post_robots_noindex_import',
			'robots_import'    => true,
		],
		'robots_nofollow'     => [
			'yoast_name'       => 'is_robots_nofollow',
			'transform_method' => 'post_general_robots_import',
			'robots_import'    => true,
			'robot_type'       => 'nofollow',
		],
		'robots_noarchive'    => [
			'yoast_name'       => 'is_robots_noarchive',
			'transform_method' => 'post_general_robots_import',
			'robots_import'    => true,
			'robot_type'       => 'noarchive',
		],
		'robots_nosnippet'    => [
			'yoast_name'       => 'is_robots_nosnippet',
			'transform_method' => 'post_general_robots_import',
			'robots_import'    => true,
			'robot_type'       => 'nosnippet',
		],
		'robots_noimageindex' => [
			'yoast_name'       => 'is_robots_noimageindex',
			'transform_method' => 'post_general_robots_import',
			'robots_import'    => true,
			'robot_type'       => 'noimageindex',
		],
	];

	/**
	 * Represents the indexables repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $indexable_repository;

	/**
	 * The WordPress database instance.
	 *
	 * @var wpdb
	 */
	protected $wpdb;

	/**
	 * The image helper.
	 *
	 * @var Image_Helper
	 */
	protected $image;

	/**
	 * The indexable_to_postmeta helper.
	 *
	 * @var Indexable_To_Postmeta_Helper
	 */
	protected $indexable_to_postmeta;

	/**
	 * The indexable helper.
	 *
	 * @var Indexable_Helper
	 */
	protected $indexable_helper;

	/**
	 * The social images provider service.
	 *
	 * @var Aioseo_Social_Images_Provider_Service
	 */
	protected $social_images_provider;

	/**
	 * Class constructor.
	 *
	 * @param Indexable_Repository                  $indexable_repository   The indexables repository.
	 * @param wpdb                                  $wpdb                   The WordPress database instance.
	 * @param Import_Cursor_Helper                  $import_cursor          The import cursor helper.
	 * @param Indexable_Helper                      $indexable_helper       The indexable helper.
	 * @param Indexable_To_Postmeta_Helper          $indexable_to_postmeta  The indexable_to_postmeta helper.
	 * @param Options_Helper                        $options                The options helper.
	 * @param Image_Helper                          $image                  The image helper.
	 * @param Sanitization_Helper                   $sanitization           The sanitization helper.
	 * @param Aioseo_Replacevar_Service             $replacevar_handler     The replacevar handler.
	 * @param Aioseo_Robots_Provider_Service        $robots_provider        The robots provider service.
	 * @param Aioseo_Robots_Transformer_Service     $robots_transformer     The robots transfomer service.
	 * @param Aioseo_Social_Images_Provider_Service $social_images_provider The social images provider service.
	 */
	public function __construct(
		Indexable_Repository $indexable_repository,
		wpdb $wpdb,
		Import_Cursor_Helper $import_cursor,
		Indexable_Helper $indexable_helper,
		Indexable_To_Postmeta_Helper $indexable_to_postmeta,
		Options_Helper $options,
		Image_Helper $image,
		Sanitization_Helper $sanitization,
		Aioseo_Replacevar_Service $replacevar_handler,
		Aioseo_Robots_Provider_Service $robots_provider,
		Aioseo_Robots_Transformer_Service $robots_transformer,
		Aioseo_Social_Images_Provider_Service $social_images_provider
	) {
		parent::__construct( $import_cursor, $options, $sanitization, $replacevar_handler, $robots_provider, $robots_transformer );

		$this->indexable_repository   = $indexable_repository;
		$this->wpdb                   = $wpdb;
		$this->image                  = $image;
		$this->indexable_helper       = $indexable_helper;
		$this->indexable_to_postmeta  = $indexable_to_postmeta;
		$this->social_images_provider = $social_images_provider;
	}

	// phpcs:disable WordPress.DB.PreparedSQL.NotPrepared -- Reason: They are already prepared.

	/**
	 * Returns the total number of unimported objects.
	 *
	 * @return int The total number of unimported objects.
	 */
	public function get_total_unindexed() {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			return 0;
		}

		$limit                = false;
		$just_detect          = true;
		$indexables_to_create = $this->wpdb->get_col( $this->query( $limit, $just_detect ) );

		$number_of_indexables_to_create = \count( $indexables_to_create );
		$completed                      = $number_of_indexables_to_create === 0;
		$this->set_completed( $completed );

		return $number_of_indexables_to_create;
	}

	/**
	 * Returns the limited number of unimported objects.
	 *
	 * @param int $limit The maximum number of unimported objects to be returned.
	 *
	 * @return int|false The limited number of unindexed posts. False if the query fails.
	 */
	public function get_limited_unindexed_count( $limit ) {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			return 0;
		}

		$just_detect          = true;
		$indexables_to_create = $this->wpdb->get_col( $this->query( $limit, $just_detect ) );

		$number_of_indexables_to_create = \count( $indexables_to_create );
		$completed                      = $number_of_indexables_to_create === 0;
		$this->set_completed( $completed );

		return $number_of_indexables_to_create;
	}

	/**
	 * Imports AIOSEO meta data and creates the respective Yoast indexables and postmeta.
	 *
	 * @return Indexable[]|false An array of created indexables or false if aioseo data was not found.
	 */
	public function index() {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			return false;
		}

		$limit              = $this->get_limit();
		$aioseo_indexables  = $this->wpdb->get_results( $this->query( $limit ), \ARRAY_A );
		$created_indexables = [];

		$completed = \count( $aioseo_indexables ) === 0;
		$this->set_completed( $completed );

		// Let's build the list of fields to check their defaults, to identify whether we're gonna import AIOSEO data in the indexable or not.
		$check_defaults_fields = [];
		foreach ( $this->aioseo_to_yoast_map as $yoast_mapping ) {
			// We don't want to check all the imported fields.
			if ( ! \in_array( $yoast_mapping['yoast_name'], [ 'open_graph_image', 'twitter_image' ], true ) ) {
				$check_defaults_fields[] = $yoast_mapping['yoast_name'];
			}
		}

		$last_indexed_aioseo_id = 0;
		foreach ( $aioseo_indexables as $aioseo_indexable ) {
			$last_indexed_aioseo_id = $aioseo_indexable['id'];

			$indexable = $this->indexable_repository->find_by_id_and_type( $aioseo_indexable['post_id'], 'post' );

			// Let's ensure that the current post id represents something that we want to index (eg. *not* shop_order).
			if ( ! \is_a( $indexable, 'Yoast\WP\SEO\Models\Indexable' ) ) {
				continue;
			}

			if ( $this->indexable_helper->check_if_default_indexable( $indexable, $check_defaults_fields ) ) {
				$indexable = $this->map( $indexable, $aioseo_indexable );
				$indexable->save();

				// To ensure that indexables can be rebuild after a reset, we have to store the data in the postmeta table too.
				$this->indexable_to_postmeta->map_to_postmeta( $indexable );
			}

			$last_indexed_aioseo_id = $aioseo_indexable['id'];

			$created_indexables[] = $indexable;
		}

		$cursor_id = $this->get_cursor_id();
		$this->import_cursor->set_cursor( $cursor_id, $last_indexed_aioseo_id );

		return $created_indexables;
	}

	// phpcs:enable WordPress.DB.PreparedSQL.NotPrepared

	/**
	 * Maps AIOSEO meta data to Yoast meta data.
	 *
	 * @param Indexable $indexable        The Yoast indexable.
	 * @param array     $aioseo_indexable The AIOSEO indexable.
	 *
	 * @return Indexable The created indexables.
	 */
	public function map( $indexable, $aioseo_indexable ) {
		foreach ( $this->aioseo_to_yoast_map as $aioseo_key => $yoast_mapping ) {
			// For robots import.
			if ( isset( $yoast_mapping['robots_import'] ) && $yoast_mapping['robots_import'] ) {
				$yoast_mapping['subtype']                  = $indexable->object_sub_type;
				$indexable->{$yoast_mapping['yoast_name']} = $this->transform_import_data( $yoast_mapping['transform_method'], $aioseo_indexable, $aioseo_key, $yoast_mapping, $indexable );

				continue;
			}

			// For social images, like open graph and twitter image.
			if ( isset( $yoast_mapping['social_image_import'] ) && $yoast_mapping['social_image_import'] ) {
				$image_url = $this->transform_import_data( $yoast_mapping['transform_method'], $aioseo_indexable, $aioseo_key, $yoast_mapping, $indexable );

				// Update the indexable's social image only where there's actually a url to import, so as not to lose the social images that we came up with when we originally built the indexable.
				if ( ! empty( $image_url ) ) {
					$indexable->{$yoast_mapping['yoast_name']} = $image_url;

					$image_source_key             = $yoast_mapping['social_setting_prefix_yoast'] . 'image_source';
					$indexable->$image_source_key = 'imported';

					$image_id_key             = $yoast_mapping['social_setting_prefix_yoast'] . 'image_id';
					$indexable->$image_id_key = $this->image->get_attachment_by_url( $image_url );

					if ( $yoast_mapping['yoast_name'] === 'open_graph_image' ) {
						$indexable->open_graph_image_meta = null;
					}
				}
				continue;
			}

			// For twitter import, take the respective open graph data if the appropriate setting is enabled.
			if ( isset( $yoast_mapping['twitter_import'] ) && $yoast_mapping['twitter_import'] && $aioseo_indexable['twitter_use_og'] ) {
				$aioseo_indexable['twitter_title']       = $aioseo_indexable['og_title'];
				$aioseo_indexable['twitter_description'] = $aioseo_indexable['og_description'];
			}

			if ( ! empty( $aioseo_indexable[ $aioseo_key ] ) ) {
				$indexable->{$yoast_mapping['yoast_name']} = $this->transform_import_data( $yoast_mapping['transform_method'], $aioseo_indexable, $aioseo_key, $yoast_mapping, $indexable );
			}
		}

		return $indexable;
	}

	/**
	 * Transforms the data to be imported.
	 *
	 * @param string    $transform_method The method that is going to be used for transforming the data.
	 * @param array     $aioseo_indexable The data of the AIOSEO indexable data that is being imported.
	 * @param string    $aioseo_key       The name of the specific set of data that is going to be transformed.
	 * @param array     $yoast_mapping    Extra details for the import of the specific data that is going to be transformed.
	 * @param Indexable $indexable        The Yoast indexable that we are going to import the transformed data into.
	 *
	 * @return string|bool|null The transformed data to be imported.
	 */
	protected function transform_import_data( $transform_method, $aioseo_indexable, $aioseo_key, $yoast_mapping, $indexable ) {
		return \call_user_func( [ $this, $transform_method ], $aioseo_indexable, $aioseo_key, $yoast_mapping, $indexable );
	}

	/**
	 * Returns the number of objects that will be imported in a single importing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_aioseo_post_indexation_limit' - Allow filtering the number of posts indexed during each indexing pass.
		 *
		 * @param int $max_posts The maximum number of posts indexed.
		 */
		$limit = \apply_filters( 'wpseo_aioseo_post_indexation_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}

	/**
	 * Populates the needed data array based on which columns we use from the AIOSEO indexable table.
	 *
	 * @return array The needed data array that contains all the needed columns.
	 */
	public function get_needed_data() {
		$needed_data = \array_keys( $this->aioseo_to_yoast_map );
		\array_push( $needed_data, 'id', 'post_id', 'robots_default', 'og_image_custom_url', 'og_image_type', 'twitter_image_custom_url', 'twitter_image_type', 'twitter_use_og' );

		return $needed_data;
	}

	/**
	 * Populates the needed robot data array to be used in validating against its structure.
	 *
	 * @return array The needed data array that contains all the needed columns.
	 */
	public function get_needed_robot_data() {
		$needed_robot_data = [];

		foreach ( $this->aioseo_to_yoast_map as $yoast_mapping ) {
			if ( isset( $yoast_mapping['robot_type'] ) ) {
				$needed_robot_data[] = $yoast_mapping['robot_type'];
			}
		}

		return $needed_robot_data;
	}

	/**
	 * Creates a query for gathering AiOSEO data from the database.
	 *
	 * @param int  $limit       The maximum number of unimported objects to be returned.
	 * @param bool $just_detect Whether we want to just detect if there are unimported objects. If false, we want to actually import them too.
	 *
	 * @return string The query to use for importing or counting the number of items to import.
	 */
	public function query( $limit = false, $just_detect = false ) {
		$table = $this->aioseo_helper->get_table();

		$select_statement = 'id';
		if ( ! $just_detect ) {
			// If we want to import too, we need the actual needed data from AIOSEO indexables.
			$needed_data = $this->get_needed_data();

			$select_statement = \implode( ', ', $needed_data );
		}

		$cursor_id = $this->get_cursor_id();
		$cursor    = $this->import_cursor->get_cursor( $cursor_id );

		/**
		 * Filter 'wpseo_aioseo_post_cursor' - Allow filtering the value of the aioseo post import cursor.
		 *
		 * @param int $import_cursor The value of the aioseo post import cursor.
		 */
		$cursor = \apply_filters( 'wpseo_aioseo_post_import_cursor', $cursor );

		$replacements = [ $cursor ];

		$limit_statement = '';
		if ( ! empty( $limit ) ) {
			$replacements[]  = $limit;
			$limit_statement = ' LIMIT %d';
		}

		// phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Reason: There is no unescaped user input.
		return $this->wpdb->prepare(
			"SELECT {$select_statement} FROM {$table} WHERE id > %d ORDER BY id{$limit_statement}",
			$replacements
		);
		// phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Minimally transforms data to be imported.
	 *
	 * @param array  $aioseo_data All of the AIOSEO data to be imported.
	 * @param string $aioseo_key  The AIOSEO key that contains the setting we're working with.
	 *
	 * @return string The transformed meta data.
	 */
	public function simple_import_post( $aioseo_data, $aioseo_key ) {
		return $this->simple_import( $aioseo_data[ $aioseo_key ] );
	}

	/**
	 * Transforms URL to be imported.
	 *
	 * @param array  $aioseo_data All of the AIOSEO data to be imported.
	 * @param string $aioseo_key  The AIOSEO key that contains the setting we're working with.
	 *
	 * @return string The transformed URL.
	 */
	public function url_import_post( $aioseo_data, $aioseo_key ) {
		return $this->url_import( $aioseo_data[ $aioseo_key ] );
	}

	/**
	 * Plucks the keyphrase to be imported from the AIOSEO array of keyphrase meta data.
	 *
	 * @param array  $aioseo_data All of the AIOSEO data to be imported.
	 * @param string $aioseo_key  The AIOSEO key that contains the setting we're working with, aka keyphrases.
	 *
	 * @return string|null The plucked keyphrase.
	 */
	public function keyphrase_import( $aioseo_data, $aioseo_key ) {
		$meta_data = \json_decode( $aioseo_data[ $aioseo_key ], true );
		if ( ! isset( $meta_data['focus']['keyphrase'] ) ) {
			return null;
		}

		return $this->sanitization->sanitize_text_field( $meta_data['focus']['keyphrase'] );
	}

	/**
	 * Imports the post's noindex setting.
	 *
	 * @param bool $aioseo_robots_settings AIOSEO's set of robot settings for the post.
	 *
	 * @return bool|null The value of Yoast's noindex setting for the post.
	 */
	public function post_robots_noindex_import( $aioseo_robots_settings ) {
		// If robot settings defer to default settings, we have null in the is_robots_noindex field.
		if ( $aioseo_robots_settings['robots_default'] ) {
			return null;
		}

		return $aioseo_robots_settings['robots_noindex'];
	}

	/**
	 * Imports the post's robots setting.
	 *
	 * @param bool   $aioseo_robots_settings AIOSEO's set of robot settings for the post.
	 * @param string $aioseo_key             The AIOSEO key that contains the robot setting we're working with.
	 * @param array  $mapping                The mapping of the setting we're working with.
	 *
	 * @return bool|null The value of Yoast's noindex setting for the post.
	 */
	public function post_general_robots_import( $aioseo_robots_settings, $aioseo_key, $mapping ) {
		$mapping = $this->enhance_mapping( $mapping );

		if ( $aioseo_robots_settings['robots_default'] ) {
			// Let's first get the subtype's setting value and then transform it taking into consideration whether it defers to global defaults.
			$subtype_setting = $this->robots_provider->get_subtype_robot_setting( $mapping );
			return $this->robots_transformer->transform_robot_setting( $mapping['robot_type'], $subtype_setting, $mapping );
		}

		return $aioseo_robots_settings[ $aioseo_key ];
	}

	/**
	 * Enhances the mapping of the setting we're working with, with type and the option name, so that we can retrieve the settings for the object we're working with.
	 *
	 * @param array $mapping The mapping of the setting we're working with.
	 *
	 * @return array The enhanced mapping.
	 */
	public function enhance_mapping( $mapping = [] ) {
		$mapping['type']        = 'postTypes';
		$mapping['option_name'] = 'aioseo_options_dynamic';

		return $mapping;
	}

	/**
	 * Imports the og and twitter image url.
	 *
	 * @param bool      $aioseo_social_image_settings AIOSEO's set of social image settings for the post.
	 * @param string    $aioseo_key                   The AIOSEO key that contains the robot setting we're working with.
	 * @param array     $mapping                      The mapping of the setting we're working with.
	 * @param Indexable $indexable                    The Yoast indexable we're importing into.
	 *
	 * @return bool|null The url of the social image we're importing, null if there's none.
	 */
	public function social_image_url_import( $aioseo_social_image_settings, $aioseo_key, $mapping, $indexable ) {
		if ( $mapping['social_setting_prefix_aioseo'] === 'twitter_' && $aioseo_social_image_settings['twitter_use_og'] ) {
			$mapping['social_setting_prefix_aioseo'] = 'og_';
		}

		$social_setting = \rtrim( $mapping['social_setting_prefix_aioseo'], '_' );

		$image_type = $aioseo_social_image_settings[ $mapping['social_setting_prefix_aioseo'] . 'image_type' ];

		if ( $image_type === 'default' ) {
			$image_type = $this->social_images_provider->get_default_social_image_source( $social_setting );
		}

		switch ( $image_type ) {
			case 'attach':
				$image_url = $this->social_images_provider->get_first_attached_image( $indexable->object_id );
				break;
			case 'auto':
				if ( $this->social_images_provider->get_featured_image( $indexable->object_id ) ) {
					// If there's a featured image, lets not import it, as our indexable calculation has already set that as active social image. That way we achieve dynamicality.
					return null;
				}
				$image_url = $this->social_images_provider->get_auto_image( $indexable->object_id );
				break;
			case 'content':
				$image_url = $this->social_images_provider->get_first_image_in_content( $indexable->object_id );
				break;
			case 'custom_image':
				$image_url = $aioseo_social_image_settings[ $mapping['social_setting_prefix_aioseo'] . 'image_custom_url' ];
				break;
			case 'featured':
				return null; // Our auto-calculation when the indexable was built/updated has taken care of it, so it's not needed to transfer any data now.
			case 'author':
				return null;
			case 'custom':
				return null;
			case 'default':
				$image_url = $this->social_images_provider->get_default_custom_social_image( $social_setting );
				break;
			default:
				$image_url = $aioseo_social_image_settings[ $mapping['social_setting_prefix_aioseo'] . 'image_url' ];
				break;
		}

		if ( empty( $image_url ) ) {
			$image_url = $this->social_images_provider->get_default_custom_social_image( $social_setting );
		}

		if ( empty( $image_url ) ) {
			return null;
		}

		return $this->sanitization->sanitize_url( $image_url, null );
	}
}
                                                                 src/actions/importing/aioseo/aioseo-posttype-defaults-settings-importing-action.php                 0000644                 00000005540 15122266557 0033116 0                                                                                                    ustar 00                                                                                plugins/wordpress-seo-extended                                                                                                                                         <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

/**
 * Importing action for AIOSEO posttype defaults settings data.
 *
 * @phpcs:disable Yoast.NamingConventions.ObjectNameDepth.MaxExceeded
 */
class Aioseo_Posttype_Defaults_Settings_Importing_Action extends Abstract_Aioseo_Settings_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'posttype_default_settings';

	/**
	 * The option_name of the AIOSEO option that contains the settings.
	 */
	public const SOURCE_OPTION_NAME = 'aioseo_options_dynamic';

	/**
	 * The map of aioseo_options to yoast settings.
	 *
	 * @var array
	 */
	protected $aioseo_options_to_yoast_map = [];

	/**
	 * The tab of the aioseo settings we're working with.
	 *
	 * @var string
	 */
	protected $settings_tab = 'postTypes';

	/**
	 * Builds the mapping that ties AOISEO option keys with Yoast ones and their data transformation method.
	 *
	 * @return void
	 */
	protected function build_mapping() {
		$post_type_objects = \get_post_types( [ 'public' => true ], 'objects' );

		foreach ( $post_type_objects as $pt ) {
			// Use all the custom post types that are public.
			$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/title' ]                       = [
				'yoast_name'       => 'title-' . $pt->name,
				'transform_method' => 'simple_import',
			];
			$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/metaDescription' ]             = [
				'yoast_name'       => 'metadesc-' . $pt->name,
				'transform_method' => 'simple_import',
			];
			$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/advanced/showMetaBox' ]        = [
				'yoast_name'       => 'display-metabox-pt-' . $pt->name,
				'transform_method' => 'simple_boolean_import',
			];
			$this->aioseo_options_to_yoast_map[ '/' . $pt->name . '/advanced/robotsMeta/noindex' ] = [
				'yoast_name'       => 'noindex-' . $pt->name,
				'transform_method' => 'import_noindex',
				'type'             => 'postTypes',
				'subtype'          => $pt->name,
				'option_name'      => 'aioseo_options_dynamic',
			];

			if ( $pt->name === 'attachment' ) {
				$this->aioseo_options_to_yoast_map['/attachment/redirectAttachmentUrls'] = [
					'yoast_name'       => 'disable-attachment',
					'transform_method' => 'import_redirect_attachment',
				];
			}
		}
	}

	/**
	 * Transforms the redirect_attachment setting.
	 *
	 * @param string $redirect_attachment The redirect_attachment setting.
	 *
	 * @return bool The transformed redirect_attachment setting.
	 */
	public function import_redirect_attachment( $redirect_attachment ) {
		switch ( $redirect_attachment ) {
			case 'disabled':
				return false;

			case 'attachment':
			case 'attachment_parent':
			default:
				return true;
		}
	}
}
                                                                                                                                                                wordpress-seo-extended/src/actions/importing/aioseo/aioseo-taxonomy-settings-importing-action.php   0000644                 00000007645 15122266557 0031310 0                                                                                                    ustar 00                                                                                plugins                                                                                                                                                                <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

/**
 * Importing action for AIOSEO taxonomies settings data.
 */
class Aioseo_Taxonomy_Settings_Importing_Action extends Abstract_Aioseo_Settings_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'taxonomy_settings';

	/**
	 * The option_name of the AIOSEO option that contains the settings.
	 */
	public const SOURCE_OPTION_NAME = 'aioseo_options_dynamic';

	/**
	 * The map of aioseo_options to yoast settings.
	 *
	 * @var array
	 */
	protected $aioseo_options_to_yoast_map = [];

	/**
	 * The tab of the aioseo settings we're working with.
	 *
	 * @var string
	 */
	protected $settings_tab = 'taxonomies';

	/**
	 * Additional mapping between AiOSEO replace vars and Yoast replace vars.
	 *
	 * @var array
	 *
	 * @see https://yoast.com/help/list-available-snippet-variables-yoast-seo/
	 */
	protected $replace_vars_edited_map = [
		'#breadcrumb_404_error_format'         => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_archive_post_type_format' => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_archive_post_type_name'   => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_author_display_name'      => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_author_first_name'        => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_blog_page_title'          => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_label'                    => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_link'                     => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_search_result_format'     => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_search_string'            => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_separator'                => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#breadcrumb_taxonomy_title'           => '', // Empty string, as AIOSEO shows nothing for that tag.
		'#taxonomy_title'                      => '%%term_title%%',
	];

	/**
	 * Builds the mapping that ties AOISEO option keys with Yoast ones and their data transformation method.
	 *
	 * @return void
	 */
	protected function build_mapping() {
		$taxonomy_objects = \get_taxonomies( [ 'public' => true ], 'object' );

		foreach ( $taxonomy_objects as $tax ) {
			// Use all the public taxonomies.
			$this->aioseo_options_to_yoast_map[ '/' . $tax->name . '/title' ]                       = [
				'yoast_name'       => 'title-tax-' . $tax->name,
				'transform_method' => 'simple_import',
			];
			$this->aioseo_options_to_yoast_map[ '/' . $tax->name . '/metaDescription' ]             = [
				'yoast_name'       => 'metadesc-tax-' . $tax->name,
				'transform_method' => 'simple_import',
			];
			$this->aioseo_options_to_yoast_map[ '/' . $tax->name . '/advanced/robotsMeta/noindex' ] = [
				'yoast_name'       => 'noindex-tax-' . $tax->name,
				'transform_method' => 'import_noindex',
				'type'             => 'taxonomies',
				'subtype'          => $tax->name,
				'option_name'      => 'aioseo_options_dynamic',
			];
		}
	}

	/**
	 * Returns a setting map of the robot setting for post category taxonomies.
	 *
	 * @return array The setting map of the robot setting for post category taxonomies.
	 */
	public function pluck_robot_setting_from_mapping() {
		$this->build_mapping();

		foreach ( $this->aioseo_options_to_yoast_map as $setting ) {
			// Return the first archive setting map.
			if ( $setting['transform_method'] === 'import_noindex' && isset( $setting['subtype'] ) && $setting['subtype'] === 'category' ) {
				return $setting;
			}
		}

		return [];
	}
}
                                                                                           plugins/wordpress-seo-extended/src/actions/importing/aioseo/aioseo-validate-data-action.php         0000644                 00000021157 15122266557 0026337 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

// phpcs:disable Yoast.NamingConventions.NamespaceName.TooLong -- Given it's a very specific case.
namespace Yoast\WP\SEO\Actions\Importing\Aioseo;

use wpdb;
use Yoast\WP\SEO\Actions\Importing\Abstract_Aioseo_Importing_Action;
use Yoast\WP\SEO\Exceptions\Importing\Aioseo_Validation_Exception;
use Yoast\WP\SEO\Helpers\Options_Helper;

/**
 * Importing action for validating AIOSEO data before the import occurs.
 */
class Aioseo_Validate_Data_Action extends Abstract_Aioseo_Importing_Action {

	/**
	 * The plugin of the action.
	 */
	public const PLUGIN = 'aioseo';

	/**
	 * The type of the action.
	 */
	public const TYPE = 'validate_data';

	/**
	 * The WordPress database instance.
	 *
	 * @var wpdb
	 */
	protected $wpdb;

	/**
	 * The Post Importing action.
	 *
	 * @var Aioseo_Posts_Importing_Action
	 */
	protected $post_importing_action;

	/**
	 * The settings importing actions.
	 *
	 * @var array
	 */
	protected $settings_importing_actions;

	/**
	 * Class constructor.
	 *
	 * @param wpdb                                               $wpdb                              The WordPress database instance.
	 * @param Options_Helper                                     $options                           The options helper.
	 * @param Aioseo_Custom_Archive_Settings_Importing_Action    $custom_archive_action             The Custom Archive Settings importing action.
	 * @param Aioseo_Default_Archive_Settings_Importing_Action   $default_archive_action            The Default Archive Settings importing action.
	 * @param Aioseo_General_Settings_Importing_Action           $general_settings_action           The General Settings importing action.
	 * @param Aioseo_Posttype_Defaults_Settings_Importing_Action $posttype_defaults_settings_action The Posttype Defaults Settings importing action.
	 * @param Aioseo_Taxonomy_Settings_Importing_Action          $taxonomy_settings_action          The Taxonomy Settings importing action.
	 * @param Aioseo_Posts_Importing_Action                      $post_importing_action             The Post importing action.
	 */
	public function __construct(
		wpdb $wpdb,
		Options_Helper $options,
		Aioseo_Custom_Archive_Settings_Importing_Action $custom_archive_action,
		Aioseo_Default_Archive_Settings_Importing_Action $default_archive_action,
		Aioseo_General_Settings_Importing_Action $general_settings_action,
		Aioseo_Posttype_Defaults_Settings_Importing_Action $posttype_defaults_settings_action,
		Aioseo_Taxonomy_Settings_Importing_Action $taxonomy_settings_action,
		Aioseo_Posts_Importing_Action $post_importing_action
	) {
		$this->wpdb                       = $wpdb;
		$this->options                    = $options;
		$this->post_importing_action      = $post_importing_action;
		$this->settings_importing_actions = [
			$custom_archive_action,
			$default_archive_action,
			$general_settings_action,
			$posttype_defaults_settings_action,
			$taxonomy_settings_action,
		];
	}

	/**
	 * Just checks if the action has been completed in the past.
	 *
	 * @return int 1 if it hasn't been completed in the past, 0 if it has.
	 */
	public function get_total_unindexed() {
		return ( ! $this->get_completed() ) ? 1 : 0;
	}

	/**
	 * Just checks if the action has been completed in the past.
	 *
	 * @param int $limit The maximum number of unimported objects to be returned. Not used, exists to comply with the interface.
	 *
	 * @return int 1 if it hasn't been completed in the past, 0 if it has.
	 */
	public function get_limited_unindexed_count( $limit ) {
		return ( ! $this->get_completed() ) ? 1 : 0;
	}

	/**
	 * Validates AIOSEO data.
	 *
	 * @return array An array of validated data or false if aioseo data did not pass validation.
	 *
	 * @throws Aioseo_Validation_Exception If the validation fails.
	 */
	public function index() {
		if ( $this->get_completed() ) {
			return [];
		}

		$validated_aioseo_table    = $this->validate_aioseo_table();
		$validated_aioseo_settings = $this->validate_aioseo_settings();
		$validated_robot_settings  = $this->validate_robot_settings();

		if ( $validated_aioseo_table === false || $validated_aioseo_settings === false || $validated_robot_settings === false ) {
			throw new Aioseo_Validation_Exception();
		}

		$this->set_completed( true );

		return [
			'validated_aioseo_table'    => $validated_aioseo_table,
			'validated_aioseo_settings' => $validated_aioseo_settings,
			'validated_robot_settings'  => $validated_robot_settings,
		];
	}

	/**
	 * Validates the AIOSEO indexable table.
	 *
	 * @return bool Whether the AIOSEO table exists and has the structure we expect.
	 */
	public function validate_aioseo_table() {
		if ( ! $this->aioseo_helper->aioseo_exists() ) {
			return false;
		}

		$table       = $this->aioseo_helper->get_table();
		$needed_data = $this->post_importing_action->get_needed_data();

		$aioseo_columns = $this->wpdb->get_col(
			"SHOW COLUMNS FROM {$table}", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Reason: There is no unescaped user input.
			0
		);

		return $needed_data === \array_intersect( $needed_data, $aioseo_columns );
	}

	/**
	 * Validates the AIOSEO settings from the options table.
	 *
	 * @return bool Whether the AIOSEO settings from the options table exist and have the structure we expect.
	 */
	public function validate_aioseo_settings() {
		foreach ( $this->settings_importing_actions as $settings_import_action ) {
			$aioseo_settings = \json_decode( \get_option( $settings_import_action->get_source_option_name(), '' ), true );

			if ( ! $settings_import_action->isset_settings_tab( $aioseo_settings ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Validates the AIOSEO robots settings from the options table.
	 *
	 * @return bool Whether the AIOSEO robots settings from the options table exist and have the structure we expect.
	 */
	public function validate_robot_settings() {
		if ( $this->validate_post_robot_settings() && $this->validate_default_robot_settings() ) {
			return true;
		}

		return false;
	}

	/**
	 * Validates the post AIOSEO robots settings from the options table.
	 *
	 * @return bool Whether the post AIOSEO robots settings from the options table exist and have the structure we expect.
	 */
	public function validate_post_robot_settings() {
		$post_robot_mapping = $this->post_importing_action->enhance_mapping();
		// We're gonna validate against posttype robot settings only for posts, assuming the robot settings stay the same for other post types.
		$post_robot_mapping['subtype'] = 'post';

		// Let's get both the aioseo_options and the aioseo_options_dynamic options.
		$aioseo_global_settings = $this->aioseo_helper->get_global_option();
		$aioseo_posts_settings  = \json_decode( \get_option( $post_robot_mapping['option_name'], '' ), true );

		$needed_robots_data = $this->post_importing_action->get_needed_robot_data();
		\array_push( $needed_robots_data, 'default', 'noindex' );

		foreach ( $needed_robots_data as $robot_setting ) {
			// Validate against global settings.
			if ( ! isset( $aioseo_global_settings['searchAppearance']['advanced']['globalRobotsMeta'][ $robot_setting ] ) ) {
				return false;
			}

			// Validate against posttype settings.
			if ( ! isset( $aioseo_posts_settings['searchAppearance'][ $post_robot_mapping['type'] ][ $post_robot_mapping['subtype'] ]['advanced']['robotsMeta'][ $robot_setting ] ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Validates the default AIOSEO robots settings for search appearance settings from the options table.
	 *
	 * @return bool Whether the AIOSEO robots settings for search appearance settings from the options table exist and have the structure we expect.
	 */
	public function validate_default_robot_settings() {

		foreach ( $this->settings_importing_actions as $settings_import_action ) {
			$robot_setting_map = $settings_import_action->pluck_robot_setting_from_mapping();

			// Some actions return empty robot settings, let's not validate against those.
			if ( ! empty( $robot_setting_map ) ) {
				$aioseo_settings = \json_decode( \get_option( $robot_setting_map['option_name'], '' ), true );

				if ( ! isset( $aioseo_settings['searchAppearance'][ $robot_setting_map['type'] ][ $robot_setting_map['subtype'] ]['advanced']['robotsMeta']['default'] ) ) {
					return false;
				}
			}
		}

		return true;
	}

	/**
	 * Used nowhere. Exists to comply with the interface.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_aioseo_cleanup_limit' - Allow filtering the number of validations during each action pass.
		 *
		 * @param int $limit The maximum number of validations.
		 */
		$limit = \apply_filters( 'wpseo_aioseo_validation_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                 plugins/wordpress-seo-extended/src/actions/importing/deactivate-conflicting-plugins-action.php      0000644                 00000010417 15122266557 0027165 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Importing;

use Yoast\WP\SEO\Conditionals\Updated_Importer_Framework_Conditional;
use Yoast\WP\SEO\Config\Conflicting_Plugins;
use Yoast\WP\SEO\Helpers\Import_Cursor_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Sanitization_Helper;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Replacevar_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Provider_Service;
use Yoast\WP\SEO\Services\Importing\Aioseo\Aioseo_Robots_Transformer_Service;
use Yoast\WP\SEO\Services\Importing\Conflicting_Plugins_Service;

/**
 * Deactivates plug-ins that cause conflicts with Yoast SEO.
 */
class Deactivate_Conflicting_Plugins_Action extends Abstract_Aioseo_Importing_Action {

	/**
	 * The plugin the class deals with.
	 *
	 * @var string
	 */
	public const PLUGIN = 'conflicting-plugins';

	/**
	 * The type the class deals with.
	 *
	 * @var string
	 */
	public const TYPE = 'deactivation';

	/**
	 * The replacevar handler.
	 *
	 * @var Aioseo_Replacevar_Service
	 */
	protected $replacevar_handler;

	/**
	 * Knows all plugins that might possibly conflict.
	 *
	 * @var Conflicting_Plugins_Service
	 */
	protected $conflicting_plugins;

	/**
	 * The list of conflicting plugins
	 *
	 * @var array
	 */
	protected $detected_plugins;

	/**
	 * Class constructor.
	 *
	 * @param Import_Cursor_Helper              $import_cursor               The import cursor helper.
	 * @param Options_Helper                    $options                     The options helper.
	 * @param Sanitization_Helper               $sanitization                The sanitization helper.
	 * @param Aioseo_Replacevar_Service         $replacevar_handler          The replacevar handler.
	 * @param Aioseo_Robots_Provider_Service    $robots_provider             The robots provider service.
	 * @param Aioseo_Robots_Transformer_Service $robots_transformer          The robots transfomer service.
	 * @param Conflicting_Plugins_Service       $conflicting_plugins_service The Conflicting plugins Service.
	 */
	public function __construct(
		Import_Cursor_Helper $import_cursor,
		Options_Helper $options,
		Sanitization_Helper $sanitization,
		Aioseo_Replacevar_Service $replacevar_handler,
		Aioseo_Robots_Provider_Service $robots_provider,
		Aioseo_Robots_Transformer_Service $robots_transformer,
		Conflicting_Plugins_Service $conflicting_plugins_service
	) {
		parent::__construct( $import_cursor, $options, $sanitization, $replacevar_handler, $robots_provider, $robots_transformer );

		$this->conflicting_plugins = $conflicting_plugins_service;
		$this->detected_plugins    = [];
	}

	/**
	 * Get the total number of conflicting plugins.
	 *
	 * @return int
	 */
	public function get_total_unindexed() {
		return \count( $this->get_detected_plugins() );
	}

	/**
	 * Returns whether the updated importer framework is enabled.
	 *
	 * @return bool True if the updated importer framework is enabled.
	 */
	public function is_enabled() {
		$updated_importer_framework_conditional = \YoastSEO()->classes->get( Updated_Importer_Framework_Conditional::class );

		return $updated_importer_framework_conditional->is_met();
	}

	/**
	 * Deactivate conflicting plugins.
	 *
	 * @return array
	 */
	public function index() {
		$detected_plugins = $this->get_detected_plugins();
		$this->conflicting_plugins->deactivate_conflicting_plugins( $detected_plugins );

		// We need to conform to the interface, so we report that no indexables were created.
		return [];
	}

	/**
	 * {@inheritDoc}
	 */
	public function get_limit() {
		return \count( Conflicting_Plugins::all_plugins() );
	}

	/**
	 * Returns the total number of unindexed objects up to a limit.
	 *
	 * @param int $limit The maximum.
	 *
	 * @return int The total number of unindexed objects.
	 */
	public function get_limited_unindexed_count( $limit ) {
		$count = \count( $this->get_detected_plugins() );
		return ( $count <= $limit ) ? $count : $limit;
	}

	/**
	 * Returns all detected plugins.
	 *
	 * @return array The detected plugins.
	 */
	protected function get_detected_plugins() {
		// The active plugins won't change much. We can reuse the result for the duration of the request.
		if ( \count( $this->detected_plugins ) < 1 ) {
			$this->detected_plugins = $this->conflicting_plugins->detect_conflicting_plugins();
		}
		return $this->detected_plugins;
	}
}
                                                                                                                                                                                                                                                 plugins/wordpress-seo-extended/src/actions/importing/importing-action-interface.php                 0000644                 00000001614 15122266557 0025045 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Importing;

use Yoast\WP\SEO\Actions\Indexing\Limited_Indexing_Action_Interface;

interface Importing_Action_Interface extends Importing_Indexation_Action_Interface, Limited_Indexing_Action_Interface {

	/**
	 * Returns the name of the plugin we import from.
	 *
	 * @return string The plugin name.
	 */
	public function get_plugin();

	/**
	 * Returns the type of data we import.
	 *
	 * @return string The type of data.
	 */
	public function get_type();

	/**
	 * Whether or not this action is capable of importing given a specific plugin and type.
	 *
	 * @param string|null $plugin The name of the plugin being imported.
	 * @param string|null $type   The component of the plugin being imported.
	 *
	 * @return bool True if the action can import the given plugin's data of the given type.
	 */
	public function is_compatible_with( $plugin = null, $type = null );
}
                                                                                                                    plugins/wordpress-seo-extended/src/actions/importing/importing-indexation-action-interface.php      0000644                 00000001445 15122266557 0027207 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Importing;

/**
 * Interface definition of reindexing action for indexables.
 */
interface Importing_Indexation_Action_Interface {

	/**
	 * Returns the total number of unindexed objects.
	 *
	 * @return int The total number of unindexed objects.
	 */
	public function get_total_unindexed();

	/**
	 * Indexes a number of objects.
	 *
	 * NOTE: ALWAYS use limits, this method is intended to be called multiple times over several requests.
	 *
	 * For indexing that requires JavaScript simply return the objects that should be indexed.
	 *
	 * @return array The reindexed objects.
	 */
	public function index();

	/**
	 * Returns the number of objects that will be indexed in a single indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit();
}
                                                                                                                                                                                                                           plugins/wordpress-seo-extended/src/actions/indexables/indexable-head-action.php                     0000644                 00000007417 15122266557 0024046 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexables;

use Yoast\WP\SEO\Surfaces\Meta_Surface;
use Yoast\WP\SEO\Surfaces\Values\Meta;

/**
 * Get head action for indexables.
 */
class Indexable_Head_Action {

	/**
	 * Caches the output.
	 *
	 * @var mixed
	 */
	protected $cache;

	/**
	 * The meta surface.
	 *
	 * @var Meta_Surface
	 */
	private $meta_surface;

	/**
	 * Indexable_Head_Action constructor.
	 *
	 * @param Meta_Surface $meta_surface The meta surface.
	 */
	public function __construct( Meta_Surface $meta_surface ) {
		$this->meta_surface = $meta_surface;
	}

	/**
	 * Retrieves the head for a url.
	 *
	 * @param string $url The url to get the head for.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_url( $url ) {
		if ( $url === \trailingslashit( \get_home_url() ) ) {
			return $this->with_404_fallback( $this->with_cache( 'home_page' ) );
		}
		return $this->with_404_fallback( $this->with_cache( 'url', $url ) );
	}

	/**
	 * Retrieves the head for a post.
	 *
	 * @param int $id The id.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_post( $id ) {
		return $this->with_404_fallback( $this->with_cache( 'post', $id ) );
	}

	/**
	 * Retrieves the head for a term.
	 *
	 * @param int $id The id.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_term( $id ) {
		return $this->with_404_fallback( $this->with_cache( 'term', $id ) );
	}

	/**
	 * Retrieves the head for an author.
	 *
	 * @param int $id The id.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_author( $id ) {
		return $this->with_404_fallback( $this->with_cache( 'author', $id ) );
	}

	/**
	 * Retrieves the head for a post type archive.
	 *
	 * @param int $type The id.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_post_type_archive( $type ) {
		return $this->with_404_fallback( $this->with_cache( 'post_type_archive', $type ) );
	}

	/**
	 * Retrieves the head for the posts page.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_posts_page() {
		return $this->with_404_fallback( $this->with_cache( 'posts_page' ) );
	}

	/**
	 * Retrieves the head for the 404 page. Always sets the status to 404.
	 *
	 * @return object Object with head and status properties.
	 */
	public function for_404() {
		$meta = $this->with_cache( '404' );

		if ( ! $meta ) {
			return (object) [
				'html'   => '',
				'json'   => [],
				'status' => 404,
			];
		}

		$head = $meta->get_head();

		return (object) [
			'html'   => $head->html,
			'json'   => $head->json,
			'status' => 404,
		];
	}

	/**
	 * Retrieves the head for a successful page load.
	 *
	 * @param object $head The calculated Yoast head.
	 *
	 * @return object The presentations and status code 200.
	 */
	protected function for_200( $head ) {
		return (object) [
			'html'   => $head->html,
			'json'   => $head->json,
			'status' => 200,
		];
	}

	/**
	 * Returns the head with 404 fallback
	 *
	 * @param Meta|false $meta The meta object.
	 *
	 * @return object The head response.
	 */
	protected function with_404_fallback( $meta ) {
		if ( $meta === false ) {
			return $this->for_404();
		}
		else {
			return $this->for_200( $meta->get_head() );
		}
	}

	/**
	 * Retrieves a value from the meta surface cached.
	 *
	 * @param string $type     The type of value to retrieve.
	 * @param string $argument Optional. The argument for the value.
	 *
	 * @return Meta The meta object.
	 */
	protected function with_cache( $type, $argument = '' ) {
		if ( ! isset( $this->cache[ $type ][ $argument ] ) ) {
			$this->cache[ $type ][ $argument ] = \call_user_func( [ $this->meta_surface, "for_$type" ], $argument );
		}

		return $this->cache[ $type ][ $argument ];
	}
}
                                                                                                                                                                                                                                                 plugins/wordpress-seo-extended/src/actions/indexing/abstract-indexing-action.php                    0000644                 00000006115 15122266557 0024303 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

/**
 * Base class of indexing actions.
 */
abstract class Abstract_Indexing_Action implements Indexation_Action_Interface, Limited_Indexing_Action_Interface {

	/**
	 * The transient name.
	 *
	 * This is a trick to force derived classes to define a transient themselves.
	 *
	 * @var string
	 */
	public const UNINDEXED_COUNT_TRANSIENT = null;

	/**
	 * The transient cache key for limited counts.
	 *
	 * @var string
	 */
	public const UNINDEXED_LIMITED_COUNT_TRANSIENT = self::UNINDEXED_COUNT_TRANSIENT . '_limited';

	/**
	 * Builds a query for selecting the ID's of unindexed posts.
	 *
	 * @param bool $limit The maximum number of post IDs to return.
	 *
	 * @return string The prepared query string.
	 */
	abstract protected function get_select_query( $limit );

	/**
	 * Builds a query for counting the number of unindexed posts.
	 *
	 * @return string The prepared query string.
	 */
	abstract protected function get_count_query();

	/**
	 * Returns a limited number of unindexed posts.
	 *
	 * @param int $limit Limit the maximum number of unindexed posts that are counted.
	 *
	 * @return int The limited number of unindexed posts. 0 if the query fails.
	 */
	public function get_limited_unindexed_count( $limit ) {
		$transient = \get_transient( static::UNINDEXED_LIMITED_COUNT_TRANSIENT );
		if ( $transient !== false ) {
			return (int) $transient;
		}

		\set_transient( static::UNINDEXED_LIMITED_COUNT_TRANSIENT, 0, ( \MINUTE_IN_SECONDS * 15 ) );

		$query = $this->get_select_query( $limit );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Function get_count_query returns a prepared query.
		$unindexed_object_ids = $this->wpdb->get_col( $query );
		$count                = (int) \count( $unindexed_object_ids );

		\set_transient( static::UNINDEXED_LIMITED_COUNT_TRANSIENT, $count, ( \MINUTE_IN_SECONDS * 15 ) );

		return $count;
	}

	/**
	 * Returns the total number of unindexed posts.
	 *
	 * @return int|false The total number of unindexed posts. False if the query fails.
	 */
	public function get_total_unindexed() {
		$transient = \get_transient( static::UNINDEXED_COUNT_TRANSIENT );
		if ( $transient !== false ) {
			return (int) $transient;
		}

		// Store transient before doing the query so multiple requests won't make multiple queries.
		// Only store this for 15 minutes to ensure that if the query doesn't complete a wrong count is not kept too long.
		\set_transient( static::UNINDEXED_COUNT_TRANSIENT, 0, ( \MINUTE_IN_SECONDS * 15 ) );

		$query = $this->get_count_query();

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Function get_count_query returns a prepared query.
		$count = $this->wpdb->get_var( $query );

		if ( \is_null( $count ) ) {
			return false;
		}

		\set_transient( static::UNINDEXED_COUNT_TRANSIENT, $count, \DAY_IN_SECONDS );

		/**
		 * Action: 'wpseo_indexables_unindexed_calculated' - sets an option to timestamp when there are no unindexed indexables left.
		 *
		 * @internal
		 */
		\do_action( 'wpseo_indexables_unindexed_calculated', static::UNINDEXED_COUNT_TRANSIENT, $count );

		return (int) $count;
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                   plugins/wordpress-seo-extended/src/actions/indexing/abstract-link-indexing-action.php               0000644                 00000005414 15122266557 0025237 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use wpdb;
use Yoast\WP\SEO\Builders\Indexable_Link_Builder;
use Yoast\WP\SEO\Models\SEO_Links;
use Yoast\WP\SEO\Repositories\Indexable_Repository;

/**
 * Reindexing action for link indexables.
 */
abstract class Abstract_Link_Indexing_Action extends Abstract_Indexing_Action {

	/**
	 * The link builder.
	 *
	 * @var Indexable_Link_Builder
	 */
	protected $link_builder;

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $repository;

	/**
	 * The WordPress database instance.
	 *
	 * @var wpdb
	 */
	protected $wpdb;

	/**
	 * Indexable_Post_Indexing_Action constructor
	 *
	 * @param Indexable_Link_Builder $link_builder The indexable link builder.
	 * @param Indexable_Repository   $repository   The indexable repository.
	 * @param wpdb                   $wpdb         The WordPress database instance.
	 */
	public function __construct(
		Indexable_Link_Builder $link_builder,
		Indexable_Repository $repository,
		wpdb $wpdb
	) {
		$this->link_builder = $link_builder;
		$this->repository   = $repository;
		$this->wpdb         = $wpdb;
	}

	/**
	 * Builds links for indexables which haven't had their links indexed yet.
	 *
	 * @return SEO_Links[] The created SEO links.
	 */
	public function index() {
		$objects = $this->get_objects();

		$indexables = [];
		foreach ( $objects as $object ) {
			$indexable = $this->repository->find_by_id_and_type( $object->id, $object->type );
			if ( $indexable ) {
				$this->link_builder->build( $indexable, $object->content );
				$indexable->save();

				$indexables[] = $indexable;
			}
		}

		if ( \count( $indexables ) > 0 ) {
			\delete_transient( static::UNINDEXED_COUNT_TRANSIENT );
			\delete_transient( static::UNINDEXED_LIMITED_COUNT_TRANSIENT );
		}

		return $indexables;
	}

	/**
	 * In the case of term-links and post-links we want to use the total unindexed count, because using
	 * the limited unindexed count actually leads to worse performance.
	 *
	 * @param int|bool $limit Unused.
	 *
	 * @return int The total number of unindexed links.
	 */
	public function get_limited_unindexed_count( $limit = false ) {
		return $this->get_total_unindexed();
	}

	/**
	 * Returns the number of texts that will be indexed in a single link indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_link_indexing_limit' - Allow filtering the number of texts indexed during each link indexing pass.
		 *
		 * @param int $limit The maximum number of texts indexed.
		 */
		return \apply_filters( 'wpseo_link_indexing_limit', 5 );
	}

	/**
	 * Returns objects to be indexed.
	 *
	 * @return array Objects to be indexed, should be an array of objects with object_id, object_type and content.
	 */
	abstract protected function get_objects();
}
                                                                                                                                                                                                                                                    plugins/wordpress-seo-extended/src/actions/indexing/indexable-general-indexation-action.php         0000644                 00000007601 15122266557 0026404 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Repository;

/**
 * General reindexing action for indexables.
 */
class Indexable_General_Indexation_Action implements Indexation_Action_Interface, Limited_Indexing_Action_Interface {

	/**
	 * The transient cache key.
	 */
	public const UNINDEXED_COUNT_TRANSIENT = 'wpseo_total_unindexed_general_items';

	/**
	 * Represents the indexables repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $indexable_repository;

	/**
	 * Indexable_General_Indexation_Action constructor.
	 *
	 * @param Indexable_Repository $indexable_repository The indexables repository.
	 */
	public function __construct( Indexable_Repository $indexable_repository ) {
		$this->indexable_repository = $indexable_repository;
	}

	/**
	 * Returns the total number of unindexed objects.
	 *
	 * @return int The total number of unindexed objects.
	 */
	public function get_total_unindexed() {
		$transient = \get_transient( static::UNINDEXED_COUNT_TRANSIENT );
		if ( $transient !== false ) {
			return (int) $transient;
		}

		$indexables_to_create = $this->query();

		$result = \count( $indexables_to_create );

		\set_transient( static::UNINDEXED_COUNT_TRANSIENT, $result, \DAY_IN_SECONDS );

		/**
		 * Action: 'wpseo_indexables_unindexed_calculated' - sets an option to timestamp when there are no unindexed indexables left.
		 *
		 * @internal
		 */
		\do_action( 'wpseo_indexables_unindexed_calculated', static::UNINDEXED_COUNT_TRANSIENT, $result );

		return $result;
	}

	/**
	 * Returns a limited number of unindexed posts.
	 *
	 * @param int $limit Limit the maximum number of unindexed posts that are counted.
	 *
	 * @return int|false The limited number of unindexed posts. False if the query fails.
	 */
	public function get_limited_unindexed_count( $limit ) {
		return $this->get_total_unindexed();
	}

	/**
	 * Creates indexables for unindexed system pages, the date archive, and the homepage.
	 *
	 * @return Indexable[] The created indexables.
	 */
	public function index() {
		$indexables           = [];
		$indexables_to_create = $this->query();

		if ( isset( $indexables_to_create['404'] ) ) {
			$indexables[] = $this->indexable_repository->find_for_system_page( '404' );
		}

		if ( isset( $indexables_to_create['search'] ) ) {
			$indexables[] = $this->indexable_repository->find_for_system_page( 'search-result' );
		}

		if ( isset( $indexables_to_create['date_archive'] ) ) {
			$indexables[] = $this->indexable_repository->find_for_date_archive();
		}
		if ( isset( $indexables_to_create['home_page'] ) ) {
			$indexables[] = $this->indexable_repository->find_for_home_page();
		}

		\set_transient( static::UNINDEXED_COUNT_TRANSIENT, 0, \DAY_IN_SECONDS );

		return $indexables;
	}

	/**
	 * Returns the number of objects that will be indexed in a single indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		// This matches the maximum number of indexables created by this action.
		return 4;
	}

	/**
	 * Check which indexables already exist and return the values of the ones to create.
	 *
	 * @return array The indexable types to create.
	 */
	private function query() {
		$indexables_to_create = [];
		if ( ! $this->indexable_repository->find_for_system_page( '404', false ) ) {
			$indexables_to_create['404'] = true;
		}

		if ( ! $this->indexable_repository->find_for_system_page( 'search-result', false ) ) {
			$indexables_to_create['search'] = true;
		}

		if ( ! $this->indexable_repository->find_for_date_archive( false ) ) {
			$indexables_to_create['date_archive'] = true;
		}

		$need_home_page_indexable = ( (int) \get_option( 'page_on_front' ) === 0 && \get_option( 'show_on_front' ) === 'posts' );

		if ( $need_home_page_indexable && ! $this->indexable_repository->find_for_home_page( false ) ) {
			$indexables_to_create['home_page'] = true;
		}

		return $indexables_to_create;
	}
}
                                                                                                                               plugins/wordpress-seo-extended/src/actions/indexing/indexable-indexing-complete-action.php          0000644                 00000001316 15122266557 0026237 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\SEO\Helpers\Indexable_Helper;

/**
 * Indexing action to call when the indexable indexing process is completed.
 */
class Indexable_Indexing_Complete_Action {

	/**
	 * The options helper.
	 *
	 * @var Indexable_Helper
	 */
	protected $indexable_helper;

	/**
	 * Indexable_Indexing_Complete_Action constructor.
	 *
	 * @param Indexable_Helper $indexable_helper The indexable helper.
	 */
	public function __construct( Indexable_Helper $indexable_helper ) {
		$this->indexable_helper = $indexable_helper;
	}

	/**
	 * Wraps up the indexing process.
	 *
	 * @return void
	 */
	public function complete() {
		$this->indexable_helper->finish_indexing();
	}
}
                                                                                                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/src/actions/indexing/indexable-post-indexation-action.php            0000644                 00000013313 15122266557 0025751 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use wpdb;
use Yoast\WP\Lib\Model;
use Yoast\WP\SEO\Helpers\Post_Helper;
use Yoast\WP\SEO\Helpers\Post_Type_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Values\Indexables\Indexable_Builder_Versions;

/**
 * Reindexing action for post indexables.
 */
class Indexable_Post_Indexation_Action extends Abstract_Indexing_Action {

	/**
	 * The transient cache key.
	 *
	 * @var string
	 */
	public const UNINDEXED_COUNT_TRANSIENT = 'wpseo_total_unindexed_posts';

	/**
	 * The transient cache key for limited counts.
	 *
	 * @var string
	 */
	public const UNINDEXED_LIMITED_COUNT_TRANSIENT = self::UNINDEXED_COUNT_TRANSIENT . '_limited';

	/**
	 * The post type helper.
	 *
	 * @var Post_Type_Helper
	 */
	protected $post_type_helper;

	/**
	 * The post helper.
	 *
	 * @var Post_Helper
	 */
	protected $post_helper;

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $repository;

	/**
	 * The WordPress database instance.
	 *
	 * @var wpdb
	 */
	protected $wpdb;

	/**
	 * The latest version of Post Indexables.
	 *
	 * @var int
	 */
	protected $version;

	/**
	 * Indexable_Post_Indexing_Action constructor
	 *
	 * @param Post_Type_Helper           $post_type_helper The post type helper.
	 * @param Indexable_Repository       $repository       The indexable repository.
	 * @param wpdb                       $wpdb             The WordPress database instance.
	 * @param Indexable_Builder_Versions $builder_versions The latest versions for each Indexable type.
	 * @param Post_Helper                $post_helper      The post helper.
	 */
	public function __construct(
		Post_Type_Helper $post_type_helper,
		Indexable_Repository $repository,
		wpdb $wpdb,
		Indexable_Builder_Versions $builder_versions,
		Post_Helper $post_helper
	) {
		$this->post_type_helper = $post_type_helper;
		$this->repository       = $repository;
		$this->wpdb             = $wpdb;
		$this->version          = $builder_versions->get_latest_version_for_type( 'post' );
		$this->post_helper      = $post_helper;
	}

	/**
	 * Creates indexables for unindexed posts.
	 *
	 * @return Indexable[] The created indexables.
	 */
	public function index() {
		$query = $this->get_select_query( $this->get_limit() );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Function get_select_query returns a prepared query.
		$post_ids = $this->wpdb->get_col( $query );

		$indexables = [];
		foreach ( $post_ids as $post_id ) {
			$indexables[] = $this->repository->find_by_id_and_type( (int) $post_id, 'post' );
		}

		if ( \count( $indexables ) > 0 ) {
			\delete_transient( static::UNINDEXED_COUNT_TRANSIENT );
			\delete_transient( static::UNINDEXED_LIMITED_COUNT_TRANSIENT );
		}

		return $indexables;
	}

	/**
	 * Returns the number of posts that will be indexed in a single indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_post_indexation_limit' - Allow filtering the amount of posts indexed during each indexing pass.
		 *
		 * @param int $limit The maximum number of posts indexed.
		 */
		$limit = \apply_filters( 'wpseo_post_indexation_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}

	/**
	 * Builds a query for counting the number of unindexed posts.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_count_query() {
		$indexable_table = Model::get_table_name( 'Indexable' );

		$post_types             = $this->post_type_helper->get_indexable_post_types();
		$excluded_post_statuses = $this->post_helper->get_excluded_post_statuses();
		$replacements           = \array_merge(
			$post_types,
			$excluded_post_statuses
		);

		$replacements[] = $this->version;

		// Warning: If this query is changed, makes sure to update the query in get_select_query as well.
		// @phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
		return $this->wpdb->prepare(
			"
			SELECT COUNT(P.ID)
			FROM {$this->wpdb->posts} AS P
			WHERE P.post_type IN (" . \implode( ', ', \array_fill( 0, \count( $post_types ), '%s' ) ) . ')
			AND P.post_status NOT IN (' . \implode( ', ', \array_fill( 0, \count( $excluded_post_statuses ), '%s' ) ) . ")
			AND P.ID not in (
				SELECT I.object_id from $indexable_table as I
				WHERE I.object_type = 'post'
				AND I.version = %d )",
			$replacements
		);
	}

	/**
	 * Builds a query for selecting the ID's of unindexed posts.
	 *
	 * @param bool $limit The maximum number of post IDs to return.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_select_query( $limit = false ) {
		$indexable_table = Model::get_table_name( 'Indexable' );

		$post_types             = $this->post_type_helper->get_indexable_post_types();
		$excluded_post_statuses = $this->post_helper->get_excluded_post_statuses();
		$replacements           = \array_merge(
			$post_types,
			$excluded_post_statuses
		);
		$replacements[]         = $this->version;

		$limit_query = '';
		if ( $limit ) {
			$limit_query    = 'LIMIT %d';
			$replacements[] = $limit;
		}

		// Warning: If this query is changed, makes sure to update the query in get_count_query as well.
		// @phpcs:ignore WordPress.DB.PreparedSQLPlaceholders.ReplacementsWrongNumber
		return $this->wpdb->prepare(
			"
			SELECT P.ID
			FROM {$this->wpdb->posts} AS P
			WHERE P.post_type IN (" . \implode( ', ', \array_fill( 0, \count( $post_types ), '%s' ) ) . ')
			AND P.post_status NOT IN (' . \implode( ', ', \array_fill( 0, \count( $excluded_post_statuses ), '%s' ) ) . ")
			AND P.ID not in (
				SELECT I.object_id from $indexable_table as I
				WHERE I.object_type = 'post'
				AND I.version = %d )
			$limit_query",
			$replacements
		);
	}
}
                                                                                                                                                                                                                                                                                                                     wordpress-seo-extended/src/actions/indexing/indexable-post-type-archive-indexation-action.php       0000644                 00000013737 15122266557 0030302 0                                                                                                    ustar 00                                                                                plugins                                                                                                                                                                <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\SEO\Builders\Indexable_Builder;
use Yoast\WP\SEO\Helpers\Post_Type_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Values\Indexables\Indexable_Builder_Versions;

/**
 * Reindexing action for post type archive indexables.
 *
 * @phpcs:disable Yoast.NamingConventions.ObjectNameDepth.MaxExceeded
 */
class Indexable_Post_Type_Archive_Indexation_Action implements Indexation_Action_Interface, Limited_Indexing_Action_Interface {


	/**
	 * The transient cache key.
	 */
	public const UNINDEXED_COUNT_TRANSIENT = 'wpseo_total_unindexed_post_type_archives';

	/**
	 * The post type helper.
	 *
	 * @var Post_Type_Helper
	 */
	protected $post_type;

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $repository;

	/**
	 * The indexable builder.
	 *
	 * @var Indexable_Builder
	 */
	protected $builder;

	/**
	 * The current version of the post type archive indexable builder.
	 *
	 * @var int
	 */
	protected $version;

	/**
	 * Indexation_Post_Type_Archive_Action constructor.
	 *
	 * @param Indexable_Repository       $repository The indexable repository.
	 * @param Indexable_Builder          $builder    The indexable builder.
	 * @param Post_Type_Helper           $post_type  The post type helper.
	 * @param Indexable_Builder_Versions $versions   The current versions of all indexable builders.
	 */
	public function __construct(
		Indexable_Repository $repository,
		Indexable_Builder $builder,
		Post_Type_Helper $post_type,
		Indexable_Builder_Versions $versions
	) {
		$this->repository = $repository;
		$this->builder    = $builder;
		$this->post_type  = $post_type;
		$this->version    = $versions->get_latest_version_for_type( 'post-type-archive' );
	}

	/**
	 * Returns the total number of unindexed post type archives.
	 *
	 * @param int $limit Limit the number of counted objects.
	 *
	 * @return int The total number of unindexed post type archives.
	 */
	public function get_total_unindexed( $limit = false ) {
		$transient = \get_transient( static::UNINDEXED_COUNT_TRANSIENT );
		if ( $transient !== false ) {
			return (int) $transient;
		}

		\set_transient( static::UNINDEXED_COUNT_TRANSIENT, 0, \DAY_IN_SECONDS );

		$result = \count( $this->get_unindexed_post_type_archives( $limit ) );

		\set_transient( static::UNINDEXED_COUNT_TRANSIENT, $result, \DAY_IN_SECONDS );

		/**
		 * Action: 'wpseo_indexables_unindexed_calculated' - sets an option to timestamp when there are no unindexed indexables left.
		 *
		 * @internal
		 */
		\do_action( 'wpseo_indexables_unindexed_calculated', static::UNINDEXED_COUNT_TRANSIENT, $result );

		return $result;
	}

	/**
	 * Creates indexables for post type archives.
	 *
	 * @return Indexable[] The created indexables.
	 */
	public function index() {
		$unindexed_post_type_archives = $this->get_unindexed_post_type_archives( $this->get_limit() );

		$indexables = [];
		foreach ( $unindexed_post_type_archives as $post_type_archive ) {
			$indexables[] = $this->builder->build_for_post_type_archive( $post_type_archive );
		}

		if ( \count( $indexables ) > 0 ) {
			\delete_transient( static::UNINDEXED_COUNT_TRANSIENT );
		}

		return $indexables;
	}

	/**
	 * Returns the number of post type archives that will be indexed in a single indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_post_type_archive_indexation_limit' - Allow filtering the number of posts indexed during each indexing pass.
		 *
		 * @param int $limit The maximum number of posts indexed.
		 */
		$limit = \apply_filters( 'wpseo_post_type_archive_indexation_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}

	/**
	 * Retrieves the list of post types for which no indexable for its archive page has been made yet.
	 *
	 * @param int|false $limit Limit the number of retrieved indexables to this number.
	 *
	 * @return array The list of post types for which no indexable for its archive page has been made yet.
	 */
	protected function get_unindexed_post_type_archives( $limit = false ) {
		$post_types_with_archive_pages = $this->get_post_types_with_archive_pages();
		$indexed_post_types            = $this->get_indexed_post_type_archives();

		$unindexed_post_types = \array_diff( $post_types_with_archive_pages, $indexed_post_types );

		if ( $limit ) {
			return \array_slice( $unindexed_post_types, 0, $limit );
		}

		return $unindexed_post_types;
	}

	/**
	 * Returns the names of all the post types that have archive pages.
	 *
	 * @return array The list of names of all post types that have archive pages.
	 */
	protected function get_post_types_with_archive_pages() {
		// We only want to index archive pages of public post types that have them.
		$post_types_with_archive = $this->post_type->get_indexable_post_archives();

		// We only need the post type names, not the objects.
		$post_types = [];
		foreach ( $post_types_with_archive as $post_type_with_archive ) {
			$post_types[] = $post_type_with_archive->name;
		}

		return $post_types;
	}

	/**
	 * Retrieves the list of post type names for which an archive indexable exists.
	 *
	 * @return array The list of names of post types with unindexed archive pages.
	 */
	protected function get_indexed_post_type_archives() {
		$results = $this->repository->query()
			->select( 'object_sub_type' )
			->where( 'object_type', 'post-type-archive' )
			->where_equal( 'version', $this->version )
			->find_array();

		if ( $results === false ) {
			return [];
		}

		$callback = static function ( $result ) {
			return $result['object_sub_type'];
		};

		return \array_map( $callback, $results );
	}

	/**
	 * Returns a limited number of unindexed posts.
	 *
	 * @param int $limit Limit the maximum number of unindexed posts that are counted.
	 *
	 * @return int|false The limited number of unindexed posts. False if the query fails.
	 */
	public function get_limited_unindexed_count( $limit ) {
		return $this->get_total_unindexed( $limit );
	}
}
                                 plugins/wordpress-seo-extended/src/actions/indexing/indexable-term-indexation-action.php            0000644                 00000011752 15122266557 0025740 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use wpdb;
use Yoast\WP\Lib\Model;
use Yoast\WP\SEO\Helpers\Taxonomy_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Values\Indexables\Indexable_Builder_Versions;

/**
 * Reindexing action for term indexables.
 */
class Indexable_Term_Indexation_Action extends Abstract_Indexing_Action {

	/**
	 * The transient cache key.
	 */
	public const UNINDEXED_COUNT_TRANSIENT = 'wpseo_total_unindexed_terms';

	/**
	 * The transient cache key for limited counts.
	 *
	 * @var string
	 */
	public const UNINDEXED_LIMITED_COUNT_TRANSIENT = self::UNINDEXED_COUNT_TRANSIENT . '_limited';

	/**
	 * The post type helper.
	 *
	 * @var Taxonomy_Helper
	 */
	protected $taxonomy;

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $repository;

	/**
	 * The WordPress database instance.
	 *
	 * @var wpdb
	 */
	protected $wpdb;

	/**
	 * The latest version of the Indexable term builder
	 *
	 * @var int
	 */
	protected $version;

	/**
	 * Indexable_Term_Indexation_Action constructor
	 *
	 * @param Taxonomy_Helper            $taxonomy         The taxonomy helper.
	 * @param Indexable_Repository       $repository       The indexable repository.
	 * @param wpdb                       $wpdb             The WordPress database instance.
	 * @param Indexable_Builder_Versions $builder_versions The latest versions of all indexable builders.
	 */
	public function __construct(
		Taxonomy_Helper $taxonomy,
		Indexable_Repository $repository,
		wpdb $wpdb,
		Indexable_Builder_Versions $builder_versions
	) {
		$this->taxonomy   = $taxonomy;
		$this->repository = $repository;
		$this->wpdb       = $wpdb;
		$this->version    = $builder_versions->get_latest_version_for_type( 'term' );
	}

	/**
	 * Creates indexables for unindexed terms.
	 *
	 * @return Indexable[] The created indexables.
	 */
	public function index() {
		$query = $this->get_select_query( $this->get_limit() );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Function get_select_query returns a prepared query.
		$term_ids = $this->wpdb->get_col( $query );

		$indexables = [];
		foreach ( $term_ids as $term_id ) {
			$indexables[] = $this->repository->find_by_id_and_type( (int) $term_id, 'term' );
		}

		if ( \count( $indexables ) > 0 ) {
			\delete_transient( static::UNINDEXED_COUNT_TRANSIENT );
			\delete_transient( static::UNINDEXED_LIMITED_COUNT_TRANSIENT );
		}

		return $indexables;
	}

	/**
	 * Returns the number of terms that will be indexed in a single indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit() {
		/**
		 * Filter 'wpseo_term_indexation_limit' - Allow filtering the number of terms indexed during each indexing pass.
		 *
		 * @param int $limit The maximum number of terms indexed.
		 */
		$limit = \apply_filters( 'wpseo_term_indexation_limit', 25 );

		if ( ! \is_int( $limit ) || $limit < 1 ) {
			$limit = 25;
		}

		return $limit;
	}

	/**
	 * Builds a query for counting the number of unindexed terms.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_count_query() {
		$indexable_table   = Model::get_table_name( 'Indexable' );
		$taxonomy_table    = $this->wpdb->term_taxonomy;
		$public_taxonomies = $this->taxonomy->get_indexable_taxonomies();

		$taxonomies_placeholders = \implode( ', ', \array_fill( 0, \count( $public_taxonomies ), '%s' ) );

		$replacements = [ $this->version ];
		\array_push( $replacements, ...$public_taxonomies );

		// Warning: If this query is changed, makes sure to update the query in get_count_query as well.
		return $this->wpdb->prepare(
			"
			SELECT COUNT(term_id)
			FROM {$taxonomy_table} AS T
			LEFT JOIN $indexable_table AS I
				ON T.term_id = I.object_id
				AND I.object_type = 'term'
				AND I.version = %d
			WHERE I.object_id IS NULL
				AND taxonomy IN ($taxonomies_placeholders)",
			$replacements
		);
	}

	/**
	 * Builds a query for selecting the ID's of unindexed terms.
	 *
	 * @param bool $limit The maximum number of term IDs to return.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_select_query( $limit = false ) {
		$indexable_table   = Model::get_table_name( 'Indexable' );
		$taxonomy_table    = $this->wpdb->term_taxonomy;
		$public_taxonomies = $this->taxonomy->get_indexable_taxonomies();
		$placeholders      = \implode( ', ', \array_fill( 0, \count( $public_taxonomies ), '%s' ) );

		$replacements = [ $this->version ];
		\array_push( $replacements, ...$public_taxonomies );

		$limit_query = '';
		if ( $limit ) {
			$limit_query    = 'LIMIT %d';
			$replacements[] = $limit;
		}

		// Warning: If this query is changed, makes sure to update the query in get_count_query as well.
		return $this->wpdb->prepare(
			"
			SELECT term_id
			FROM {$taxonomy_table} AS T
			LEFT JOIN $indexable_table AS I
				ON T.term_id = I.object_id
				AND I.object_type = 'term'
				AND I.version = %d
			WHERE I.object_id IS NULL
				AND taxonomy IN ($placeholders)
			$limit_query",
			$replacements
		);
	}
}
                      plugins/wordpress-seo-extended/src/actions/indexing/indexation-action-interface.php                 0000644                 00000001432 15122266557 0024772 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

/**
 * Interface definition of reindexing action for indexables.
 */
interface Indexation_Action_Interface {

	/**
	 * Returns the total number of unindexed objects.
	 *
	 * @return int The total number of unindexed objects.
	 */
	public function get_total_unindexed();

	/**
	 * Indexes a number of objects.
	 *
	 * NOTE: ALWAYS use limits, this method is intended to be called multiple times over several requests.
	 *
	 * For indexing that requires JavaScript simply return the objects that should be indexed.
	 *
	 * @return array The reindexed objects.
	 */
	public function index();

	/**
	 * Returns the number of objects that will be indexed in a single indexing pass.
	 *
	 * @return int The limit.
	 */
	public function get_limit();
}
                                                                                                                                                                                                                                      plugins/wordpress-seo-extended/src/actions/indexing/indexing-complete-action.php                    0000644                 00000001227 15122266557 0024307 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\SEO\Helpers\Indexing_Helper;

/**
 * Indexing action to call when the indexing is completed.
 */
class Indexing_Complete_Action {

	/**
	 * The indexing helper.
	 *
	 * @var Indexing_Helper
	 */
	protected $indexing_helper;

	/**
	 * Indexing_Complete_Action constructor.
	 *
	 * @param Indexing_Helper $indexing_helper The indexing helper.
	 */
	public function __construct( Indexing_Helper $indexing_helper ) {
		$this->indexing_helper = $indexing_helper;
	}

	/**
	 * Wraps up the indexing process.
	 *
	 * @return void
	 */
	public function complete() {
		$this->indexing_helper->complete();
	}
}
                                                                                                                                                                                                                                                                                                                                                                         plugins/wordpress-seo-extended/src/actions/indexing/indexing-prepare-action.php                     0000644                 00000001264 15122266557 0024136 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\SEO\Helpers\Indexing_Helper;

/**
 * Class Indexing_Prepare_Action.
 *
 * Action for preparing the indexing routine.
 */
class Indexing_Prepare_Action {

	/**
	 * The indexing helper.
	 *
	 * @var Indexing_Helper
	 */
	protected $indexing_helper;

	/**
	 * Action for preparing the indexing routine.
	 *
	 * @param Indexing_Helper $indexing_helper The indexing helper.
	 */
	public function __construct(
		Indexing_Helper $indexing_helper
	) {
		$this->indexing_helper = $indexing_helper;
	}

	/**
	 * Prepares the indexing routine.
	 *
	 * @return void
	 */
	public function prepare() {
		$this->indexing_helper->prepare();
	}
}
                                                                                                                                                                                                                                                                                                                                            plugins/wordpress-seo-extended/src/actions/indexing/limited-indexing-action-interface.php           0000644                 00000000763 15122266557 0026070 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

/**
 * Interface definition of a reindexing action for indexables that have a limited unindexed count.
 */
interface Limited_Indexing_Action_Interface {

	/**
	 * Returns a limited number of unindexed posts.
	 *
	 * @param int $limit Limit the maximum number of unindexed posts that are counted.
	 *
	 * @return int|false The limited number of unindexed posts. False if the query fails.
	 */
	public function get_limited_unindexed_count( $limit );
}
             plugins/wordpress-seo-extended/src/actions/indexing/post-link-indexing-action.php                   0000644                 00000007532 15122266557 0024424 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\Lib\Model;
use Yoast\WP\SEO\Helpers\Post_Type_Helper;

/**
 * Reindexing action for post link indexables.
 */
class Post_Link_Indexing_Action extends Abstract_Link_Indexing_Action {

	/**
	 * The transient name.
	 *
	 * @var string
	 */
	public const UNINDEXED_COUNT_TRANSIENT = 'wpseo_unindexed_post_link_count';

	/**
	 * The transient cache key for limited counts.
	 *
	 * @var string
	 */
	public const UNINDEXED_LIMITED_COUNT_TRANSIENT = self::UNINDEXED_COUNT_TRANSIENT . '_limited';

	/**
	 * The post type helper.
	 *
	 * @var Post_Type_Helper
	 */
	protected $post_type_helper;

	/**
	 * Sets the required helper.
	 *
	 * @required
	 *
	 * @param Post_Type_Helper $post_type_helper The post type helper.
	 *
	 * @return void
	 */
	public function set_helper( Post_Type_Helper $post_type_helper ) {
		$this->post_type_helper = $post_type_helper;
	}

	/**
	 * Returns objects to be indexed.
	 *
	 * @return array Objects to be indexed.
	 */
	protected function get_objects() {
		$query = $this->get_select_query( $this->get_limit() );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Function get_select_query returns a prepared query.
		$posts = $this->wpdb->get_results( $query );

		return \array_map(
			static function ( $post ) {
				return (object) [
					'id'      => (int) $post->ID,
					'type'    => 'post',
					'content' => $post->post_content,
				];
			},
			$posts
		);
	}

	/**
	 * Builds a query for counting the number of unindexed post links.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_count_query() {
		$public_post_types = $this->post_type_helper->get_indexable_post_types();
		$indexable_table   = Model::get_table_name( 'Indexable' );
		$links_table       = Model::get_table_name( 'SEO_Links' );

		// Warning: If this query is changed, makes sure to update the query in get_select_query as well.
		return $this->wpdb->prepare(
			"SELECT COUNT(P.ID)
			FROM {$this->wpdb->posts} AS P
			LEFT JOIN $indexable_table AS I
				ON P.ID = I.object_id
				AND I.link_count IS NOT NULL
				AND I.object_type = 'post'
			LEFT JOIN $links_table AS L
				ON L.post_id = P.ID
				AND L.target_indexable_id IS NULL
				AND L.type = 'internal'
				AND L.target_post_id IS NOT NULL
				AND L.target_post_id != 0
			WHERE ( I.object_id IS NULL OR L.post_id IS NOT NULL )
				AND P.post_status = 'publish'
				AND P.post_type IN (" . \implode( ', ', \array_fill( 0, \count( $public_post_types ), '%s' ) ) . ')',
			$public_post_types
		);
	}

	/**
	 * Builds a query for selecting the ID's of unindexed post links.
	 *
	 * @param int|false $limit The maximum number of post link IDs to return.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_select_query( $limit = false ) {
		$public_post_types = $this->post_type_helper->get_indexable_post_types();
		$indexable_table   = Model::get_table_name( 'Indexable' );
		$links_table       = Model::get_table_name( 'SEO_Links' );
		$replacements      = $public_post_types;

		$limit_query = '';
		if ( $limit ) {
			$limit_query    = 'LIMIT %d';
			$replacements[] = $limit;
		}

		// Warning: If this query is changed, makes sure to update the query in get_count_query as well.
		return $this->wpdb->prepare(
			"
			SELECT P.ID, P.post_content
			FROM {$this->wpdb->posts} AS P
			LEFT JOIN $indexable_table AS I
				ON P.ID = I.object_id
				AND I.link_count IS NOT NULL
				AND I.object_type = 'post'
			LEFT JOIN $links_table AS L
				ON L.post_id = P.ID
				AND L.target_indexable_id IS NULL
				AND L.type = 'internal'
				AND L.target_post_id IS NOT NULL
				AND L.target_post_id != 0
			WHERE ( I.object_id IS NULL OR L.post_id IS NOT NULL )
				AND P.post_status = 'publish'
				AND P.post_type IN (" . \implode( ', ', \array_fill( 0, \count( $public_post_types ), '%s' ) ) . ")
			$limit_query",
			$replacements
		);
	}
}
                                                                                                                                                                      plugins/wordpress-seo-extended/src/actions/indexing/term-link-indexing-action.php                   0000644                 00000006442 15122266557 0024405 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Indexing;

use Yoast\WP\Lib\Model;
use Yoast\WP\SEO\Helpers\Taxonomy_Helper;

/**
 * Reindexing action for term link indexables.
 */
class Term_Link_Indexing_Action extends Abstract_Link_Indexing_Action {

	/**
	 * The transient name.
	 *
	 * @var string
	 */
	public const UNINDEXED_COUNT_TRANSIENT = 'wpseo_unindexed_term_link_count';

	/**
	 * The transient cache key for limited counts.
	 *
	 * @var string
	 */
	public const UNINDEXED_LIMITED_COUNT_TRANSIENT = self::UNINDEXED_COUNT_TRANSIENT . '_limited';

	/**
	 * The post type helper.
	 *
	 * @var Taxonomy_Helper
	 */
	protected $taxonomy_helper;

	/**
	 * Sets the required helper.
	 *
	 * @required
	 *
	 * @param Taxonomy_Helper $taxonomy_helper The taxonomy helper.
	 *
	 * @return void
	 */
	public function set_helper( Taxonomy_Helper $taxonomy_helper ) {
		$this->taxonomy_helper = $taxonomy_helper;
	}

	/**
	 * Returns objects to be indexed.
	 *
	 * @return array Objects to be indexed.
	 */
	protected function get_objects() {
		$query = $this->get_select_query( $this->get_limit() );

		// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Function get_select_query returns a prepared query.
		$terms = $this->wpdb->get_results( $query );

		return \array_map(
			static function ( $term ) {
				return (object) [
					'id'      => (int) $term->term_id,
					'type'    => 'term',
					'content' => $term->description,
				];
			},
			$terms
		);
	}

	/**
	 * Builds a query for counting the number of unindexed term links.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_count_query() {
		$public_taxonomies = $this->taxonomy_helper->get_indexable_taxonomies();
		$placeholders      = \implode( ', ', \array_fill( 0, \count( $public_taxonomies ), '%s' ) );
		$indexable_table   = Model::get_table_name( 'Indexable' );

		// Warning: If this query is changed, makes sure to update the query in get_select_query as well.
		return $this->wpdb->prepare(
			"
			SELECT COUNT(T.term_id)
			FROM {$this->wpdb->term_taxonomy} AS T
			LEFT JOIN $indexable_table AS I
				ON T.term_id = I.object_id
				AND I.object_type = 'term'
				AND I.link_count IS NOT NULL
			WHERE I.object_id IS NULL
				AND T.taxonomy IN ($placeholders)",
			$public_taxonomies
		);
	}

	/**
	 * Builds a query for selecting the ID's of unindexed term links.
	 *
	 * @param int|false $limit The maximum number of term link IDs to return.
	 *
	 * @return string The prepared query string.
	 */
	protected function get_select_query( $limit = false ) {
		$public_taxonomies = $this->taxonomy_helper->get_indexable_taxonomies();

		$indexable_table = Model::get_table_name( 'Indexable' );
		$replacements    = $public_taxonomies;

		$limit_query = '';
		if ( $limit ) {
			$limit_query    = 'LIMIT %d';
			$replacements[] = $limit;
		}

		// Warning: If this query is changed, makes sure to update the query in get_count_query as well.
		return $this->wpdb->prepare(
			"
			SELECT T.term_id, T.description
			FROM {$this->wpdb->term_taxonomy} AS T
			LEFT JOIN $indexable_table AS I
				ON T.term_id = I.object_id
				AND I.object_type = 'term'
				AND I.link_count IS NOT NULL
			WHERE I.object_id IS NULL
				AND T.taxonomy IN (" . \implode( ', ', \array_fill( 0, \count( $public_taxonomies ), '%s' ) ) . ")
			$limit_query",
			$replacements
		);
	}
}
                                                                                                                                                                                                                              plugins/wordpress-seo-extended/src/actions/integrations-action.php                                  0000644                 00000002366 15122266557 0021602 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions;

use Yoast\WP\SEO\Helpers\Options_Helper;

/**
 * Class Integrations_Action.
 */
class Integrations_Action {

	/**
	 * The Options_Helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * Integrations_Action constructor.
	 *
	 * @param Options_Helper $options_helper The WPSEO options helper.
	 */
	public function __construct( Options_Helper $options_helper ) {
		$this->options_helper = $options_helper;
	}

	/**
	 * Sets an integration state.
	 *
	 * @param string $integration_name The name of the integration to activate/deactivate.
	 * @param bool   $value            The value to store.
	 *
	 * @return object The response object.
	 */
	public function set_integration_active( $integration_name, $value ) {
		$option_name  = $integration_name . '_integration_active';
		$success      = true;
		$option_value = $this->options_helper->get( $option_name );

		if ( $option_value !== $value ) {
			$success = $this->options_helper->set( $option_name, $value );
		}

		if ( $success ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}
		return (object) [
			'success' => false,
			'status'  => 500,
			'error'   => 'Could not save the option in the database',
		];
	}
}
                                                                                                                                                                                                                                                                          plugins/wordpress-seo-extended/src/actions/semrush/semrush-login-action.php                         0000644                 00000002360 15122266557 0023350 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\SEMrush;

use Yoast\WP\SEO\Config\SEMrush_Client;
use Yoast\WP\SEO\Exceptions\OAuth\Authentication_Failed_Exception;

/**
 * Class SEMrush_Login_Action
 */
class SEMrush_Login_Action {

	/**
	 * The SEMrush_Client instance.
	 *
	 * @var SEMrush_Client
	 */
	protected $client;

	/**
	 * SEMrush_Login_Action constructor.
	 *
	 * @param SEMrush_Client $client The API client.
	 */
	public function __construct( SEMrush_Client $client ) {
		$this->client = $client;
	}

	/**
	 * Authenticates with SEMrush to request the necessary tokens.
	 *
	 * @param string $code The authentication code to use to request a token with.
	 *
	 * @return object The response object.
	 */
	public function authenticate( $code ) {
		// Code has already been validated at this point. No need to do that again.
		try {
			$tokens = $this->client->request_tokens( $code );

			return (object) [
				'tokens' => $tokens->to_array(),
				'status' => 200,
			];
		} catch ( Authentication_Failed_Exception $e ) {
			return $e->get_response();
		}
	}

	/**
	 * Performs the login request, if necessary.
	 *
	 * @return void
	 */
	public function login() {
		if ( $this->client->has_valid_tokens() ) {
			return;
		}

		// Prompt with login screen.
	}
}
                                                                                                                                                                                                                                                                                plugins/wordpress-seo-extended/src/actions/semrush/semrush-options-action.php                       0000644                 00000002140 15122266557 0023727 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\SEMrush;

use Yoast\WP\SEO\Helpers\Options_Helper;

/**
 * Class SEMrush_Options_Action
 */
class SEMrush_Options_Action {

	/**
	 * The Options_Helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * SEMrush_Options_Action constructor.
	 *
	 * @param Options_Helper $options_helper The WPSEO options helper.
	 */
	public function __construct( Options_Helper $options_helper ) {
		$this->options_helper = $options_helper;
	}

	/**
	 * Stores SEMrush country code in the WPSEO options.
	 *
	 * @param string $country_code The country code to store.
	 *
	 * @return object The response object.
	 */
	public function set_country_code( $country_code ) {
		// The country code has already been validated at this point. No need to do that again.
		$success = $this->options_helper->set( 'semrush_country_code', $country_code );

		if ( $success ) {
			return (object) [
				'success' => true,
				'status'  => 200,
			];
		}
		return (object) [
			'success' => false,
			'status'  => 500,
			'error'   => 'Could not save option in the database',
		];
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                plugins/wordpress-seo-extended/src/actions/semrush/semrush-phrases-action.php                       0000644                 00000004270 15122266557 0023707 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\SEMrush;

use Exception;
use Yoast\WP\SEO\Config\SEMrush_Client;

/**
 * Class SEMrush_Phrases_Action
 */
class SEMrush_Phrases_Action {

	/**
	 * The transient cache key.
	 */
	public const TRANSIENT_CACHE_KEY = 'wpseo_semrush_related_keyphrases_%s_%s';

	/**
	 * The SEMrush keyphrase URL.
	 *
	 * @var string
	 */
	public const KEYPHRASES_URL = 'https://oauth.semrush.com/api/v1/keywords/phrase_fullsearch';

	/**
	 * The SEMrush_Client instance.
	 *
	 * @var SEMrush_Client
	 */
	protected $client;

	/**
	 * SEMrush_Phrases_Action constructor.
	 *
	 * @param SEMrush_Client $client The API client.
	 */
	public function __construct( SEMrush_Client $client ) {
		$this->client = $client;
	}

	/**
	 * Gets the related keyphrases and data based on the passed keyphrase and database country code.
	 *
	 * @param string $keyphrase The keyphrase to search for.
	 * @param string $database  The database's country code.
	 *
	 * @return object The response object.
	 */
	public function get_related_keyphrases( $keyphrase, $database ) {
		try {
			$transient_key = \sprintf( static::TRANSIENT_CACHE_KEY, $keyphrase, $database );
			$transient     = \get_transient( $transient_key );

			if ( $transient !== false ) {
				return $this->to_result_object( $transient );
			}

			$options = [
				'params' => [
					'phrase'         => $keyphrase,
					'database'       => $database,
					'export_columns' => 'Ph,Nq,Td',
					'display_limit'  => 10,
					'display_offset' => 0,
					'display_sort'   => 'nq_desc',
					'display_filter' => '%2B|Nq|Lt|1000',
				],
			];

			$results = $this->client->get( self::KEYPHRASES_URL, $options );

			\set_transient( $transient_key, $results, \DAY_IN_SECONDS );

			return $this->to_result_object( $results );
		} catch ( Exception $e ) {
			return (object) [
				'error'  => $e->getMessage(),
				'status' => $e->getCode(),
			];
		}
	}

	/**
	 * Converts the passed dataset to an object.
	 *
	 * @param array $result The result dataset to convert to an object.
	 *
	 * @return object The result object.
	 */
	protected function to_result_object( $result ) {
		return (object) [
			'results' => $result['data'],
			'status'  => $result['status'],
		];
	}
}
                                                                                                                                                                                                                                                                                                                                        plugins/wordpress-seo-extended/src/actions/wincher/wincher-account-action.php                       0000644                 00000004743 15122266557 0023625 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Wincher;

use Exception;
use Yoast\WP\SEO\Config\Wincher_Client;
use Yoast\WP\SEO\Helpers\Options_Helper;

/**
 * Class Wincher_Account_Action
 */
class Wincher_Account_Action {

	public const ACCOUNT_URL          = 'https://api.wincher.com/beta/account';
	public const UPGRADE_CAMPAIGN_URL = 'https://api.wincher.com/v1/yoast/upgrade-campaign';

	/**
	 * The Wincher_Client instance.
	 *
	 * @var Wincher_Client
	 */
	protected $client;

	/**
	 * The Options_Helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * Wincher_Account_Action constructor.
	 *
	 * @param Wincher_Client $client         The API client.
	 * @param Options_Helper $options_helper The options helper.
	 */
	public function __construct( Wincher_Client $client, Options_Helper $options_helper ) {
		$this->client         = $client;
		$this->options_helper = $options_helper;
	}

	/**
	 * Checks the account limit for tracking keyphrases.
	 *
	 * @return object The response object.
	 */
	public function check_limit() {
		// Code has already been validated at this point. No need to do that again.
		try {
			$results = $this->client->get( self::ACCOUNT_URL );

			$usage   = $results['limits']['keywords']['usage'];
			$limit   = $results['limits']['keywords']['limit'];
			$history = $results['limits']['history_days'];

			return (object) [
				'canTrack'    => \is_null( $limit ) || $usage < $limit,
				'limit'       => $limit,
				'usage'       => $usage,
				'historyDays' => $history,
				'status'      => 200,
			];
		} catch ( Exception $e ) {
			return (object) [
				'status' => $e->getCode(),
				'error'  => $e->getMessage(),
			];
		}
	}

	/**
	 * Gets the upgrade campaign.
	 *
	 * @return object The response object.
	 */
	public function get_upgrade_campaign() {
		try {
			$result   = $this->client->get( self::UPGRADE_CAMPAIGN_URL );
			$type     = ( $result['type'] ?? null );
			$months   = ( $result['months'] ?? null );
			$discount = ( $result['value'] ?? null );

			// We display upgrade discount only if it's a rate discount and positive months/discount.
			if ( $type === 'RATE' && $months && $discount ) {

				return (object) [
					'discount'  => $discount,
					'months'    => $months,
					'status'    => 200,
				];
			}

			return (object) [
				'discount'  => null,
				'months'    => null,
				'status'    => 200,
			];
		} catch ( Exception $e ) {
			return (object) [
				'status' => $e->getCode(),
				'error'  => $e->getMessage(),
			];
		}
	}
}
                             plugins/wordpress-seo-extended/src/actions/wincher/wincher-keyphrases-action.php                    0000644                 00000022227 15122266557 0024344 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Wincher;

use Exception;
use WP_Post;
use WPSEO_Utils;
use Yoast\WP\SEO\Config\Wincher_Client;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Repositories\Indexable_Repository;

/**
 * Class Wincher_Keyphrases_Action
 */
class Wincher_Keyphrases_Action {

	/**
	 * The Wincher keyphrase URL for bulk addition.
	 *
	 * @var string
	 */
	public const KEYPHRASES_ADD_URL = 'https://api.wincher.com/beta/websites/%s/keywords/bulk';

	/**
	 * The Wincher tracked keyphrase retrieval URL.
	 *
	 * @var string
	 */
	public const KEYPHRASES_URL = 'https://api.wincher.com/beta/yoast/%s';

	/**
	 * The Wincher delete tracked keyphrase URL.
	 *
	 * @var string
	 */
	public const KEYPHRASE_DELETE_URL = 'https://api.wincher.com/beta/websites/%s/keywords/%s';

	/**
	 * The Wincher_Client instance.
	 *
	 * @var Wincher_Client
	 */
	protected $client;

	/**
	 * The Options_Helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * The Indexable_Repository instance.
	 *
	 * @var Indexable_Repository
	 */
	protected $indexable_repository;

	/**
	 * Wincher_Keyphrases_Action constructor.
	 *
	 * @param Wincher_Client       $client               The API client.
	 * @param Options_Helper       $options_helper       The options helper.
	 * @param Indexable_Repository $indexable_repository The indexables repository.
	 */
	public function __construct(
		Wincher_Client $client,
		Options_Helper $options_helper,
		Indexable_Repository $indexable_repository
	) {
		$this->client               = $client;
		$this->options_helper       = $options_helper;
		$this->indexable_repository = $indexable_repository;
	}

	/**
	 * Sends the tracking API request for one or more keyphrases.
	 *
	 * @param string|array $keyphrases One or more keyphrases that should be tracked.
	 * @param Object       $limits     The limits API call response data.
	 *
	 * @return Object The reponse object.
	 */
	public function track_keyphrases( $keyphrases, $limits ) {
		try {
			$endpoint = \sprintf(
				self::KEYPHRASES_ADD_URL,
				$this->options_helper->get( 'wincher_website_id' )
			);

			// Enforce arrrays to ensure a consistent way of preparing the request.
			if ( ! \is_array( $keyphrases ) ) {
				$keyphrases = [ $keyphrases ];
			}

			// Calculate if the user would exceed their limit.
			// phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase -- To ensure JS code style, this can be ignored.
			if ( ! $limits->canTrack || $this->would_exceed_limits( $keyphrases, $limits ) ) {
				$response = [
					'limit'  => $limits->limit,
					'error'  => 'Account limit exceeded',
					'status' => 400,
				];

				return $this->to_result_object( $response );
			}

			$formatted_keyphrases = \array_values(
				\array_map(
					static function ( $keyphrase ) {
						return [
							'keyword' => $keyphrase,
							'groups'  => [],
						];
					},
					$keyphrases
				)
			);

			$results = $this->client->post( $endpoint, WPSEO_Utils::format_json_encode( $formatted_keyphrases ) );

			if ( ! \array_key_exists( 'data', $results ) ) {
				return $this->to_result_object( $results );
			}

			// The endpoint returns a lot of stuff that we don't want/need.
			$results['data'] = \array_map(
				static function ( $keyphrase ) {
					return [
						'id'         => $keyphrase['id'],
						'keyword'    => $keyphrase['keyword'],
					];
				},
				$results['data']
			);

			$results['data'] = \array_combine(
				\array_column( $results['data'], 'keyword' ),
				\array_values( $results['data'] )
			);

			return $this->to_result_object( $results );
		} catch ( Exception $e ) {
			return (object) [
				'error'  => $e->getMessage(),
				'status' => $e->getCode(),
			];
		}
	}

	/**
	 * Sends an untrack request for the passed keyword ID.
	 *
	 * @param int $keyphrase_id The ID of the keyphrase to untrack.
	 *
	 * @return object The response object.
	 */
	public function untrack_keyphrase( $keyphrase_id ) {
		try {
			$endpoint = \sprintf(
				self::KEYPHRASE_DELETE_URL,
				$this->options_helper->get( 'wincher_website_id' ),
				$keyphrase_id
			);

			$this->client->delete( $endpoint );

			return (object) [
				'status' => 200,
			];
		} catch ( Exception $e ) {
			return (object) [
				'error'  => $e->getMessage(),
				'status' => $e->getCode(),
			];
		}
	}

	/**
	 * Gets the keyphrase data for the passed keyphrases.
	 * Retrieves all available data if no keyphrases are provided.
	 *
	 * @param array|null  $used_keyphrases The currently used keyphrases. Optional.
	 * @param string|null $permalink       The current permalink. Optional.
	 * @param string|null $start_at        The position start date. Optional.
	 *
	 * @return object The keyphrase chart data.
	 */
	public function get_tracked_keyphrases( $used_keyphrases = null, $permalink = null, $start_at = null ) {
		try {
			if ( $used_keyphrases === null ) {
				$used_keyphrases = $this->collect_all_keyphrases();
			}

			// If we still have no keyphrases the API will return an error, so
			// don't even bother sending a request.
			if ( empty( $used_keyphrases ) ) {
				return $this->to_result_object(
					[
						'data'   => [],
						'status' => 200,
					]
				);
			}

			$endpoint = \sprintf(
				self::KEYPHRASES_URL,
				$this->options_helper->get( 'wincher_website_id' )
			);

			$results = $this->client->post(
				$endpoint,
				WPSEO_Utils::format_json_encode(
					[
						'keywords' => $used_keyphrases,
						'url'      => $permalink,
						'start_at' => $start_at,
					]
				),
				[
					'timeout' => 60,
				]
			);

			if ( ! \array_key_exists( 'data', $results ) ) {
				return $this->to_result_object( $results );
			}

			$results['data'] = $this->filter_results_by_used_keyphrases( $results['data'], $used_keyphrases );

			// Extract the positional data and assign it to the keyphrase.
			$results['data'] = \array_combine(
				\array_column( $results['data'], 'keyword' ),
				\array_values( $results['data'] )
			);

			return $this->to_result_object( $results );
		} catch ( Exception $e ) {
			return (object) [
				'error'  => $e->getMessage(),
				'status' => $e->getCode(),
			];
		}
	}

	/**
	 * Collects the keyphrases associated with the post.
	 *
	 * @param WP_Post $post The post object.
	 *
	 * @return array The keyphrases.
	 */
	public function collect_keyphrases_from_post( $post ) {
		$keyphrases        = [];
		$primary_keyphrase = $this->indexable_repository
			->query()
			->select( 'primary_focus_keyword' )
			->where( 'object_id', $post->ID )
			->find_one();

		if ( $primary_keyphrase ) {
			$keyphrases[] = $primary_keyphrase->primary_focus_keyword;
		}

		/**
		 * Filters the keyphrases collected by the Wincher integration from the post.
		 *
		 * @param array $keyphrases The keyphrases array.
		 * @param int   $post_id    The ID of the post.
		 */
		return \apply_filters( 'wpseo_wincher_keyphrases_from_post', $keyphrases, $post->ID );
	}

	/**
	 * Collects all keyphrases known to Yoast.
	 *
	 * @return array
	 */
	protected function collect_all_keyphrases() {
		// Collect primary keyphrases first.
		$keyphrases = \array_column(
			$this->indexable_repository
				->query()
				->select( 'primary_focus_keyword' )
				->where_not_null( 'primary_focus_keyword' )
				->where( 'object_type', 'post' )
				->where_not_equal( 'post_status', 'trash' )
				->distinct()
				->find_array(),
			'primary_focus_keyword'
		);

		/**
		 * Filters the keyphrases collected by the Wincher integration from all the posts.
		 *
		 * @param array $keyphrases The keyphrases array.
		 */
		$keyphrases = \apply_filters( 'wpseo_wincher_all_keyphrases', $keyphrases );

		// Filter out empty entries.
		return \array_filter( $keyphrases );
	}

	/**
	 * Filters the results based on the passed keyphrases.
	 *
	 * @param array $results         The results to filter.
	 * @param array $used_keyphrases The used keyphrases.
	 *
	 * @return array The filtered results.
	 */
	protected function filter_results_by_used_keyphrases( $results, $used_keyphrases ) {
		return \array_filter(
			$results,
			static function ( $result ) use ( $used_keyphrases ) {
				return \in_array( $result['keyword'], \array_map( 'strtolower', $used_keyphrases ), true );
			}
		);
	}

	/**
	 * Determines whether the amount of keyphrases would mean the user exceeds their account limits.
	 *
	 * @param string|array $keyphrases The keyphrases to be added.
	 * @param object       $limits     The current account limits.
	 *
	 * @return bool Whether the limit is exceeded.
	 */
	protected function would_exceed_limits( $keyphrases, $limits ) {
		if ( ! \is_array( $keyphrases ) ) {
			$keyphrases = [ $keyphrases ];
		}

		if ( \is_null( $limits->limit ) ) {
			return false;
		}

		return ( \count( $keyphrases ) + $limits->usage ) > $limits->limit;
	}

	/**
	 * Converts the passed dataset to an object.
	 *
	 * @param array $result The result dataset to convert to an object.
	 *
	 * @return object The result object.
	 */
	protected function to_result_object( $result ) {
		if ( \array_key_exists( 'data', $result ) ) {
			$result['results'] = (object) $result['data'];

			unset( $result['data'] );
		}

		if ( \array_key_exists( 'message', $result ) ) {
			$result['error'] = $result['message'];

			unset( $result['message'] );
		}

		return (object) $result;
	}
}
                                                                                                                                                                                                                                                                                                                                                                         plugins/wordpress-seo-extended/src/actions/wincher/wincher-login-action.php                         0000644                 00000003336 15122266557 0023276 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Actions\Wincher;

use Yoast\WP\SEO\Config\Wincher_Client;
use Yoast\WP\SEO\Exceptions\OAuth\Authentication_Failed_Exception;
use Yoast\WP\SEO\Helpers\Options_Helper;

/**
 * Class Wincher_Login_Action
 */
class Wincher_Login_Action {

	/**
	 * The Wincher_Client instance.
	 *
	 * @var Wincher_Client
	 */
	protected $client;

	/**
	 * The Options_Helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * Wincher_Login_Action constructor.
	 *
	 * @param Wincher_Client $client         The API client.
	 * @param Options_Helper $options_helper The options helper.
	 */
	public function __construct( Wincher_Client $client, Options_Helper $options_helper ) {
		$this->client         = $client;
		$this->options_helper = $options_helper;
	}

	/**
	 * Returns the authorization URL.
	 *
	 * @return object The response object.
	 */
	public function get_authorization_url() {
		return (object) [
			'status'    => 200,
			'url'       => $this->client->get_authorization_url(),
		];
	}

	/**
	 * Authenticates with Wincher to request the necessary tokens.
	 *
	 * @param string $code       The authentication code to use to request a token with.
	 * @param string $website_id The website id associated with the code.
	 *
	 * @return object The response object.
	 */
	public function authenticate( $code, $website_id ) {
		// Code has already been validated at this point. No need to do that again.
		try {
			$tokens = $this->client->request_tokens( $code );
			$this->options_helper->set( 'wincher_website_id', $website_id );

			return (object) [
				'tokens' => $tokens->to_array(),
				'status' => 200,
			];
		} catch ( Authentication_Failed_Exception $e ) {
			return $e->get_response();
		}
	}
}
                                                                                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/src/analytics/application/missing-indexables-collector.php           0000644                 00000004521 15122266557 0026217 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Analytics\Application;

use WPSEO_Collection;
use Yoast\WP\SEO\Actions\Indexing\Indexation_Action_Interface;
use Yoast\WP\SEO\Analytics\Domain\Missing_Indexable_Bucket;
use Yoast\WP\SEO\Analytics\Domain\Missing_Indexable_Count;

/**
 * Manages the collection of the missing indexable data.
 *
 * @makePublic
 */
class Missing_Indexables_Collector implements WPSEO_Collection {

	/**
	 * All the indexation actions.
	 *
	 * @var array<Indexation_Action_Interface>
	 */
	private $indexation_actions;

	/**
	 * The collector constructor.
	 *
	 * @param Indexation_Action_Interface ...$indexation_actions All the Indexation actions.
	 */
	public function __construct( Indexation_Action_Interface ...$indexation_actions ) {
		$this->indexation_actions = $indexation_actions;
		$this->add_additional_indexing_actions();
	}

	/**
	 * Gets the data for the tracking collector.
	 *
	 * @return array The list of missing indexables.
	 */
	public function get() {
		$missing_indexable_bucket = new Missing_Indexable_Bucket();
		foreach ( $this->indexation_actions as $indexation_action ) {
			$missing_indexable_count = new Missing_Indexable_Count( \get_class( $indexation_action ), $indexation_action->get_total_unindexed() );
			$missing_indexable_bucket->add_missing_indexable_count( $missing_indexable_count );
		}

		return [ 'missing_indexables' => $missing_indexable_bucket->to_array() ];
	}

	/**
	 * Adds additional indexing actions to count from the 'wpseo_indexable_collector_add_indexation_actions' filter.
	 *
	 * @return void
	 */
	private function add_additional_indexing_actions() {
		/**
		 * Filter: Adds the possibility to add additional indexation actions to be included in the count routine.
		 *
		 * @internal
		 * @param Indexation_Action_Interface $actions This filter expects a list of Indexation_Action_Interface instances
		 *                                             and expects only Indexation_Action_Interface implementations to be
		 *                                             added to the list.
		 */
		$indexing_actions = (array) \apply_filters( 'wpseo_indexable_collector_add_indexation_actions', $this->indexation_actions );

		$this->indexation_actions = \array_filter(
			$indexing_actions,
			static function ( $indexing_action ) {
				return \is_a( $indexing_action, Indexation_Action_Interface::class );
			}
		);
	}
}
                                                                                                                                                                               plugins/wordpress-seo-extended/src/analytics/application/to-be-cleaned-indexables-collector.php     0000644                 00000010124 15122266557 0027141 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Analytics\Application;

use WPSEO_Collection;
use Yoast\WP\SEO\Analytics\Domain\To_Be_Cleaned_Indexable_Bucket;
use Yoast\WP\SEO\Analytics\Domain\To_Be_Cleaned_Indexable_Count;
use Yoast\WP\SEO\Repositories\Indexable_Cleanup_Repository;

/**
 * Collects data about to-be-cleaned indexables.
 *
 * @makePublic
 */
class To_Be_Cleaned_Indexables_Collector implements WPSEO_Collection {

	/**
	 * The cleanup query repository.
	 *
	 * @var Indexable_Cleanup_Repository
	 */
	private $indexable_cleanup_repository;

	/**
	 * The constructor.
	 *
	 * @param Indexable_Cleanup_Repository $indexable_cleanup_repository The Indexable cleanup repository.
	 */
	public function __construct( Indexable_Cleanup_Repository $indexable_cleanup_repository ) {
		$this->indexable_cleanup_repository = $indexable_cleanup_repository;
	}

	/**
	 * Gets the data for the collector.
	 *
	 * @return array
	 */
	public function get() {
		$to_be_cleaned_indexable_bucket = new To_Be_Cleaned_Indexable_Bucket();
		$cleanup_tasks                  = [
			'indexables_with_post_object_type_and_shop_order_object_sub_type' => $this->indexable_cleanup_repository->count_indexables_with_object_type_and_object_sub_type( 'post', 'shop_order' ),
			'indexables_with_auto-draft_post_status'            => $this->indexable_cleanup_repository->count_indexables_with_post_status( 'auto-draft' ),
			'indexables_for_non_publicly_viewable_post'         => $this->indexable_cleanup_repository->count_indexables_for_non_publicly_viewable_post(),
			'indexables_for_non_publicly_viewable_taxonomies'   => $this->indexable_cleanup_repository->count_indexables_for_non_publicly_viewable_taxonomies(),
			'indexables_for_non_publicly_viewable_post_type_archive_pages' => $this->indexable_cleanup_repository->count_indexables_for_non_publicly_post_type_archive_pages(),
			'indexables_for_authors_archive_disabled'           => $this->indexable_cleanup_repository->count_indexables_for_authors_archive_disabled(),
			'indexables_for_authors_without_archive'            => $this->indexable_cleanup_repository->count_indexables_for_authors_without_archive(),
			'indexables_for_object_type_and_source_table_users' => $this->indexable_cleanup_repository->count_indexables_for_orphaned_users(),
			'indexables_for_object_type_and_source_table_posts' => $this->indexable_cleanup_repository->count_indexables_for_object_type_and_source_table( 'posts', 'ID', 'post' ),
			'indexables_for_object_type_and_source_table_terms' => $this->indexable_cleanup_repository->count_indexables_for_object_type_and_source_table( 'terms', 'term_id', 'term' ),
			'orphaned_from_table_indexable_hierarchy'           => $this->indexable_cleanup_repository->count_orphaned_from_table( 'Indexable_Hierarchy', 'indexable_id' ),
			'orphaned_from_table_indexable_id'                  => $this->indexable_cleanup_repository->count_orphaned_from_table( 'SEO_Links', 'indexable_id' ),
			'orphaned_from_table_target_indexable_id'           => $this->indexable_cleanup_repository->count_orphaned_from_table( 'SEO_Links', 'target_indexable_id' ),
		];

		foreach ( $cleanup_tasks as $name => $count ) {
			if ( $count !== null ) {
				$count_object = new To_Be_Cleaned_Indexable_Count( $name, $count );
				$to_be_cleaned_indexable_bucket->add_to_be_cleaned_indexable_count( $count_object );
			}
		}

		$this->add_additional_counts( $to_be_cleaned_indexable_bucket );

		return [ 'to_be_cleaned_indexables' => $to_be_cleaned_indexable_bucket->to_array() ];
	}

	/**
	 * Allows additional tasks to be added via the 'wpseo_add_cleanup_counts_to_indexable_bucket' action.
	 *
	 * @param To_Be_Cleaned_Indexable_Bucket $to_be_cleaned_indexable_bucket The current bucket with data.
	 *
	 * @return void
	 */
	private function add_additional_counts( $to_be_cleaned_indexable_bucket ) {
		/**
		 * Action: Adds the possibility to add additional to be cleaned objects.
		 *
		 * @internal
		 * @param To_Be_Cleaned_Indexable_Bucket $bucket An indexable cleanup bucket. New values are instances of To_Be_Cleaned_Indexable_Count.
		 */
		\do_action( 'wpseo_add_cleanup_counts_to_indexable_bucket', $to_be_cleaned_indexable_bucket );
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                            plugins/wordpress-seo-extended/src/analytics/domain/missing-indexable-bucket.php                    0000644                 00000001731 15122266557 0024267 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Analytics\Domain;

/**
 * A collection domain object.
 */
class Missing_Indexable_Bucket {

	/**
	 * All the missing indexable count objects.
	 *
	 * @var array<Missing_Indexable_Count>
	 */
	private $missing_indexable_counts;

	/**
	 * The constructor.
	 */
	public function __construct() {
		$this->missing_indexable_counts = [];
	}

	/**
	 * Adds a missing indexable count object to this bucket.
	 *
	 * @param Missing_Indexable_Count $missing_indexable_count The missing indexable count object.
	 *
	 * @return void
	 */
	public function add_missing_indexable_count( Missing_Indexable_Count $missing_indexable_count ): void {
		$this->missing_indexable_counts[] = $missing_indexable_count;
	}

	/**
	 * Returns the array representation of all indexable counts.
	 *
	 * @return array
	 */
	public function to_array() {
		return \array_map(
			static function ( $item ) {
				return $item->to_array();
			},
			$this->missing_indexable_counts
		);
	}
}
                                       plugins/wordpress-seo-extended/src/analytics/domain/missing-indexable-count.php                     0000644                 00000002336 15122266557 0024144 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Analytics\Domain;

/**
 * Domain object that holds indexable count information.
 */
class Missing_Indexable_Count {

	/**
	 * The indexable type that is represented by this.
	 *
	 * @var string
	 */
	private $indexable_type;

	/**
	 * The amount of missing indexables.
	 *
	 * @var int
	 */
	private $count;

	/**
	 * The constructor.
	 *
	 * @param string $indexable_type The indexable type that is represented by this.
	 * @param int    $count          The amount of missing indexables.
	 */
	public function __construct( $indexable_type, $count ) {
		$this->indexable_type = $indexable_type;
		$this->count          = $count;
	}

	/**
	 * Returns an array representation of the data.
	 *
	 * @return array Returns both values in an array format.
	 */
	public function to_array() {
		return [
			'indexable_type' => $this->get_indexable_type(),
			'count'          => $this->get_count(),
		];
	}

	/**
	 * Gets the indexable type.
	 *
	 * @return string Returns the indexable type.
	 */
	public function get_indexable_type() {
		return $this->indexable_type;
	}

	/**
	 * Gets the count.
	 *
	 * @return int Returns the amount of missing indexables.
	 */
	public function get_count() {
		return $this->count;
	}
}
                                                                                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/src/analytics/domain/to-be-cleaned-indexable-bucket.php              0000644                 00000002062 15122266557 0025213 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Analytics\Domain;

/**
 * A collection domain object.
 */
class To_Be_Cleaned_Indexable_Bucket {

	/**
	 * All the to be cleaned indexable count objects.
	 *
	 * @var array<To_Be_Cleaned_Indexable_Count>
	 */
	private $to_be_cleaned_indexable_counts;

	/**
	 * The constructor.
	 */
	public function __construct() {
		$this->to_be_cleaned_indexable_counts = [];
	}

	/**
	 * Adds a 'to be cleaned' indexable count object to this bucket.
	 *
	 * @param To_Be_Cleaned_Indexable_Count $to_be_cleaned_indexable_counts The to be cleaned indexable count object.
	 *
	 * @return void
	 */
	public function add_to_be_cleaned_indexable_count( To_Be_Cleaned_Indexable_Count $to_be_cleaned_indexable_counts ) {
		$this->to_be_cleaned_indexable_counts[] = $to_be_cleaned_indexable_counts;
	}

	/**
	 * Returns the array representation of all indexable counts.
	 *
	 * @return array
	 */
	public function to_array() {
		return \array_map(
			static function ( $item ) {
				return $item->to_array();
			},
			$this->to_be_cleaned_indexable_counts
		);
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/src/analytics/domain/to-be-cleaned-indexable-count.php               0000644                 00000002231 15122266557 0025064 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Analytics\Domain;

/**
 * The to be cleaned indexable domain object.
 */
class To_Be_Cleaned_Indexable_Count {

	/**
	 * The cleanup task that is represented by this.
	 *
	 * @var string
	 */
	private $cleanup_name;

	/**
	 * The amount of missing indexables.
	 *
	 * @var int
	 */
	private $count;

	/**
	 * The constructor.
	 *
	 * @param string $cleanup_name The indexable type that is represented by this.
	 * @param int    $count        The amount of missing indexables.
	 */
	public function __construct( $cleanup_name, $count ) {
		$this->cleanup_name = $cleanup_name;
		$this->count        = $count;
	}

	/**
	 * Returns an array representation of the data.
	 *
	 * @return array Returns both values in an array format.
	 */
	public function to_array() {
		return [
			'cleanup_name' => $this->get_cleanup_name(),
			'count'        => $this->get_count(),
		];
	}

	/**
	 * Gets the name.
	 *
	 * @return string
	 */
	public function get_cleanup_name() {
		return $this->cleanup_name;
	}

	/**
	 * Gets the count.
	 *
	 * @return int Returns the amount of missing indexables.
	 */
	public function get_count() {
		return $this->count;
	}
}
                                                                                                                                                                                                                                                                                                                                                                       wordpress-seo-extended/src/analytics/user-interface/last-completed-indexation-integration.php       0000644                 00000003102 15122266557 0030370 0                                                                                                    ustar 00                                                                                plugins                                                                                                                                                                <?php

namespace Yoast\WP\SEO\Analytics\User_Interface;

use Yoast\WP\SEO\Conditionals\No_Conditionals;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Integrations\Integration_Interface;

/**
 * Handles setting a timestamp when the indexation of a specific indexation action is completed.
 */
class Last_Completed_Indexation_Integration implements Integration_Interface {

	use No_Conditionals;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper The options helper.
	 */
	private $options_helper;

	/**
	 * The constructor.
	 *
	 * @param Options_Helper $options_helper The options helper.
	 */
	public function __construct( Options_Helper $options_helper ) {
		$this->options_helper = $options_helper;
	}

	/**
	 * Registers action hook to maybe save an option.
	 *
	 * @return void
	 */
	public function register_hooks(): void {
		\add_action(
			'wpseo_indexables_unindexed_calculated',
			[
				$this,
				'maybe_set_indexables_unindexed_calculated',
			],
			10,
			2
		);
	}

	/**
	 * Saves a timestamp option when there are no unindexed indexables.
	 *
	 * @param string $indexable_name The name of the indexable that is being checked.
	 * @param int    $count          The amount of missing indexables.
	 *
	 * @return void
	 */
	public function maybe_set_indexables_unindexed_calculated( string $indexable_name, int $count ): void {
		if ( $count === 0 ) {
			$no_index                    = $this->options_helper->get( 'last_known_no_unindexed', [] );
			$no_index[ $indexable_name ] = \time();
			$this->options_helper->set( 'last_known_no_unindexed', $no_index );
		}
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/src/builders/indexable-author-builder.php                            0000644                 00000017117 15122266557 0022651 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Builders;

use Yoast\WP\SEO\Exceptions\Indexable\Author_Not_Built_Exception;
use Yoast\WP\SEO\Helpers\Author_Archive_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Post_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Values\Indexables\Indexable_Builder_Versions;

/**
 * Author Builder for the indexables.
 *
 * Formats the author meta to indexable format.
 */
class Indexable_Author_Builder {

	use Indexable_Social_Image_Trait;

	/**
	 * The author archive helper.
	 *
	 * @var Author_Archive_Helper
	 */
	private $author_archive;

	/**
	 * The latest version of the Indexable_Author_Builder.
	 *
	 * @var int
	 */
	protected $version;

	/**
	 * Holds the options helper instance.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * Holds the taxonomy helper instance.
	 *
	 * @var Post_Helper
	 */
	protected $post_helper;

	/**
	 * Indexable_Author_Builder constructor.
	 *
	 * @param Author_Archive_Helper      $author_archive The author archive helper.
	 * @param Indexable_Builder_Versions $versions       The Indexable version manager.
	 * @param Options_Helper             $options_helper The options helper.
	 * @param Post_Helper                $post_helper    The post helper.
	 */
	public function __construct(
		Author_Archive_Helper $author_archive,
		Indexable_Builder_Versions $versions,
		Options_Helper $options_helper,
		Post_Helper $post_helper
	) {
		$this->author_archive = $author_archive;
		$this->version        = $versions->get_latest_version_for_type( 'user' );
		$this->options_helper = $options_helper;
		$this->post_helper    = $post_helper;
	}

	/**
	 * Formats the data.
	 *
	 * @param int       $user_id   The user to retrieve the indexable for.
	 * @param Indexable $indexable The indexable to format.
	 *
	 * @return Indexable The extended indexable.
	 *
	 * @throws Author_Not_Built_Exception When author is not built.
	 */
	public function build( $user_id, Indexable $indexable ) {
		$exception = $this->check_if_user_should_be_indexed( $user_id );
		if ( $exception ) {
			throw $exception;
		}

		$meta_data = $this->get_meta_data( $user_id );

		$indexable->object_id              = $user_id;
		$indexable->object_type            = 'user';
		$indexable->permalink              = \get_author_posts_url( $user_id );
		$indexable->title                  = $meta_data['wpseo_title'];
		$indexable->description            = $meta_data['wpseo_metadesc'];
		$indexable->is_cornerstone         = false;
		$indexable->is_robots_noindex      = ( $meta_data['wpseo_noindex_author'] === 'on' );
		$indexable->is_robots_nofollow     = null;
		$indexable->is_robots_noarchive    = null;
		$indexable->is_robots_noimageindex = null;
		$indexable->is_robots_nosnippet    = null;
		$indexable->is_public              = ( $indexable->is_robots_noindex ) ? false : null;
		$indexable->has_public_posts       = $this->author_archive->author_has_public_posts( $user_id );
		$indexable->blog_id                = \get_current_blog_id();

		$this->reset_social_images( $indexable );
		$this->handle_social_images( $indexable );

		$timestamps                      = $this->get_object_timestamps( $user_id );
		$indexable->object_published_at  = $timestamps->published_at;
		$indexable->object_last_modified = $timestamps->last_modified;

		$indexable->version = $this->version;

		return $indexable;
	}

	/**
	 * Retrieves the meta data for this indexable.
	 *
	 * @param int $user_id The user to retrieve the meta data for.
	 *
	 * @return array List of meta entries.
	 */
	protected function get_meta_data( $user_id ) {
		$keys = [
			'wpseo_title',
			'wpseo_metadesc',
			'wpseo_noindex_author',
		];

		$output = [];
		foreach ( $keys as $key ) {
			$output[ $key ] = $this->get_author_meta( $user_id, $key );
		}

		return $output;
	}

	/**
	 * Retrieves the author meta.
	 *
	 * @param int    $user_id The user to retrieve the indexable for.
	 * @param string $key     The meta entry to retrieve.
	 *
	 * @return string|null The value of the meta field.
	 */
	protected function get_author_meta( $user_id, $key ) {
		$value = \get_the_author_meta( $key, $user_id );
		if ( \is_string( $value ) && $value === '' ) {
			return null;
		}

		return $value;
	}

	/**
	 * Finds an alternative image for the social image.
	 *
	 * @param Indexable $indexable The indexable.
	 *
	 * @return array|bool False when not found, array with data when found.
	 */
	protected function find_alternative_image( Indexable $indexable ) {
		$gravatar_image = \get_avatar_url(
			$indexable->object_id,
			[
				'size'   => 500,
				'scheme' => 'https',
			]
		);
		if ( $gravatar_image ) {
			return [
				'image'  => $gravatar_image,
				'source' => 'gravatar-image',
			];
		}

		return false;
	}

	/**
	 * Returns the timestamps for a given author.
	 *
	 * @param int $author_id The author ID.
	 *
	 * @return object An object with last_modified and published_at timestamps.
	 */
	protected function get_object_timestamps( $author_id ) {
		global $wpdb;
		$post_statuses = $this->post_helper->get_public_post_statuses();

		$replacements   = [];
		$replacements[] = 'post_modified_gmt';
		$replacements[] = 'post_date_gmt';
		$replacements[] = $wpdb->posts;
		$replacements[] = 'post_status';
		$replacements   = \array_merge( $replacements, $post_statuses );
		$replacements[] = 'post_password';
		$replacements[] = 'post_author';
		$replacements[] = $author_id;

		//phpcs:disable WordPress.DB.PreparedSQLPlaceholders -- %i placeholder is still not recognized.
		//phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery -- Reason: Most performant way.
		//phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- Reason: No relevant caches.
		return $wpdb->get_row(
			$wpdb->prepare(
				'
				SELECT MAX(p.%i) AS last_modified, MIN(p.%i) AS published_at
				FROM %i AS p
				WHERE p.%i IN (' . \implode( ', ', \array_fill( 0, \count( $post_statuses ), '%s' ) ) . ")
					AND p.%i = ''
					AND p.%i = %d
				",
				$replacements
			)
		);
		//phpcs:enable
	}

	/**
	 * Checks if the user should be indexed.
	 * Returns an exception with an appropriate message if not.
	 *
	 * @param string $user_id The user id.
	 *
	 * @return Author_Not_Built_Exception|null The exception if it should not be indexed, or `null` if it should.
	 */
	protected function check_if_user_should_be_indexed( $user_id ) {
		$exception = null;

		if ( $this->author_archive->are_disabled() ) {
			$exception = Author_Not_Built_Exception::author_archives_are_disabled( $user_id );
		}

		// We will check if the author has public posts the WP way, instead of the indexable way, to make sure we get proper results even if SEO optimization is not run.
		// In case the user has no public posts, we check if the user should be indexed anyway.
		if ( $this->options_helper->get( 'noindex-author-noposts-wpseo', false ) === true && $this->author_archive->author_has_public_posts_wp( $user_id ) === false ) {
			$exception = Author_Not_Built_Exception::author_archives_are_not_indexed_for_users_without_posts( $user_id );
		}

		/**
		 * Filter: Include or exclude a user from being build and saved as an indexable.
		 * Return an `Author_Not_Built_Exception` when the indexable should not be build, with an appropriate message telling why it should not be built.
		 * Return `null` if the indexable should be build.
		 *
		 * @param Author_Not_Built_Exception|null $exception An exception if the indexable is not being built, `null` if the indexable should be built.
		 * @param string                          $user_id   The ID of the user that should or should not be excluded.
		 */
		return \apply_filters( 'wpseo_should_build_and_save_user_indexable', $exception, $user_id );
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                 plugins/wordpress-seo-extended/src/builders/indexable-builder.php                                   0000644                 00000034776 15122266557 0021363 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Builders;

use Yoast\WP\SEO\Exceptions\Indexable\Not_Built_Exception;
use Yoast\WP\SEO\Exceptions\Indexable\Source_Exception;
use Yoast\WP\SEO\Helpers\Indexable_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Services\Indexables\Indexable_Version_Manager;

/**
 * Builder for the indexables.
 *
 * Creates all the indexables.
 */
class Indexable_Builder {

	/**
	 * The author builder.
	 *
	 * @var Indexable_Author_Builder
	 */
	private $author_builder;

	/**
	 * The post builder.
	 *
	 * @var Indexable_Post_Builder
	 */
	private $post_builder;

	/**
	 * The term builder.
	 *
	 * @var Indexable_Term_Builder
	 */
	private $term_builder;

	/**
	 * The home page builder.
	 *
	 * @var Indexable_Home_Page_Builder
	 */
	private $home_page_builder;

	/**
	 * The post type archive builder.
	 *
	 * @var Indexable_Post_Type_Archive_Builder
	 */
	private $post_type_archive_builder;

	/**
	 * The data archive builder.
	 *
	 * @var Indexable_Date_Archive_Builder
	 */
	private $date_archive_builder;

	/**
	 * The system page builder.
	 *
	 * @var Indexable_System_Page_Builder
	 */
	private $system_page_builder;

	/**
	 * The indexable hierarchy builder.
	 *
	 * @var Indexable_Hierarchy_Builder
	 */
	private $hierarchy_builder;

	/**
	 * The primary term builder
	 *
	 * @var Primary_Term_Builder
	 */
	private $primary_term_builder;

	/**
	 * The link builder
	 *
	 * @var Indexable_Link_Builder
	 */
	private $link_builder;

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	private $indexable_repository;

	/**
	 * The indexable helper.
	 *
	 * @var Indexable_Helper
	 */
	protected $indexable_helper;

	/**
	 * The Indexable Version Manager.
	 *
	 * @var Indexable_Version_Manager
	 */
	protected $version_manager;

	/**
	 * Returns the instance of this class constructed through the ORM Wrapper.
	 *
	 * @param Indexable_Author_Builder            $author_builder            The author builder for creating missing indexables.
	 * @param Indexable_Post_Builder              $post_builder              The post builder for creating missing indexables.
	 * @param Indexable_Term_Builder              $term_builder              The term builder for creating missing indexables.
	 * @param Indexable_Home_Page_Builder         $home_page_builder         The front page builder for creating missing indexables.
	 * @param Indexable_Post_Type_Archive_Builder $post_type_archive_builder The post type archive builder for creating missing indexables.
	 * @param Indexable_Date_Archive_Builder      $date_archive_builder      The date archive builder for creating missing indexables.
	 * @param Indexable_System_Page_Builder       $system_page_builder       The search result builder for creating missing indexables.
	 * @param Indexable_Hierarchy_Builder         $hierarchy_builder         The hierarchy builder for creating the indexable hierarchy.
	 * @param Primary_Term_Builder                $primary_term_builder      The primary term builder for creating primary terms for posts.
	 * @param Indexable_Helper                    $indexable_helper          The indexable helper.
	 * @param Indexable_Version_Manager           $version_manager           The indexable version manager.
	 * @param Indexable_Link_Builder              $link_builder              The link builder for creating missing SEO links.
	 */
	public function __construct(
		Indexable_Author_Builder $author_builder,
		Indexable_Post_Builder $post_builder,
		Indexable_Term_Builder $term_builder,
		Indexable_Home_Page_Builder $home_page_builder,
		Indexable_Post_Type_Archive_Builder $post_type_archive_builder,
		Indexable_Date_Archive_Builder $date_archive_builder,
		Indexable_System_Page_Builder $system_page_builder,
		Indexable_Hierarchy_Builder $hierarchy_builder,
		Primary_Term_Builder $primary_term_builder,
		Indexable_Helper $indexable_helper,
		Indexable_Version_Manager $version_manager,
		Indexable_Link_Builder $link_builder
	) {
		$this->author_builder            = $author_builder;
		$this->post_builder              = $post_builder;
		$this->term_builder              = $term_builder;
		$this->home_page_builder         = $home_page_builder;
		$this->post_type_archive_builder = $post_type_archive_builder;
		$this->date_archive_builder      = $date_archive_builder;
		$this->system_page_builder       = $system_page_builder;
		$this->hierarchy_builder         = $hierarchy_builder;
		$this->primary_term_builder      = $primary_term_builder;
		$this->indexable_helper          = $indexable_helper;
		$this->version_manager           = $version_manager;
		$this->link_builder              = $link_builder;
	}

	/**
	 * Sets the indexable repository. Done to avoid circular dependencies.
	 *
	 * @required
	 *
	 * @param Indexable_Repository $indexable_repository The indexable repository.
	 *
	 * @return void
	 */
	public function set_indexable_repository( Indexable_Repository $indexable_repository ) {
		$this->indexable_repository = $indexable_repository;
	}

	/**
	 * Creates a clean copy of an Indexable to allow for later database operations.
	 *
	 * @param Indexable $indexable The Indexable to copy.
	 *
	 * @return bool|Indexable
	 */
	protected function deep_copy_indexable( $indexable ) {
		return $this->indexable_repository
			->query()
			->create( $indexable->as_array() );
	}

	/**
	 * Creates an indexable by its ID and type.
	 *
	 * @param int            $object_id   The indexable object ID.
	 * @param string         $object_type The indexable object type.
	 * @param Indexable|bool $indexable   Optional. An existing indexable to overwrite.
	 *
	 * @return bool|Indexable Instance of indexable. False when unable to build.
	 */
	public function build_for_id_and_type( $object_id, $object_type, $indexable = false ) {
		$defaults = [
			'object_type' => $object_type,
			'object_id'   => $object_id,
		];

		$indexable = $this->build( $indexable, $defaults );

		return $indexable;
	}

	/**
	 * Creates an indexable for the homepage.
	 *
	 * @param Indexable|bool $indexable Optional. An existing indexable to overwrite.
	 *
	 * @return Indexable The home page indexable.
	 */
	public function build_for_home_page( $indexable = false ) {
		return $this->build( $indexable, [ 'object_type' => 'home-page' ] );
	}

	/**
	 * Creates an indexable for the date archive.
	 *
	 * @param Indexable|bool $indexable Optional. An existing indexable to overwrite.
	 *
	 * @return Indexable The date archive indexable.
	 */
	public function build_for_date_archive( $indexable = false ) {
		return $this->build( $indexable, [ 'object_type' => 'date-archive' ] );
	}

	/**
	 * Creates an indexable for a post type archive.
	 *
	 * @param string         $post_type The post type.
	 * @param Indexable|bool $indexable Optional. An existing indexable to overwrite.
	 *
	 * @return Indexable The post type archive indexable.
	 */
	public function build_for_post_type_archive( $post_type, $indexable = false ) {
		$defaults = [
			'object_type'     => 'post-type-archive',
			'object_sub_type' => $post_type,
		];
		return $this->build( $indexable, $defaults );
	}

	/**
	 * Creates an indexable for a system page.
	 *
	 * @param string         $page_type The type of system page.
	 * @param Indexable|bool $indexable Optional. An existing indexable to overwrite.
	 *
	 * @return Indexable The search result indexable.
	 */
	public function build_for_system_page( $page_type, $indexable = false ) {
		$defaults = [
			'object_type'     => 'system-page',
			'object_sub_type' => $page_type,
		];
		return $this->build( $indexable, $defaults );
	}

	/**
	 * Ensures we have a valid indexable. Creates one if false is passed.
	 *
	 * @param Indexable|false $indexable The indexable.
	 * @param array           $defaults  The initial properties of the Indexable.
	 *
	 * @return Indexable The indexable.
	 */
	protected function ensure_indexable( $indexable, $defaults = [] ) {
		if ( ! $indexable ) {
			return $this->indexable_repository->query()->create( $defaults );
		}

		return $indexable;
	}

	/**
	 * Saves and returns an indexable (on production environments only).
	 *
	 * @param Indexable      $indexable        The indexable.
	 * @param Indexable|null $indexable_before The indexable before possible changes.
	 *
	 * @return Indexable The indexable.
	 */
	protected function save_indexable( $indexable, $indexable_before = null ) {
		$intend_to_save = $this->indexable_helper->should_index_indexables();

		/**
		 * Filter: 'wpseo_should_save_indexable' - Allow developers to enable / disable
		 * saving the indexable when the indexable is updated. Warning: overriding
		 * the intended action may cause problems when moving from a staging to a
		 * production environment because indexable permalinks may get set incorrectly.
		 *
		 * @param bool      $intend_to_save True if YoastSEO intends to save the indexable.
		 * @param Indexable $indexable      The indexable to be saved.
		 */
		$intend_to_save = \apply_filters( 'wpseo_should_save_indexable', $intend_to_save, $indexable );

		if ( ! $intend_to_save ) {
			return $indexable;
		}

		// Save the indexable before running the WordPress hook.
		$indexable->save();

		if ( $indexable_before ) {
			/**
			 * Action: 'wpseo_save_indexable' - Allow developers to perform an action
			 * when the indexable is updated.
			 *
			 * @param Indexable $indexable        The saved indexable.
			 * @param Indexable $indexable_before The indexable before saving.
			 */
			\do_action( 'wpseo_save_indexable', $indexable, $indexable_before );
		}

		return $indexable;
	}

	/**
	 * Build and author indexable from an author id if it does not exist yet, or if the author indexable needs to be upgraded.
	 *
	 * @param int $author_id The author id.
	 *
	 * @return Indexable|false The author indexable if it has been built, `false` if it could not be built.
	 */
	protected function maybe_build_author_indexable( $author_id ) {
		$author_indexable = $this->indexable_repository->find_by_id_and_type(
			$author_id,
			'user',
			false
		);
		if ( ! $author_indexable || $this->version_manager->indexable_needs_upgrade( $author_indexable ) ) {
			// Try to build the author.
			$author_defaults  = [
				'object_type' => 'user',
				'object_id'   => $author_id,
			];
			$author_indexable = $this->build( $author_indexable, $author_defaults );
		}
		return $author_indexable;
	}

	/**
	 * Checks if the indexable type is one that is not supposed to have object ID for.
	 *
	 * @param string $type The type of the indexable.
	 *
	 * @return bool Whether the indexable type is one that is not supposed to have object ID for.
	 */
	protected function is_type_with_no_id( $type ) {
		return \in_array( $type, [ 'home-page', 'date-archive', 'post-type-archive', 'system-page' ], true );
	}

	// phpcs:disable Squiz.Commenting.FunctionCommentThrowTag.Missing -- Exceptions are handled by the catch statement in the method.

	/**
	 * Rebuilds an Indexable from scratch.
	 *
	 * @param Indexable  $indexable The Indexable to (re)build.
	 * @param array|null $defaults  The object type of the Indexable.
	 *
	 * @return Indexable|false The resulting Indexable.
	 */
	public function build( $indexable, $defaults = null ) {
		// Backup the previous Indexable, if there was one.
		$indexable_before = ( $indexable ) ? $this->deep_copy_indexable( $indexable ) : null;

		// Make sure we have an Indexable to work with.
		$indexable = $this->ensure_indexable( $indexable, $defaults );

		try {
			if ( $indexable->object_id === 0 ) {
				throw Not_Built_Exception::invalid_object_id( $indexable->object_id );
			}
			switch ( $indexable->object_type ) {

				case 'post':
					$indexable = $this->post_builder->build( $indexable->object_id, $indexable );

					// Save indexable, to make sure it can be queried when building related objects like the author indexable and hierarchy.
					$indexable = $this->save_indexable( $indexable, $indexable_before );

					// For attachments, we have to make sure to patch any potentially previously cleaned up SEO links.
					if ( \is_a( $indexable, Indexable::class ) && $indexable->object_sub_type === 'attachment' ) {
						$this->link_builder->patch_seo_links( $indexable );
					}

					// Always rebuild the primary term.
					$this->primary_term_builder->build( $indexable->object_id );

					// Always rebuild the hierarchy; this needs the primary term to run correctly.
					$this->hierarchy_builder->build( $indexable );

					$this->maybe_build_author_indexable( $indexable->author_id );

					// The indexable is already saved, so return early.
					return $indexable;

				case 'user':
					$indexable = $this->author_builder->build( $indexable->object_id, $indexable );
					break;

				case 'term':
					$indexable = $this->term_builder->build( $indexable->object_id, $indexable );

					// Save indexable, to make sure it can be queried when building hierarchy.
					$indexable = $this->save_indexable( $indexable, $indexable_before );

					$this->hierarchy_builder->build( $indexable );

					// The indexable is already saved, so return early.
					return $indexable;

				case 'home-page':
					$indexable = $this->home_page_builder->build( $indexable );
					break;

				case 'date-archive':
					$indexable = $this->date_archive_builder->build( $indexable );
					break;

				case 'post-type-archive':
					$indexable = $this->post_type_archive_builder->build( $indexable->object_sub_type, $indexable );
					break;

				case 'system-page':
					$indexable = $this->system_page_builder->build( $indexable->object_sub_type, $indexable );
					break;
			}

			return $this->save_indexable( $indexable, $indexable_before );
		}
		catch ( Source_Exception $exception ) {
			if ( ! $this->is_type_with_no_id( $indexable->object_type ) && ( ! isset( $indexable->object_id ) || \is_null( $indexable->object_id ) ) ) {
				return false;
			}

			/**
			 * The current indexable could not be indexed. Create a placeholder indexable, so we can
			 * skip this indexable in future indexing runs.
			 *
			 * @var Indexable $indexable
			 */
			$indexable = $this->ensure_indexable(
				$indexable,
				[
					'object_id'   => $indexable->object_id,
					'object_type' => $indexable->object_type,
					'post_status' => 'unindexed',
					'version'     => 0,
				]
			);
			// If we already had an existing indexable, mark it as unindexed. We cannot rely on its validity anymore.
			$indexable->post_status = 'unindexed';
			// Make sure that the indexing process doesn't get stuck in a loop on this broken indexable.
			$indexable = $this->version_manager->set_latest( $indexable );

			return $this->save_indexable( $indexable, $indexable_before );
		}
		catch ( Not_Built_Exception $exception ) {
			return false;
		}
	}

	// phpcs:enable
}
  plugins/wordpress-seo-extended/src/builders/indexable-date-archive-builder.php                      0000644                 00000003234 15122266557 0023676 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Builders;

use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Values\Indexables\Indexable_Builder_Versions;

/**
 * Date Archive Builder for the indexables.
 *
 * Formats the date archive meta to indexable format.
 */
class Indexable_Date_Archive_Builder {

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	private $options;

	/**
	 * The latest version of the Indexable_Date_Archive_Builder.
	 *
	 * @var int
	 */
	protected $version;

	/**
	 * Indexable_Date_Archive_Builder constructor.
	 *
	 * @param Options_Helper             $options  The options helper.
	 * @param Indexable_Builder_Versions $versions The latest version for all indexable builders.
	 */
	public function __construct(
		Options_Helper $options,
		Indexable_Builder_Versions $versions
	) {
		$this->options = $options;
		$this->version = $versions->get_latest_version_for_type( 'date-archive' );
	}

	/**
	 * Formats the data.
	 *
	 * @param Indexable $indexable The indexable to format.
	 *
	 * @return Indexable The extended indexable.
	 */
	public function build( $indexable ) {
		$indexable->object_type       = 'date-archive';
		$indexable->title             = $this->options->get( 'title-archive-wpseo' );
		$indexable->description       = $this->options->get( 'metadesc-archive-wpseo' );
		$indexable->is_robots_noindex = $this->options->get( 'noindex-archive-wpseo' );
		$indexable->is_public         = ( (int) $indexable->is_robots_noindex !== 1 );
		$indexable->blog_id           = \get_current_blog_id();
		$indexable->permalink         = null;
		$indexable->version           = $this->version;

		return $indexable;
	}
}
                                                                                                                                                                                                                                                                                                                                                                    plugins/wordpress-seo-extended/src/builders/indexable-hierarchy-builder.php                         0000644                 00000025437 15122266557 0023331 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Builders;

use WP_Post;
use WP_Term;
use WPSEO_Meta;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Post_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Repositories\Indexable_Hierarchy_Repository;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Repositories\Primary_Term_Repository;

/**
 * Builder for the indexables hierarchy.
 *
 * Builds the indexable hierarchy for indexables.
 */
class Indexable_Hierarchy_Builder {

	/**
	 * Holds a list of indexables where the ancestors are saved for.
	 *
	 * @var array
	 */
	protected $saved_ancestors = [];

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	private $indexable_repository;

	/**
	 * The indexable hierarchy repository.
	 *
	 * @var Indexable_Hierarchy_Repository
	 */
	private $indexable_hierarchy_repository;

	/**
	 * The primary term repository.
	 *
	 * @var Primary_Term_Repository
	 */
	private $primary_term_repository;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	private $options;

	/**
	 * Holds the Post_Helper instance.
	 *
	 * @var Post_Helper
	 */
	private $post;

	/**
	 * Indexable_Author_Builder constructor.
	 *
	 * @param Indexable_Hierarchy_Repository $indexable_hierarchy_repository The indexable hierarchy repository.
	 * @param Primary_Term_Repository        $primary_term_repository        The primary term repository.
	 * @param Options_Helper                 $options                        The options helper.
	 * @param Post_Helper                    $post                           The post helper.
	 */
	public function __construct(
		Indexable_Hierarchy_Repository $indexable_hierarchy_repository,
		Primary_Term_Repository $primary_term_repository,
		Options_Helper $options,
		Post_Helper $post
	) {
		$this->indexable_hierarchy_repository = $indexable_hierarchy_repository;
		$this->primary_term_repository        = $primary_term_repository;
		$this->options                        = $options;
		$this->post                           = $post;
	}

	/**
	 * Sets the indexable repository. Done to avoid circular dependencies.
	 *
	 * @required
	 *
	 * @param Indexable_Repository $indexable_repository The indexable repository.
	 *
	 * @return void
	 */
	public function set_indexable_repository( Indexable_Repository $indexable_repository ) {
		$this->indexable_repository = $indexable_repository;
	}

	/**
	 * Builds the ancestor hierarchy for an indexable.
	 *
	 * @param Indexable $indexable The indexable.
	 *
	 * @return Indexable The indexable.
	 */
	public function build( Indexable $indexable ) {
		if ( $this->hierarchy_is_built( $indexable ) ) {
			return $indexable;
		}

		$this->indexable_hierarchy_repository->clear_ancestors( $indexable->id );

		$indexable_id = $this->get_indexable_id( $indexable );
		$ancestors    = [];
		if ( $indexable->object_type === 'post' ) {
			$this->add_ancestors_for_post( $indexable_id, $indexable->object_id, $ancestors );
		}

		if ( $indexable->object_type === 'term' ) {
			$this->add_ancestors_for_term( $indexable_id, $indexable->object_id, $ancestors );
		}
		$indexable->ancestors     = \array_reverse( \array_values( $ancestors ) );
		$indexable->has_ancestors = ! empty( $ancestors );
		if ( $indexable->id ) {
			$this->save_ancestors( $indexable );
		}

		return $indexable;
	}

	/**
	 * Checks if a hierarchy is built already for the given indexable.
	 *
	 * @param Indexable $indexable The indexable to check.
	 *
	 * @return bool True when indexable has a built hierarchy.
	 */
	protected function hierarchy_is_built( Indexable $indexable ) {
		if ( \in_array( $indexable->id, $this->saved_ancestors, true ) ) {
			return true;
		}

		$this->saved_ancestors[] = $indexable->id;

		return false;
	}

	/**
	 * Saves the ancestors.
	 *
	 * @param Indexable $indexable The indexable.
	 *
	 * @return void
	 */
	private function save_ancestors( $indexable ) {
		if ( empty( $indexable->ancestors ) ) {
			$this->indexable_hierarchy_repository->add_ancestor( $indexable->id, 0, 0 );
			return;
		}
		$depth = \count( $indexable->ancestors );
		foreach ( $indexable->ancestors as $ancestor ) {
			$this->indexable_hierarchy_repository->add_ancestor( $indexable->id, $ancestor->id, $depth );
			--$depth;
		}
	}

	/**
	 * Adds ancestors for a post.
	 *
	 * @param int   $indexable_id The indexable id, this is the id of the original indexable.
	 * @param int   $post_id      The post id, this is the id of the post currently being evaluated.
	 * @param int[] $parents      The indexable IDs of all parents.
	 *
	 * @return void
	 */
	private function add_ancestors_for_post( $indexable_id, $post_id, &$parents ) {
		$post = $this->post->get_post( $post_id );

		if ( ! isset( $post->post_parent ) ) {
			return;
		}

		if ( $post->post_parent !== 0 && $this->post->get_post( $post->post_parent ) !== null ) {
			$ancestor = $this->indexable_repository->find_by_id_and_type( $post->post_parent, 'post' );
			if ( $this->is_invalid_ancestor( $ancestor, $indexable_id, $parents ) ) {
				return;
			}

			$parents[ $this->get_indexable_id( $ancestor ) ] = $ancestor;

			$this->add_ancestors_for_post( $indexable_id, $ancestor->object_id, $parents );

			return;
		}

		$primary_term_id = $this->find_primary_term_id_for_post( $post );

		if ( $primary_term_id === 0 ) {
			return;
		}

		$ancestor = $this->indexable_repository->find_by_id_and_type( $primary_term_id, 'term' );
		if ( $this->is_invalid_ancestor( $ancestor, $indexable_id, $parents ) ) {
			return;
		}

		$parents[ $this->get_indexable_id( $ancestor ) ] = $ancestor;

		$this->add_ancestors_for_term( $indexable_id, $ancestor->object_id, $parents );
	}

	/**
	 * Adds ancestors for a term.
	 *
	 * @param int   $indexable_id The indexable id, this is the id of the original indexable.
	 * @param int   $term_id      The term id, this is the id of the term currently being evaluated.
	 * @param int[] $parents      The indexable IDs of all parents.
	 *
	 * @return void
	 */
	private function add_ancestors_for_term( $indexable_id, $term_id, &$parents = [] ) {
		$term         = \get_term( $term_id );
		$term_parents = $this->get_term_parents( $term );

		foreach ( $term_parents as $parent ) {
			$ancestor = $this->indexable_repository->find_by_id_and_type( $parent->term_id, 'term' );
			if ( $this->is_invalid_ancestor( $ancestor, $indexable_id, $parents ) ) {
				continue;
			}

			$parents[ $this->get_indexable_id( $ancestor ) ] = $ancestor;
		}
	}

	/**
	 * Gets the primary term ID for a post.
	 *
	 * @param WP_Post $post The post.
	 *
	 * @return int The primary term ID. 0 if none exists.
	 */
	private function find_primary_term_id_for_post( $post ) {
		$main_taxonomy = $this->options->get( 'post_types-' . $post->post_type . '-maintax' );

		if ( ! $main_taxonomy || $main_taxonomy === '0' ) {
			return 0;
		}

		$primary_term_id = $this->get_primary_term_id( $post->ID, $main_taxonomy );

		if ( $primary_term_id ) {
			$term = \get_term( $primary_term_id );
			if ( $term !== null && ! \is_wp_error( $term ) ) {
				return $primary_term_id;
			}
		}

		$terms = \get_the_terms( $post->ID, $main_taxonomy );

		if ( ! \is_array( $terms ) || empty( $terms ) ) {
			return 0;
		}

		return $this->find_deepest_term_id( $terms );
	}

	/**
	 * Find the deepest term in an array of term objects.
	 *
	 * @param array $terms Terms set.
	 *
	 * @return int The deepest term ID.
	 */
	private function find_deepest_term_id( $terms ) {
		/*
		 * Let's find the deepest term in this array, by looping through and then
		 * unsetting every term that is used as a parent by another one in the array.
		 */
		$terms_by_id = [];
		foreach ( $terms as $term ) {
			$terms_by_id[ $term->term_id ] = $term;
		}
		foreach ( $terms as $term ) {
			unset( $terms_by_id[ $term->parent ] );
		}

		/*
		 * As we could still have two subcategories, from different parent categories,
		 * let's pick the one with the lowest ordered ancestor.
		 */
		$parents_count = -1;
		$term_order    = 9999; // Because ASC.
		$deepest_term  = \reset( $terms_by_id );
		foreach ( $terms_by_id as $term ) {
			$parents = $this->get_term_parents( $term );

			$new_parents_count = \count( $parents );

			if ( $new_parents_count < $parents_count ) {
				continue;
			}

			$parent_order = 9999; // Set default order.
			foreach ( $parents as $parent ) {
				if ( $parent->parent === 0 && isset( $parent->term_order ) ) {
					$parent_order = $parent->term_order;
				}
			}

			// Check if parent has lowest order.
			if ( $new_parents_count > $parents_count || $parent_order < $term_order ) {
				$term_order   = $parent_order;
				$deepest_term = $term;
			}

			$parents_count = $new_parents_count;
		}

		return $deepest_term->term_id;
	}

	/**
	 * Get a term's parents.
	 *
	 * @param WP_Term $term Term to get the parents for.
	 *
	 * @return WP_Term[] An array of all this term's parents.
	 */
	private function get_term_parents( $term ) {
		$tax     = $term->taxonomy;
		$parents = [];
		while ( (int) $term->parent !== 0 ) {
			$term      = \get_term( $term->parent, $tax );
			$parents[] = $term;
		}

		return $parents;
	}

	/**
	 * Checks if an ancestor is valid to add.
	 *
	 * @param Indexable $ancestor     The ancestor (presumed indexable) to check.
	 * @param int       $indexable_id The indexable id we're adding ancestors for.
	 * @param int[]     $parents      The indexable ids of the parents already added.
	 *
	 * @return bool
	 */
	private function is_invalid_ancestor( $ancestor, $indexable_id, $parents ) {
		// If the ancestor is not an Indexable, it is invalid by default.
		if ( ! \is_a( $ancestor, 'Yoast\WP\SEO\Models\Indexable' ) ) {
			return true;
		}

		// Don't add ancestors if they're unindexed, already added or the same as the main object.
		if ( $ancestor->post_status === 'unindexed' ) {
			return true;
		}

		$ancestor_id = $this->get_indexable_id( $ancestor );
		if ( \array_key_exists( $ancestor_id, $parents ) ) {
			return true;
		}

		if ( $ancestor_id === $indexable_id ) {
			return true;
		}

		return false;
	}

	/**
	 * Returns the ID for an indexable. Catches situations where the id is null due to errors.
	 *
	 * @param Indexable $indexable The indexable.
	 *
	 * @return string|int A unique ID for the indexable.
	 */
	private function get_indexable_id( Indexable $indexable ) {
		if ( $indexable->id === 0 ) {
			return "{$indexable->object_type}:{$indexable->object_id}";
		}

		return $indexable->id;
	}

	/**
	 * Returns the primary term id of a post.
	 *
	 * @param int    $post_id       The post ID.
	 * @param string $main_taxonomy The main taxonomy.
	 *
	 * @return int The ID of the primary term.
	 */
	private function get_primary_term_id( $post_id, $main_taxonomy ) {
		$primary_term = $this->primary_term_repository->find_by_post_id_and_taxonomy( $post_id, $main_taxonomy, false );

		if ( $primary_term ) {
			return $primary_term->term_id;
		}

		return \get_post_meta( $post_id, WPSEO_Meta::$meta_prefix . 'primary_' . $main_taxonomy, true );
	}
}
                                                                                                                                                                                                                                 plugins/wordpress-seo-extended/src/builders/indexable-home-page-builder.php                         0000644                 00000010542 15122266557 0023204 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Builders;

use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Post_Helper;
use Yoast\WP\SEO\Helpers\Url_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Values\Indexables\Indexable_Builder_Versions;

/**
 * Homepage Builder for the indexables.
 *
 * Formats the homepage meta to indexable format.
 */
class Indexable_Home_Page_Builder {

	use Indexable_Social_Image_Trait;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	protected $options;

	/**
	 * The URL helper.
	 *
	 * @var Url_Helper
	 */
	protected $url_helper;

	/**
	 * The latest version of the Indexable-Home-Page-Builder.
	 *
	 * @var int
	 */
	protected $version;

	/**
	 * Holds the taxonomy helper instance.
	 *
	 * @var Post_Helper
	 */
	protected $post_helper;

	/**
	 * Indexable_Home_Page_Builder constructor.
	 *
	 * @param Options_Helper             $options     The options helper.
	 * @param Url_Helper                 $url_helper  The url helper.
	 * @param Indexable_Builder_Versions $versions    Knows the latest version of each Indexable type.
	 * @param Post_Helper                $post_helper The post helper.
	 */
	public function __construct(
		Options_Helper $options,
		Url_Helper $url_helper,
		Indexable_Builder_Versions $versions,
		Post_Helper $post_helper
	) {
		$this->options     = $options;
		$this->url_helper  = $url_helper;
		$this->version     = $versions->get_latest_version_for_type( 'home-page' );
		$this->post_helper = $post_helper;
	}

	/**
	 * Formats the data.
	 *
	 * @param Indexable $indexable The indexable to format.
	 *
	 * @return Indexable The extended indexable.
	 */
	public function build( $indexable ) {
		$indexable->object_type      = 'home-page';
		$indexable->title            = $this->options->get( 'title-home-wpseo' );
		$indexable->breadcrumb_title = $this->options->get( 'breadcrumbs-home' );
		$indexable->permalink        = $this->url_helper->home();
		$indexable->blog_id          = \get_current_blog_id();
		$indexable->description      = $this->options->get( 'metadesc-home-wpseo' );
		if ( empty( $indexable->description ) ) {
			$indexable->description = \get_bloginfo( 'description' );
		}

		$indexable->is_robots_noindex = \get_option( 'blog_public' ) === '0';

		$indexable->open_graph_title       = $this->options->get( 'open_graph_frontpage_title' );
		$indexable->open_graph_image       = $this->options->get( 'open_graph_frontpage_image' );
		$indexable->open_graph_image_id    = $this->options->get( 'open_graph_frontpage_image_id' );
		$indexable->open_graph_description = $this->options->get( 'open_graph_frontpage_desc' );

		// Reset the OG image source & meta.
		$indexable->open_graph_image_source = null;
		$indexable->open_graph_image_meta   = null;

		// When the image or image id is set.
		if ( $indexable->open_graph_image || $indexable->open_graph_image_id ) {
			$indexable->open_graph_image_source = 'set-by-user';

			$this->set_open_graph_image_meta_data( $indexable );
		}

		$timestamps                      = $this->get_object_timestamps();
		$indexable->object_published_at  = $timestamps->published_at;
		$indexable->object_last_modified = $timestamps->last_modified;

		$indexable->version = $this->version;

		return $indexable;
	}

	/**
	 * Returns the timestamps for the homepage.
	 *
	 * @return object An object with last_modified and published_at timestamps.
	 */
	protected function get_object_timestamps() {
		global $wpdb;
		$post_statuses = $this->post_helper->get_public_post_statuses();

		$replacements   = [];
		$replacements[] = 'post_modified_gmt';
		$replacements[] = 'post_date_gmt';
		$replacements[] = $wpdb->posts;
		$replacements[] = 'post_status';
		$replacements   = \array_merge( $replacements, $post_statuses );
		$replacements[] = 'post_password';
		$replacements[] = 'post_type';

		//phpcs:disable WordPress.DB.PreparedSQLPlaceholders -- %i placeholder is still not recognized.
		//phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery -- Reason: Most performant way.
		//phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching -- Reason: No relevant caches.
		return $wpdb->get_row(
			$wpdb->prepare(
				'
			SELECT MAX(p.%i) AS last_modified, MIN(p.%i) AS published_at
			FROM %i AS p
			WHERE p.%i IN (' . \implode( ', ', \array_fill( 0, \count( $post_statuses ), '%s' ) ) . ")
				AND p.%i = ''
				AND p.%i = 'post'
			",
				$replacements
			)
		);
		//phpcs:enable
	}
}
                                                                                                                                                              plugins/wordpress-seo-extended/src/builders/indexable-link-builder.php                              0000644                 00000051660 15122266557 0022305 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php

namespace Yoast\WP\SEO\Builders;

use DOMDocument;
use WP_HTML_Tag_Processor;
use WPSEO_Image_Utils;
use Yoast\WP\SEO\Helpers\Image_Helper;
use Yoast\WP\SEO\Helpers\Options_Helper;
use Yoast\WP\SEO\Helpers\Post_Helper;
use Yoast\WP\SEO\Helpers\Url_Helper;
use Yoast\WP\SEO\Models\Indexable;
use Yoast\WP\SEO\Models\SEO_Links;
use Yoast\WP\SEO\Repositories\Indexable_Repository;
use Yoast\WP\SEO\Repositories\SEO_Links_Repository;

/**
 * Indexable link builder.
 */
class Indexable_Link_Builder {

	/**
	 * The SEO links repository.
	 *
	 * @var SEO_Links_Repository
	 */
	protected $seo_links_repository;

	/**
	 * The url helper.
	 *
	 * @var Url_Helper
	 */
	protected $url_helper;

	/**
	 * The image helper.
	 *
	 * @var Image_Helper
	 */
	protected $image_helper;

	/**
	 * The post helper.
	 *
	 * @var Post_Helper
	 */
	protected $post_helper;

	/**
	 * The options helper.
	 *
	 * @var Options_Helper
	 */
	protected $options_helper;

	/**
	 * The indexable repository.
	 *
	 * @var Indexable_Repository
	 */
	protected $indexable_repository;

	/**
	 * Indexable_Link_Builder constructor.
	 *
	 * @param SEO_Links_Repository $seo_links_repository The SEO links repository.
	 * @param Url_Helper           $url_helper           The URL helper.
	 * @param Post_Helper          $post_helper          The post helper.
	 * @param Options_Helper       $options_helper       The options helper.
	 */
	public function __construct(
		SEO_Links_Repository $seo_links_repository,
		Url_Helper $url_helper,
		Post_Helper $post_helper,
		Options_Helper $options_helper
	) {
		$this->seo_links_repository = $seo_links_repository;
		$this->url_helper           = $url_helper;
		$this->post_helper          = $post_helper;
		$this->options_helper       = $options_helper;
	}

	/**
	 * Sets the indexable repository.
	 *
	 * @required
	 *
	 * @param Indexable_Repository $indexable_repository The indexable repository.
	 * @param Image_Helper         $image_helper         The image helper.
	 *
	 * @return void
	 */
	public function set_dependencies(
		Indexable_Repository $indexable_repository,
		Image_Helper $image_helper
	) {
		$this->indexable_repository = $indexable_repository;
		$this->image_helper         = $image_helper;
	}

	/**
	 * Builds the links for a post.
	 *
	 * @param Indexable $indexable The indexable.
	 * @param string    $content   The content. Expected to be unfiltered.
	 *
	 * @return SEO_Links[] The created SEO links.
	 */
	public function build( $indexable, $content ) {
		global $post;
		if ( $indexable->object_type === 'post' ) {
			$post_backup = $post;
			// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- To setup the post we need to do this explicitly.
			$post = $this->post_helper->get_post( $indexable->object_id );
			\setup_postdata( $post );
			$content = \apply_filters( 'the_content', $content );
			\wp_reset_postdata();
			// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited -- To setup the post we need to do this explicitly.
			$post = $post_backup;
		}

		$content = \str_replace( ']]>', ']]&gt;', $content );
		$links   = $this->gather_links( $content );
		$images  = $this->gather_images( $content );

		if ( empty( $links ) && empty( $images ) ) {
			$indexable->link_count = 0;
			$this->update_related_indexables( $indexable, [] );

			return [];
		}

		$links = $this->create_links( $indexable, $links, $images );

		$this->update_related_indexables( $indexable, $links );

		$indexable->link_count = $this->get_internal_link_count( $links );

		return $links;
	}

	/**
	 * Deletes all SEO links for an indexable.
	 *
	 * @param Indexable $indexable The indexable.
	 *
	 * @return void
	 */
	public function delete( $indexable ) {
		$links = ( $this->seo_links_repository->find_all_by_indexable_id( $indexable->id ) );
		$this->seo_links_repository->delete_all_by_indexable_id( $indexable->id );

		$linked_indexable_ids = [];
		foreach ( $links as $link ) {
			if ( $link->target_indexable_id ) {
				$linked_indexable_ids[] = $link->target_indexable_id;
			}
		}

		$this->update_incoming_links_for_related_indexables( $linked_indexable_ids );
	}

	/**
	 * Fixes existing SEO links that are supposed to have a target indexable but don't, because of prior indexable
	 * cleanup.
	 *
	 * @param Indexable $indexable The indexable to be the target of SEO Links.
	 *
	 * @return void
	 */
	public function patch_seo_links( Indexable $indexable ) {
		if ( ! empty( $indexable->id ) && ! empty( $indexable->object_id ) ) {
			$links = $this->seo_links_repository->find_all_by_target_post_id( $indexable->object_id );

			$updated_indexable = false;
			foreach ( $links as $link ) {
				if ( \is_a( $link, SEO_Links::class ) && empty( $link->target_indexable_id ) ) {
					// Since that post ID exists in an SEO link but has no target_indexable_id, it's probably because of prior indexable cleanup.
					$this->seo_links_repository->update_target_indexable_id( $link->id, $indexable->id );
					$updated_indexable = true;
				}
			}

			if ( $updated_indexable ) {
				$updated_indexable_id = [ $indexable->id ];
				$this->update_incoming_links_for_related_indexables( $updated_indexable_id );
			}
		}
	}

	/**
	 * Gathers all links from content.
	 *
	 * @param string $content The content.
	 *
	 * @return string[] An array of urls.
	 */
	protected function gather_links( $content ) {
		if ( \strpos( $content, 'href' ) === false ) {
			// Nothing to do.
			return [];
		}

		$links  = [];
		$regexp = '<a\s[^>]*href=("??)([^" >]*?)\1[^>]*>';
		// Used modifiers iU to match case insensitive and make greedy quantifiers lazy.
		if ( \preg_match_all( "/$regexp/iU", $content, $matches, \PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$links[] = \trim( $match[2], "'" );
			}
		}

		return $links;
	}

	/**
	 * Gathers all images from content with WP's WP_HTML_Tag_Processor() and returns them along with their IDs, if
	 * possible.
	 *
	 * @param string $content The content.
	 *
	 * @return int[] An associated array of image IDs, keyed by their URL.
	 */
	protected function gather_images_wp( $content ) {
		$processor = new WP_HTML_Tag_Processor( $content );
		$images    = [];

		$query = [
			'tag_name' => 'img',
		];

		/**
		 * Filter 'wpseo_image_attribute_containing_id' - Allows filtering what attribute will be used to extract image IDs from.
		 *
		 * Defaults to "class", which is where WP natively stores the image IDs, in a `wp-image-<ID>` format.
		 *
		 * @api string The attribute to be used to extract image IDs from.
		 */
		$attribute = \apply_filters( 'wpseo_image_attribute_containing_id', 'class' );

		while ( $processor->next_tag( $query ) ) {
			$src     = \htmlentities( $processor->get_attribute( 'src' ), ( \ENT_QUOTES | \ENT_SUBSTITUTE | \ENT_HTML401 ), \get_bloginfo( 'charset' ) );
			$classes = $processor->get_attribute( $attribute );
			$id      = $this->extract_id_of_classes( $classes );

			$images[ $src ] = $id;
		}

		return $images;
	}

	/**
	 * Gathers all images from content with DOMDocument() and returns them along with their IDs, if possible.
	 *
	 * @param string $content The content.
	 *
	 * @return int[] An associated array of image IDs, keyed by their URL.
	 */
	protected function gather_images_domdocument( $content ) {
		$images  = [];
		$charset = \get_bloginfo( 'charset' );

		/**
		 * Filter 'wpseo_image_attribute_containing_id' - Allows filtering what attribute will be used to extract image IDs from.
		 *
		 * Defaults to "class", which is where WP natively stores the image IDs, in a `wp-image-<ID>` format.
		 *
		 * @api string The attribute to be used to extract image IDs from.
		 */
		$attribute = \apply_filters( 'wpseo_image_attribute_containing_id', 'class' );

		\libxml_use_internal_errors( true );
		$post_dom = new DOMDocument();
		$post_dom->loadHTML( '<?xml encoding="' . $charset . '">' . $content );
		\libxml_clear_errors();

		foreach ( $post_dom->getElementsByTagName( 'img' ) as $img ) {
			$src     = \htmlentities( $img->getAttribute( 'src' ), ( \ENT_QUOTES | \ENT_SUBSTITUTE | \ENT_HTML401 ), $charset );
			$classes = $img->getAttribute( $attribute );
			$id      = $this->extract_id_of_classes( $classes );

			$images[ $src ] = $id;
		}

		return $images;
	}

	/**
	 * Extracts image ID out of the image's classes.
	 *
	 * @param string $classes The classes assigned to the image.
	 *
	 * @return int The ID that's extracted from the classes.
	 */
	protected function extract_id_of_classes( $classes ) {
		if ( ! $classes ) {
			return 0;
		}

		/**
		 * Filter 'wpseo_extract_id_pattern' - Allows filtering the regex patern to be used to extract image IDs from class/attribute names.
		 *
		 * Defaults to the pattern that extracts image IDs from core's `wp-image-<ID>` native format in image classes.
		 *
		 * @api string The regex pattern to be used to extract image IDs from class names. Empty string if the whole class/attribute should be returned.
		 */
		$pattern = \apply_filters( 'wpseo_extract_id_pattern', '/(?<!\S)wp-image-(\d+)(?!\S)/i' );

		if ( $pattern === '' ) {
			return (int) $classes;
		}

		$matches = [];

		if ( \preg_match( $pattern, $classes, $matches ) ) {
			return (int) $matches[1];
		}

		return 0;
	}

	/**
	 * Gathers all images from content.
	 *
	 * @param string $content The content.
	 *
	 * @return int[] An associated array of image IDs, keyed by their URLs.
	 */
	protected function gather_images( $content ) {

		/**
		 * Filter 'wpseo_force_creating_and_using_attachment_indexables' - Filters if we should use attachment indexables to find all content images. Instead of scanning the content.
		 *
		 * The default value is false.
		 *
		 * @since 21.1
		 */
		$should_not_parse_content = \apply_filters( 'wpseo_force_creating_and_using_attachment_indexables', false );

		/**
		 * Filter 'wpseo_force_skip_image_content_parsing' - Filters if we should force skip scanning the content to parse images.
		 * This filter can be used if the regex gives a faster result than scanning the code.
		 *
		 * The default value is false.
		 *
		 * @since 21.1
		 */
		$should_not_parse_content = \apply_filters( 'wpseo_force_skip_image_content_parsing', $should_not_parse_content );
		if ( ! $should_not_parse_content && \class_exists( WP_HTML_Tag_Processor::class ) ) {
			return $this->gather_images_wp( $content );
		}

		if ( ! $should_not_parse_content && \class_exists( DOMDocument::class ) ) {
			return $this->gather_images_DOMDocument( $content );
		}

		if ( \strpos( $content, 'src' ) === false ) {
			// Nothing to do.
			return [];
		}

		$images = [];
		$regexp = '<img\s[^>]*src=("??)([^" >]*?)\\1[^>]*>';
		// Used modifiers iU to match case insensitive and make greedy quantifiers lazy.
		if ( \preg_match_all( "/$regexp/iU", $content, $matches, \PREG_SET_ORDER ) ) {
			foreach ( $matches as $match ) {
				$images[ $match[2] ] = 0;
			}
		}

		return $images;
	}

	/**
	 * Creates link models from lists of URLs and image sources.
	 *
	 * @param Indexable $indexable The indexable.
	 * @param string[]  $links     The link URLs.
	 * @param int[]     $images    The image sources.
	 *
	 * @return SEO_Links[] The link models.
	 */
	protected function create_links( $indexable, $links, $images ) {
		$home_url    = \wp_parse_url( \home_url() );
		$current_url = \wp_parse_url( $indexable->permalink );
		$links       = \array_map(
			function ( $link ) use ( $home_url, $indexable ) {
				return $this->create_internal_link( $link, $home_url, $indexable );
			},
			$links
		);
		// Filter out links to the same page with a fragment or query.
		$links = \array_filter(
			$links,
			function ( $link ) use ( $current_url ) {
				return $this->filter_link( $link, $current_url );
			}
		);

		$image_links = [];
		foreach ( $images as $image_url => $image_id ) {
			$image_links[] = $this->create_internal_link( $image_url, $home_url, $indexable, true, $image_id );
		}

		return \array_merge( $links, $image_links );
	}

	/**
	 * Get the post ID based on the link's type and its target's permalink.
	 *
	 * @param string $type      The type of link (either SEO_Links::TYPE_INTERNAL or SEO_Links::TYPE_INTERNAL_IMAGE).
	 * @param string $permalink The permalink of the link's target.
	 *
	 * @return int The post ID.
	 */
	protected function get_post_id( $type, $permalink ) {
		if ( $type === SEO_Links::TYPE_INTERNAL ) {
			return \url_to_postid( $permalink );
		}

		return $this->image_helper->get_attachment_by_url( $permalink );
	}

	/**
	 * Creates an internal link.
	 *
	 * @param string    $url       The url of the link.
	 * @param array     $home_url  The home url, as parsed by wp_parse_url.
	 * @param Indexable $indexable The indexable of the post containing the link.
	 * @param bool      $is_image  Whether or not the link is an image.
	 * @param int       $image_id  The ID of the internal image.
	 *
	 * @return SEO_Links The created link.
	 */
	protected function create_internal_link( $url, $home_url, $indexable, $is_image = false, $image_id = 0 ) {
		$parsed_url = \wp_parse_url( $url );
		$link_type  = $this->url_helper->get_link_type( $parsed_url, $home_url, $is_image );

		/**
		 * ORM representing a link in the SEO Links table.
		 *
		 * @var SEO_Links $model
		 */
		$model = $this->seo_links_repository->query()->create(
			[
				'url'          => $url,
				'type'         => $link_type,
				'indexable_id' => $indexable->id,
				'post_id'      => $indexable->object_id,
			]
		);

		$model->parsed_url = $parsed_url;

		if ( $model->type === SEO_Links::TYPE_INTERNAL ) {
			$permalink = $this->build_permalink( $url, $home_url );

			return $this->enhance_link_from_indexable( $model, $permalink );
		}

		if ( $model->type === SEO_Links::TYPE_INTERNAL_IMAGE ) {
			$permalink = $this->build_permalink( $url, $home_url );

			/** The `wpseo_force_creating_and_using_attachment_indexables` filter is documented in indexable-link-builder.php */
			if ( ! $this->options_helper->get( 'disable-attachment' ) || \apply_filters( 'wpseo_force_creating_and_using_attachment_indexables', false ) ) {
				$model = $this->enhance_link_from_indexable( $model, $permalink );
			}
			else {
				$target_post_id = ( $image_id !== 0 ) ? $image_id : WPSEO_Image_Utils::get_attachment_by_url( $permalink );

				if ( ! empty( $target_post_id ) ) {
					$model->target_post_id = $target_post_id;
				}
			}

			if ( $model->target_post_id ) {
				$file = \get_attached_file( $model->target_post_id );

				if ( $file ) {
					if ( \file_exists( $file ) ) {
						$model->size = \filesize( $file );
					}
					else {
						$model->size = null;
					}

					[ , $width, $height ] = \wp_get_attachment_image_src( $model->target_post_id, 'full' );
					$model->width         = $width;
					$model->height        = $height;
				}
				else {
					$model->width  = 0;
					$model->height = 0;
					$model->size   = 0;
				}
			}
		}

		return $model;
	}

	/**
	 * Enhances the link model with information from its indexable.
	 *
	 * @param SEO_Links $model     The link's model.
	 * @param string    $permalink The link's permalink.
	 *
	 * @return SEO_Links The enhanced link model.
	 */
	protected function enhance_link_from_indexable( $model, $permalink ) {
		$target = $this->indexable_repository->find_by_permalink( $permalink );

		if ( ! $target ) {
			// If target indexable cannot be found, create one based on the post's post ID.
			$post_id = $this->get_post_id( $model->type, $permalink );
			if ( $post_id && $post_id !== 0 ) {
				$target = $this->indexable_repository->find_by_id_and_type( $post_id, 'post' );
			}
		}

		if ( ! $target ) {
			return $model;
		}

		$model->target_indexable_id = $target->id;
		if ( $target->object_type === 'post' ) {
			$model->target_post_id = $target->object_id;
		}

		if ( $model->target_indexable_id ) {
			$model->language = $target->language;
			$model->region   = $target->region;
		}

		return $model;
	}

	/**
	 * Builds the link's permalink.
	 *
	 * @param string $url      The url of the link.
	 * @param array  $home_url The home url, as parsed by wp_parse_url.
	 *
	 * @return string The link's permalink.
	 */
	protected function build_permalink( $url, $home_url ) {
		$permalink = $this->get_permalink( $url, $home_url );

		if ( $this->url_helper->is_relative( $permalink ) ) {
			// Make sure we're checking against the absolute URL, and add a trailing slash if the site has a trailing slash in its permalink settings.
			$permalink = $this->url_helper->ensure_absolute_url( \user_trailingslashit( $permalink ) );
		}

		return $permalink;
	}

	/**
	 * Filters out links that point to the same page with a fragment or query.
	 *
	 * @param SEO_Links $link        The link.
	 * @param array     $current_url The url of the page the link is on, as parsed by wp_parse_url.
	 *
	 * @return bool Whether or not the link should be filtered.
	 */
	protected function filter_link( SEO_Links $link, $current_url ) {
		$url = $link->parsed_url;

		// Always keep external links.
		if ( $link->type === SEO_Links::TYPE_EXTERNAL ) {
			return true;
		}

		// Always keep links with an empty path or pointing to other pages.
		if ( isset( $url['path'] ) ) {
			return empty( $url['path'] ) || $url['path'] !== $current_url['path'];
		}

		// Only keep links to the current page without a fragment or query.
		return ( ! isset( $url['fragment'] ) && ! isset( $url['query'] ) );
	}

	/**
	 * Updates the link counts for related indexables.
	 *
	 * @param Indexable   $indexable The indexable.
	 * @param SEO_Links[] $links     The link models.
	 *
	 * @return void
	 */
	protected function update_related_indexables( $indexable, $links ) {
		// Old links were only stored by post id, so remove all old seo links for this post that have no indexable id.
		// This can be removed if we ever fully clear all seo links.
		if ( $indexable->object_type === 'post' ) {
			$this->seo_links_repository->delete_all_by_post_id_where_indexable_id_null( $indexable->object_id );
		}

		$updated_indexable_ids = [];
		$old_links             = $this->seo_links_repository->find_all_by_indexable_id( $indexable->id );

		$links_to_remove = $this->links_diff( $old_links, $links );
		$links_to_add    = $this->links_diff( $links, $old_links );

		if ( ! empty( $links_to_remove ) ) {
			$this->seo_links_repository->delete_many_by_id( \wp_list_pluck( $links_to_remove, 'id' ) );
		}

		if ( ! empty( $links_to_add ) ) {
			$this->seo_links_repository->insert_many( $links_to_add );
		}

		foreach ( $links_to_add as $link ) {
			if ( $link->target_indexable_id ) {
				$updated_indexable_ids[] = $link->target_indexable_id;
			}
		}
		foreach ( $links_to_remove as $link ) {
			if ( $link->target_indexable_id ) {
				$updated_indexable_ids[] = $link->target_indexable_id;
			}
		}

		$this->update_incoming_links_for_related_indexables( $updated_indexable_ids );
	}

	/**
	 * Creates a diff between two arrays of SEO links, based on urls.
	 *
	 * @param SEO_Links[] $links_a The array to compare.
	 * @param SEO_Links[] $links_b The array to compare against.
	 *
	 * @return SEO_Links[] Links that are in $links_a, but not in $links_b.
	 */
	protected function links_diff( $links_a, $links_b ) {
		return \array_udiff(
			$links_a,
			$links_b,
			static function ( SEO_Links $link_a, SEO_Links $link_b ) {
				return \strcmp( $link_a->url, $link_b->url );
			}
		);
	}

	/**
	 * Returns the number of internal links in an array of link models.
	 *
	 * @param SEO_Links[] $links The link models.
	 *
	 * @return int The number of internal links.
	 */
	protected function get_internal_link_count( $links ) {
		$internal_link_count = 0;

		foreach ( $links as $link ) {
			if ( $link->type === SEO_Links::TYPE_INTERNAL ) {
				++$internal_link_count;
			}
		}

		return $internal_link_count;
	}

	/**
	 * Returns a cleaned permalink for a given link.
	 *
	 * @param string $link     The raw URL.
	 * @param array  $home_url The home URL, as parsed by wp_parse_url.
	 *
	 * @return string The cleaned permalink.
	 */
	protected function get_permalink( $link, $home_url ) {
		// Get rid of the #anchor.
		$url_split = \explode( '#', $link );
		$link      = $url_split[0];

		// Get rid of URL ?query=string.
		$url_split = \explode( '?', $link );
		$link      = $url_split[0];

		// Set the correct URL scheme.
		$link = \set_url_scheme( $link, $home_url['scheme'] );

		// Add 'www.' if it is absent and should be there.
		if ( \strpos( $home_url['host'], 'www.' ) === 0 && \strpos( $link, '://www.' ) === false ) {
			$link = \str_replace( '://', '://www.', $link );
		}

		// Strip 'www.' if it is present and shouldn't be.
		if ( \strpos( $home_url['host'], 'www.' ) !== 0 ) {
			$link = \str_replace( '://www.', '://', $link );
		}

		return $link;
	}

	/**
	 * Updates incoming link counts for related indexables.
	 *
	 * @param int[] $related_indexable_ids The IDs of all related indexables.
	 *
	 * @return void
	 */
	protected function update_incoming_links_for_related_indexables( $related_indexable_ids ) {
		if ( empty