ram bool $force Whether to send the tracking data ignoring the two
	 *                    weeks time threshold. Default false.
	 *
	 * @return void
	 */
	public function send( $force = false ) {
		if ( ! $this->should_send_tracking( $force ) ) {
			return;
		}

		// Set a 'content-type' header of 'application/json'.
		$tracking_request_args = [
			'headers' => [
				'content-type:' => 'application/json',
			],
		];

		$collector = $this->get_collector();

		$request = new WPSEO_Remote_Request( $this->endpoint, $tracking_request_args );
		$request->set_body( $collector->get_as_json() );
		$request->send();

		update_option( $this->option_name, $this->current_time, 'yes' );
	}

	/**
	 * Determines whether to send the tracking data.
	 *
	 * Returns false if tracking is disabled or the current page is one of the
	 * admin plugins pages. Returns true when there's no tracking data stored or
	 * the data was sent more than two weeks ago. The two weeks interval is set
	 * when instantiating the class.
	 *
	 * @param bool $ignore_time_treshhold Whether to send the tracking data ignoring
	 *                                    the two weeks time treshhold. Default false.
	 *
	 * @return bool True when tracking data should be sent.
	 */
	protected function should_send_tracking( $ignore_time_treshhold = false ) {
		global $pagenow;

		// Only send tracking on the main site of a multi-site instance. This returns true on non-multisite installs.
		if ( is_network_admin() || ! is_main_site() ) {
			return false;
		}

		// Because we don't want to possibly block plugin actions with our routines.
		if ( in_array( $pagenow, [ 'plugins.php', 'plugin-install.php', 'plugin-editor.php' ], true ) ) {
			return false;
		}

		$last_time = get_option( $this->option_name );

		// When tracking data haven't been sent yet or when sending data is forced.
		if ( ! $last_time || $ignore_time_treshhold ) {
			return true;
		}

		return $this->exceeds_treshhold( $this->current_time - $last_time );
	}

	/**
	 * Checks if the given amount of seconds exceeds the set threshold.
	 *
	 * @param int $seconds The amount of seconds to check.
	 *
	 * @return bool True when seconds is bigger than threshold.
	 */
	protected function exceeds_treshhold( $seconds ) {
		return ( $seconds > $this->threshold );
	}

	/**
	 * Returns the collector for collecting the data.
	 *
	 * @return WPSEO_Collector The instance of the collector.
	 */
	public function get_collector() {
		$collector = new WPSEO_Collector();
		$collector->add_collection( new WPSEO_Tracking_Default_Data() );
		$collector->add_collection( new WPSEO_Tracking_Server_Data() );
		$collector->add_collection( new WPSEO_Tracking_Theme_Data() );
		$collector->add_collection( new WPSEO_Tracking_Plugin_Data() );
		$collector->add_collection( new WPSEO_Tracking_Settings_Data() );
		$collector->add_collection( new WPSEO_Tracking_Addon_Data() );
		$collector->add_collection( YoastSEO()->classes->get( Missing_Indexables_Collector::class ) );
		$collector->add_collection( YoastSEO()->classes->get( To_Be_Cleaned_Indexables_Collector::class ) );

		return $collector;
	}

	/**
	 * See if we should run tracking at all.
	 *
	 * @return bool True when we can track, false when we can't.
	 */
	private function tracking_enabled() {
		// Check if we're allowing tracking.
		$tracking = WPSEO_Options::get( 'tracking' );

		if ( $tracking === false ) {
			return false;
		}

		// Save this state.
		if ( $tracking === null ) {
			/**
			 * Filter: 'wpseo_enable_tracking' - Enables the data tracking of Yoast SEO Premium and add-ons.
			 *
			 * @param string $is_enabled The enabled state. Default is false.
			 */
			$tracking = apply_filters( 'wpseo_enable_tracking', false );

			WPSEO_Options::set( 'tracking', $tracking );
		}

		if ( $tracking === false ) {
			return false;
		}

		if ( ! YoastSEO()->helpers->environment->is_production_mode() ) {
			return false;
		}

		return true;
	}
}
                                                                                                                                                         plugins/wordpress-seo-extended/admin/tracking/class-tracking-plugin-data.php                        0000644                 00000004042 15122266555 0023363 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Tracking
 */

/**
 * Represents the plugin data.
 */
class WPSEO_Tracking_Plugin_Data implements WPSEO_Collection {

	/**
	 * Plugins with auto updating enabled.
	 *
	 * @var array
	 */
	private $auto_update_plugin_list;

	/**
	 * Returns the collection data.
	 *
	 * @return array The collection data.
	 */
	public function get() {
		return [
			'plugins' => $this->get_plugin_data(),
		];
	}

	/**
	 * Returns all plugins.
	 *
	 * @return array The formatted plugins.
	 */
	protected function get_plugin_data() {

		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugins = wp_get_active_and_valid_plugins();

		$plugins = array_map( 'get_plugin_data', $plugins );
		$this->set_auto_update_plugin_list();
		$plugins = array_map( [ $this, 'format_plugin' ], $plugins );

		$plugin_data = [];
		foreach ( $plugins as $plugin ) {
			$plugin_key                 = sanitize_title( $plugin['name'] );
			$plugin_data[ $plugin_key ] = $plugin;
		}

		return $plugin_data;
	}

	/**
	 * Sets all auto updating plugin data so it can be used in the tracking list.
	 *
	 * @return void
	 */
	public function set_auto_update_plugin_list() {

		$auto_update_plugins      = [];
		$auto_update_plugin_files = get_option( 'auto_update_plugins' );
		if ( $auto_update_plugin_files ) {
			foreach ( $auto_update_plugin_files as $auto_update_plugin ) {
				$data                                 = get_plugin_data( WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $auto_update_plugin );
				$auto_update_plugins[ $data['Name'] ] = $data;
			}
		}

		$this->auto_update_plugin_list = $auto_update_plugins;
	}

	/**
	 * Formats the plugin array.
	 *
	 * @param array $plugin The plugin details.
	 *
	 * @return array The formatted array.
	 */
	protected function format_plugin( array $plugin ) {

		return [
			'name'          => $plugin['Name'],
			'version'       => $plugin['Version'],
			'auto_updating' => array_key_exists( $plugin['Name'], $this->auto_update_plugin_list ),
		];
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/admin/tracking/class-tracking-server-data.php                        0000644                 00000003757 15122266555 0023407 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Tracking
 */

/**
 * Represents the server data.
 */
class WPSEO_Tracking_Server_Data implements WPSEO_Collection {

	/**
	 * Returns the collection data.
	 *
	 * @return array The collection data.
	 */
	public function get() {
		return [
			'server' => $this->get_server_data(),
		];
	}

	/**
	 * Returns the values with server details.
	 *
	 * @return array Array with the value.
	 */
	protected function get_server_data() {
		$server_data = [];

		// Validate if the server address is a valid IP-address.
		$ipaddress = isset( $_SERVER['SERVER_ADDR'] ) ? filter_var( wp_unslash( $_SERVER['SERVER_ADDR'] ), FILTER_VALIDATE_IP ) : '';
		if ( $ipaddress ) {
			$server_data['ip']       = $ipaddress;
			$server_data['Hostname'] = gethostbyaddr( $ipaddress );
		}

		$server_data['os']            = function_exists( 'php_uname' ) ? php_uname() : PHP_OS;
		$server_data['PhpVersion']    = PHP_VERSION;
		$server_data['CurlVersion']   = $this->get_curl_info();
		$server_data['PhpExtensions'] = $this->get_php_extensions();

		return $server_data;
	}

	/**
	 * Returns details about the curl version.
	 *
	 * @return array|null The curl info. Or null when curl isn't available..
	 */
	protected function get_curl_info() {
		if ( ! function_exists( 'curl_version' ) ) {
			return null;
		}

		$curl = curl_version();

		$ssl_support = true;
		if ( ! $curl['features'] && CURL_VERSION_SSL ) {
			$ssl_support = false;
		}

		return [
			'version'    => $curl['version'],
			'sslSupport' => $ssl_support,
		];
	}

	/**
	 * Returns a list with php extensions.
	 *
	 * @return array Returns the state of the php extensions.
	 */
	protected function get_php_extensions() {
		return [
			'imagick'   => extension_loaded( 'imagick' ),
			'filter'    => extension_loaded( 'filter' ),
			'bcmath'    => extension_loaded( 'bcmath' ),
			'pcre'      => extension_loaded( 'pcre' ),
			'xml'       => extension_loaded( 'xml' ),
			'pdo_mysql' => extension_loaded( 'pdo_mysql' ),
		];
	}
}
                 plugins/wordpress-seo-extended/admin/tracking/class-tracking-settings-data.php                      0000644                 00000015377 15122266555 0023742 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Tracking
 */

/**
 * Collects anonymized settings data.
 */
class WPSEO_Tracking_Settings_Data implements WPSEO_Collection {

	/**
	 * The options that need to be anonymized before they can be sent elsewhere.
	 *
	 * @var array All of the option_names which need to be
	 * anonymized before they can be sent elsewhere.
	 */
	private $anonymous_settings = [
		'baiduverify',
		'googleverify',
		'msverify',
		'yandexverify',
		'myyoast-oauth',
		'website_name',
		'alternate_website_name',
		'company_logo',
		'company_name',
		'company_alternate_name',
		'person_name',
		'person_logo',
		'person_logo_id',
		'company_logo_id',
		'facebook_site',
		'instagram_url',
		'linkedin_url',
		'myspace_url',
		'og_default_image',
		'og_default_image_id',
		'og_frontpage_title',
		'og_frontpage_desc',
		'og_frontpage_image',
		'og_frontpage_image_id',
		'open_graph_frontpage_title',
		'open_graph_frontpage_desc',
		'open_graph_frontpage_image',
		'open_graph_frontpage_image_id',
		'other_social_urls',
		'mastodon_url',
		'pinterest_url',
		'pinterestverify',
		'twitter_site',
		'youtube_url',
		'wikipedia_url',
		'semrush_tokens',
		'zapier_api_key',
		'wincher_tokens',
		'wincher_website_id',
		'least_readab