or delete query.
	 *
	 * @var string
	 */
	protected $meta_key;

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys;

	/**
	 * Class constructor.
	 */
	public function __construct() {}

	/**
	 * Returns the string for the plugin we're importing from.
	 *
	 * @return string Plugin name.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Imports the settings and post meta data from another SEO plugin.
	 *
	 * @return WPSEO_Import_Status Import status object.
	 */
	public function run_import() {
		$this->status = new WPSEO_Import_Status( 'import', false );

		if ( ! $this->detect() ) {
			return $this->status;
		}

		$this->status->set_status( $this->import() );

		// Flush the entire cache, as we no longer know what's valid and what's not.
		wp_cache_flush();

		return $this->status;
	}

	/**
	 * Handles post meta data to import.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		return $this->meta_keys_clone( $this->clone_keys );
	}

	/**
	 * Removes the plugin data from the database.
	 *
	 * @return WPSEO_Import_Status Import status object.
	 */
	public function run_cleanup() {
		$this->status = new WPSEO_Import_Status( 'cleanup', false );

		if ( ! $this->detect() ) {
			return $this->status;
		}

		return $this->status->set_status( $this->cleanup() );
	}

	/**
	 * Removes the plugin data from the database.
	 *
	 * @return bool Cleanup status.
	 */
	protected function cleanup() {
		global $wpdb;
		if ( empty( $this->meta_key ) ) {
			return true;
		}
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE %s",
				$this->meta_key
			)
		);
		$result = $wpdb->__get( 'result' );
		if ( ! $result ) {
			$this->cleanup_error_msg();
		}

		return $result;
	}

	/**
	 * Sets the status message for when a cleanup has gone bad.
	 *
	 * @return void
	 */
	protected function cleanup_error_msg() {
		/* translators: %s is replaced with the plugin's name. */
		$this->status->set_msg( sprintf( __( 'Cleanup of %s data failed.', 'wordpress-seo' ), $this->plugin_name ) );
	}

	/**
	 * Detects whether an import for this plugin is needed.
	 *
	 * @return WPSEO_Import_Status Import status object.
	 */
	public function run_detect() {
		$this->status = new WPSEO_Import_Status( 'detect', false );

		if ( ! $this->detect() ) {
			return $this->status;
		}

		return $this->status->set_status( true );
	}

	/**
	 * Detects whether there is post meta data to import.
	 *
	 * @return bool Boolean indicating whether there is something to import.
	 */
	protected function detect() {
		global $wpdb;

		$meta_keys = wp_list_pluck( $this->clone_keys, 'old_key' );
		$result    = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) AS `count`
					FROM {$wpdb->postmeta}
					WHERE meta_key IN ( " . implode( ', ', array_fill( 0, count( $meta_keys ), '%s' ) ) . ' )',
				$meta_keys
			)
		);

		if ( $result === '0' ) {
			return false;
		}

		return true;
	}

	/**
	 * Helper function to clone meta keys and (optionally) change their values in bulk.
	 *
	 * @param string $old_key        The existing meta key.
	 * @param string $new_key        The new meta key.
	 * @param array  $replace_values An array, keys old value, values new values.
	 *
	 * @return bool Clone status.
	 */
	protected function meta_key_clone( $old_key, $new_key, $replace_values = [] ) {
		global $wpdb;

		// First we create a temp table with all the values for meta_key.
		$result = $wpdb->query(
			$wpdb->prepare(
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange -- This is intentional + temporary.
				"CREATE TEMPORARY TABLE tmp_meta_table SELECT * FROM {$wpdb->postmeta} WHERE meta_key = %s",
				$old_key
			)
		);
		if ( $result === false ) {
			$this->set_missing_db_rights_status();
			return false;
		}

		// Delete all the values in our temp table for posts that already have data for $new_key.
		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM tmp_meta_table WHERE post_id IN ( SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s )",
				WPSEO_Meta::$meta_prefix . $new_key
			)
		);

		/*
		 * We set meta_id to NULL so on re-insert into the postmeta table, MYSQL can set
		 * new meta_id's and we don't get duplicates.
		 */
		$wpdb->query( 'UPDATE tmp_meta_table SET meta_id = NULL' );

		// Now we rename the meta_key.
		$wpdb->query(
			$wpdb->prepare(
				'UPDATE tmp_meta_table SET meta_key = %s',
				WPSEO_Meta::$meta_prefix . $new_key
			)
		);

		$this->meta_key_clone_replace( $replace_values );

		// With everything done, we insert all our newly cloned lines into the postmeta table.
		$wpdb->query( "INSERT INTO {$wpdb->postmeta} SELECT * FROM tmp_meta_table" );

		// Now we drop our temporary table.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.SchemaChange -- This is intentional + a temporary table.
		$wpdb->query( 'DROP TEMPORARY TABLE IF EXISTS tmp_meta_table' );

		return true;
	}

	/**
	 * Clones multiple meta keys.
	 *
	 * @param array $clone_keys The keys to clone.
	 *
	 * @return bool Success status.
	 */
	protected function meta_keys_clone( $clone_keys ) {
		foreach ( $clone_keys as $clone_key ) {
			$result = $this->meta_key_clone( $clone_key['old_key'], $clone_key['new_key'], ( $clone_key['convert'] ?? [] ) );
			if ( ! $result ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Sets the import status to false and returns a message about why it failed.
	 *
	 * @return void
	 */
	protected function set_missing_db_rights_status() {
		$this->status->set_status( false );
		/* translators: %s is replaced with Yoast SEO. */
		$this->status->set_msg( sprintf( __( 'The %s importer functionality uses temporary database tables. It seems your WordPress install does not have the capability to do this, please consult your hosting provider.', 'wordpress-seo' ), 'Yoast SEO' ) );
	}

	/**
	 * Helper function to search for a key in an array and maybe save it as a meta field.
	 *
	 * @param string $plugin_key The key in the $data array to check.
	 * @param string $yoast_key  The identifier we use in our meta settings.
	 * @param array  $data       The array of data for this post to sift through.
	 * @param int    $post_id    The post ID.
	 *
	 * @return void
	 */
	protected function import_meta_helper( $plugin_key, $yoast_key, $data, $post_id ) {
		if ( ! empty( $data[ $plugin_key ] ) ) {
			$this->maybe_save_post_meta( $yoast_key, $data[ $plugin_key ], $post_id );
		}
	}

	/**
	 * Saves a post meta value if it doesn't already exist.
	 *
	 * @param string $new_key The key to save.
	 * @param mixed  $value   The value to set the key to.
	 * @param int    $post_id The Post to save the meta for.
	 *
	 * @return void
	 */
	protected function maybe_save_post_meta( $new_key, $value, $post_id ) {
		// Big. Fat. Sigh. Mostly used for _yst_is_cornerstone, but might be useful for other hidden meta's.
		$key        = WPSEO_Meta::$meta_prefix . $new_key;
		$wpseo_meta = true;
		if ( substr( $new_key, 0, 1 ) === '_' ) {
			$key        = $new_key;
			$wpseo_meta = false;
		}

		$existing_value = get_post_meta( $post_id, $key, true );
		if ( empty( $existing_value ) ) {
			if ( $wpseo_meta ) {
				WPSEO_Meta::set_value( $new_key, $value, $post_id );
				return;
			}
			update_post_meta( $post_id, $new_key, $value );
		}
	}

	/**
	 * Replaces values in our temporary table according to our settings.
	 *
	 * @param array $replace_values Key value pair of values to replace with other values.
	 *
	 * @return void
	 */
	protected function meta_key_clone_replace( $replace_values ) {
		global $wpdb;

		// Now we replace values if needed.
		if ( is_array( $replace_values ) && $replace_values !== [] ) {
			foreach ( $replace_values as $old_value => $new_value ) {
				$wpdb->query(
					$wpdb->prepare(
						'UPDATE tmp_meta_table SET meta_value = %s WHERE meta_value = %s',
						$new_value,
						$old_value
					)
				);
			}
		}
	}
}
                                                                                                                                                                                                                                plugins/wordpress-seo-extended/admin/import/plugins/class-import-aioseo.php                         0000644                 00000004717 15122266554 0023346 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from All in One SEO Pack, versions 3 and under.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean All in One SEO Pack post metadata, versions 3 and under.
 */
class WPSEO_Import_AIOSEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'All In One SEO Pack';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_aioseop_%';

	/**
	 * OpenGraph keys to import.
	 *
	 * @var array
	 */
	protected $import_keys = [
		'aioseop_opengraph_settings_title'             => 'opengraph-title',
		'aioseop_opengraph_settings_desc'              => 'opengraph-description',
		'aioseop_opengraph_settings_customimg'         => 'opengraph-image',
		'aioseop_opengraph_settings_customimg_twitter' => 'twitter-image',
	];

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_aioseop_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_aioseop_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_aioseop_noindex',
			'new_key' => 'meta-robots-noindex',
			'convert' => [ 'on' => 1 ],
		],
		[
			'old_key' => '_aioseop_nofollow',
			'new_key' => 'meta-robots-nofollow',
			'convert' => [ 'on' => 1 ],
		],
	];

	/**
	 * Import All In One SEO meta values.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		$status = parent::import();
		if ( $status ) {
			$this->import_opengraph();
		}
		return $status;
	}

	/**
	 * Imports the OpenGraph and Twitter settings for all posts.
	 *
	 * @return bool
	 */
	protected function import_opengraph() {
		$query_posts = new WP_Query( 'post_type=any&meta_key=_aioseop_opengraph_settings&order=ASC&fields=ids&nopaging=true' );

		if ( ! empty( $query_posts->posts ) ) {
			foreach ( array_values( $query_posts->posts ) as $post_id ) {
				$this->import_post_opengraph( $post_id );
			}
		}

		return true;
	}

	/**
	 * Imports the OpenGraph and Twitter settings for a single post.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return void
	 */
	private function import_post_opengraph( $post_id ) {
		$meta = get_post_meta( $post_id, '_aioseop_opengraph_settings', true );
		$meta = maybe_unserialize( $meta );

		foreach ( $this->import_keys as $old_key => $new_key ) {
			$this->maybe_save_post_meta( $new_key, $meta[ $old_key ], $post_id );
		}
	}
}
                                                 plugins/wordpress-seo-extended/admin/import/plugins/class-import-aioseo-v4.php                      0000644                 00000022156 15122266554 0023672 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from All in One SEO Pack, versions 4 and up.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

use Yoast\WP\SEO\Actions\Importing\Aioseo\Aioseo_Cleanup_Action;
use Yoast\WP\SEO\Actions\Importing\Aioseo\Aioseo_Posts_Importing_Action;

/**
 * Class with functionality to import & clean All in One SEO Pack post metadata, versions 4 and up.
 */
class WPSEO_Import_AIOSEO_V4 extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'All In One SEO Pack';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_aioseo_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_aioseo_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_aioseo_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_aioseo_og_title',
			'new_key' => 'opengraph-title',
		],
		[
			'old_key' => '_aioseo_og_description',
			'new_key' => 'opengraph-description',
		],
		[
			'old_key' => '_aioseo_twitter_title',
			'new_key' => 'twitter-title',
		],
		[
			'old_key' => '_aioseo_twitter_description',
			'new_key' => 'twitter-description',
		],
	];

	/**
	 * Mapping between the AiOSEO replace vars and the Yoast replace vars.
	 *
	 * @var array
	 *
	 * @see https://yoast.com/help/list-available-snippet-variables-yoast-seo/
	 */
	protected $replace_vars = [
		// They key is the AiOSEO replace var, the value is the Yoast replace var (see class-wpseo-replace-vars).
		'#author_first_name' => '%%author_first_name%%',
		'#author_last_name'  => '%%author_last_name%%',
		'#author_name'       => '%%name%%',
		'#categories'        => '%%category%%',
		'#current_date'      => '%%currentdate%%',
		'#current_day'       => '%%currentday%%',
		'#current_month'     => '%%currentmonth%%',
		'#current_year'      => '%%currentyear%%',
		'#permalink'         => '%%permalink%%',
		'#post_content'      => '%%post_content%%',
		'#post_date'         => '%%date%%',
		'#post_day'          => '%%post_day%%',
		'#post_month'        => '%%post_month%%',
		'#post_title'        => '%%title%%',
		'#post_year'         => '%%post_year%%',
		'#post_excerpt_only' => '%%excerpt_only%%',
		'#post_excerpt'      => '%%excerpt%%',
		'#separator_sa'      => '%%sep%%',
		'#site_title'        => '%%sitename%%',
		'#tagline'           => '%%sitedesc%%',
		'#taxonomy_title'    => '%%category_title%%',
	];

	/**
	 * Replaces the AiOSEO variables in our temporary table with Yoast variables (replace vars).
	 *
	 * @param array $replace_values Key value pair of values to replace with other values. This is only used in the base class but not here.
	 *                              That is because this class doesn't have any `convert` keys in `$clone_keys`.
	 *                              For that reason, we're overwriting the base class' `meta_key_clone_replace()` function without executing that base functionality.
	 *
	 * @return void
	 */
	protected function meta_key_clone_replace( $replace_values ) {
		global $wpdb;

		// At this point we're already looping through all the $clone_keys (this happens in meta_keys_clone() in the abstract class).
		// Now, we'll also loop through the replace_vars array, which holds the mappings between the AiOSEO variables and the Yoast variables.
		// We'll replace all the AiOSEO variables in the temporary table with their Yoast equivalents.
		foreach ( $this->replace_vars as $aioseo_variable => $yoast_variable ) {
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Reason: We need this query and this is done at many other places as well, for example class-import-rankmath.
			$wpdb->query(
				$wpdb->prepare(
					'UPDATE tmp_meta_table SET meta_value = REPLACE( meta_value, %s, %s )',
					$aioseo_variable,
					$yoast_variable
				)
			);
		}

		// The AiOSEO custom fields take the form of `#custom_field-myfield`.
		// These should be mapped to %%cf_myfield%%.
		$meta_values_with_custom_fields = $this->get_meta_values_with_custom_field_or_taxonomy( $wpdb, 'custom_field' );
		$unique_custom_fields           = $this->get_unique_custom_fields_or_taxonomies( $meta_values_with_custom_fields, 'custom_field' );
		$this->replace_custom_field_or_taxonomy_replace_vars( $unique_custom_fields, $wpdb, 'custom_field', 'cf' );

		// Map `#tax_name-{tax-slug}` to `%%ct_{tax-slug}%%``.
		$meta_values_with_custom_taxonomies = $this->get_meta_values_with_custom_field_or_taxonomy( $wpdb, 'tax_name' );
		$unique_custom_taxonomies           = $this->get_unique_custom_fields_or_taxonomies( $meta_values_with_custom_taxonomies, 'tax_name' );
		$this->replace_custom_field_or_taxonomy_replace_vars( $unique_custom_taxonomies, $wpdb, 'tax_name', 'ct' );
	}

	/**
	 * Filters out all unique custom fields/taxonomies/etc. used in an AiOSEO replace var.
	 *
	 * @param string[] $meta_values   An array of all the meta values that
	 *                                contain one or more AIOSEO custom field replace vars
	 *                                (in the form `#custom_field-xyz`).
	 * @param string   $aioseo_prefix The AiOSEO prefix to use
	 *                                (e.g. `custom-field` for custom fields or `tax_name` for custom taxonomies).
	 *
	 * @return string[] An array of all the unique custom fields/taxonomies/etc. used in the replace vars.
	 *                  E.g. `xyz` in the above example.
	 */
	protected function get_unique_custom_fields_or_taxonomies( $meta_values, $aioseo_prefix ) {
		$unique_custom_fields_or_taxonomies = [];

		foreach ( $meta_values as $meta_value ) {
			// Find all custom field replace vars, store them in `$matches`.
			preg_match_all(
				"/#$aioseo_prefix-([\w-]+)/",
				$meta_value,
				$matches
			);

			/*
			 * `$matches[1]` contain the captured matches of the
			 * first capturing group (the `([\w-]+)` in the regex above).
			 */
			$custom_fields_or_taxonomies = $matches[1];

			foreach ( $custom_fields_or_taxonomies as $custom_field_or_taxonomy ) {
				$unique_custom_fields_or_taxonomies[ trim( $custom_field_or_taxonomy ) ] = 1;
			}
		}

		return array_keys( $unique_custom_fields_or_taxonomies );
	}

	/**
	 * Replaces every AIOSEO custom field/taxonomy/etc. replace var with the Yoast version.
	 *
	 * E.g. `#custom_field-xyz` becomes `%%cf_xyz%%`.
	 *
	 * @param string[] $unique_custom_fields_or_taxonomies An array of unique custom fields to replace the replace vars of.
	 * @param wpdb     $wpdb                               The WordPress database object.
	 * @param string   $aioseo_prefix                      The AiOSEO prefix to use
	 *                                                     (e.g. `custom-field` for custom fields or `tax_name` for custom taxonomies).
	 * @param string   $yoast_prefix                       The Yoast prefix to use (e.g. `cf` for custom fields).
	 *
	 * @return void
	 */
	protected function replace_custom_field_or_taxonomy_replace_vars( $unique_custom_fields_or_taxonomies, $wpdb, $aioseo_prefix, $yoast_prefix ) {
		foreach ( $unique_custom_fields_or_taxonomies as $unique_custom_field_or_taxonomy ) {
			$aioseo_variable = "#{$aioseo_prefix}-{$unique_custom_field_or_taxonomy}";
			$yoast_variable  = "%%{$yoast_prefix}_{$unique_custom_field_or_taxonomy}%%";

			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching
			$wpdb->query(
				$wpdb->prepare(
					'UPDATE tmp_meta_table SET meta_value = REPLACE( meta_value, %s, %s )',
					$aioseo_variable,
					$yoast_variable
				)
			);
		}
	}

	// phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

	/**
	 * Retrieve all the meta values from the temporary meta table that contain
	 * at least one AiOSEO custom field replace var.
	 *
	 * @param wpdb   $wpdb          The WordPress database object.
	 * @param string $aioseo_prefix The AiOSEO prefix to use
	 *                              (e.g. `custom-field` for custom fields or `tax_name` for custom taxonomies).
	 *
	 * @return string[] All meta values that contain at least one AioSEO custom field replace var.
	 */
	protected function get_meta_values_with_custom_field_or_taxonomy( $wpdb, $aioseo_prefix ) {
		return $wpdb->get_col(
			$wpdb->prepare(
				'SELECT meta_value FROM tmp_meta_table WHERE meta_value LIKE %s',
				"%#$aioseo_prefix-%"
			)
		);
	}

	// phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching

	/**
	 * Detects whether there is AIOSEO data to import by looking whether the AIOSEO data have been cleaned up.
	 *
	 * @return bool Boolean indicating whether there is something to import.
	 */
	protected function detect() {
		$aioseo_cleanup_action = YoastSEO()->classes->get( Aioseo_Cleanup_Action::class );
		return ( $aioseo_cleanup_action->get_total_unindexed() > 0 );
	}

	/**
	 * Import AIOSEO post data from their custom indexable table. Not currently used.
	 *
	 * @return void
	 */
	protected function import() {
		// This is overriden from the import.js and never run.
		$aioseo_posts_import_action = YoastSEO()->classes->get( Aioseo_Posts_Importing_Action::class );
		$aioseo_posts_import_action->index();
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/admin/import/plugins/class-importers.php                             0000644                 00000001634 15122266554 0022576 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Plugin_Importers.
 *
 * Object which contains all importers.
 */
class WPSEO_Plugin_Importers {

	/**
	 * List of supported importers.
	 *
	 * @var array
	 */
	private static $importers = [
		'WPSEO_Import_AIOSEO',
		'WPSEO_Import_AIOSEO_V4',
		'WPSEO_Import_Greg_SEO',
		'WPSEO_Import_HeadSpace',
		'WPSEO_Import_Jetpack_SEO',
		'WPSEO_Import_WP_Meta_SEO',
		'WPSEO_Import_Platinum_SEO',
		'WPSEO_Import_Premium_SEO_Pack',
		'WPSEO_Import_RankMath',
		'WPSEO_Import_SEOPressor',
		'WPSEO_Import_SEO_Framework',
		'WPSEO_Import_Smartcrawl_SEO',
		'WPSEO_Import_Squirrly',
		'WPSEO_Import_Ultimate_SEO',
		'WPSEO_Import_WooThemes_SEO',
		'WPSEO_Import_WPSEO',
	];

	/**
	 * Returns an array of importers available.
	 *
	 * @return array Available importers.
	 */
	public static function get() {
		return self::$importers;
	}
}
                                                                                                    plugins/wordpress-seo-extended/admin/import/plugins/class-import-greg-high-performance-seo.php      0000644                 00000001371 15122266554 0027004 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Ultimate SEO.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean Ultimate SEO post metadata.
 */
class WPSEO_Import_Greg_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = "Greg's High Performance SEO";

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_ghpseo_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_ghpseo_alternative_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_ghpseo_secondary_title',
			'new_key' => 'title',
		],
	];
}
                                                                                                                                                                                                                                                                       plugins/wordpress-seo-extended/admin/import/plugins/class-import-headspace.php                      0000644                 00000001762 15122266554 0024001 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from HeadSpace.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_HeadSpace.
 *
 * Class with functionality to import & clean HeadSpace SEO post metadata.
 */
class WPSEO_Import_HeadSpace extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'HeadSpace SEO';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_headspace_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_headspace_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_headspace_page_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_headspace_noindex',
			'new_key' => 'meta-robots-noindex',
			'convert' => [ 'on' => 1 ],
		],
		[
			'old_key' => '_headspace_nofollow',
			'new_key' => 'meta-robots-nofollow',
			'convert' => [ 'on' => 1 ],
		],
	];
}
              plugins/wordpress-seo-extended/admin/import/plugins/class-import-jetpack.php                        0000644                 00000001333 15122266554 0023477 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Jetpack's Advanced SEO settings.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_Jetpack_SEO.
 *
 * Class with functionality to import & clean Jetpack SEO post metadata.
 */
class WPSEO_Import_Jetpack_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'Jetpack';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = 'advanced_seo_description';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => 'advanced_seo_description',
			'new_key' => 'metadesc',
		],
	];
}
                                                                                                                                                                                                                                                                                                     plugins/wordpress-seo-extended/admin/import/plugins/class-import-platinum-seo-pack.php              0000644                 00000005510 15122266554 0025410 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Platinum SEO Pack.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean Ultimate SEO post metadata.
 */
class WPSEO_Import_Platinum_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'Platinum SEO Pack';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = 'title';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => 'description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => 'title',
			'new_key' => 'title',
		],
	];

	/**
	 * Runs the import of post meta keys stored by Platinum SEO Pack.
	 *
	 * @return bool
	 */
	protected function import() {
		$return = parent::import();
		if ( $return ) {
			$this->import_robots_meta();
		}

		return $return;
	}

	/**
	 * Cleans up all the meta values Platinum SEO pack creates.
	 *
	 * @return bool
	 */
	protected function cleanup() {
		$this->meta_key = 'title';
		parent::cleanup();

		$this->meta_key = 'description';
		parent::cleanup();

		$this->meta_key = 'metarobots';
		parent::cleanup();

		return true;
	}

	/**
	 * Finds all the robotsmeta fields to import and deals with them.
	 *
	 * There are four potential values that Platinum SEO stores:
	 * - index,folllow
	 * - index,nofollow
	 * - noindex,follow
	 * - noindex,nofollow
	 *
	 * We only have to deal with the latter 3, the first is our default.
	 *
	 * @return void
	 */
	protected function import_robots_meta() {
		$this->import_by_meta_robots( 'index,nofollow', [ 'nofollow' ] );
		$this->import_by_meta_robots( 'noindex,follow', [ 'noindex' ] );
		$this->import_by_meta_robots( 'noindex,nofollow', [ 'noindex', 'nofollow' ] );
	}

	/**
	 * Imports the values for all index, nofollow posts.
	 *
	 * @param string $value The meta robots value to find posts for.
	 * @param array  $metas The meta field(s) to save.
	 *
	 * @return void
	 */
	protected function import_by_meta_robots( $value, $metas ) {
		$posts = $this->find_posts_by_robots_meta( $value );
		if ( ! $posts ) {
			return;
		}

		foreach ( $posts as $post_id ) {
			foreach ( $metas as $meta ) {
				$this->maybe_save_post_meta( 'meta-robots-' . $meta, 1, $post_id );
			}
		}
	}

	/**
	 * Finds posts by a given meta robots value.
	 *
	 * @param string $meta_value Robots meta value.
	 *
	 * @return array|bool Array of Post IDs on success, false on failure.
	 */
	protected function find_posts_by_robots_meta( $meta_value ) {
		$posts = get_posts(
			[
				'post_type'  => 'any',
				'meta_key'   => 'robotsmeta',
				'meta_value' => $meta_value,
				'order'      => 'ASC',
				'fields'     => 'ids',
				'nopaging'   => true,
			]
		);
		if ( empty( $posts ) ) {
			return false;
		}
		return $posts;
	}
}
                                                                                                                                                                                        plugins/wordpress-seo-extended/admin/import/plugins/class-import-premium-seo-pack.php               0000644                 00000001455 15122266554 0025241 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Premium SEO Pack.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean Premium SEO Pack post metadata.
 */
class WPSEO_Import_Premium_SEO_Pack extends WPSEO_Import_Squirrly {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'Premium SEO Pack';

	/**
	 * WPSEO_Import_Premium_SEO_Pack constructor.
	 */
	public function __construct() {
		parent::__construct();

		global $wpdb;
		$this->table_name = $wpdb->prefix . 'psp';
		$this->meta_key   = '';
	}

	/**
	 * Returns the query to return an identifier for the posts to import.
	 *
	 * @return string
	 */
	protected function retrieve_posts_query() {
		return "SELECT URL AS identifier FROM {$this->table_name} WHERE blog_id = %d";
	}
}
                                                                                                                                                                                                                   plugins/wordpress-seo-extended/admin/import/plugins/class-import-rankmath.php                       0000644                 00000011255 15122266554 0023667 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from RankMath.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import RankMath post metadata.
 */
class WPSEO_Import_RankMath extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'RankMath';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = 'rank_math_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => 'rank_math_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => 'rank_math_title',
			'new_key' => 'title',
		],
		[
			'old_key' => 'rank_math_canonical_url',
			'new_key' => 'canonical',
		],
		[
			'old_key' => 'rank_math_primary_category',
			'new_key' => 'primary_category',
		],
		[
			'old_key' => 'rank_math_facebook_title',
			'new_key' => 'opengraph-title',
		],
		[
			'old_key' => 'rank_math_facebook_description',
			'new_key' => 'opengraph-description',
		],
		[
			'old_key' => 'rank_math_facebook_image',
			'new_key' => 'opengraph-image',
		],
		[
			'old_key' => 'rank_math_facebook_image_id',
			'new_key' => 'opengraph-image-id',
		],
		[
			'old_key' => 'rank_math_twitter_title',
			'new_key' => 'twitter-title',
		],
		[
			'old_key' => 'rank_math_twitter_description',
			'new_key' => 'twitter-description',
		],
		[
			'old_key' => 'rank_math_twitter_image',
			'new_key' => 'twitter-image',
		],
		[
			'old_key' => 'rank_math_twitter_image_id',
			'new_key' => 'twitter-image-id',
		],
		[
			'old_key' => 'rank_math_focus_keyword',
			'new_key' => 'focuskw',
		],
	];

	/**
	 * Handles post meta data to import.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		global $wpdb;
		// Replace % with %% as their variables are the same except for that.
		$wpdb->query( "UPDATE $wpdb->postmeta SET meta_value = REPLACE( meta_value, '%', '%%' ) WHERE meta_key IN ( 'rank_math_description', 'rank_math_title' )" );

		$this->import_meta_robots();
		$return = $this->meta_keys_clone( $this->clone_keys );

		// Return %% to % so our import is non-destructive.
		$wpdb->query( "UPDATE $wpdb->postmeta SET meta_value = REPLACE( meta_value, '%%', '%' ) WHERE meta_key IN ( 'rank_math_description', 'rank_math_title' )" );

		if ( $return ) {
			$this->import_settings();
		}

		return $return;
	}

	/**
	 * RankMath stores robots meta quite differently, so we have to parse it out.
	 *
	 * @return void
	 */
	private function import_meta_robots() {
		global $wpdb;
		$post_metas = $wpdb->get_results( "SELECT post_id, meta_value FROM $wpdb->postmeta WHERE meta_key = 'rank_math_robots'" );
		foreach ( $post_metas as $post_meta ) {
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions -- Reason: We can't control the form in which Rankmath sends the data.
			$robots_values = unserialize( $post_meta->meta_value );
			foreach ( [ 'noindex', 'nofollow' ] as $directive ) {
				$directive_key = array_search( $directive, $robots_values, true );
				if ( $directive_key !== false ) {
					update_post_meta( $post_meta->post_id, '_yoast_wpseo_meta-robots-' . $directive, 1 );
					unset( $robots_values[ $directive_key ] );
				}
			}
			if ( count( $robots_values ) > 0 ) {
				$value = implode( ',', $robots_values );
				update_post_meta( $post_meta->post_id, '_yoast_wpseo_meta-robots-adv', $value );
			}
		}
	}

	/**
	 * Imports some of the RankMath settings.
	 *
	 * @return void
	 */
	private function import_settings() {
		$settings = [
			'title_separator'      => 'separator',
			'homepage_title'       => 'title-home-wpseo',
			'homepage_description' => 'metadesc-home-wpseo',
			'author_archive_title' => 'title-author-wpseo',
			'date_archive_title'   => 'title-archive-wpseo',
			'search_title'         => 'title-search-wpseo',
			'404_title'            => 'title-404-wpseo',
			'pt_post_title'        => 'title-post',
			'pt_page_title'        => 'title-page',
		];
		$options  = get_option( 'rank-math-options-titles' );

		foreach ( $settings as $import_setting_key => $setting_key ) {
			if ( ! empty( $options[ $import_setting_key ] ) ) {
				$value = $options[ $import_setting_key ];
				// Make sure replace vars work.
				$value = str_replace( '%', '%%', $value );
				WPSEO_Options::set( $setting_key, $value );
			}
		}
	}

	/**
	 * Removes the plugin data from the database.
	 *
	 * @return bool Cleanup status.
	 */
	protected function cleanup() {
		$return = parent::cleanup();
		if ( $return ) {
			global $wpdb;
			$wpdb->query( "DELETE FROM $wpdb->options WHERE option_name LIKE 'rank-math-%'" );
			$wpdb->query( "DELETE FROM $wpdb->options WHERE option_name LIKE '%rank_math%'" );
		}

		return $return;
	}
}
                                                                                                                                                                                                                                                                                                                                                   plugins/wordpress-seo-extended/admin/import/plugins/class-import-seo-framework.php                  0000644                 00000003464 15122266554 0024646 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from SEO Framework.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean SEO Framework post metadata.
 */
class WPSEO_Import_SEO_Framework extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'The SEO Framework';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_genesis_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_genesis_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_genesis_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_genesis_noindex',
			'new_key' => 'meta-robots-noindex',
		],
		[
			'old_key' => '_genesis_nofollow',
			'new_key' => 'meta-robots-nofollow',
		],
		[
			'old_key' => '_genesis_canonical_uri',
			'new_key' => 'canonical',
		],
		[
			'old_key' => '_open_graph_title',
			'new_key' => 'opengraph-title',
		],
		[
			'old_key' => '_open_graph_description',
			'new_key' => 'opengraph-description',
		],
		[
			'old_key' => '_social_image_url',
			'new_key' => 'opengraph-image',
		],
		[
			'old_key' => '_twitter_title',
			'new_key' => 'twitter-title',
		],
		[
			'old_key' => '_twitter_description',
			'new_key' => 'twitter-description',
		],
	];

	/**
	 * Removes all the metadata set by the SEO Framework plugin.
	 *
	 * @return bool
	 */
	protected function cleanup() {
		$set1 = parent::cleanup();

		$this->meta_key = '_social_image_%';
		$set2           = parent::cleanup();

		$this->meta_key = '_twitter_%';
		$set3           = parent::cleanup();

		$this->meta_key = '_open_graph_%';
		$set4           = parent::cleanup();

		return ( $set1 || $set2 || $set3 || $set4 );
	}
}
                                                                                                                                                                                                            plugins/wordpress-seo-extended/admin/import/plugins/class-import-seopressor.php                     0000644                 00000011445 15122266554 0024267 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from SEOPressor.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_SEOPressor.
 *
 * Class with functionality to import & clean SEOPressor post metadata.
 */
class WPSEO_Import_SEOPressor extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'SEOpressor';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_seop_settings';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_seop_settings',
		],
	];

	/**
	 * Imports the post meta values to Yoast SEO.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		// Query for all the posts that have an _seop_settings meta set.
		$query_posts = new WP_Query( 'post_type=any&meta_key=_seop_settings&order=ASC&fields=ids&nopaging=true' );
		foreach ( $query_posts->posts as $post_id ) {
			$this->import_post_focus_keywords( $post_id );
			$this->import_seopressor_post_settings( $post_id );
		}

		return true;
	}

	/**
	 * Removes all the post meta fields SEOpressor creates.
	 *
	 * @return bool Cleanup status.
	 */
	protected function cleanup() {
		global $wpdb;

		// If we get to replace the data, let's do some proper cleanup.
		return $wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE '_seop_%'" );
	}

	/**
	 * Imports the data. SEOpressor stores most of the data in one post array, this loops over it.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return void
	 */
	private function import_seopressor_post_settings( $post_id ) {
		$settings = get_post_meta( $post_id, '_seop_settings', true );

		foreach (
			[
				'fb_description'   => 'opengraph-description',
				'fb_title'         => 'opengraph-title',
				'fb_type'          => 'og_type',
				'fb_img'           => 'opengraph-image',
				'meta_title'       => 'title',
				'meta_description' => 'metadesc',
				'meta_canonical'   => 'canonical',
				'tw_description'   => 'twitter-description',
				'tw_title'         => 'twitter-title',
				'tw_image'         => 'twitter-image',
			] as $seopressor_key => $yoast_key ) {
			$this->import_meta_helper( $seopressor_key, $yoast_key, $settings, $post_id );
		}

		if ( isset( $settings['meta_rules'] ) ) {
			$this->import_post_robots( $settings['meta_rules'], $post_id );
		}
	}

	/**
	 * Imports the focus keywords, and stores them for later use.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return void
	 */
	private function import_post_focus_keywords( $post_id ) {
		// Import the focus keyword.
		$focuskw = trim( get_post_meta( $post_id, '_seop_kw_1', true ) );
		$this->maybe_save_post_meta( 'focuskw', $focuskw, $post_id );

		// Import additional focus keywords for use in premium.
		$focuskw2 = trim( get_post_meta( $post_id, '_seop_kw_2', true ) );
		$focuskw3 = trim( get_post_meta( $post_id, '_seop_kw_3', true ) );

		$focus_keywords = [];
		if ( ! empty( $focuskw2 ) ) {
			$focus_keywords[] = $focuskw2;
		}
		if ( ! empty( $focuskw3 ) ) {
			$focus_keywords[] = $focuskw3;
		}

		if ( $focus_keywords !== [] ) {
			$this->maybe_save_post_meta( 'focuskeywords', WPSEO_Utils::format_json_encode( $focus_keywords ), $post_id );
		}
	}

	/**
	 * Retrieves the SEOpressor robot value and map this to Yoast SEO values.
	 *
	 * @param string $meta_rules The meta rules taken from the SEOpressor settings array.
	 * @param int    $post_id    The post id of the current post.
	 *
	 * @return void
	 */
	private function import_post_robots( $meta_rules, $post_id ) {
		$seopressor_robots = explode( '#|#|#', $meta_rules );
		$robot_value       = $this->get_robot_value( $seopressor_robots );

		// Saving the new meta values for Yoast SEO.
		$this->maybe_save_post_meta( 'meta-robots-noindex', $robot_value['index'], $post_id );
		$this->maybe_save_post_meta( 'meta-robots-nofollow', $robot_value['follow'], $post_id );
		$this->maybe_save_post_meta( 'meta-robots-adv', $robot_value['advanced'], $post_id );
	}

	/**
	 * Gets the robot config by given SEOpressor robots value.
	 *
	 * @param array $seopressor_robots The value in SEOpressor that needs to be converted to the Yoast format.
	 *
	 * @return array The robots values in Yoast format.
	 */
	private function get_robot_value( $seopressor_robots ) {
		$return = [
			'index'    => 2,
			'follow'   => 0,
			'advanced' => '',
		];

		if ( in_array( 'noindex', $seopressor_robots, true ) ) {
			$return['index'] = 1;
		}
		if ( in_array( 'nofollow', $seopressor_robots, true ) ) {
			$return['follow'] = 1;
		}
		foreach ( [ 'noarchive', 'nosnippet', 'noimageindex' ] as $needle ) {
			if ( in_array( $needle, $seopressor_robots, true ) ) {
				$return['advanced'] .= $needle . ',';
			}
		}
		$return['advanced'] = rtrim( $return['advanced'], ',' );

		return $return;
	}
}
                                                                                                                                                                                                                           plugins/wordpress-seo-extended/admin/import/plugins/class-import-smartcrawl.php                     0000644                 00000006221 15122266554 0024236 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Smartcrawl SEO.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean Smartcrawl SEO post metadata.
 */
class WPSEO_Import_Smartcrawl_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'Smartcrawl SEO';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_wds_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_wds_metadesc',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_wds_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_wds_canonical',
			'new_key' => 'canonical',
		],
		[
			'old_key' => '_wds_focus-keywords',
			'new_key' => 'focuskw',
		],
		[
			'old_key' => '_wds_meta-robots-noindex',
			'new_key' => 'meta-robots-noindex',
		],
		[
			'old_key' => '_wds_meta-robots-nofollow',
			'new_key' => 'meta-robots-nofollow',
		],
	];

	/**
	 * Used for importing Twitter and Facebook meta's.
	 *
	 * @var array
	 */
	protected $social_keys = [];

	/**
	 * Handles post meta data to import.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		$return = parent::import();
		if ( $return ) {
			$this->import_opengraph();
			$this->import_twitter();
		}

		return $return;
	}

	/**
	 * Imports the OpenGraph meta keys saved by Smartcrawl.
	 *
	 * @return bool Import status.
	 */
	protected function import_opengraph() {
		$this->social_keys = [
			'title'       => 'opengraph-title',
			'description' => 'opengraph-description',
			'images'      => 'opengraph-image',
		];
		return $this->post_find_import( '_wds_opengraph' );
	}

	/**
	 * Imports the Twitter meta keys saved by Smartcrawl.
	 *
	 * @return bool Import status.
	 */
	protected function import_twitter() {
		$this->social_keys = [
			'title'       => 'twitter-title',
			'description' => 'twitter-description',
		];
		return $this->post_find_import( '_wds_twitter' );
	}

	/**
	 * Imports a post's serialized post meta values.
	 *
	 * @param int    $post_id Post ID.
	 * @param string $key     The meta key to import.
	 *
	 * @return void
	 */
	protected function import_serialized_post_meta( $post_id, $key ) {
		$data = get_post_meta( $post_id, $key, true );
		$data = maybe_unserialize( $data );
		foreach ( $this->social_keys as $key => $meta_key ) {
			if ( ! isset( $data[ $key ] ) ) {
				return;
			}
			$value = $data[ $key ];
			if ( is_array( $value ) ) {
				$value = $value[0];
			}
			$this->maybe_save_post_meta( $meta_key, $value, $post_id );
		}
	}

	/**
	 * Finds all the posts with a certain meta key and imports its values.
	 *
	 * @param string $key The meta key to search for.
	 *
	 * @return bool Import status.
	 */
	protected function post_find_import( $key ) {
		$query_posts = new WP_Query( 'post_type=any&meta_key=' . $key . '&order=ASC&fields=ids&nopaging=true' );

		if ( empty( $query_posts->posts ) ) {
			return false;
		}

		foreach ( array_values( $query_posts->posts ) as $post_id ) {
			$this->import_serialized_post_meta( $post_id, $key );
		}

		return true;
	}
}
                                                                                                                                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/admin/import/plugins/class-import-squirrly.php                       0000644                 00000012054 15122266554 0023752 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Squirrly.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean Squirrly post metadata.
 */
class WPSEO_Import_Squirrly extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'Squirrly SEO';

	/**
	 * Holds the name of the table Squirrly uses to store data.
	 *
	 * @var string
	 */
	protected $table_name;

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_sq_post_keyword';

	/**
	 * Data to import from (and the target to field) the serialized array stored in the SEO field in the Squirrly table.
	 *
	 * @var array
	 */
	protected $seo_field_keys = [
		'noindex'        => 'meta-robots-noindex',
		'nofollow'       => 'meta-robots-nofollow',
		'title'          => 'title',
		'description'    => 'metadesc',
		'canonical'      => 'canonical',
		'cornerstone'    => '_yst_is_cornerstone',
		'tw_media'       => 'twitter-image',
		'tw_title'       => 'twitter-title',
		'tw_description' => 'twitter-description',
		'og_title'       => 'opengraph-title',
		'og_description' => 'opengraph-description',
		'og_media'       => 'opengraph-image',
		'focuskw'        => 'focuskw',
	];

	/**
	 * WPSEO_Import_Squirrly constructor.
	 */
	public function __construct() {
		parent::__construct();

		global $wpdb;
		$this->table_name = $wpdb->prefix . 'qss';
	}

	/**
	 * Imports the post meta values to Yoast SEO.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		$results = $this->retrieve_posts();
		foreach ( $results as $post ) {
			$return = $this->import_post_values( $post->identifier );
			if ( ! $return ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Retrieve the posts from the Squirrly Database.
	 *
	 * @return array Array of post IDs from the DB.
	 */
	protected function retrieve_posts() {
		global $wpdb;
		return $wpdb->get_results(
			$wpdb->prepare(
				$this->retrieve_posts_query(),
				get_current_blog_id()
			)
		);
	}

	/**
	 * Returns the query to return an identifier for the posts to import.
	 *
	 * @return string Query to get post ID's from the DB.
	 */
	protected function retrieve_posts_query() {
		return "SELECT post_id AS identifier FROM {$this->table_name} WHERE blog_id = %d";
	}

	/**
	 * Removes the DB table and the post meta field Squirrly creates.
	 *
	 * @return bool Cleanup status.
	 */
	protected function cleanup() {
		global $wpdb;

		// If we can clean, let's clean.
		$wpdb->query( "DROP TABLE {$this->table_name}" );

		// This removes the post meta field for the focus keyword from the DB.
		parent::cleanup();

		// If we can still see the table, something went wrong.
		if ( $this->detect() ) {
			$this->cleanup_error_msg();
			return false;
		}

		return true;
	}

	/**
	 * Detects whether there is post meta data to import.
	 *
	 * @return bool Boolean indicating whether there is something to import.
	 */
	protected function detect() {
		global $wpdb;

		$result = $wpdb->get_var( "SHOW TABLES LIKE '{$this->table_name}'" );
		if ( is_wp_error( $result ) || is_null( $result ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Imports the data of a post out of Squirrly's DB table.
	 *
	 * @param mixed $post_identifier Post identifier, can be ID or string.
	 *
	 * @return bool Import status.
	 */
	private function import_post_values( $post_identifier ) {
		$data = $this->retrieve_post_data( $post_identifier );
		if ( ! $data ) {
			return false;
		}

		if ( ! is_numeric( $post_identifier ) ) {
			$post_id = url_to_postid( $post_identifier );
		}

		if ( is_numeric( $post_identifier ) ) {
			$post_id         = (int) $post_identifier;
			$data['focuskw'] = $this->maybe_add_focus_kw( $post_identifier );
		}

		foreach ( $this->seo_field_keys as $squirrly_key => $yoast_key ) {
			$this->import_meta_helper( $squirrly_key, $yoast_key, $data, $post_id );
		}
		return true;
	}

	/**
	 * Retrieves the Squirrly SEO data for a post from the DB.
	 *
	 * @param int $post_identifier Post ID.
	 *
	 * @return array|bool Array of data or false.
	 */
	private function retrieve_post_data( $post_identifier ) {
		global $wpdb;

		if ( is_numeric( $post_identifier ) ) {
			$post_identifier = (int) $post_identifier;
			$query_where     = 'post_id = %d';
		}
		if ( ! is_numeric( $post_identifier ) ) {
			$query_where = 'URL = %s';
		}

		$replacements = [
			get_current_blog_id(),
			$post_identifier,
		];

		$data = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT seo FROM {$this->table_name} WHERE blog_id = %d AND " . $query_where,
				$replacements
			)
		);
		if ( ! $data || is_wp_error( $data ) ) {
			return false;
		}
		$data = maybe_unserialize( $data );
		return $data;
	}

	/**
	 * Squirrly stores the focus keyword in post meta.
	 *
	 * @param int $post_id Post ID.
	 *
	 * @return string The focus keyword.
	 */
	private function maybe_add_focus_kw( $post_id ) {
		$focuskw = get_post_meta( $post_id, '_sq_post_keyword', true );
		if ( $focuskw ) {
			$focuskw = json_decode( $focuskw );
			return $focuskw->keyword;
		}
		return '';
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    plugins/wordpress-seo-extended/admin/import/plugins/class-import-ultimate-seo.php                   0000644                 00000002262 15122266554 0024470 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from Ultimate SEO.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean Ultimate SEO post metadata.
 */
class WPSEO_Import_Ultimate_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'Ultimate SEO';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_su_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_su_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_su_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_su_og_title',
			'new_key' => 'opengraph-title',
		],
		[
			'old_key' => '_su_og_description',
			'new_key' => 'opengraph-description',
		],
		[
			'old_key' => '_su_og_image',
			'new_key' => 'opengraph-image',
		],
		[
			'old_key' => '_su_meta_robots_noindex',
			'new_key' => 'meta-robots-noindex',
			'convert' => [ 'on' => 1 ],
		],
		[
			'old_key' => '_su_meta_robots_nofollow',
			'new_key' => 'meta-robots-nofollow',
			'convert' => [ 'on' => 1 ],
		],
	];
}
                                                                                                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/admin/import/plugins/class-import-woothemes-seo.php                  0000644                 00000004751 15122266554 0024663 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from WooThemes SEO.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_WooThemes_SEO
 *
 * Class with functionality to import & clean WooThemes SEO post metadata.
 */
class WPSEO_Import_WooThemes_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'WooThemes SEO';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = 'seo_title';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => 'seo_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => 'seo_title',
			'new_key' => 'title',
		],
		[
			'old_key' => 'seo_noindex',
			'new_key' => 'meta-robots-noindex',
		],
		[
			'old_key' => 'seo_follow',
			'new_key' => 'meta-robots-nofollow',
		],
	];

	/**
	 * Holds the meta fields we can delete after import.
	 *
	 * @var array
	 */
	protected $cleanup_metas = [
		'seo_follow',
		'seo_noindex',
		'seo_title',
		'seo_description',
		'seo_keywords',
	];

	/**
	 * Holds the options we can delete after import.
	 *
	 * @var array
	 */
	protected $cleanup_options = [
		'seo_woo_archive_layout',
		'seo_woo_single_layout',
		'seo_woo_page_layout',
		'seo_woo_wp_title',
		'seo_woo_meta_single_desc',
		'seo_woo_meta_single_key',
		'seo_woo_home_layout',
	];

	/**
	 * Cleans up the WooThemes SEO settings.
	 *
	 * @return bool Cleanup status.
	 */
	protected function cleanup() {
		$result = $this->cleanup_meta();
		if ( $result ) {
			$this->cleanup_options();
		}
		return $result;
	}

	/**
	 * Removes the Woo Options from the database.
	 *
	 * @return void
	 */
	private function cleanup_options() {
		foreach ( $this->cleanup_options as $option ) {
			delete_option( $option );
		}
	}

	/**
	 * Removes the post meta fields from the database.
	 *
	 * @return bool Cleanup status.
	 */
	private function cleanup_meta() {
		foreach ( $this->cleanup_metas as $key ) {
			$result = $this->cleanup_meta_key( $key );
			if ( ! $result ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Removes a single meta field from the postmeta table in the database.
	 *
	 * @param string $key The meta_key to delete.
	 *
	 * @return bool Cleanup status.
	 */
	private function cleanup_meta_key( $key ) {
		global $wpdb;

		$wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$wpdb->postmeta} WHERE meta_key = %s",
				$key
			)
		);
		return $wpdb->__get( 'result' );
	}
}
                       plugins/wordpress-seo-extended/admin/import/plugins/class-import-wp-meta-seo.php                    0000644                 00000003056 15122266554 0024220 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from WP Meta SEO.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class with functionality to import & clean WP Meta SEO post metadata.
 */
class WPSEO_Import_WP_Meta_SEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'WP Meta SEO';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_metaseo_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_metaseo_metadesc',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_metaseo_metatitle',
			'new_key' => 'title',
		],
		[
			'old_key' => '_metaseo_metaopengraph-title',
			'new_key' => 'opengraph-title',
		],
		[
			'old_key' => '_metaseo_metaopengraph-desc',
			'new_key' => 'opengraph-description',
		],
		[
			'old_key' => '_metaseo_metaopengraph-image',
			'new_key' => 'opengraph-image',
		],
		[
			'old_key' => '_metaseo_metatwitter-title',
			'new_key' => 'twitter-title',
		],
		[
			'old_key' => '_metaseo_metatwitter-desc',
			'new_key' => 'twitter-description',
		],
		[
			'old_key' => '_metaseo_metatwitter-image',
			'new_key' => 'twitter-image',
		],
		[
			'old_key' => '_metaseo_metaindex',
			'new_key' => 'meta-robots-noindex',
			'convert' => [
				'index'   => 0,
				'noindex' => 1,
			],
		],
		[
			'old_key' => '_metaseo_metafollow',
			'new_key' => 'meta-robots-nofollow',
			'convert' => [
				'follow'   => 0,
				'nofollow' => 1,
			],
		],
	];
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/admin/import/plugins/class-import-wpseo.php                          0000644                 00000016366 15122266554 0023227 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * File with the class to handle data from wpSEO.de.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_WPSEO.
 *
 * Class with functionality to import & clean wpSEO.de post metadata.
 */
class WPSEO_Import_WPSEO extends WPSEO_Plugin_Importer {

	/**
	 * The plugin name.
	 *
	 * @var string
	 */
	protected $plugin_name = 'wpSEO.de';

	/**
	 * Meta key, used in SQL LIKE clause for delete query.
	 *
	 * @var string
	 */
	protected $meta_key = '_wpseo_edit_%';

	/**
	 * Array of meta keys to detect and import.
	 *
	 * @var array
	 */
	protected $clone_keys = [
		[
			'old_key' => '_wpseo_edit_description',
			'new_key' => 'metadesc',
		],
		[
			'old_key' => '_wpseo_edit_title',
			'new_key' => 'title',
		],
		[
			'old_key' => '_wpseo_edit_canonical',
			'new_key' => 'canonical',
		],
		[
			'old_key' => '_wpseo_edit_og_title',
			'new_key' => 'opengraph-title',
		],
		[
			'old_key' => '_wpseo_edit_og_description',
			'new_key' => 'opengraph-description',
		],
		[
			'old_key' => '_wpseo_edit_og_image',
			'new_key' => 'opengraph-image',
		],
		[
			'old_key' => '_wpseo_edit_twittercard_title',
			'new_key' => 'twitter-title',
		],
		[
			'old_key' => '_wpseo_edit_twittercard_description',
			'new_key' => 'twitter-description',
		],
		[
			'old_key' => '_wpseo_edit_twittercard_image',
			'new_key' => 'twitter-image',
		],
	];

	/**
	 * The values 1 - 6 are the configured values from wpSEO. This array will map the values of wpSEO to our values.
	 *
	 * There are some double array like 1-6 and 3-4. The reason is they only set the index value. The follow value is
	 * the default we use in the cases there isn't a follow value present.
	 *
	 * @var array
	 */
	private $robot_values = [
		// In wpSEO: index, follow.
		1 => [
			'index'  => 2,
			'follow' => 0,
		],
		// In wpSEO: index, nofollow.
		2 => [
			'index'  => 2,
			'follow' => 1,
		],
		// In wpSEO: noindex.
		3 => [
			'index'  => 1,
			'follow' => 0,
		],
		// In wpSEO: noindex, follow.
		4 => [
			'index'  => 1,
			'follow' => 0,
		],
		// In wpSEO: noindex, nofollow.
		5 => [
			'index'  => 1,
			'follow' => 1,
		],
		// In wpSEO: index.
		6 => [
			'index'  => 2,
			'follow' => 0,
		],
	];

	/**
	 * Imports wpSEO settings.
	 *
	 * @return bool Import success status.
	 */
	protected function import() {
		$status = parent::import();
		if ( $status ) {
			$this->import_post_robots();
			$this->import_taxonomy_metas();
		}

		return $status;
	}

	/**
	 * Removes wpseo.de post meta's.
	 *
	 * @return bool Cleanup status.
	 */
	protected function cleanup() {
		$this->cleanup_term_meta();
		$result = $this->cleanup_post_meta();
		return $result;
	}

	/**
	 * Detects whether there is post meta data to import.
	 *
	 * @return bool Boolean indicating whether there is something to import.
	 */
	protected function detect() {
		if ( parent::detect() ) {
			return true;
		}

		global $wpdb;
		$count = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE 'wpseo_category_%'" );
		if ( $count !== '0' ) {
			return true;
		}

		return false;
	}

	/**
	 * Imports the robot values from WPSEO plugin. These have to be converted to the Yoast format.
	 *
	 * @return void
	 */
	private function import_post_robots() {
		$query_posts = new WP_Query( 'post_type=any&meta_key=_wpseo_edit_robots&order=ASC&fields=ids&nopaging=true' );

		if ( ! empty( $query_posts->posts ) ) {
			foreach ( array_values( $query_posts->posts ) as $post_id ) {
				$this->import_post_robot( $post_id );
			}
		}
	}

	/**
	 * Gets the wpSEO robot value and map this to Yoast SEO values.
	 *
	 * @param int $post_id The post id of the current post.
	 *
	 * @return void
	 */
	private function import_post_robot( $post_id ) {
		$wpseo_robots = get_post_meta( $post_id, '_wpseo_edit_robots', true );
		$robot_value  = $this->get_robot_value( $wpseo_robots );

		// Saving the new meta values for Yoast SEO.
		$this->maybe_save_post_meta( 'meta-robots-noindex', $robot_value['index'], $post_id );
		$this->maybe_save_post_meta( 'meta-robots-nofollow', $robot_value['follow'], $post_id );
	}

	/**
	 * Imports the taxonomy metas from wpSEO.
	 *
	 * @return void
	 */
	private function import_taxonomy_metas() {
		$terms    = get_terms(
			[
				'taxonomy'   => get_taxonomies(),
				'hide_empty' => false,
			]
		);
		$tax_meta = get_option( 'wpseo_taxonomy_meta' );

		foreach ( $terms as $term ) {
			$this->import_taxonomy_description( $tax_meta, $term->taxonomy, $term->term_id );
			$this->import_taxonomy_robots( $tax_meta, $term->taxonomy, $term->term_id );
		}

		update_option( 'wpseo_taxonomy_meta', $tax_meta );
	}

	/**
	 * Imports the meta description to Yoast SEO.
	 *
	 * @param array  $tax_meta The array with the current metadata.
	 * @param string $taxonomy String with the name of the taxonomy.
	 * @param string $term_id  The ID of the current term.
	 *
	 * @return void
	 */
	private function import_taxonomy_description( &$tax_meta, $taxonomy, $term_id ) {
		$description = get_option( 'wpseo_' . $taxonomy . '_' . $term_id, false );
		if ( $description !== false ) {
			// Import description.
			$tax_meta[ $taxonomy ][ $term_id ]['wpseo_desc'] = $description;
		}
	}

	/**
	 * Imports the robot value to Yoast SEO.
	 *
	 * @param array  $tax_meta The array with the current metadata.
	 * @param string $taxonomy String with the name of the taxonomy.
	 * @param string $term_id  The ID of the current term.
	 *
	 * @return void
	 */
	private function import_taxonomy_robots( &$tax_meta, $taxonomy, $term_id ) {
		$wpseo_robots = get_option( 'wpseo_' . $taxonomy . '_' . $term_id . '_robots', false );
		if ( $wpseo_robots === false ) {
			return;
		}
		// The value 1, 2 and 6 are the index values in wpSEO.
		$new_robot_value = 'noindex';

		if ( in_array( (int) $wpseo_robots, [ 1, 2, 6 ], true ) ) {
			$new_robot_value = 'index';
		}

		$tax_meta[ $taxonomy ][ $term_id ]['wpseo_noindex'] = $new_robot_value;
	}

	/**
	 * Deletes the wpSEO taxonomy meta data.
	 *
	 * @param string $taxonomy String with the name of the taxonomy.
	 * @param string $term_id  The ID of the current term.
	 *
	 * @return void
	 */
	private function delete_taxonomy_metas( $taxonomy, $term_id ) {
		delete_option( 'wpseo_' . $taxonomy . '_' . $term_id );
		delete_option( 'wpseo_' . $taxonomy . '_' . $term_id . '_robots' );
	}

	/**
	 * Gets the robot config by given wpSEO robots value.
	 *
	 * @param string $wpseo_robots The value in wpSEO that needs to be converted to the Yoast format.
	 *
	 * @return string The correct robot value.
	 */
	private function get_robot_value( $wpseo_robots ) {
		if ( array_key_exists( $wpseo_robots, $this->robot_values ) ) {
			return $this->robot_values[ $wpseo_robots ];
		}

		return $this->robot_values[1];
	}

	/**
	 * Deletes wpSEO postmeta from the database.
	 *
	 * @return bool Cleanup status.
	 */
	private function cleanup_post_meta() {
		global $wpdb;

		// If we get to replace the data, let's do some proper cleanup.
		return $wpdb->query( "DELETE FROM {$wpdb->postmeta} WHERE meta_key LIKE '_wpseo_edit_%'" );
	}

	/**
	 * Cleans up the wpSEO term meta.
	 *
	 * @return void
	 */
	private function cleanup_term_meta() {
		$terms = get_terms(
			[
				'taxonomy'   => get_taxonomies(),
				'hide_empty' => false,
			]
		);

		foreach ( $terms as $term ) {
			$this->delete_taxonomy_metas( $term->taxonomy, $term->term_id );
		}
	}
}
                                                                                                                                                                                                                                                                          plugins/wordpress-seo-extended/admin/index.php                                                      0000644                 00000000046 15122266554 0015557 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * Nothing to see here.
 */
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          plugins/wordpress-seo-extended/admin/interface-collection.php                                       0000644                 00000000401 15122266554 0020534 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin
 */

/**
 * Interface that represents a collection.
 */
interface WPSEO_Collection {

	/**
	 * Returns the collection data.
	 *
	 * @return array The collection data.
	 */
	public function get();
}
                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/admin/interface-installable.php                                      0000644                 00000000376 15122266554 0020706 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin
 */

/**
 * Represents the interface for an installable object.
 */
interface WPSEO_Installable {

	/**
	 * Runs the installation routine.
	 *
	 * @return void
	 */
	public function install();
}
                                                                                                                                                                                                                                                                  plugins/wordpress-seo-extended/admin/listeners/class-listener.php                                   0000644                 00000000460 15122266554 0021410 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Listeners
 */

/**
 * Dictates the required methods for a Listener implementation.
 */
interface WPSEO_Listener {

	/**
	 * Listens to an argument in the request URL and triggers an action.
	 *
	 * @return void
	 */
	public function listen();
}
                                                                                                                                                                                                                plugins/wordpress-seo-extended/admin/menu/class-admin-menu.php                                      0000644                 00000007653 15122266554 0020564 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Menu
 */

/**
 * Registers the admin menu on the left of the admin area.
 */
class WPSEO_Admin_Menu extends WPSEO_Base_Menu {

	/**
	 * Registers all hooks to WordPress.
	 *
	 * @return void
	 */
	public function register_hooks() {
		// Needs the lower than default priority so other plugins can hook underneath it without issue.
		add_action( 'admin_menu', [ $this, 'register_settings_page' ], 5 );
	}

	/**
	 * Registers the menu item submenus.
	 *
	 * @return void
	 */
	public function register_settings_page() {
		$manage_capability   = $this->get_manage_capability();
		$page_identifier     = $this->get_page_identifier();
		$admin_page_callback = $this->get_admin_page_callback();

		// Get all submenu pages.
		$submenu_pages = $this->get_submenu_pages();

		foreach ( $submenu_pages as $submenu_page ) {
			if ( WPSEO_Capability_Utils::current_user_can( $submenu_page[3] ) ) {
				$manage_capability   = $submenu_page[3];
				$page_identifier     = $submenu_page[4];
				$admin_page_callback = $submenu_page[5];
				break;
			}
		}

		foreach ( $submenu_pages as $index => $submenu_page ) {
			$submenu_pages[ $index ][0] = $page_identifier;
		}

		/*
		 * The current user has the capability to control anything.
		 * This means that all submenus and dashboard can be shown.
		 */
		global $admin_page_hooks;

		add_menu_page(
			'Yoast SEO: ' . __( 'Dashboard', 'wordpress-seo' ),
			'Yoast SEO ' . $this->get_notification_counter(),
			$manage_capability,
			$page_identifier,
			$admin_page_callback,
			$this->get_icon_svg(),
			99
		);

		// Wipe notification bits from hooks.
		// phpcs:ignore WordPress.WP.GlobalVariablesOverride -- This is a deliberate action.
		$admin_page_hooks[ $page_identifier ] = 'seo';

		// Add submenu items to the main menu if possible.
		$this->register_submenu_pages( $submenu_pages );
	}

	/**
	 * Returns the list of registered submenu pages.
	 *
	 * @return array List of registered submenu pages.
	 */
	public function get_submenu_pages() {
		global $wpseo_admin;

		$search_console_callback = null;

		// Account for when the available submenu pages are requested from outside the admin.
		if ( isset( $wpseo_admin ) ) {
			$google_search_console   = new WPSEO_GSC();
			$search_console_callback = [ $google_search_console, 'display' ];
		}

		// Submenu pages.
		$submenu_pages = [
			$this->get_submenu_page( __( 'General', 'wordpress-seo' ), $this->get_page_identifier() ),
			$this->get_submenu_page(
				__( 'Search Console', 'wordpress-seo' ),
				'wpseo_search_console',
				$search_console_callback
			),
			$this->get_submenu_page( __( 'Tools', 'wordpress-seo' ), 'wpseo_tools' ),
			$this->get_submenu_page( $this->get_license_page_title(), 'wpseo_licenses' ),
		];

		/**
		 * Filter: 'wpseo_submenu_pages' - Collects all submenus that need to be shown.
		 *
		 * @param array $submenu_pages List with all submenu pages.
		 */
		return (array) apply_filters( 'wpseo_submenu_pages', $submenu_pages );
	}

	/**
	 * Returns the notification count in HTML format.
	 *
	 * @return string The notification count in HTML format.
	 */
	protected function get_notification_counter() {
		$notification_center = Yoast_Notification_Center::get();
		$notification_count  = $notification_center->get_notification_count();

		// Add main page.
		/* translators: Hidden accessibility text; %s: number of notifications. */
		$notifications = sprintf( _n( '%s notification', '%s notifications', $notification_count, 'wordpress-seo' ), number_format_i18n( $notification_count ) );

		return sprintf( '<span class="update-plugins count-%1$d"><span class="plugin-count" aria-hidden="true">%1$d</span><span class="screen-reader-text">%2$s</span></span>', $notification_count, $notifications );
	}

	/**
	 * Returns the capability that is required to manage all options.
	 *
	 * @return string Capability to check against.
	 */
	protected function get_manage_capability() {
		return 'wpseo_manage_options';
	}
}
                                                                                     plugins/wordpress-seo-extended/admin/menu/class-base-menu.php                                       0000644                 00000021106 15122266554 0020373 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Menu
 */

use Yoast\WP\SEO\Promotions\Application\Promotion_Manager;

/**
 * Admin menu base class.
 */
abstract class WPSEO_Base_Menu implements WPSEO_WordPress_Integration {

	/**
	 * A menu.
	 *
	 * @var WPSEO_Menu
	 */
	protected $menu;

	/**
	 * Constructs the Admin Menu.
	 *
	 * @param WPSEO_Menu $menu Menu to use.
	 */
	public function __construct( WPSEO_Menu $menu ) {
		$this->menu = $menu;
	}

	/**
	 * Returns the list of registered submenu pages.
	 *
	 * @return array List of registered submenu pages.
	 */
	abstract public function get_submenu_pages();

	/**
	 * Creates a submenu formatted array.
	 *
	 * @param string          $page_title Page title to use.
	 * @param string          $page_slug  Page slug to use.
	 * @param callable|null   $callback   Optional. Callback which handles the page request.
	 * @param callable[]|null $hook       Optional. Hook to trigger when the page is registered.
	 *
	 * @return array Formatted submenu.
	 */
	protected function get_submenu_page( $page_title, $page_slug, $callback = null, $hook = null ) {
		if ( $callback === null ) {
			$callback = $this->get_admin_page_callback();
		}

		return [
			$this->get_page_identifier(),
			'',
			$page_title,
			$this->get_manage_capability(),
			$page_slug,
			$callback,
			$hook,
		];
	}

	/**
	 * Registers submenu pages as menu pages.
	 *
	 * This method should only be used if the user does not have the required capabilities
	 * to access the parent menu page.
	 *
	 * @param array $submenu_pages List of submenu pages to register.
	 *
	 * @return void
	 */
	protected function register_menu_pages( $submenu_pages ) {
		if ( ! is_array( $submenu_pages ) || empty( $submenu_pages ) ) {
			return;
		}

		// Loop through submenu pages and add them.
		array_walk( $submenu_pages, [ $this, 'register_menu_page' ] );
	}

	/**
	 * Registers submenu pages.
	 *
	 * @param array $submenu_pages List of submenu pages to register.
	 *
	 * @return void
	 */
	protected function register_submenu_pages( $submenu_pages ) {
		if ( ! is_array( $submenu_pages ) || empty( $submenu_pages ) ) {
			return;
		}

		// Loop through submenu pages and add them.
		array_walk( $submenu_pages, [ $this, 'register_submenu_page' ] );
	}

	/**
	 * Registers a submenu page as a menu page.
	 *
	 * This method should only be used if the user does not have the required capabilities
	 * to access the parent menu page.
	 *
	 * @param array $submenu_page {
	 *     Submenu page definition.
	 *
	 *     @type string   $0 Parent menu page slug.
	 *     @type string   $1 Page title, currently unused.
	 *     @type string   $2 Title to display in the menu.
	 *     @type string   $3 Required capability to access the page.
	 *     @type string   $4 Page slug.
	 *     @type callable $5 Callback to run when the page is rendered.
	 *     @type array    $6 Optional. List of callbacks to run when the page is loaded.
	 * }
	 *
	 * @return void
	 */
	protected function register_menu_page( $submenu_page ) {

		// If the submenu page requires the general manage capability, it must be added as an actual submenu page.
		if ( $submenu_page[3] === $this->get_manage_capability() ) {
			return;
		}

		$page_title = 'Yoast SEO: ' . $submenu_page[2];

		// Register submenu page as menu page.
		$hook_suffix = add_menu_page(
			$page_title,
			$submenu_page[2],
			$submenu_page[3],
			$submenu_page[4],
			$submenu_page[5],
			$this->get_icon_svg(),
			99
		);

		// If necessary, add hooks for the submenu page.
		if ( isset( $submenu_page[6] ) && ( is_array( $submenu_page[6] ) ) ) {
			$this->add_page_hooks( $hook_suffix, $submenu_page[6] );
		}
	}

	/**
	 * Registers a submenu page.
	 *
	 * This method will override the capability of the page to automatically use the
	 * general manage capability. Use the `register_menu_page()` method if the submenu
	 * page should actually use a different capability.
	 *
	 * @param array $submenu_page {
	 *     Submenu page definition.
	 *
	 *     @type string   $0 Parent menu page slug.
	 *     @type string   $1 Page title, currently unused.
	 *     @type string   $2 Title to display in the menu.
	 *     @type string   $3 Required capability to access the page.
	 *     @type string   $4 Page slug.
	 *     @type callable $5 Callback to run when the page is rendered.
	 *     @type array    $6 Optional. List of callbacks to run when the page is loaded.
	 * }
	 *
	 * @return void
	 */
	protected function register_submenu_page( $submenu_page ) {
		$page_title = $submenu_page[2];

		// We cannot use $submenu_page[1] because add-ons define that, so hard-code this value.
		if ( $submenu_page[4] === 'wpseo_licenses' ) {
			$page_title = $this->get_license_page_title();
		}

		/*
		 * Handle the Google Search Console special case by passing a fake parent
		 * page slug. This way, the sub-page is stil registered and can be accessed
		 * directly. Its menu item won't be displayed.
		 */
		if ( $submenu_page[4] === 'wpseo_search_console' ) {
			// Set the parent page slug to a non-existing one.
			$submenu_page[0] = 'wpseo_fake_menu_parent_page_slug';
		}

		$page_title .= ' - Yoast SEO';

		// Register submenu page.
		$hook_suffix = add_submenu_page(
			$submenu_page[0],
			$page_title,
			$submenu_page[2],
			$submenu_page[3],
			$submenu_page[4],
			$submenu_page[5]
		);

		// If necessary, add hooks for the submenu page.
		if ( isset( $submenu_page[6] ) && ( is_array( $submenu_page[6] ) ) ) {
			$this->add_page_hooks( $hook_suffix, $submenu_page[6] );
		}
	}

	/**
	 * Adds hook callbacks for a given admin page hook suffix.
	 *
	 * @param string $hook_suffix Admin page hook suffix, as returned by `add_menu_page()`
	 *                            or `add_submenu_page()`.
	 * @param array  $callbacks   Callbacks to add.
	 *
	 * @return void
	 */
	protected function add_page_hooks( $hook_suffix, array $callbacks ) {
		foreach ( $callbacks as $callback ) {
			add_action( 'load-' . $hook_suffix, $callback );
		}
	}

	/**
	 * Gets the main admin page identifier.
	 *
	 * @return string Admin page identifier.
	 */
	protected function get_page_identifier() {
		return $this->menu->get_page_identifier();
	}

	/**
	 * Checks whether the current user has capabilities to manage all options.
	 *
	 * @return bool True if capabilities are sufficient, false otherwise.
	 */
	protected function check_manage_capability() {
		return WPSEO_Capability_Utils::current_user_can( $this->get_manage_capability() );
	}

	/**
	 * Returns the capability that is required to manage all options.
	 *
	 * @return string Capability to check against.
	 */
	abstract protected function get_manage_capability();

	/**
	 * Returns the page handler callback.
	 *
	 * @return array Callback page handler.
	 */
	protected function get_admin_page_callback() {
		return [ $this->menu, 'load_page' ];
	}

	/**
	 * Returns the page title to use for the licenses page.
	 *
	 * @return string The title for the license page.
	 */
	protected function get_license_page_title() {
		static $title = null;

		if ( $title === null ) {
			$title = __( 'Premium', 'wordpress-seo' );
		}

		if ( YoastSEO()->classes->get( Promotion_Manager::class )->is( 'black-friday-2023-promotion' ) && ! YoastSEO()->helpers->product->is_premium() ) {
			$title = __( 'Premium', 'wordpress-seo' ) . '<span class="yoast-menu-bf-sale-badge">' . __( '30% OFF', 'wordpress-seo' ) . '</span>';
		}

		return $title;
	}

	/**
	 * Returns a base64 URL for the svg for use in the menu.
	 *
	 * @param bool $base64 Whether or not to return base64'd output.
	 *
	 * @return string SVG icon.
	 */
	public function get_icon_svg( $base64 = true ) {
		$svg = '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" xml:space="preserve" width="100%" height="100%" style="fill:#82878c" viewBox="0 0 512 512" role="img" aria-hidden="true" focusable="false"><g><g><g><g><path d="M203.6,395c6.8-17.4,6.8-36.6,0-54l-79.4-204h70.9l47.7,149.4l74.8-207.6H116.4c-41.8,0-76,34.2-76,76V357c0,41.8,34.2,76,76,76H173C189,424.1,197.6,410.3,203.6,395z"/></g><g><path d="M471.6,154.8c0-41.8-34.2-76-76-76h-3L285.7,365c-9.6,26.7-19.4,49.3-30.3,68h216.2V154.8z"/></g></g><path stroke-width="2.974" stroke-miterlimit="10" d="M338,1.3l-93.3,259.1l-42.1-131.9h-89.1l83.8,215.2c6,15.5,6,32.5,0,48c-7.4,19-19,37.3-53,41.9l-7.2,1v76h8.3c81.7,0,118.9-57.2,149.6-142.9L431.6,1.3H338z M279.4,362c-32.9,92-67.6,128.7-125.7,131.8v-45c37.5-7.5,51.3-31,59.1-51.1c7.5-19.3,7.5-40.7,0-60l-75-192.7h52.8l53.3,166.8l105.9-294h58.1L279.4,362z"/></g></g></svg>';

		if ( $base64 ) {
			//phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- This encoding is intended.
			return 'data:image/svg+xml;base64,' . base64_encode( $svg );
		}

		return $svg;
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                          plugins/wordpress-seo-extended/admin/menu/class-menu.php                                            0000644                 00000004232 15122266554 0017464 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Menu
 */

/**
 * Registers the regular admin menu and network admin menu implementations.
 */
class WPSEO_Menu implements WPSEO_WordPress_Integration {

	/**
	 * The page identifier used in WordPress to register the admin page.
	 *
	 * !DO NOT CHANGE THIS!
	 *
	 * @var string
	 */
	public const PAGE_IDENTIFIER = 'wpseo_dashboard';

	/**
	 * List of classes that add admin functionality.
	 *
	 * @var array
	 */
	protected $admin_features;

	/**
	 * Registers all hooks to WordPress.
	 *
	 * @return void
	 */
	public function register_hooks() {
		$admin_menu = new WPSEO_Admin_Menu( $this );
		$admin_menu->register_hooks();

		if ( WPSEO_Utils::is_plugin_network_active() ) {
			$network_admin_menu = new WPSEO_Network_Admin_Menu( $this );
			$network_admin_menu->register_hooks();
		}

		$capability_normalizer = new WPSEO_Submenu_Capability_Normalize();
		$capability_normalizer->register_hooks();
	}

	/**
	 * Returns the main menu page identifier.
	 *
	 * @return string Page identifier to use.
	 */
	public function get_page_identifier() {
		return self::PAGE_IDENTIFIER;
	}

	/**
	 * Loads the requested admin settings page.
	 *
	 * @return void
	 */
	public function load_page() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
		if ( isset( $_GET['page'] ) && is_string( $_GET['page'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
			$page = sanitize_text_field( wp_unslash( $_GET['page'] ) );
			$this->show_page( $page );
		}
	}

	/**
	 * Shows an admin settings page.
	 *
	 * @param string $page Page to display.
	 *
	 * @return void
	 */
	protected function show_page( $page ) {
		switch ( $page ) {
			case 'wpseo_tools':
				require_once WPSEO_PATH . 'admin/pages/tools.php';
				break;

			case 'wpseo_licenses':
				require_once WPSEO_PATH . 'admin/pages/licenses.php';
				break;

			case 'wpseo_files':
				require_once WPSEO_PATH . 'admin/views/tool-file-editor.php';
				break;

			default:
				require_once WPSEO_PATH . 'admin/pages/dashboard.php';
				break;
		}
	}
}
                                                                                                                                                                                                                                                                                                                                                                      plugins/wordpress-seo-extended/admin/menu/class-network-admin-menu.php                              0000644                 00000004377 15122266554 0022253 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Menu
 */

/**
 * Network Admin Menu handler.
 */
class WPSEO_Network_Admin_Menu extends WPSEO_Base_Menu {

	/**
	 * Registers all hooks to WordPress.
	 *
	 * @return void
	 */
	public function register_hooks() {
		// Needs the lower than default priority so other plugins can hook underneath it without issue.
		add_action( 'network_admin_menu', [ $this, 'register_settings_page' ], 5 );
	}

	/**
	 * Register the settings page for the Network settings.
	 *
	 * @return void
	 */
	public function register_settings_page() {
		if ( ! $this->check_manage_capability() ) {
			return;
		}

		add_menu_page(
			__( 'Network Settings', 'wordpress-seo' ) . ' - Yoast SEO',
			'Yoast SEO',
			$this->get_manage_capability(),
			$this->get_page_identifier(),
			[ $this, 'network_config_page' ],
			$this->get_icon_svg()
		);

		$submenu_pages = $this->get_submenu_pages();
		$this->register_submenu_pages( $submenu_pages );
	}

	/**
	 * Returns the list of registered submenu pages.
	 *
	 * @return array List of registered submenu pages.
	 */
	public function get_submenu_pages() {

		// Submenu pages.
		$submenu_pages = [
			$this->get_submenu_page(
				__( 'General', 'wordpress-seo' ),
				$this->get_page_identifier(),
				[ $this, 'network_config_page' ]
			),
		];

		if ( WPSEO_Utils::allow_system_file_edit() === true ) {
			$submenu_pages[] = $this->get_submenu_page( __( 'Edit Files', 'wordpress-seo' ), 'wpseo_files' );
		}

		$submenu_pages[] = $this->get_submenu_page( __( 'Extensions', 'wordpress-seo' ), 'wpseo_licenses' );

		return $submenu_pages;
	}

	/**
	 * Loads the form for the network configuration page.
	 *
	 * @return void
	 */
	public function network_config_page() {
		require_once WPSEO_PATH . 'admin/pages/network.php';
	}

	/**
	 * Checks whether the current user has capabilities to manage all options.
	 *
	 * @return bool True if capabilities are sufficient, false otherwise.
	 */
	protected function check_manage_capability() {
		return current_user_can( $this->get_manage_capability() );
	}

	/**
	 * Returns the capability that is required to manage all options.
	 *
	 * @return string Capability to check against.
	 */
	protected function get_manage_capability() {
		return 'wpseo_manage_network_options';
	}
}
                                                                                                                                                                       