lugins, [ $this, 'clear_error' ] );
		}
	}

	/**
	 * Setting an error on the screen.
	 *
	 * @param string $plugin_section          Type of conflict group (such as Open Graph or sitemap).
	 * @param string $readable_plugin_section This is the value for the translation.
	 *
	 * @return void
	 */
	protected function set_error( $plugin_section, $readable_plugin_section ) {

		$notification_center = Yoast_Notification_Center::get();

		foreach ( $this->active_conflicting_plugins[ $plugin_section ] as $plugin_file ) {

			$plugin_name = $this->get_plugin_name( $plugin_file );

			$error_message = '';
			/* translators: %1$s: 'Facebook & Open Graph' plugin name(s) of possibly conflicting plugin(s), %2$s to Yoast SEO */
			$error_message .= '<p>' . sprintf( __( 'The %1$s plugin might cause issues when used in conjunction with %2$s.', 'wordpress-seo' ), '<em>' . $plugin_name . '</em>', 'Yoast SEO' ) . '</p>';
			$error_message .= '<p>' . sprintf( $readable_plugin_section, 'Yoast SEO', $plugin_name ) . '</p>';

			/* translators: %s: 'Facebook' plugin name of possibly conflicting plugin */
			$error_message .= '<a class="button button-primary" href="' . wp_nonce_url( 'plugins.php?action=deactivate&amp;plugin=' . $plugin_file . '&amp;plugin_status=all', 'deactivate-plugin_' . $plugin_file ) . '">' . sprintf( __( 'Deactivate %s', 'wordpress-seo' ), $this->get_plugin_name( $plugin_file ) ) . '</a> ';

			$identifier = $this->get_notification_identifier( $plugin_file );

			// Add the message to the notifications center.
			$notification_center->add_notification(
				new Yoast_Notification(
					$error_message,
					[
						'type' => Yoast_Notification::ERROR,
						'id'   => 'wpseo-conflict-' . $identifier,
					]
				)
			);
		}
	}

	/**
	 * Clear the notification for a plugin.
	 *
	 * @param string $plugin_file Clear the optional notification for this plugin.
	 *
	 * @return void
	 */
	public function clear_error( $plugin_file ) {
		$identifier = $this->get_notification_identifier( $plugin_file );

		$notification_center = Yoast_Notification_Center::get();
		$notification_center->remove_notification_by_id( 'wpseo-conflict-' . $identifier );
	}

	/**
	 * Loop through the $this->plugins to check if one of the plugins is active.
	 *
	 * This method will store the active plugins in $this->active_plugins.
	 *
	 * @return void
	 */
	protected function search_active_plugins() {
		foreach ( $this->plugins as $plugin_section => $plugins ) {
			$this->check_plugins_active( $plugins, $plugin_section );
		}
	}

	/**
	 * Loop through plugins and check if each plugin is active.
	 *
	 * @param array  $plugins        Set of plugins.
	 * @param string $plugin_section Type of conflict group (such as Open Graph or sitemap).
	 *
	 * @return void
	 */
	protected function check_plugins_active( $plugins, $plugin_section ) {
		foreach ( $plugins as $plugin ) {
			if ( $this->check_plugin_is_active( $plugin ) ) {
				$this->add_active_plugin( $plugin_section, $plugin );
			}
		}
	}

	/**
	 * Check if given plugin exists in array with all_active_plugins.
	 *
	 * @param string $plugin Plugin basename string.
	 *
	 * @return bool
	 */
	protected function check_plugin_is_active( $plugin ) {
		return in_array( $plugin, $this->all_active_plugins, true );
	}

	/**
	 * Add plugin to the list of active plugins.
	 *
	 * This method will check first if key $plugin_section exists, if not it will create an empty array
	 * If $plugin itself doesn't exist it will be added.
	 *
	 * @param string $plugin_section Type of conflict group (such as Open Graph or sitemap).
	 * @param string $plugin         Plugin basename string.
	 *
	 * @return void
	 */
	protected function add_active_plugin( $plugin_section, $plugin ) {
		if ( ! array_key_exists( $plugin_section, $this->active_conflicting_plugins ) ) {
			$this->active_conflicting_plugins[ $plugin_section ] = [];
		}

		if ( ! in_array( $plugin, $this->active_conflicting_plugins[ $plugin_section ], true ) ) {
			$this->active_conflicting_plugins[ $plugin_section ][] = $plugin;
		}
	}

	/**
	 * Search in $this->plugins for the given $plugin.
	 *
	 * If there is a result it will return the plugin category.
	 *
	 * @param string $plugin Plugin basename string.
	 *
	 * @return int|string
	 */
	protected function find_plugin_category( $plugin ) {
		foreach ( $this->plugins as $plugin_section => $plugins ) {
			if ( in_array( $plugin, $plugins, true ) ) {
				return $plugin_section;
			}
		}
	}

	/**
	 * Get plugin name from file.
	 *
	 * @param string $plugin Plugin path relative to plugins directory.
	 *
	 * @return string|bool Plugin name or false when no name is set.
	 */
	protected function get_plugin_name( $plugin ) {
		$plugin_details = get_plugin_data( WP_PLUGIN_DIR . '/' . $plugin );

		if ( $plugin_details['Name'] !== '' ) {
			return $plugin_details['Name'];
		}

		return false;
	}

	/**
	 * When being in the deactivation process the currently deactivated plugin has to be removed.
	 *
	 * @return void
	 */
	private function remove_deactivated_plugin() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: On the deactivation screen the nonce is already checked by WordPress itself.
		if ( ! isset( $_GET['plugin'] ) || ! is_string( $_GET['plugin'] ) ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: On the deactivation screen the nonce is already checked by WordPress itself.
		$deactivated_plugin = sanitize_text_field( wp_unslash( $_GET['plugin'] ) );
		$key_to_remove      = array_search( $deactivated_plugin, $this->all_active_plugins, true );

		if ( $key_to_remove !== false ) {
			unset( $this->all_active_plugins[ $key_to_remove ] );
		}
	}

	/**
	 * Get the identifier from the plugin file.
	 *
	 * @param string $plugin_file Plugin file to get Identifier from.
	 *
	 * @return string
	 */
	private function get_notification_identifier( $plugin_file ) {
		return md5( $plugin_file );
	}
}
                                                                                                                                                                                plugins/wordpress-seo-extended/admin/endpoints/class-endpoint-file-size.php                         0000644                 00000003341 15122266554 0023264 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Endpoints
 */

/**
 * Represents an implementation of the WPSEO_Endpoint interface to register one or multiple endpoints.
 */
class WPSEO_Endpoint_File_Size implements WPSEO_Endpoint {

	/**
	 * The namespace of the REST route.
	 *
	 * @var string
	 */
	public const REST_NAMESPACE = 'yoast/v1';

	/**
	 * The route of the endpoint to retrieve the file size.
	 *
	 * @var string
	 */
	public const ENDPOINT_SINGULAR = 'file_size';

	/**
	 * The name of the capability needed to retrieve data using the endpoints.
	 *
	 * @var string
	 */
	public const CAPABILITY_RETRIEVE = 'manage_options';

	/**
	 * The service provider.
	 *
	 * @var WPSEO_File_Size_Service
	 */
	private $service;

	/**
	 * Sets the service provider.
	 *
	 * @param WPSEO_File_Size_Service $service The service provider.
	 */
	public function __construct( WPSEO_File_Size_Service $service ) {
		$this->service = $service;
	}

	/**
	 * Registers the routes for the endpoints.
	 *
	 * @return void
	 */
	public function register() {
		$route_args = [
			'methods'             => 'GET',
			'args'                => [
				'url' => [
					'required'    => true,
					'type'        => 'string',
					'description' => 'The url to retrieve',
				],
			],
			'callback'            => [
				$this->service,
				'get',
			],
			'permission_callback' => [
				$this,
				'can_retrieve_data',
			],
		];
		register_rest_route( self::REST_NAMESPACE, self::ENDPOINT_SINGULAR, $route_args );
	}

	/**
	 * Determines whether or not data can be retrieved for the registered endpoints.
	 *
	 * @return bool Whether or not data can be retrieved.
	 */
	public function can_retrieve_data() {
		return current_user_can( self::CAPABILITY_RETRIEVE );
	}
}
                                                                                                                                                                                                                                                                                               plugins/wordpress-seo-extended/admin/endpoints/class-endpoint.php                                   0000644                 00000000727 15122266554 0021404 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Endpoints
 */

/**
 * Dictates the required methods for an Endpoint implementation.
 */
interface WPSEO_Endpoint {

	/**
	 * Registers the routes for the endpoints.
	 *
	 * @return void
	 */
	public function register();

	/**
	 * Determines whether or not data can be retrieved for the registered endpoints.
	 *
	 * @return bool Whether or not data can be retrieved.
	 */
	public function can_retrieve_data();
}
                                         plugins/wordpress-seo-extended/admin/endpoints/class-endpoint-statistics.php                        0000644                 00000003167 15122266554 0023575 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Statistics
 */

/**
 * Represents an implementation of the WPSEO_Endpoint interface to register one or multiple endpoints.
 */
class WPSEO_Endpoint_Statistics implements WPSEO_Endpoint {

	/**
	 * The namespace of the REST route.
	 *
	 * @var string
	 */
	public const REST_NAMESPACE = 'yoast/v1';

	/**
	 * The route of the statistics endpoint.
	 *
	 * @var string
	 */
	public const ENDPOINT_RETRIEVE = 'statistics';

	/**
	 * The name of the capability needed to retrieve data using the endpoints.
	 *
	 * @var string
	 */
	public const CAPABILITY_RETRIEVE = 'read';

	/**
	 * Service to use.
	 *
	 * @var WPSEO_Statistics_Service
	 */
	protected $service;

	/**
	 * Constructs the WPSEO_Endpoint_Statistics class and sets the service to use.
	 *
	 * @param WPSEO_Statistics_Service $service Service to use.
	 */
	public function __construct( WPSEO_Statistics_Service $service ) {
		$this->service = $service;
	}

	/**
	 * Registers the REST routes that are available on the endpoint.
	 *
	 * @return void
	 */
	public function register() {
		// Register fetch config.
		$route_args = [
			'methods'             => 'GET',
			'callback'            => [ $this->service, 'get_statistics' ],
			'permission_callback' => [ $this, 'can_retrieve_data' ],
		];
		register_rest_route( self::REST_NAMESPACE, self::ENDPOINT_RETRIEVE, $route_args );
	}

	/**
	 * Determines whether or not data can be retrieved for the registered endpoints.
	 *
	 * @return bool Whether or not data can be retrieved.
	 */
	public function can_retrieve_data() {
		return current_user_can( self::CAPABILITY_RETRIEVE );
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                         plugins/wordpress-seo-extended/admin/exceptions/class-file-size-exception.php                       0000644                 00000002102 15122266554 0023612 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Exceptions
 */

/**
 * Represents named methods for exceptions.
 */
class WPSEO_File_Size_Exception extends Exception {

	/**
	 * Gets the exception for an externally hosted file.
	 *
	 * @param string $file_url The file url.
	 *
	 * @return WPSEO_File_Size_Exception Instance of the exception.
	 */
	public static function externally_hosted( $file_url ) {
		$message = sprintf(
			/* translators: %1$s expands to the requested url */
			__( 'Cannot get the size of %1$s because it is hosted externally.', 'wordpress-seo' ),
			$file_url
		);

		return new self( $message );
	}

	/**
	 * Gets the exception for when a unknown error occurs.
	 *
	 * @param string $file_url The file url.
	 *
	 * @return WPSEO_File_Size_Exception Instance of the exception.
	 */
	public static function unknown_error( $file_url ) {
		$message = sprintf(
			/* translators: %1$s expands to the requested url */
			__( 'Cannot get the size of %1$s because of unknown reasons.', 'wordpress-seo' ),
			$file_url
		);

		return new self( $message );
	}
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/admin/filters/class-abstract-post-filter.php                         0000644                 00000012471 15122266554 0023301 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Filters
 */

/**
 * Class WPSEO_Abstract_Post_Filter.
 */
abstract class WPSEO_Abstract_Post_Filter implements WPSEO_WordPress_Integration {

	/**
	 * The filter's query argument.
	 *
	 * @var string
	 */
	public const FILTER_QUERY_ARG = 'yoast_filter';

	/**
	 * Modify the query based on the FILTER_QUERY_ARG variable in $_GET.
	 *
	 * @param string $where Query variables.
	 *
	 * @return string The modified query.
	 */
	abstract public function filter_posts( $where );

	/**
	 * Returns the query value this filter uses.
	 *
	 * @return string The query value this filter uses.
	 */
	abstract public function get_query_val();

	/**
	 * Returns the total number of posts that match this filter.
	 *
	 * @return int The total number of posts that match this filter.
	 */
	abstract protected function get_post_total();

	/**
	 * Returns the label for this filter.
	 *
	 * @return string The label for this filter.
	 */
	abstract protected function get_label();

	/**
	 * Registers the hooks.
	 *
	 * @return void
	 */
	public function register_hooks() {
		add_action( 'admin_init', [ $this, 'add_filter_links' ], 11 );

		add_filter( 'posts_where', [ $this, 'filter_posts' ] );

		if ( $this->is_filter_active() ) {
			add_action( 'restrict_manage_posts', [ $this, 'render_hidden_input' ] );
		}

		if ( $this->is_filter_active() && $this->get_explanation() !== null ) {
			add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_explanation_assets' ] );
		}
	}

	/**
	 * Adds the filter links to the view_edit screens to give the user a filter link.
	 *
	 * @return void
	 */
	public function add_filter_links() {
		foreach ( $this->get_post_types() as $post_type ) {
			add_filter( 'views_edit-' . $post_type, [ $this, 'add_filter_link' ] );
		}
	}

	/**
	 * Enqueues the necessary assets to display a filter explanation.
	 *
	 * @return void
	 */
	public function enqueue_explanation_assets() {
		$asset_manager = new WPSEO_Admin_Asset_Manager();
		$asset_manager->enqueue_script( 'filter-explanation' );
		$asset_manager->enqueue_style( 'filter-explanation' );
		$asset_manager->localize_script(
			'filter-explanation',
			'yoastFilterExplanation',
			[ 'text' => $this->get_explanation() ]
		);
	}

	/**
	 * Adds a filter link to the views.
	 *
	 * @param array<string, string> $views Array with the views.
	 *
	 * @return array<string, string> Array of views including the added view.
	 */
	public function add_filter_link( $views ) {
		$views[ 'yoast_' . $this->get_query_val() ] = sprintf(
			'<a href="%1$s"%2$s>%3$s</a> (%4$s)',
			esc_url( $this->get_filter_url() ),
			( $this->is_filter_active() ) ? ' class="current" aria-current="page"' : '',
			$this->get_label(),
			$this->get_post_total()
		);

		return $views;
	}

	/**
	 * Returns a text explaining this filter. Null if no explanation is necessary.
	 *
	 * @return string|null The explanation or null.
	 */
	protected function get_explanation() {
		return null;
	}

	/**
	 * Renders a hidden input to preserve this filter's state when using sub-filters.
	 *
	 * @return void
	 */
	public function render_hidden_input() {
		echo '<input type="hidden" name="' . esc_attr( self::FILTER_QUERY_ARG ) . '" value="' . esc_attr( $this->get_query_val() ) . '">';
	}

	/**
	 * Returns an url to edit.php with post_type and this filter as the query arguments.
	 *
	 * @return string The url to activate this filter.
	 */
	protected function get_filter_url() {
		$query_args = [
			self::FILTER_QUERY_ARG => $this->get_query_val(),
			'post_type'            => $this->get_current_post_type(),
		];

		return add_query_arg( $query_args, 'edit.php' );
	}

	/**
	 * Returns true when the filter is active.
	 *
	 * @return bool Whether the filter is active.
	 */
	protected function is_filter_active() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
		if ( isset( $_GET[ self::FILTER_QUERY_ARG ] ) && is_string( $_GET[ self::FILTER_QUERY_ARG ] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
			return sanitize_text_field( wp_unslash( $_GET[ self::FILTER_QUERY_ARG ] ) ) === $this->get_query_val();
		}
		return false;
	}

	/**
	 * Returns the current post type.
	 *
	 * @return string The current post type.
	 */
	protected function get_current_post_type() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
		if ( isset( $_GET['post_type'] ) && is_string( $_GET['post_type'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
			$post_type = sanitize_text_field( wp_unslash( $_GET['post_type'] ) );
			if ( ! empty( $post_type ) ) {
				return $post_type;
			}
		}
		return 'post';
	}

	/**
	 * Returns the post types to which this filter should be added.
	 *
	 * @return array The post types to which this filter should be added.
	 */
	protected function get_post_types() {
		return WPSEO_Post_Type::get_accessible_post_types();
	}

	/**
	 * Checks if the post type is supported.
	 *
	 * @param string $post_type Post type to check against.
	 *
	 * @return bool True when it is supported.
	 */
	protected function is_supported_post_type( $post_type ) {
		return in_array( $post_type, $this->get_post_types(), true );
	}
}
                                                                                                                                                                                                       plugins/wordpress-seo-extended/admin/filters/class-cornerstone-filter.php                           0000644                 00000007275 15122266554 0023062 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin
 */

/**
 * Registers the filter for filtering posts by cornerstone content.
 */
class WPSEO_Cornerstone_Filter extends WPSEO_Abstract_Post_Filter {

	/**
	 * Name of the meta value.
	 *
	 * @var string
	 */
	public const META_NAME = 'is_cornerstone';

	/**
	 * Registers the hooks.
	 *
	 * @return void
	 */
	public function register_hooks() {
		parent::register_hooks();

		add_filter( 'wpseo_cornerstone_post_types', [ 'WPSEO_Post_Type', 'filter_attachment_post_type' ] );
		add_filter( 'wpseo_cornerstone_post_types', [ $this, 'filter_metabox_disabled' ] );
	}

	/**
	 * Returns the query value this filter uses.
	 *
	 * @return string The query value this filter uses.
	 */
	public function get_query_val() {
		return 'cornerstone';
	}

	/**
	 * Modify the query based on the seo_filter variable in $_GET.
	 *
	 * @param string $where Query variables.
	 *
	 * @return string The modified query.
	 */
	public function filter_posts( $where ) {
		if ( $this->is_filter_active() ) {
			global $wpdb;

			$where .= $wpdb->prepare(
				" AND {$wpdb->posts}.ID IN ( SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = '1' ) ",
				WPSEO_Meta::$meta_prefix . self::META_NAME
			);
		}

		return $where;
	}

	/**
	 * Filters the post types that have the metabox disabled.
	 *
	 * @param array $post_types The post types to filter.
	 *
	 * @return array The filtered post types.
	 */
	public function filter_metabox_disabled( $post_types ) {
		$filtered_post_types = [];
		foreach ( $post_types as $post_type_key => $post_type ) {
			if ( ! WPSEO_Post_Type::has_metabox_enabled( $post_type_key ) ) {
				continue;
			}

			$filtered_post_types[ $post_type_key ] = $post_type;
		}

		return $filtered_post_types;
	}

	/**
	 * Returns the label for this filter.
	 *
	 * @return string The label for this filter.
	 */
	protected function get_label() {
		return __( 'Cornerstone content', 'wordpress-seo' );
	}

	/**
	 * Returns a text explaining this filter.
	 *
	 * @return string|null The explanation.
	 */
	protected function get_explanation() {
		$post_type_object = get_post_type_object( $this->get_current_post_type() );

		if ( $post_type_object === null ) {
			return null;
		}

		return sprintf(
			/* translators: %1$s expands to the posttype label, %2$s expands anchor to blog post about cornerstone content, %3$s expands to </a> */
			__( 'Mark the most important %1$s as \'cornerstone content\' to improve your site structure. %2$sLearn more about cornerstone content%3$s.', 'wordpress-seo' ),
			strtolower( $post_type_object->labels->name ),
			'<a href="' . WPSEO_Shortlinker::get( 'https://yoa.st/1i9' ) . '" target="_blank">',
			'</a>'
		);
	}

	/**
	 * Returns the total amount of articles marked as cornerstone content.
	 *
	 * @return int
	 */
	protected function get_post_total() {
		global $wpdb;

		return (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT( 1 )
					FROM {$wpdb->postmeta}
					WHERE post_id IN( SELECT ID FROM {$wpdb->posts} WHERE post_type = %s ) AND
					meta_key = %s AND meta_value = '1'
				",
				$this->get_current_post_type(),
				WPSEO_Meta::$meta_prefix . self::META_NAME
			)
		);
	}

	/**
	 * Returns the post types to which this filter should be added.
	 *
	 * @return array The post types to which this filter should be added.
	 */
	protected function get_post_types() {
		/**
		 * Filter: 'wpseo_cornerstone_post_types' - Filters post types to exclude the cornerstone feature for.
		 *
		 * @param array $post_types The accessible post types to filter.
		 */
		$post_types = apply_filters( 'wpseo_cornerstone_post_types', parent::get_post_types() );
		if ( ! is_array( $post_types ) ) {
			return [];
		}

		return $post_types;
	}
}
                                                                                                                                                                                                                                                                                                                                   plugins/wordpress-seo-extended/admin/formatter/class-metabox-formatter.php                          0000644                 00000034302 15122266554 0023220 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Formatter
 */

use Yoast\WP\SEO\Conditionals\Third_Party\Polylang_Conditional;
use Yoast\WP\SEO\Conditionals\Third_Party\TranslatePress_Conditional;
use Yoast\WP\SEO\Conditionals\Third_Party\WPML_Conditional;
use Yoast\WP\SEO\Config\Schema_Types;
use Yoast\WP\SEO\Config\SEMrush_Client;
use Yoast\WP\SEO\Editors\Application\Analysis_Features\Enabled_Analysis_Features_Repository;
use Yoast\WP\SEO\Exceptions\OAuth\Authentication_Failed_Exception;
use Yoast\WP\SEO\Exceptions\OAuth\Tokens\Empty_Property_Exception;
use Yoast\WP\SEO\Exceptions\OAuth\Tokens\Empty_Token_Exception;

/**
 * This class forces needed methods for the metabox localization.
 */
class WPSEO_Metabox_Formatter {

	/**
	 * Object that provides formatted values.
	 *
	 * @var WPSEO_Metabox_Formatter_Interface
	 */
	private $formatter;

	/**
	 * Setting the formatter property.
	 *
	 * @param WPSEO_Metabox_Formatter_Interface $formatter Object that provides the formatted values.
	 */
	public function __construct( WPSEO_Metabox_Formatter_Interface $formatter ) {
		$this->formatter = $formatter;
	}

	/**
	 * Returns the values.
	 *
	 * @return array<string,string|array<string|int|bool>|bool|int>
	 */
	public function get_values() {
		$defaults = $this->get_defaults();
		$values   = $this->formatter->get_values();

		return ( $values + $defaults );
	}

	/**
	 * Returns array with all the values always needed by a scraper object.
	 *
	 * @return array<string,string|array<string|int|bool>|bool|int> Default settings for the metabox.
	 */
	private function get_defaults() {
		$schema_types      = new Schema_Types();
		$is_wincher_active = YoastSEO()->helpers->wincher->is_active();
		$host              = YoastSEO()->helpers->url->get_url_host( get_site_url() );

		$defaults = [
			'author_name'                        => get_the_author_meta( 'display_name' ),
			'site_name'                          => YoastSEO()->meta->for_current_page()->site_name,
			'sitewide_social_image'              => WPSEO_Options::get( 'og_default_image' ),
			'search_url'                         => '',
			'post_edit_url'                      => '',
			'base_url'                           => '',
			'contentTab'                         => __( 'Readability', 'wordpress-seo' ),
			'keywordTab'                         => __( 'Keyphrase:', 'wordpress-seo' ),
			'removeKeyword'                      => __( 'Remove keyphrase', 'wordpress-seo' ),
			'contentLocale'                      => get_locale(),
			'userLocale'                         => get_user_locale(),
			'translations'                       => $this->get_translations(),
			'keyword_usage'                      => [],
			'title_template'                     => '',
			'metadesc_template'                  => '',
			'semrushIntegrationActive'           => WPSEO_Options::get( 'semrush_integration_active', true ) ? 1 : 0,
			'intl'                               => $this->get_content_analysis_component_translations(),
			'isRtl'                              => is_rtl(),
			'isPremium'                          => YoastSEO()->helpers->product->is_premium(),
			'siteIconUrl'                        => get_site_icon_url(),
			'countryCode'                        => WPSEO_Options::get( 'semrush_country_code', false ),
			'SEMrushLoginStatus'                 => WPSEO_Options::get( 'semrush_integration_active', true ) ? $this->get_semrush_login_status() : false,
			'showSocial'                         => [
				'facebook' => WPSEO_Options::get( 'opengraph', false ),
				'twitter'  => WPSEO_Options::get( 'twitter', false ),
			],
			'schema'                             => [
				'displayFooter'      => WPSEO_Capability_Utils::current_user_can( 'wpseo_manage_options' ),
				'pageTypeOptions'    => $schema_types->get_page_type_options(),
				'articleTypeOptions' => $schema_types->get_article_type_options(),
			],
			'twitterCardType'                    => 'summary_large_image',
			'publish_box'                        => [
				'labels' => [
					'keyword'            => [
						'na'   => sprintf(
							/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the SEO score. */
							__( '%1$sSEO%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-seo-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Not available', 'wordpress-seo' ) . '</strong>'
						),
						'bad'  => sprintf(
							/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the SEO score. */
							__( '%1$sSEO%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-seo-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Needs improvement', 'wordpress-seo' ) . '</strong>'
						),
						'ok'   => sprintf(
							/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the SEO score. */
							__( '%1$sSEO%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-seo-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'OK', 'wordpress-seo' ) . '</strong>'
						),
						'good' => sprintf(
							/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the SEO score. */
							__( '%1$sSEO%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-seo-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Good', 'wordpress-seo' ) . '</strong>'
						),
					],
					'content'            => [
						'na'   => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the readability score. */
							__( '%1$sReadability%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-readability-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Not available', 'wordpress-seo' ) . '</strong>'
						),
						'bad'  => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the readability score. */
							__( '%1$sReadability%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-readability-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Needs improvement', 'wordpress-seo' ) . '</strong>'
						),
						'ok'   => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the readability score. */
							__( '%1$sReadability%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-readability-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'OK', 'wordpress-seo' ) . '</strong>'
						),
						'good' => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the readability score. */
							__( '%1$sReadability%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-readability-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Good', 'wordpress-seo' ) . '</strong>'
						),
					],
					'inclusive-language' => [
						'na'   => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the inclusive language score. */
							__( '%1$sInclusive language%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-inclusive-language-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Not available', 'wordpress-seo' ) . '</strong>'
						),
						'bad'  => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the inclusive language score. */
							__( '%1$sInclusive language%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-inclusive-language-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Needs improvement', 'wordpress-seo' ) . '</strong>'
						),
						'ok'   => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the inclusive language score. */
							__( '%1$sInclusive language%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-inclusive-language-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Potentially non-inclusive', 'wordpress-seo' ) . '</strong>'
						),
						'good' => sprintf(
						/* translators: %1$s expands to the opening anchor tag, %2$s to the closing anchor tag, %3$s to the inclusive language score. */
							__( '%1$sInclusive language%2$s: %3$s', 'wordpress-seo' ),
							'<a href="#yoast-inclusive-language-analysis-collapsible-metabox">',
							'</a>',
							'<strong>' . __( 'Good', 'wordpress-seo' ) . '</strong>'
						),
					],
				],
			],
			/**
			 * Filter to determine if the markers should be enabled or not.
			 *
			 * @param bool $showMarkers Should the markers being enabled. Default = true.
			 */
			'show_markers'                       => apply_filters( 'wpseo_enable_assessment_markers', true ),
			'markdownEnabled'                    => $this->is_markdown_enabled(),
			'analysisHeadingTitle'               => __( 'Analysis', 'wordpress-seo' ),
			'zapierIntegrationActive'            => WPSEO_Options::get( 'zapier_integration_active', false ) ? 1 : 0,
			'zapierConnectedStatus'              => ! empty( WPSEO_Options::get( 'zapier_subscription', [] ) ) ? 1 : 0,
			'wincherIntegrationActive'           => ( $is_wincher_active ) ? 1 : 0,
			'wincherLoginStatus'                 => ( $is_wincher_active ) ? YoastSEO()->helpers->wincher->login_status() : false,
			'wincherWebsiteId'                   => WPSEO_Options::get( 'wincher_website_id', '' ),
			'wincherAutoAddKeyphrases'           => WPSEO_Options::get( 'wincher_automatically_add_keyphrases', false ),
			'wordproofIntegrationActive'         => YoastSEO()->helpers->wordproof->is_active() ? 1 : 0,
			'multilingualPluginActive'           => $this->multilingual_plugin_active(),
			'getJetpackBoostPrePublishLink'      => WPSEO_Shortlinker::get( 'https://yoa.st/jetpack-boost-get-prepublish?domain=' . $host ),
			'upgradeJetpackBoostPrePublishLink'  => WPSEO_Shortlinker::get( 'https://yoa.st/jetpack-boost-upgrade-prepublish?domain=' . $host ),
			'woocommerceUpsellSchemaLink'        => WPSEO_Shortlinker::get( 'https://yoa.st/product-schema-metabox' ),
			'woocommerceUpsellGooglePreviewLink' => WPSEO_Shortlinker::get( 'https://yoa.st/product-google-preview-metabox' ),
		];

		$enabled_features_repo = YoastSEO()->classes->get( Enabled_Analysis_Features_Repository::class );

		$enabled_features = $enabled_features_repo->get_enabled_features()->parse_to_legacy_array();

		return array_merge( $defaults, $enabled_features );
	}

	/**
	 * Returns required yoast-component translations.
	 *
	 * @return string[]
	 */
	private function get_content_analysis_component_translations() {
		// Esc_html is not needed because React already handles HTML in the (translations of) these strings.
		return [
			'locale'                                         => get_user_locale(),
			'content-analysis.errors'                        => __( 'Errors', 'wordpress-seo' ),
			'content-analysis.problems'                      => __( 'Problems', 'wordpress-seo' ),
			'content-analysis.improvements'                  => __( 'Improvements', 'wordpress-seo' ),
			'content-analysis.considerations'                => __( 'Considerations', 'wordpress-seo' ),
			'content-analysis.good'                          => __( 'Good results', 'wordpress-seo' ),
			'content-analysis.highlight'                     => __( 'Highlight this result in the text', 'wordpress-seo' ),
			'content-analysis.nohighlight'                   => __( 'Remove highlight from the text', 'wordpress-seo' ),
			'content-analysis.disabledButton'                => __( 'Marks are disabled in current view', 'wordpress-seo' ),
			/* translators: Hidden accessibility text. */
			'a11yNotice.opensInNewTab'                       => __( '(Opens in a new browser tab)', 'wordpress-seo' ),
		];
	}

	/**
	 * Returns Jed compatible YoastSEO.js translations.
	 *
	 * @return string[]
	 */
	private function get_translations() {
		$locale = get_user_locale();

		$file = WPSEO_PATH . 'languages/wordpress-seo-' . $locale . '.json';
		if ( file_exists( $file ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents -- Retrieving a local file.
			$file = file_get_contents( $file );
			if ( is_string( $file ) && $file !== '' ) {
				return json_decode( $file, true );
			}
		}

		return [];
	}

	/**
	 * Checks if Jetpack's markdown module is enabled.
	 * Can be extended to work with other plugins that parse markdown in the content.
	 *
	 * @return bool
	 */
	private function is_markdown_enabled() {
		$is_markdown = false;

		if ( class_exists( 'Jetpack' ) && method_exists( 'Jetpack', 'get_active_modules' ) ) {
			$active_modules = Jetpack::get_active_modules();

			// First at all, check if Jetpack's markdown module is active.
			$is_markdown = in_array( 'markdown', $active_modules, true );
		}

		/**
		 * Filters whether markdown support is active in the readability- and seo-analysis.
		 *
		 * @since 11.3
		 *
		 * @param array $is_markdown Is markdown support for Yoast SEO active.
		 */
		return apply_filters( 'wpseo_is_markdown_enabled', $is_markdown );
	}

	/**
	 * Checks if the user is logged in to SEMrush.
	 *
	 * @return bool The SEMrush login status.
	 */
	private function get_semrush_login_status() {
		try {
			$semrush_client = YoastSEO()->classes->get( SEMrush_Client::class );
		} catch ( Empty_Property_Exception $e ) {
			// Return false if token is malformed (empty property).
			return false;
		}

		// Get token (and refresh it if it's expired).
		try {
			$semrush_client->get_tokens();
		} catch ( Authentication_Failed_Exception $e ) {
			return false;
		} catch ( Empty_Token_Exception $e ) {
			return false;
		}

		return $semrush_client->has_valid_tokens();
	}

	/**
	 * Checks whether a multilingual plugin is currently active. Currently, we only check the following plugins: WPML, Polylang, and TranslatePress.
	 *
	 * @return bool Whether a multilingual plugin is currently active.
	 */
	private function multilingual_plugin_active() {
		$wpml_active           = YoastSEO()->classes->get( WPML_Conditional::class )->is_met();
		$polylang_active       = YoastSEO()->classes->get( Polylang_Conditional::class )->is_met();
		$translatepress_active = YoastSEO()->classes->get( TranslatePress_Conditional::class )->is_met();

		return ( $wpml_active || $polylang_active || $translatepress_active );
	}
}
                                                                                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/admin/formatter/class-post-metabox-formatter.php                     0000644                 00000021247 15122266554 0024207 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Formatter
 */

/**
 * This class provides data for the post metabox by return its values for localization.
 */
class WPSEO_Post_Metabox_Formatter implements WPSEO_Metabox_Formatter_Interface {

	/**
	 * Holds the WordPress Post.
	 *
	 * @var WP_Post
	 */
	private $post;

	/**
	 * The permalink to follow.
	 *
	 * @var string
	 */
	private $permalink;

	/**
	 * Whether we must return social templates values.
	 *
	 * @var bool
	 */
	private $use_social_templates = false;

	/**
	 * Constructor.
	 *
	 * @param WP_Post|array $post      Post object.
	 * @param array         $options   Title options to use.
	 * @param string        $structure The permalink to follow.
	 */
	public function __construct( $post, array $options, $structure ) {
		$this->post      = $post;
		$this->permalink = $structure;

		$this->use_social_templates = $this->use_social_templates();
	}

	/**
	 * Determines whether the social templates should be used.
	 *
	 * @return bool Whether the social templates should be used.
	 */
	public function use_social_templates() {
		return WPSEO_Options::get( 'opengraph', false ) === true;
	}

	/**
	 * Returns the translated values.
	 *
	 * @return array
	 */
	public function get_values() {

		$values = [
			'search_url'          => $this->search_url(),
			'post_edit_url'       => $this->edit_url(),
			'base_url'            => $this->base_url_for_js(),
			'metaDescriptionDate' => '',
		];

		if ( $this->post instanceof WP_Post ) {
			$keyword_usage = $this->get_focus_keyword_usage();

			$values_to_set = [
				'keyword_usage'               => $keyword_usage,
				'keyword_usage_post_types'    => $this->get_post_types_for_all_ids( $keyword_usage ),
				'title_template'              => $this->get_title_template(),
				'title_template_no_fallback'  => $this->get_title_template( false ),
				'metadesc_template'           => $this->get_metadesc_template(),
				'metaDescriptionDate'         => $this->get_metadesc_date(),
				'first_content_image'         => $this->get_image_url(),
				'social_title_template'       => $this->get_social_title_template(),
				'social_description_template' => $this->get_social_description_template(),
				'social_image_template'       => $this->get_social_image_template(),
				'isInsightsEnabled'           => $this->is_insights_enabled(),
			];

			$values = ( $values_to_set + $values );
		}

		/**
		 * Filter: 'wpseo_post_edit_values' - Allows changing the values Yoast SEO uses inside the post editor.
		 *
		 * @param array   $values The key-value map Yoast SEO uses inside the post editor.
		 * @param WP_Post $post   The post opened in the editor.
		 */
		return apply_filters( 'wpseo_post_edit_values', $values, $this->post );
	}

	/**
	 * Gets the image URL for the post's social preview.
	 *
	 * @return string|null The image URL for the social preview.
	 */
	protected function get_image_url() {
		return WPSEO_Image_Utils::get_first_usable_content_image_for_post( $this->post->ID );
	}

	/**
	 * Returns the url to search for keyword for the post.
	 *
	 * @return string
	 */
	private function search_url() {
		return admin_url( 'edit.php?seo_kw_filter={keyword}' );
	}

	/**
	 * Returns the url to edit the taxonomy.
	 *
	 * @return string
	 */
	private function edit_url() {
		return admin_url( 'post.php?post={id}&action=edit' );
	}

	/**
	 * Returns a base URL for use in the JS, takes permalink structure into account.
	 *
	 * @return string
	 */
	private function base_url_for_js() {
		global $pagenow;

		// The default base is the home_url.
		$base_url = home_url( '/', null );

		if ( $pagenow === 'post-new.php' ) {
			return $base_url;
		}

		// If %postname% is the last tag, just strip it and use that as a base.
		if ( preg_match( '#%postname%/?$#', $this->permalink ) === 1 ) {
			$base_url = preg_replace( '#%postname%/?$#', '', $this->permalink );
		}

		// If %pagename% is the last tag, just strip it and use that as a base.
		if ( preg_match( '#%pagename%/?$#', $this->permalink ) === 1 ) {
			$base_url = preg_replace( '#%pagename%/?$#', '', $this->permalink );
		}

		return $base_url;
	}

	/**
	 * Counts the number of given keywords used for other posts other than the given post_id.
	 *
	 * @return array The keyword and the associated posts that use it.
	 */
	private function get_focus_keyword_usage() {
		$keyword = WPSEO_Meta::get_value( 'focuskw', $this->post->ID );
		$usage   = [ $keyword => $this->get_keyword_usage_for_current_post( $keyword ) ];

		/**
		 * Allows enhancing the array of posts' that share their focus keywords with the post's related keywords.
		 *
		 * @param array $usage   The array of posts' ids that share their focus keywords with the post.
		 * @param int   $post_id The id of the post we're finding the usage of related keywords for.
		 */
		return apply_filters( 'wpseo_posts_for_related_keywords', $usage, $this->post->ID );
	}

	/**
	 * Retrieves the post types for the given post IDs.
	 *
	 * @param array $post_ids_per_keyword An associative array with keywords as keys and an array of post ids where those keywords are used.
	 * @return array The post types for the given post IDs.
	 */
	private function get_post_types_for_all_ids( $post_ids_per_keyword ) {

		$post_type_per_keyword_result = [];
		foreach ( $post_ids_per_keyword as $keyword => $post_ids ) {
			$post_type_per_keyword_result[ $keyword ] = WPSEO_Meta::post_types_for_ids( $post_ids );
		}

		return $post_type_per_keyword_result;
	}

	/**
	 * Gets the keyword usage for the current post and the specified keyword.
	 *
	 * @param string $keyword The keyword to check the usage of.
	 *
	 * @return array The post IDs which use the passed keyword.
	 */
	protected function get_keyword_usage_for_current_post( $keyword ) {
		return WPSEO_Meta::keyword_usage( $keyword, $this->post->ID );
	}

	/**
	 * Retrieves the title template.
	 *
	 * @param bool $fallback Whether to return the hardcoded fallback if the template value is empty.
	 *
	 * @return string The title template.
	 */
	private function get_title_template( $fallback = true ) {
		$title = $this->get_template( 'title' );

		if ( $title === '' && $fallback === true ) {
			return '%%title%% %%page%% %%sep%% %%sitename%%';
		}

		return $title;
	}

	/**
	 * Retrieves the metadesc template.
	 *
	 * @return string The metadesc template.
	 */
	private function get_metadesc_template() {
		return $this->get_template( 'metadesc' );
	}

	/**
	 * Retrieves the social title template.
	 *
	 * @return string The social title template.
	 */
	private function get_social_title_template() {
		if ( $this->use_social_templates ) {
			return $this->get_social_template( 'title' );
		}

		return '';
	}

	/**
	 * Retrieves the social description template.
	 *
	 * @return string The social description template.
	 */
	private function get_social_description_template() {
		if ( $this->use_social_templates ) {
			return $this->get_social_template( 'description' );
		}

		return '';
	}

	/**
	 * Retrieves the social image template.
	 *
	 * @return string The social description template.
	 */
	private function get_social_image_template() {
		if ( $this->use_social_templates ) {
			return $this->get_social_template( 'image-url' );
		}

		return '';
	}

	/**
	 * Retrieves a template.
	 *
	 * @param string $template_option_name The name of the option in which the template you want to get is saved.
	 *
	 * @return string
	 */
	private function get_template( $template_option_name ) {
		$needed_option = $template_option_name . '-' . $this->post->post_type;

		if ( WPSEO_Options::get( $needed_option, '' ) !== '' ) {
			return WPSEO_Options::get( $needed_option );
		}

		return '';
	}

	/**
	 * Retrieves a social template.
	 *
	 * @param string $template_option_name The name of the option in which the template you want to get is saved.
	 *
	 * @return string
	 */
	private function get_social_template( $template_option_name ) {
		/**
		 * Filters the social template value for a given post type.
		 *
		 * @param string $template             The social template value, defaults to empty string.
		 * @param string $template_option_name The subname of the option in which the template you want to get is saved.
		 * @param string $post_type            The name of the post type.
		 */
		return apply_filters( 'wpseo_social_template_post_type', '', $template_option_name, $this->post->post_type );
	}

	/**
	 * Determines the date to be displayed in the snippet preview.
	 *
	 * @return string
	 */
	private function get_metadesc_date() {
		return YoastSEO()->helpers->date->format_translated( $this->post->post_date, 'M j, Y' );
	}

	/**
	 * Determines whether the insights feature is enabled for this post.
	 *
	 * @return bool
	 */
	protected function is_insights_enabled() {
		return WPSEO_Options::get( 'enable_metabox_insights', false );
	}
}
                                                                                                                                                                                                                                                                                                                                                         plugins/wordpress-seo-extended/admin/formatter/class-term-metabox-formatter.php                     0000644                 00000015340 15122266554 0024166 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Formatter
 */

/**
 * This class provides data for the term metabox by return its values for localization.
 */
class WPSEO_Term_Metabox_Formatter implements WPSEO_Metabox_Formatter_Interface {

	/**
	 * The term the metabox formatter is for.
	 *
	 * @var WP_Term|stdClass
	 */
	private $term;

	/**
	 * The term's taxonomy.
	 *
	 * @var stdClass
	 */
	private $taxonomy;

	/**
	 * Whether we must return social templates values.
	 *
	 * @var bool
	 */
	private $use_social_templates = false;

	/**
	 * Array with the WPSEO_Titles options.
	 *
	 * @var array
	 */
	protected $options;

	/**
	 * WPSEO_Taxonomy_Scraper constructor.
	 *
	 * @param stdClass         $taxonomy Taxonomy.
	 * @param WP_Term|stdClass $term     Term.
	 */
	public function __construct( $taxonomy, $term ) {
		$this->taxonomy = $taxonomy;
		$this->term     = $term;

		$this->use_social_templates = $this->use_social_templates();
	}

	/**
	 * Determines whether the social templates should be used.
	 *
	 * @return bool Whether the social templates should be used.
	 */
	public function use_social_templates() {
		return WPSEO_Options::get( 'opengraph', false ) === true;
	}

	/**
	 * Returns the translated values.
	 *
	 * @return array
	 */
	public function get_values() {
		$values = [];

		// Todo: a column needs to be added on the termpages to add a filter for the keyword, so this can be used in the focus keyphrase doubles.
		if ( is_object( $this->term ) && property_exists( $this->term, 'taxonomy' ) ) {
			$values = [
				'search_url'                  => $this->search_url(),
				'post_edit_url'               => $this->edit_url(),
				'base_url'                    => $this->base_url_for_js(),
				'taxonomy'                    => $this->term->taxonomy,
				'keyword_usage'               => $this->get_focus_keyword_usage(),
				'title_template'              => $this->get_title_template(),
				'title_template_no_fallback'  => $this->get_title_template( false ),
				'metadesc_template'           => $this->get_metadesc_template(),
				'first_content_image'         => $this->get_image_url(),
				'semrushIntegrationActive'    => 0,
				'social_title_template'       => $this->get_social_title_template(),
				'social_description_template' => $this->get_social_description_template(),
				'social_image_template'       => $this->get_social_image_template(),
				'wincherIntegrationActive'    => 0,
				'isInsightsEnabled'           => $this->is_insights_enabled(),
			];
		}

		return $values;
	}

	/**
	 * Gets the image URL for the term's social preview.
	 *
	 * @return string|null The image URL for the social preview.
	 */
	protected function get_image_url() {
		return WPSEO_Image_Utils::get_first_content_image_for_term( $this->term->term_id );
	}

	/**
	 * Returns the url to search for keyword for the taxonomy.
	 *
	 * @return string
	 */
	private function search_url() {
		return admin_url( 'edit-tags.php?taxonomy=' . $this->term->taxonomy . '&seo_kw_filter={keyword}' );
	}

	/**
	 * Returns the url to edit the taxonomy.
	 *
	 * @return string
	 */
	private function edit_url() {
		return admin_url( 'term.php?action=edit&taxonomy=' . $this->term->taxonomy . '&tag_ID={id}' );
	}

	/**
	 * Returns a base URL for use in the JS, takes permalink structure into account.
	 *
	 * @return string
	 */
	private function base_url_for_js() {

		$base_url = home_url( '/', null );
		if ( ! WPSEO_Options::get( 'stripcategorybase', false ) ) {
			if ( $this->taxonomy->rewrite ) {
				$base_url = trailingslashit( $base_url . $this->taxonomy->rewrite['slug'] );
			}
		}

		return $base_url;
	}

	/**
	 * Counting the number of given keyword used for other term than given term_id.
	 *
	 * @return array
	 */
	private function get_focus_keyword_usage() {
		$focuskw = WPSEO_Taxonomy_Meta::get_term_meta( $this->term, $this->term->taxonomy, 'focuskw' );

		return WPSEO_Taxonomy_Meta::get_keyword_usage( $focuskw, $this->term->term_id, $this->term->taxonomy );
	}

	/**
	 * Retrieves the title template.
	 *
	 * @param bool $fallback Whether to return the hardcoded fallback if the template value is empty.
	 *
	 * @return string The title template.
	 */
	private function get_title_template( $fallback = true ) {
		$title = $this->get_template( 'title' );

		if ( $title === '' && $fallback === true ) {
			/* translators: %s expands to the variable used for term title. */
			$archives = sprintf( __( '%s Archives', 'wordpress-seo' ), '%%term_title%%' );
			return $archives . ' %%page%% %%sep%% %%sitename%%';
		}

		return $title;
	}

	/**
	 * Retrieves the metadesc template.
	 *
	 * @return string The metadesc template.
	 */
	private function get_metadesc_template() {
		return $this->get_template( 'metadesc' );
	}

	/**
	 * Retrieves the social title template.
	 *
	 * @return string The social title template.
	 */
	private function get_social_title_template() {
		if ( $this->use_social_templates ) {
			return $this->get_social_template( 'title' );
		}

		return '';
	}

	/**
	 * Retrieves the social description template.
	 *
	 * @return string The social description template.
	 */
	private function get_social_description_template() {
		if ( $this->use_social_templates ) {
			return $this->get_social_template( 'description' );
		}

		return '';
	}

	/**
	 * Retrieves the social image template.
	 *
	 * @return string The social description template.
	 */
	private function get_social_image_template() {
		if ( $this->use_social_templates ) {
			return $this->get_social_template( 'image-url' );
		}

		return '';
	}

	/**
	 * Retrieves a template.
	 *
	 * @param string $template_option_name The name of the option in which the template you want to get is saved.
	 *
	 * @return string
	 */
	private function get_template( $template_option_name ) {
		$needed_option = $template_option_name . '-tax-' . $this->term->taxonomy;
		return WPSEO_Options::get( $needed_option, '' );
	}

	/**
	 * Retrieves a social template.
	 *
	 * @param string $template_option_name The name of the option in which the template you want to get is saved.
	 *
	 * @return string
	 */
	private function get_social_template( $template_option_name ) {
		/**
		 * Filters the social template value for a given taxonomy.
		 *
		 * @param string $template             The social template value, defaults to empty string.
		 * @param string $template_option_name The subname of the option in which the template you want to get is saved.
		 * @param string $taxonomy             The name of the taxonomy.
		 */
		return apply_filters( 'wpseo_social_template_taxonomy', '', $template_option_name, $this->term->taxonomy );
	}

	/**
	 * Determines whether the insights feature is enabled for this taxonomy.
	 *
	 * @return bool
	 */
	protected function is_insights_enabled() {
		return WPSEO_Options::get( 'enable_metabox_insights', false );
	}
}
                                                                                                                                                                                                                                                                                                plugins/wordpress-seo-extended/admin/formatter/interface-metabox-formatter.php                      0000644                 00000000402 15122266554 0024045 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Formatter
 */

/**
 * Interface to force get_values.
 */
interface WPSEO_Metabox_Formatter_Interface {

	/**
	 * Returns formatter values.
	 *
	 * @return array
	 */
	public function get_values();
}
                                                                                                                                                                                                                                                              plugins/wordpress-seo-extended/admin/google_search_console/class-gsc.php                            0000644                 00000000672 15122266554 0022657 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\admin\google_search_console
 */

/**
 * Class WPSEO_GSC.
 */
class WPSEO_GSC {

	/**
	 * The option where data will be stored.
	 *
	 * @var string
	 */
	public const OPTION_WPSEO_GSC = 'wpseo-gsc';

	/**
	 * Outputs the HTML for the redirect page.
	 *
	 * @return void
	 */
	public function display() {
		require_once WPSEO_PATH . 'admin/google_search_console/views/gsc-display.php';
	}
}
                                                                      plugins/wordpress-seo-extended/admin/google_search_console/views/gsc-display.php                    0000644                 00000004472 15122266554 0024356 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Google_Search_Console
 */

// Admin header.
Yoast_Form::get_instance()->admin_header( false, 'wpseo-gsc', false, 'yoast_wpseo_gsc_options' );

// GSC Error notification.
$gsc_url                 = 'https://search.google.com/search-console/index';
$gsc_post_url            = 'https://yoa.st/google-search-console-deprecated';
$gsc_style_alert         = '
	display: flex;
	align-items: baseline;
	position: relative;
	padding: 16px;
	border: 1px solid rgba(0, 0, 0, 0.2);
	font-size: 14px;
	font-weight: 400;
	line-height: 1.5;
	margin: 16px 0;
	color: #450c11;
	background: #f8d7da;
';
$gsc_style_alert_icon    = 'display: block; margin-right: 8px;';
$gsc_style_alert_content = 'max-width: 600px;';
$gsc_style_alert_link    = 'color: #004973;';
$gsc_notification        = sprintf(
	/* Translators: %1$s: expands to opening anchor tag, %2$s expands to closing anchor tag. */
	__( 'Google has discontinued its Crawl Errors API. Therefore, any possible crawl errors you might have cannot be displayed here anymore. %1$sRead our statement on this for further information%2$s.', 'wordpress-seo' ),
	'<a style="' . $gsc_style_alert_link . '" href="' . WPSEO_Shortlinker::get( $gsc_post_url ) . '" target="_blank" rel="noopener">',
	WPSEO_Admin_Utils::get_new_tab_message() . '</a>'
);
$gsc_notification .= '<br/><br/>';
$gsc_notification .= sprintf(
	/* Translators: %1$s: expands to opening anchor tag, %2$s expands to closing anchor tag. */
	__( 'To view your current crawl errors, %1$splease visit Google Search Console%2$s.', 'wordpress-seo' ),
	'<a style="' . $gsc_style_alert_link . '" href="' . $gsc_url . '" target="_blank" rel="noopener noreferrer">',
	WPSEO_Admin_Utils::get_new_tab_message() . '</a>'
);
?>
	<div style="<?php echo $gsc_style_alert; ?>">
	<span style="<?php echo $gsc_style_alert_icon; ?>">
		<svg xmlns="http://www.w3.org/2000/svg" width="12" height="14" viewBox="0 0 12 14" role="img" aria-hidden="true"
			focusable="false" fill="#450c11">
			<path
				d="M6 1q1.6 0 3 .8T11.2 4t.8 3-.8 3T9 12.2 6 13t-3-.8T.8 10 0 7t.8-3T3 1.8 6 1zm1 9.7V9.3 9L6.7 9H5l-.1.3V10.9l.3.1h1.6l.1-.3zm0-2.6L7 3.2v-.1L6.8 3H5 5l-.1.2.1 4.9.3.2h1.4l.2-.1Q7 8 6.9 8z"></path>
		</svg>
	</span>
		<span style="<?php echo $gsc_style_alert_content; ?>"><?php echo $gsc_notification; ?></span>
	</div>
<?php
                                                                                                                                                                                                      plugins/wordpress-seo-extended/admin/google_search_console/views/gsc-redirect-nopremium.php         0000644                 00000001757 15122266554 0026526 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * This is the view for the modal box that appears when premium isn't loaded.
 *
 * @package WPSEO\Admin\Google_Search_Console
 */

_deprecated_file( __FILE__, 'Yoast SEO 9.5' );

echo '<h1 class="wpseo-redirect-url-title">';
printf(
	/* Translators: %s: expands to Yoast SEO Premium */
	esc_html__( 'Creating redirects is a %s feature', 'wordpress-seo' ),
	'Yoast SEO Premium'
);
echo '</h1>';
echo '<p>';
printf(
	/* Translators: %1$s: expands to 'Yoast SEO Premium', %2$s: links to Yoast SEO Premium plugin page. */
	esc_html__( 'To be able to create a redirect and fix this issue, you need %1$s. You can buy the plugin, including one year of support and updates, on %2$s.', 'wordpress-seo' ),
	'Yoast SEO Premium',
	'<a href="' . esc_url( WPSEO_Shortlinker::get( 'https://yoa.st/redirects' ) ) . '" target="_blank">yoast.com</a>'
);
echo '</p>';
echo '<button type="button" class="button wpseo-redirect-close">' . esc_html__( 'Close', 'wordpress-seo' ) . '</button>';
                 plugins/wordpress-seo-extended/admin/import/class-import-detector.php                               0000644                 00000001355 15122266554 0022212 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_Plugins_Detector.
 *
 * Class with functionality to detect whether we should import from another SEO plugin.
 */
class WPSEO_Import_Plugins_Detector {

	/**
	 * Plugins we need to import from.
	 *
	 * @var array
	 */
	public $needs_import = [];

	/**
	 * Detects whether we need to import anything.
	 *
	 * @return void
	 */
	public function detect() {
		foreach ( WPSEO_Plugin_Importers::get() as $importer_class ) {
			$importer = new $importer_class();
			$detect   = new WPSEO_Import_Plugin( $importer, 'detect' );
			if ( $detect->status->status ) {
				$this->needs_import[ $importer_class ] = $importer->get_plugin_name();
			}
		}
	}
}
                                                                                                                                                                                                                                                                                   plugins/wordpress-seo-extended/admin/import/class-import-plugin.php                                 0000644                 00000002667 15122266554 0021706 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Import\Plugins
 */

/**
 * Class WPSEO_Import_Plugin.
 *
 * Class with functionality to import Yoast SEO settings from other plugins.
 */
class WPSEO_Import_Plugin {

	/**
	 * Holds the status of and message about imports.
	 *
	 * @var WPSEO_Import_Status
	 */
	public $status;

	/**
	 * Class with functionality to import meta data from other plugins.
	 *
	 * @var WPSEO_Plugin_Importer
	 */
	protected $importer;

	/**
	 * Import class constructor.
	 *
	 * @param WPSEO_Plugin_Importer $importer The importer that needs to perform this action.
	 * @param string                $action   The action to perform.
	 */
	public function __construct( WPSEO_Plugin_Importer $importer, $action ) {
		$this->importer = $importer;

		switch ( $action ) {
			case 'cleanup':
				$this->status = $this->importer->run_cleanup();
				break;
			case 'import':
				$this->status = $this->importer->run_import();
				break;
			case 'detect':
			default:
				$this->status = $this->importer->run_detect();
		}

		$this->status->set_msg( $this->complete_msg( $this->status->get_msg() ) );
	}

	/**
	 * Convenience function to replace %s with plugin name in import message.
	 *
	 * @param string $msg Message string.
	 *
	 * @return string Returns message with plugin name instead of replacement variables.
	 */
	protected function complete_msg( $msg ) {
		return sprintf( $msg, $this->importer->get_plugin_name() );
	}
}
                                                                         plugins/wordpress-seo-extended/admin/import/class-import-settings.php                               0000644                 00000005714 15122266554 0022244 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Import
 */

/**
 * Class WPSEO_Import_Settings.
 *
 * Class with functionality to import the Yoast SEO settings.
 */
class WPSEO_Import_Settings {

	/**
	 * Nonce action key.
	 *
	 * @var string
	 */
	public const NONCE_ACTION = 'wpseo-import-settings';

	/**
	 * Holds the import status instance.
	 *
	 * @var WPSEO_Import_Status
	 */
	public $status;

	/**
	 * Holds the old WPSEO version.
	 *
	 * @var string
	 */
	private $old_wpseo_version;

	/**
	 * Class constructor.
	 */
	public function __construct() {
		$this->status = new WPSEO_Import_Status( 'import', false );
	}

	/**
	 * Imports the data submitted by the user.
	 *
	 * @return void
	 */
	public function import() {
		check_admin_referer( self::NONCE_ACTION );

		if ( ! WPSEO_Capability_Utils::current_user_can( 'wpseo_manage_options' ) ) {
			return;
		}

		if ( ! isset( $_POST['settings_import'] ) || ! is_string( $_POST['settings_import'] ) ) {
			return;
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: The raw content will be parsed afterwards.
		$content = wp_unslash( $_POST['settings_import'] );

		if ( empty( $content ) ) {
			return;
		}

		$this->parse_options( $content );
	}

	/**
	 * Parse the options.
	 *
	 * @param string $raw_options The content to parse.
	 *
	 * @return void
	 */
	protected function parse_options( $raw_options ) {
		$options = parse_ini_string( $raw_options, true, INI_SCANNER_RAW );

		if ( is_array( $options ) && $options !== [] ) {
			$this->import_options( $options );

			return;
		}

		$this->status->set_msg( __( 'Settings could not be imported:', 'wordpress-seo' ) . ' ' . __( 'No settings found.', 'wordpress-seo' ) );
	}

	/**
	 * Parse the option group and import it.
	 *
	 * @param string $name         Name string.
	 * @param array  $option_group Option group data.
	 * @param array  $options      Options data.
	 *
	 * @return void
	 */
	protected function parse_option_group( $name, $option_group, $options ) {
		// Make sure that the imported options are cleaned/converted on import.
		$option_instance = WPSEO_Options::get_option_instance( $name );
		if ( is_object( $option_instance ) && method_exists( $option_instance, 'import' ) ) {
			$option_instance->import( $option_group, $this->old_wpseo_version, $options );
		}
	}

	/**
	 * Imports the options if found.
	 *
	 * @param array $options The options parsed from the provided settings.
	 *
	 * @return void
	 */
	protected function import_options( $options ) {
		if ( isset( $options['wpseo']['version'] ) && $options['wpseo']['version'] !== '' ) {
			$this->old_wpseo_version = $options['wpseo']['version'];
		}

		foreach ( $options as $name => $option_group ) {
			$this->parse_option_group( $name, $option_group, $options );
		}

		$this->status->set_msg( __( 'Settings successfully imported.', 'wordpress-seo' ) );
		$this->status->set_status( true );

		// Reset the cached option values.
		WPSEO_Options