al,
			$reverse,
			$help,
			$strong,
			$disabled_attribute
		);

		// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: All output is properly escaped or hardcoded in the presenter.
		echo $output;
	}

	/**
	 * Create a Text input field.
	 *
	 * @since 2.0
	 * @since 2.1 Introduced the `$attr` parameter.
	 *
	 * @param string       $variable The variable within the option to create the text input field for.
	 * @param string       $label    The label to show for the variable.
	 * @param array|string $attr     Extra attributes to add to the input field. Can be class, disabled, autocomplete.
	 *
	 * @return void
	 */
	public function textinput( $variable, $label, $attr = [] ) {
		$type = 'text';
		if ( ! is_array( $attr ) ) {
			$attr = [
				'class'    => $attr,
				'disabled' => false,
			];
		}

		$defaults = [
			'placeholder' => '',
			'class'       => '',
		];
		$attr     = wp_parse_args( $attr, $defaults );
		$val      = $this->get_field_value( $variable, '' );
		if ( isset( $attr['type'] ) && $attr['type'] === 'url' ) {
			$val  = urldecode( $val );
			$type = 'url';
		}
		$attributes = isset( $attr['autocomplete'] ) ? ' autocomplete="' . esc_attr( $attr['autocomplete'] ) . '"' : '';

		$this->label(
			$label,
			[
				'for'   => $variable,
				'class' => 'textinput',
			]
		);

		$aria_attributes = Yoast_Input_Validation::get_the_aria_invalid_attribute( $variable );

		Yoast_Input_Validation::set_error_descriptions();
		$aria_attributes .= Yoast_Input_Validation::get_the_aria_describedby_attribute( $variable );

		$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

		// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded and all other output is properly escaped.
		echo '<input', $attributes, $aria_attributes, ' class="', esc_attr( 'textinput ' . $attr['class'] ), '" placeholder="', esc_attr( $attr['placeholder'] ), '" type="', $type, '" id="', esc_attr( $variable ), '" name="', esc_attr( $this->option_name . '[' . $variable . ']' ), '" value="', esc_attr( $val ), '"', $disabled_attribute, '/>', '<br class="clear" />';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Output escaped in getter.
		echo Yoast_Input_Validation::get_the_error_description( $variable );
	}

	/**
	 * Create a Number input field.
	 *
	 * @param string       $variable The variable within the option to create the text input field for.
	 * @param string       $label    The label to show for the variable.
	 * @param array|string $attr     Extra attributes to add to the input field. Can be class, disabled, autocomplete.
	 *
	 * @return void
	 */
	public function number( $variable, $label, $attr = [] ) {
		$type     = 'number';
		$defaults = [
			'placeholder' => '',
			'class'       => 'number',
			'disabled'    => false,
			'min'         => 0,
			'max'         => 100,
		];
		$attr     = wp_parse_args( $attr, $defaults );
		$val      = $this->get_field_value( $variable, 0 );

		$this->label(
			$label,
			[
				'for'   => $variable,
				'class' => 'textinput ' . $attr['class'],
			]
		);

		$aria_attributes = Yoast_Input_Validation::get_the_aria_invalid_attribute( $variable );

		Yoast_Input_Validation::set_error_descriptions();
		$aria_attributes .= Yoast_Input_Validation::get_the_aria_describedby_attribute( $variable );

		$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

		// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded and all other output is properly escaped.
		echo '<input' . $aria_attributes . ' class="' . esc_attr( $attr['class'] ) . '" type="' . $type . '" id="', esc_attr( $variable ), '" min="', esc_attr( $attr['min'] ), '" max="', esc_attr( $attr['max'] ), '" name="', esc_attr( $this->option_name . '[' . $variable . ']' ), '" value="', esc_attr( $val ), '"', $disabled_attribute, '/>', '<br class="clear" />';
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Output escaped in getter.
		echo Yoast_Input_Validation::get_the_error_description( $variable );
	}

	/**
	 * Creates a text input field with with the ability to add content after the label.
	 *
	 * @param string $variable The variable within the option to create the text input field for.
	 * @param string $label    The label to show for the variable.
	 * @param array  $attr     Extra attributes to add to the input field.
	 *
	 * @return void
	 */
	public function textinput_extra_content( $variable, $label, $attr = [] ) {
		$type = 'text';

		$defaults = [
			'class'       => 'yoast-field-group__inputfield',
			'disabled'    => false,
		];

		$attr = wp_parse_args( $attr, $defaults );
		$val  = $this->get_field_value( $variable, '' );

		if ( isset( $attr['type'] ) && $attr['type'] === 'url' ) {
			$val  = urldecode( $val );
			$type = 'url';
		}

		echo '<div class="yoast-field-group__title">';
		$this->label(
			$label,
			[
				'for'   => $variable,
				'class' => $attr['class'] . '--label',
			]
		);

		if ( isset( $attr['extra_content'] ) ) {
			// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: may contain HTML that should not be escaped.
			echo $attr['extra_content'];
		}
		echo '</div>';

		$has_input_error = Yoast_Input_Validation::yoast_form_control_has_error( $variable );
		$aria_attributes = Yoast_Input_Validation::get_the_aria_invalid_attribute( $variable );

		Yoast_Input_Validation::set_error_descriptions();
		$aria_attributes .= Yoast_Input_Validation::get_the_aria_describedby_attribute( $variable );

		// phpcs:disable WordPress.Security.EscapeOutput -- Reason: output is properly escaped or hardcoded.
		printf(
			'<input type="%1$s" name="%2$s" id="%3$s" class="%4$s"%5$s%6$s%7$s value="%8$s"%9$s>',
			$type,
			esc_attr( $this->option_name . '[' . $variable . ']' ),
			esc_attr( $variable ),
			esc_attr( $attr['class'] ),
			isset( $attr['placeholder'] ) ? ' placeholder="' . esc_attr( $attr['placeholder'] ) . '"' : '',
			isset( $attr['autocomplete'] ) ? ' autocomplete="' . esc_attr( $attr['autocomplete'] ) . '"' : '',
			$aria_attributes,
			esc_attr( $val ),
			$this->get_disabled_attribute( $variable, $attr )
		);
		// phpcs:enable
		// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: output is properly escaped.
		echo Yoast_Input_Validation::get_the_error_description( $variable );
	}

	/**
	 * Create a textarea.
	 *
	 * @since 2.0
	 *
	 * @param string       $variable The variable within the option to create the textarea for.
	 * @param string       $label    The label to show for the variable.
	 * @param string|array $attr     The CSS class or an array of attributes to assign to the textarea.
	 *
	 * @return void
	 */
	public function textarea( $variable, $label, $attr = [] ) {
		if ( ! is_array( $attr ) ) {
			$attr = [
				'class' => $attr,
			];
		}

		$defaults = [
			'cols'     => '',
			'rows'     => '',
			'class'    => '',
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );
		$val      = $this->get_field_value( $variable, '' );

		$this->label(
			$label,
			[
				'for'   => $variable,
				'class' => 'textinput',
			]
		);

		$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

		// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded and all other output is properly escaped.
		echo '<textarea cols="' . esc_attr( $attr['cols'] ) . '" rows="' . esc_attr( $attr['rows'] ) . '" class="' . esc_attr( 'textinput ' . $attr['class'] ) . '" id="' . esc_attr( $variable ) . '" name="' . esc_attr( $this->option_name . '[' . $variable . ']' ), '"', $disabled_attribute, '>' . esc_textarea( $val ) . '</textarea><br class="clear" />';
	}

	/**
	 * Create a hidden input field.
	 *
	 * @since 2.0
	 *
	 * @param string $variable The variable within the option to create the hidden input for.
	 * @param string $id       The ID of the element.
	 * @param mixed  $val      Optional. The value to set in the input field. Otherwise the value from the options will be used.
	 *
	 * @return void
	 */
	public function hidden( $variable, $id = '', $val = null ) {
		if ( is_null( $val ) ) {
			$val = $this->get_field_value( $variable, '' );
		}

		if ( is_bool( $val ) ) {
			$val = ( $val === true ) ? 'true' : 'false';
		}

		if ( $id === '' ) {
			$id = 'hidden_' . $variable;
		}

		echo '<input type="hidden" id="' . esc_attr( $id ) . '" name="' . esc_attr( $this->option_name . '[' . $variable . ']' ), '" value="' . esc_attr( $val ) . '"/>';
	}

	/**
	 * Create a Select Box.
	 *
	 * @since 2.0
	 *
	 * @param string $variable       The variable within the option to create the select for.
	 * @param string $label          The label to show for the variable.
	 * @param array  $select_options The select options to choose from.
	 * @param string $styled         The select style. Use 'styled' to get a styled select. Default 'unstyled'.
	 * @param bool   $show_label     Whether or not to show the label, if not, it will be applied as an aria-label.
	 * @param array  $attr           Extra attributes to add to the select.
	 * @param string $help           Optional. Inline Help HTML that will be printed after the label. Default is empty.
	 *
	 * @return void
	 */
	public function select( $variable, $label, array $select_options, $styled = 'unstyled', $show_label = true, $attr = [], $help = '' ) {
		if ( empty( $select_options ) ) {
			return;
		}

		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		if ( $show_label ) {
			$this->label(
				$label,
				[
					'for'   => $variable,
					'class' => 'select',
				]
			);
			echo $help; // phpcs:ignore WordPress.Security.EscapeOutput -- Reason: The help contains HTML.
		}

		$select_name       = esc_attr( $this->option_name ) . '[' . esc_attr( $variable ) . ']';
		$active_option     = $this->get_field_value( $variable, '' );
		$wrapper_start_tag = '';
		$wrapper_end_tag   = '';

		$select = new Yoast_Input_Select( $variable, $select_name, $select_options, $active_option );
		$select->add_attribute( 'class', 'select' );

		if ( $this->is_control_disabled( $variable )
			|| ( isset( $attr['disabled'] ) && $attr['disabled'] ) ) {
			$select->add_attribute( 'disabled', 'disabled' );
		}

		if ( ! $show_label ) {
			$select->add_attribute( 'aria-label', $label );
		}

		if ( $styled === 'styled' ) {
			$wrapper_start_tag = '<span class="yoast-styled-select">';
			$wrapper_end_tag   = '</span>';
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
		echo $wrapper_start_tag;
		$select->output_html();
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
		echo $wrapper_end_tag;
		echo '<br class="clear"/>';
	}

	/**
	 * Create a File upload field.
	 *
	 * @since 2.0
	 *
	 * @param string $variable The variable within the option to create the file upload field for.
	 * @param string $label    The label to show for the variable.
	 * @param array  $attr     Extra attributes to add to the file upload input.
	 *
	 * @return void
	 */
	public function file_upload( $variable, $label, $attr = [] ) {
		$val = $this->get_field_value( $variable, '' );
		if ( is_array( $val ) ) {
			$val = $val['url'];
		}

		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		$var_esc = esc_attr( $variable );
		$this->label(
			$label,
			[
				'for'   => $variable,
				'class' => 'select',
			]
		);

		$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

		// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded and all other output is properly escaped.
		echo '<input type="file" value="' . esc_attr( $val ) . '" class="textinput" name="' . esc_attr( $this->option_name ) . '[' . $var_esc . ']" id="' . $var_esc . '"', $disabled_attribute, '/>';

		// Need to save separate array items in hidden inputs, because empty file inputs type will be deleted by settings API.
		if ( ! empty( $val ) ) {
			$this->hidden( 'file', $this->option_name . '_file' );
			$this->hidden( 'url', $this->option_name . '_url' );
			$this->hidden( 'type', $this->option_name . '_type' );
		}
		echo '<br class="clear"/>';
	}

	/**
	 * Media input.
	 *
	 * @since 2.0
	 *
	 * @param string $variable Option name.
	 * @param string $label    Label message.
	 * @param array  $attr     Extra attributes to add to the media input and buttons.
	 *
	 * @return void
	 */
	public function media_input( $variable, $label, $attr = [] ) {
		$val      = $this->get_field_value( $variable, '' );
		$id_value = $this->get_field_value( $variable . '_id', '' );

		$var_esc = esc_attr( $variable );

		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		$this->label(
			$label,
			[
				'for'   => 'wpseo_' . $variable,
				'class' => 'select',
			]
		);

		$id_field_id = 'wpseo_' . $var_esc . '_id';

		$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

		echo '<span>';
			echo '<input',
				' class="textinput"',
				' id="wpseo_', $var_esc, '"', // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
				' type="text" size="36"',
				' name="', esc_attr( $this->option_name ), '[', $var_esc, ']"', // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
				' value="', esc_attr( $val ), '"',
				' readonly="readonly"',
				' /> ';
			echo '<input',
				' id="wpseo_', $var_esc, '_button"', // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
				' class="wpseo_image_upload_button button"',
				' type="button"',
				' value="', esc_attr__( 'Upload Image', 'wordpress-seo' ), '"',
				' data-target-id="', esc_attr( $id_field_id ), '"',
				// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded.
				$disabled_attribute,
				' /> ';
			echo '<input',
				' class="wpseo_image_remove_button button"',
				' type="button"',
				' value="', esc_attr__( 'Clear Image', 'wordpress-seo' ), '"',
				// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded.
				$disabled_attribute,
				' />';
			echo '<input',
				' type="hidden"',
				' id="', esc_attr( $id_field_id ), '"',
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
				' name="', esc_attr( $this->option_name ), '[', $var_esc, '_id]"',
				' value="', esc_attr( $id_value ), '"',
				' />';
		echo '</span>';
		echo '<br class="clear"/>';
	}

	/**
	 * Create a Radio input field.
	 *
	 * @since 2.0
	 *
	 * @param string $variable    The variable within the option to create the radio button for.
	 * @param array  $values      The radio options to choose from.
	 * @param string $legend      Optional. The legend to show for the field set, if any.
	 * @param array  $legend_attr Optional. The attributes for the legend, if any.
	 * @param array  $attr        Extra attributes to add to the radio button.
	 *
	 * @return void
	 */
	public function radio( $variable, $values, $legend = '', $legend_attr = [], $attr = [] ) {
		if ( ! is_array( $values ) || $values === [] ) {
			return;
		}
		$val = $this->get_field_value( $variable, false );

		$var_esc = esc_attr( $variable );

		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
		echo '<fieldset class="yoast-form-fieldset wpseo_radio_block" id="' . $var_esc . '">';

		if ( is_string( $legend ) && $legend !== '' ) {

			$legend_defaults = [
				'id'    => '',
				'class' => 'radiogroup',
			];

			$legend_attr = wp_parse_args( $legend_attr, $legend_defaults );

			$this->legend( $legend, $legend_attr );
		}

		foreach ( $values as $key => $value ) {
			$label      = $value;
			$aria_label = '';

			if ( is_array( $value ) ) {
				$label      = ( $value['label'] ?? '' );
				$aria_label = ( $value['aria_label'] ?? '' );
			}

			$key_esc = esc_attr( $key );

			$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

			// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded and all other output is properly escaped.
			echo '<input type="radio" class="radio" id="' . $var_esc . '-' . $key_esc . '" name="' . esc_attr( $this->option_name ) . '[' . $var_esc . ']" value="' . $key_esc . '" ' . checked( $val, $key_esc, false ) . $disabled_attribute . ' />';
			$this->label(
				$label,
				[
					'for'        => $var_esc . '-' . $key_esc,
					'class'      => 'radio',
					'aria_label' => $aria_label,
				]
			);
		}
		echo '</fieldset>';
	}

	/**
	 * Create a toggle switch input field using two radio buttons.
	 *
	 * @since 3.1
	 *
	 * @param string $variable The variable within the option to create the radio buttons for.
	 * @param array  $values   Associative array of on/off keys and their values to be used as
	 *                         the label elements text for the radio buttons. Optionally, each
	 *                         value can be an array of visible label text and screen reader text.
	 * @param string $label    The visual label for the radio buttons group, used as the fieldset legend.
	 * @param string $help     Inline Help that will be printed out before the visible toggles text.
	 * @param array  $attr     Extra attributes to add to the toggle switch.
	 *
	 * @return void
	 */
	public function toggle_switch( $variable, $values, $label, $help = '', $attr = [] ) {
		if ( ! is_array( $values ) || $values === [] ) {
			return;
		}

		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		if ( isset( $attr['preserve_disabled_value'] ) && $attr['preserve_disabled_value'] ) {
			$this->hidden( $variable );
			$variable .= '_disabled';
		}

		$val = $this->get_field_value( $variable, false );
		if ( $val === true ) {
			$val = 'on';
		}
		if ( $val === false ) {
			$val = 'off';
		}

		$help_class = ! empty( $help ) ? ' switch-container__has-help' : '';

		$has_premium_upsell = ( isset( $attr['show_premium_upsell'] ) && $attr['show_premium_upsell'] && isset( $attr['premium_upsell_url'] ) && ! empty( $attr['premium_upsell_url'] ) );
		$upsell_class       = ( $has_premium_upsell ) ? ' premium-upsell' : '';

		$var_esc = esc_attr( $variable );

		printf( '<div class="%s">', esc_attr( 'switch-container' . $help_class . $upsell_class ) );
		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
		echo '<fieldset id="', $var_esc, '" class="fieldset-switch-toggle"><legend>', $label, '</legend>', $help;

		// Show disabled note if attribute does not exists or does exist and is set to true.
		if ( ! isset( $attr['show_disabled_note'] ) || ( $attr['show_disabled_note'] === true ) ) {
			if ( isset( $attr['note_when_disabled'] ) ) {
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
				echo $this->get_disabled_note( $variable, $attr['note_when_disabled'] );
			}
			else {
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
				echo $this->get_disabled_note( $variable );
			}
		}

		echo '<div class="switch-toggle switch-candy switch-yoast-seo">';

		foreach ( $values as $key => $value ) {
			$screen_reader_text_html = '';

			if ( is_array( $value ) ) {
				$screen_reader_text      = $value['screen_reader_text'];
				$screen_reader_text_html = '<span class="screen-reader-text"> ' . esc_html( $screen_reader_text ) . '</span>';
				$value                   = $value['text'];
			}

			$key_esc            = esc_attr( $key );
			$for                = $var_esc . '-' . $key_esc;
			$disabled_attribute = $this->get_disabled_attribute( $variable, $attr );

			// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: $disabled_attribute output is hardcoded and all other output is properly escaped.
			echo '<input type="radio" id="' . $for . '" name="' . esc_attr( $this->option_name ) . '[' . $var_esc . ']" value="' . $key_esc . '" ' . checked( $val, $key_esc, false ) . $disabled_attribute . ' />',
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- output escaped before.
			'<label for="', $for, '">', esc_html( $value ), $screen_reader_text_html, '</label>';
		}

		$upsell_button = '';
		if ( $has_premium_upsell ) {
			$upsell_button = '<a class="yoast-button yoast-button--buy yoast-button--small" data-action="load-nfd-ctb" data-ctb-id="f6a84663-465f-4cb5-8ba5-f7a6d72224b2" href='
							. esc_url( $attr['premium_upsell_url'] ) . ' target="_blank">'
							. esc_html__( 'Unlock with Premium!', 'wordpress-seo' )
							/* translators: Hidden accessibility text. */
							. '<span class="screen-reader-text">' . esc_html__( '(Opens in a new browser tab)', 'wordpress-seo' ) . '</span>'
							. '<span aria-hidden="true" class="yoast-button--buy__caret"></span></a>';
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- All variable output is escaped above.
		echo '<a></a></div></fieldset><div class="clear"></div>' . $upsell_button . '</div>' . PHP_EOL . PHP_EOL;
	}

	/**
	 * Creates a toggle switch to define whether an indexable should be indexed or not.
	 *
	 * @param string $variable The variable within the option to create the radio buttons for.
	 * @param string $label    The visual label for the radio buttons group, used as the fieldset legend.
	 * @param string $help     Inline Help that will be printed out before the visible toggles text.
	 * @param array  $attr     Extra attributes to add to the index switch.
	 *
	 * @return void
	 */
	public function index_switch( $variable, $label, $help = '', $attr = [] ) {
		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		$index_switch_values = [
			'off' => __( 'On', 'wordpress-seo' ),
			'on'  => __( 'Off', 'wordpress-seo' ),
		];

		$is_disabled = ( isset( $attr['disabled'] ) && $attr['disabled'] );

		$this->toggle_switch(
			$variable,
			$index_switch_values,
			sprintf(
				/* translators: %s expands to an indexable object's name, like a post type or taxonomy */
				esc_html__( 'Show %s in search results?', 'wordpress-seo' ),
				$label
			),
			$help,
			[ 'disabled' => $is_disabled ]
		);
	}

	/**
	 * Creates a toggle switch to show hide certain options.
	 *
	 * @param string $variable     The variable within the option to create the radio buttons for.
	 * @param string $label        The visual label for the radio buttons group, used as the fieldset legend.
	 * @param bool   $inverse_keys Whether or not the option keys need to be inverted to support older functions.
	 * @param string $help         Inline Help that will be printed out before the visible toggles text.
	 * @param array  $attr         Extra attributes to add to the show-hide switch.
	 *
	 * @return void
	 */
	public function show_hide_switch( $variable, $label, $inverse_keys = false, $help = '', $attr = [] ) {
		$defaults = [
			'disabled' => false,
		];
		$attr     = wp_parse_args( $attr, $defaults );

		$on_key  = ( $inverse_keys ) ? 'off' : 'on';
		$off_key = ( $inverse_keys ) ? 'on' : 'off';

		$show_hide_switch = [
			$on_key  => __( 'On', 'wordpress-seo' ),
			$off_key => __( 'Off', 'wordpress-seo' ),
		];

		$is_disabled = ( isset( $attr['disabled'] ) && $attr['disabled'] );

		$this->toggle_switch(
			$variable,
			$show_hide_switch,
			$label,
			$help,
			[ 'disabled' => $is_disabled ]
		);
	}

	/**
	 * Retrieves the value for the form field.
	 *
	 * @param string      $field_name    The field name to retrieve the value for.
	 * @param string|null $default_value The default value, when field has no value.
	 *
	 * @return mixed|null The retrieved value.
	 */
	protected function get_field_value( $field_name, $default_value = null ) {
		// On multisite subsites, the Usage tracking feature should always be set to Off.
		if ( $this->is_tracking_on_subsite( $field_name ) ) {
			return false;
		}

		return WPSEO_Options::get( $field_name, $default_value );
	}

	/**
	 * Checks whether a given control should be disabled.
	 *
	 * @param string $variable The variable within the option to check whether its control should be disabled.
	 *
	 * @return bool True if control should be disabled, false otherwise.
	 */
	protected function is_control_disabled( $variable ) {
		if ( $this->option_instance === null ) {
			return false;
		}

		// Disable the Usage tracking feature for multisite subsites.
		if ( $this->is_tracking_on_subsite( $variable ) ) {
			return true;
		}

		return $this->option_instance->is_disabled( $variable );
	}

	/**
	 * Gets the explanation note to print if a given control is disabled.
	 *
	 * @param string $variable    The variable within the option to print a disabled note for.
	 * @param string $custom_note An optional custom note to print instead.
	 *
	 * @return string Explanation note HTML string, or empty string if no note necessary.
	 */
	protected function get_disabled_note( $variable, $custom_note = '' ) {
		if ( $custom_note === '' && ! $this->is_control_disabled( $variable ) ) {
			return '';
		}
		$disabled_message = esc_html__( 'This feature has been disabled by the network admin.', 'wordpress-seo' );

		// The explanation to show when disabling the Usage tracking feature for multisite subsites.
		if ( $this->is_tracking_on_subsite( $variable ) ) {
			$disabled_message = esc_html__( 'This feature has been disabled since subsites never send tracking data.', 'wordpress-seo' );
		}

		if ( $custom_note ) {
			$disabled_message = esc_html( $custom_note );
		}

		return '<p class="disabled-note">' . $disabled_message . '</p>';
	}

	/**
	 * Determines whether we are dealing with the Usage tracking feature on a multisite subsite.
	 * This feature requires specific behavior for the toggle switch.
	 *
	 * @param string $feature_setting The feature setting.
	 *
	 * @return bool True if we are dealing with the Usage tracking feature on a multisite subsite.
	 */
	protected function is_tracking_on_subsite( $feature_setting ) {
		return ( $feature_setting === 'tracking' && ! is_network_admin() && ! is_main_site() );
	}

	/**
	 * Returns the disabled attribute HTML.
	 *
	 * @param string $variable The variable within the option of the related form element.
	 * @param array  $attr     Extra attributes added to the form element.
	 *
	 * @return string The disabled attribute HTML.
	 */
	protected function get_disabled_attribute( $variable, $attr ) {
		if ( $this->is_control_disabled( $variable ) || ( isset( $attr['disabled'] ) && $attr['disabled'] ) ) {
			return ' disabled';
		}

		return '';
	}
}
                                                                                                                                                                                                                                                                                                          plugins/wordpress-seo-extended/admin/class-yoast-input-validation.php                               0000644                 00000024765 15122266554 0022215 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin
 */

/**
 * Implements server-side user input validation.
 *
 * @since 12.0
 */
class Yoast_Input_Validation {

	/**
	 * The error descriptions.
	 *
	 * @since 12.1
	 *
	 * @var array
	 */
	private static $error_descriptions = [];

	/**
	 * Check whether an option group is a Yoast SEO setting.
	 *
	 * The normal pattern is 'yoast' . $option_name . 'options'.
	 *
	 * @since 12.0
	 *
	 * @param string $group_name The option group name.
	 *
	 * @return bool Whether or not it's an Yoast SEO option group.
	 */
	public static function is_yoast_option_group_name( $group_name ) {
		return ( strpos( $group_name, 'yoast' ) !== false );
	}

	/**
	 * Adds an error message to the document title when submitting a settings
	 * form and errors are returned.
	 *
	 * Uses the WordPress `admin_title` filter in the WPSEO_Option subclasses.
	 *
	 * @since 12.0
	 *
	 * @param string $admin_title The page title, with extra context added.
	 *
	 * @return string The modified or original admin title.
	 */
	public static function add_yoast_admin_document_title_errors( $admin_title ) {
		$errors      = get_settings_errors();
		$error_count = 0;

		foreach ( $errors as $error ) {
			// For now, filter the admin title only in the Yoast SEO settings pages.
			if ( self::is_yoast_option_group_name( $error['setting'] ) && $error['code'] !== 'settings_updated' ) {
				++$error_count;
			}
		}

		if ( $error_count > 0 ) {
			return sprintf(
				/* translators: %1$s: amount of errors, %2$s: the admin page title */
				_n( 'The form contains %1$s error. %2$s', 'The form contains %1$s errors. %2$s', $error_count, 'wordpress-seo' ),
				number_format_i18n( $error_count ),
				$admin_title
			);
		}

		return $admin_title;
	}

	/**
	 * Checks whether a specific form input field was submitted with an invalid value.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Must be the same slug-name used for the field variable and for `add_settings_error()`.
	 *
	 * @return bool Whether or not the submitted input field contained an invalid value.
	 */
	public static function yoast_form_control_has_error( $error_code ) {
		$errors = get_settings_errors();

		foreach ( $errors as $error ) {
			if ( $error['code'] === $error_code ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Sets the error descriptions.
	 *
	 * @since 12.1
	 *
	 * @param array $descriptions An associative array of error descriptions.
	 *                            For each entry, the key must be the setting variable.
	 *
	 * @return void
	 */
	public static function set_error_descriptions( $descriptions = [] ) {
		$defaults = [
			'baiduverify'     => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Baidu verification codes can only contain letters, numbers, hyphens, and underscores. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'baiduverify' )
			),
			'facebook_site'   => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the Facebook Page URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'facebook_site' )
			),
			'googleverify'    => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Google verification codes can only contain letters, numbers, hyphens, and underscores. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'googleverify' )
			),
			'instagram_url'   => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the Instagram URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'instagram_url' )
			),
			'linkedin_url'    => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the LinkedIn URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'linkedin_url' )
			),
			'msverify'        => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Bing confirmation codes can only contain letters from A to F, numbers, hyphens, and underscores. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'msverify' )
			),
			'myspace_url'     => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the MySpace URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'myspace_url' )
			),
			'pinterest_url'   => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the Pinterest URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'pinterest_url' )
			),
			'pinterestverify' => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Pinterest confirmation codes can only contain letters from A to F, numbers, hyphens, and underscores. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'pinterestverify' )
			),
			'twitter_site'    => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Twitter usernames can only contain letters, numbers, and underscores. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'twitter_site' )
			),
			'wikipedia_url'   => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the Wikipedia URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'wikipedia_url' )
			),
			'yandexverify'    => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Yandex confirmation codes can only contain letters from A to F, numbers, hyphens, and underscores. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'yandexverify' )
			),
			'youtube_url'     => sprintf(
				/* translators: %s: additional message with the submitted invalid value */
				esc_html__( 'Please check the format of the YouTube URL you entered. %s', 'wordpress-seo' ),
				self::get_dirty_value_message( 'youtube_url' )
			),
		];

		$descriptions = wp_parse_args( $descriptions, $defaults );

		self::$error_descriptions = $descriptions;
	}

	/**
	 * Gets all the error descriptions.
	 *
	 * @since 12.1
	 *
	 * @return array An associative array of error descriptions.
	 */
	public static function get_error_descriptions() {
		return self::$error_descriptions;
	}

	/**
	 * Gets a specific error description.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Code of the error set via `add_settings_error()`, normally the variable name.
	 *
	 * @return string|null The error description.
	 */
	public static function get_error_description( $error_code ) {
		if ( ! isset( self::$error_descriptions[ $error_code ] ) ) {
			return null;
		}

		return self::$error_descriptions[ $error_code ];
	}

	/**
	 * Gets the aria-invalid HTML attribute based on the submitted invalid value.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Code of the error set via `add_settings_error()`, normally the variable name.
	 *
	 * @return string The aria-invalid HTML attribute or empty string.
	 */
	public static function get_the_aria_invalid_attribute( $error_code ) {
		if ( self::yoast_form_control_has_error( $error_code ) ) {
			return ' aria-invalid="true"';
		}

		return '';
	}

	/**
	 * Gets the aria-describedby HTML attribute based on the submitted invalid value.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Code of the error set via `add_settings_error()`, normally the variable name.
	 *
	 * @return string The aria-describedby HTML attribute or empty string.
	 */
	public static function get_the_aria_describedby_attribute( $error_code ) {
		if ( self::yoast_form_control_has_error( $error_code ) && self::get_error_description( $error_code ) ) {
			return ' aria-describedby="' . esc_attr( $error_code ) . '-error-description"';
		}

		return '';
	}

	/**
	 * Gets the error description wrapped in a HTML paragraph.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Code of the error set via `add_settings_error()`, normally the variable name.
	 *
	 * @return string The error description HTML or empty string.
	 */
	public static function get_the_error_description( $error_code ) {
		$error_description = self::get_error_description( $error_code );

		if ( self::yoast_form_control_has_error( $error_code ) && $error_description ) {
			return '<p id="' . esc_attr( $error_code ) . '-error-description" class="yoast-input-validation__error-description">' . $error_description . '</p>';
		}

		return '';
	}

	/**
	 * Adds the submitted invalid value to the WordPress `$wp_settings_errors` global.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code  Code of the error set via `add_settings_error()`, normally the variable name.
	 * @param string $dirty_value The submitted invalid value.
	 *
	 * @return void
	 */
	public static function add_dirty_value_to_settings_errors( $error_code, $dirty_value ) {
		global $wp_settings_errors;

		if ( ! is_array( $wp_settings_errors ) ) {
			return;
		}

		foreach ( $wp_settings_errors as $index => $error ) {
			if ( $error['code'] === $error_code ) {
				// phpcs:ignore WordPress.WP.GlobalVariablesOverride -- This is a deliberate action.
				$wp_settings_errors[ $index ]['yoast_dirty_value'] = $dirty_value;
			}
		}
	}

	/**
	 * Gets an invalid submitted value.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Code of the error set via `add_settings_error()`, normally the variable name.
	 *
	 * @return string The submitted invalid input field value.
	 */
	public static function get_dirty_value( $error_code ) {
		$errors = get_settings_errors();

		foreach ( $errors as $error ) {
			if ( $error['code'] === $error_code && isset( $error['yoast_dirty_value'] ) ) {
				return $error['yoast_dirty_value'];
			}
		}

		return '';
	}

	/**
	 * Gets a specific invalid value message.
	 *
	 * @since 12.1
	 *
	 * @param string $error_code Code of the error set via `add_settings_error()`, normally the variable name.
	 *
	 * @return string The error invalid value message or empty string.
	 */
	public static function get_dirty_value_message( $error_code ) {
		$dirty_value = self::get_dirty_value( $error_code );

		if ( $dirty_value ) {
			return sprintf(
				/* translators: %s: form value as submitted. */
				esc_html__( 'The submitted value was: %s', 'wordpress-seo' ),
				esc_html( $dirty_value )
			);
		}

		return '';
	}
}
           plugins/wordpress-seo-extended/admin/class-yoast-network-admin.php                                  0000644                 00000023740 15122266554 0021475 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Internals
 */

/**
 * Multisite utility class for network admin functionality.
 */
class Yoast_Network_Admin implements WPSEO_WordPress_AJAX_Integration, WPSEO_WordPress_Integration {

	/**
	 * Action identifier for updating plugin network options.
	 *
	 * @var string
	 */
	public const UPDATE_OPTIONS_ACTION = 'yoast_handle_network_options';

	/**
	 * Action identifier for restoring a site.
	 *
	 * @var string
	 */
	public const RESTORE_SITE_ACTION = 'yoast_restore_site';

	/**
	 * Gets the available sites as choices, e.g. for a dropdown.
	 *
	 * @param bool $include_empty Optional. Whether to include an initial placeholder choice.
	 *                            Default false.
	 * @param bool $show_title    Optional. Whether to show the title for each site. This requires
	 *                            switching through the sites, so has performance implications for
	 *                            sites that do not use a persistent cache.
	 *                            Default false.
	 *
	 * @return array Choices as $site_id => $site_label pairs.
	 */
	public function get_site_choices( $include_empty = false, $show_title = false ) {
		$choices = [];

		if ( $include_empty ) {
			$choices['-'] = __( 'None', 'wordpress-seo' );
		}

		$criteria = [
			'deleted'    => 0,
			'network_id' => get_current_network_id(),
		];
		$sites    = get_sites( $criteria );

		foreach ( $sites as $site ) {
			$site_name = $site->domain . $site->path;
			if ( $show_title ) {
				$site_name = $site->blogname . ' (' . $site->domain . $site->path . ')';
			}
			$choices[ $site->blog_id ] = $site->blog_id . ': ' . $site_name;

			$site_states = $this->get_site_states( $site );
			if ( ! empty( $site_states ) ) {
				$choices[ $site->blog_id ] .= ' [' . implode( ', ', $site_states ) . ']';
			}
		}

		return $choices;
	}

	/**
	 * Gets the states of a site.
	 *
	 * @param WP_Site $site Site object.
	 *
	 * @return array Array of $state_slug => $state_label pairs.
	 */
	public function get_site_states( $site ) {
		$available_states = [
			'public'   => __( 'public', 'wordpress-seo' ),
			'archived' => __( 'archived', 'wordpress-seo' ),
			'mature'   => __( 'mature', 'wordpress-seo' ),
			'spam'     => __( 'spam', 'wordpress-seo' ),
			'deleted'  => __( 'deleted', 'wordpress-seo' ),
		];

		$site_states = [];
		foreach ( $available_states as $state_slug => $state_label ) {
			if ( $site->$state_slug === '1' ) {
				$site_states[ $state_slug ] = $state_label;
			}
		}

		return $site_states;
	}

	/**
	 * Handles a request to update plugin network options.
	 *
	 * This method works similar to how option updates are handled in `wp-admin/options.php` and
	 * `wp-admin/network/settings.php`.
	 *
	 * @return void
	 */
	public function handle_update_options_request() {
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Reason: Nonce verification will happen in verify_request below.
		if ( ! isset( $_POST['network_option_group'] ) || ! is_string( $_POST['network_option_group'] ) ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Reason: Nonce verification will happen in verify_request below.
		$option_group = sanitize_text_field( wp_unslash( $_POST['network_option_group'] ) );

		if ( empty( $option_group ) ) {
			return;
		}

		$this->verify_request( "{$option_group}-network-options" );

		$whitelist_options = Yoast_Network_Settings_API::get()->get_whitelist_options( $option_group );

		if ( empty( $whitelist_options ) ) {
			add_settings_error( $option_group, 'settings_updated', __( 'You are not allowed to modify unregistered network settings.', 'wordpress-seo' ), 'error' );

			$this->terminate_request();
			return;
		}

		// phpcs:disable WordPress.Security.NonceVerification -- Nonce verified via `verify_request()` above.
		foreach ( $whitelist_options as $option_name ) {
			$value = null;
			if ( isset( $_POST[ $option_name ] ) ) {
				// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: Adding sanitize_text_field around this will break the saving of settings because it expects a string: https://github.com/Yoast/wordpress-seo/issues/12440.
				$value = wp_unslash( $_POST[ $option_name ] );
			}

			WPSEO_Options::update_site_option( $option_name, $value );
		}
		// phpcs:enable WordPress.Security.NonceVerification

		$settings_errors = get_settings_errors();
		if ( empty( $settings_errors ) ) {
			add_settings_error( $option_group, 'settings_updated', __( 'Settings Updated.', 'wordpress-seo' ), 'updated' );
		}

		$this->terminate_request();
	}

	/**
	 * Handles a request to restore a site's default settings.
	 *
	 * @return void
	 */
	public function handle_restore_site_request() {
		$this->verify_request( 'wpseo-network-restore', 'restore_site_nonce' );

		$option_group = 'wpseo_ms';

		// phpcs:ignore WordPress.Security.NonceVerification -- Nonce verified via `verify_request()` above.
		$site_id = ! empty( $_POST[ $option_group ]['site_id'] ) ? (int) $_POST[ $option_group ]['site_id'] : 0;
		if ( ! $site_id ) {
			add_settings_error( $option_group, 'settings_updated', __( 'No site has been selected to restore.', 'wordpress-seo' ), 'error' );

			$this->terminate_request();
			return;
		}

		$site = get_site( $site_id );
		if ( ! $site ) {
			/* translators: %s expands to the ID of a site within a multisite network. */
			add_settings_error( $option_group, 'settings_updated', sprintf( __( 'Site with ID %d not found.', 'wordpress-seo' ), $site_id ), 'error' );
		}
		else {
			WPSEO_Options::reset_ms_blog( $site_id );

			/* translators: %s expands to the name of a site within a multisite network. */
			add_settings_error( $option_group, 'settings_updated', sprintf( __( '%s restored to default SEO settings.', 'wordpress-seo' ), esc_html( $site->blogname ) ), 'updated' );
		}

		$this->terminate_request();
	}

	/**
	 * Outputs nonce, action and option group fields for a network settings page in the plugin.
	 *
	 * @param string $option_group Option group name for the current page.
	 *
	 * @return void
	 */
	public function settings_fields( $option_group ) {
		?>
		<input type="hidden" name="network_option_group" value="<?php echo esc_attr( $option_group ); ?>" />
		<input type="hidden" name="action" value="<?php echo esc_attr( self::UPDATE_OPTIONS_ACTION ); ?>" />
		<?php
		wp_nonce_field( "$option_group-network-options" );
	}

	/**
	 * Enqueues network admin assets.
	 *
	 * @return void
	 */
	public function enqueue_assets() {
		$asset_manager = new WPSEO_Admin_Asset_Manager();
		$asset_manager->enqueue_script( 'network-admin' );

		$translations = [
			/* translators: %s: success message */
			'success_prefix' => __( 'Success: %s', 'wordpress-seo' ),
			/* translators: %s: error message */
			'error_prefix'   => __( 'Error: %s', 'wordpress-seo' ),
		];
		$asset_manager->localize_script(
			'network-admin',
			'wpseoNetworkAdminGlobalL10n',
			$translations
		);
	}

	/**
	 * Hooks in the necessary actions and filters.
	 *
	 * @return void
	 */
	public function register_hooks() {

		if ( ! $this->meets_requirements() ) {
			return;
		}

		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );

		add_action( 'admin_action_' . self::UPDATE_OPTIONS_ACTION, [ $this, 'handle_update_options_request' ] );
		add_action( 'admin_action_' . self::RESTORE_SITE_ACTION, [ $this, 'handle_restore_site_request' ] );
	}

	/**
	 * Hooks in the necessary AJAX actions.
	 *
	 * @return void
	 */
	public function register_ajax_hooks() {
		add_action( 'wp_ajax_' . self::UPDATE_OPTIONS_ACTION, [ $this, 'handle_update_options_request' ] );
		add_action( 'wp_ajax_' . self::RESTORE_SITE_ACTION, [ $this, 'handle_restore_site_request' ] );
	}

	/**
	 * Checks whether the requirements to use this class are met.
	 *
	 * @return bool True if requirements are met, false otherwise.
	 */
	public function meets_requirements() {
		return is_multisite() && is_network_admin();
	}

	/**
	 * Verifies that the current request is valid.
	 *
	 * @param string $action    Nonce action.
	 * @param string $query_arg Optional. Nonce query argument. Default '_wpnonce'.
	 *
	 * @return void
	 */
	public function verify_request( $action, $query_arg = '_wpnonce' ) {
		$has_access = current_user_can( 'wpseo_manage_network_options' );

		if ( wp_doing_ajax() ) {
			check_ajax_referer( $action, $query_arg );

			if ( ! $has_access ) {
				wp_die( -1, 403 );
			}
			return;
		}

		check_admin_referer( $action, $query_arg );

		if ( ! $has_access ) {
			wp_die( esc_html__( 'You are not allowed to perform this action.', 'wordpress-seo' ) );
		}
	}

	/**
	 * Terminates the current request by either redirecting back or sending an AJAX response.
	 *
	 * @return void
	 */
	public function terminate_request() {
		if ( wp_doing_ajax() ) {
			$settings_errors = get_settings_errors();

			if ( ! empty( $settings_errors ) && $settings_errors[0]['type'] === 'updated' ) {
				wp_send_json_success( $settings_errors, 200 );
			}

			wp_send_json_error( $settings_errors, 400 );
		}

		$this->persist_settings_errors();
		$this->redirect_back( [ 'settings-updated' => 'true' ] );
	}

	/**
	 * Persists settings errors.
	 *
	 * Settings errors are stored in a transient for 30 seconds so that this transient
	 * can be retrieved on the next page load.
	 *
	 * @return void
	 */
	protected function persist_settings_errors() {
		/*
		 * A regular transient is used here, since it is automatically cleared right after the redirect.
		 * A network transient would be cleaner, but would require a lot of copied code from core for
		 * just a minor adjustment when displaying settings errors.
		 */
		set_transient( 'settings_errors', get_settings_errors(), 30 );
	}

	/**
	 * Redirects back to the referer URL, with optional query arguments.
	 *
	 * @param array $query_args Optional. Query arguments to add to the redirect URL. Default none.
	 *
	 * @return void
	 */
	protected function redirect_back( $query_args = [] ) {
		$sendback = wp_get_referer();

		if ( ! empty( $query_args ) ) {
			$sendback = add_query_arg( $query_args, $sendback );
		}

		wp_safe_redirect( $sendback );
		exit;
	}
}
                                plugins/wordpress-seo-extended/admin/class-yoast-network-settings-api.php                           0000644                 00000010264 15122266554 0023011 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Network
 */

/**
 * Implements a network settings API for the plugin's multisite settings.
 */
class Yoast_Network_Settings_API {

	/**
	 * Registered network settings.
	 *
	 * @var array
	 */
	private $registered_settings = [];

	/**
	 * Options whitelist, keyed by option group.
	 *
	 * @var array
	 */
	private $whitelist_options = [];

	/**
	 * The singleton instance of this class.
	 *
	 * @var Yoast_Network_Settings_API
	 */
	private static $instance = null;

	/**
	 * Registers a network setting and its data.
	 *
	 * @param string $option_group The group the network option is part of.
	 * @param string $option_name  The name of the network option to sanitize and save.
	 * @param array  $args         {
	 *     Optional. Data used to describe the network setting when registered.
	 *
	 *     @type callable $sanitize_callback A callback function that sanitizes the network option's value.
	 *     @type mixed    $default           Default value when calling `get_network_option()`.
	 * }
	 *
	 * @return void
	 */
	public function register_setting( $option_group, $option_name, $args = [] ) {

		$defaults = [
			'group'             => $option_group,
			'sanitize_callback' => null,
		];
		$args     = wp_parse_args( $args, $defaults );

		if ( ! isset( $this->whitelist_options[ $option_group ] ) ) {
			$this->whitelist_options[ $option_group ] = [];
		}

		$this->whitelist_options[ $option_group ][] = $option_name;

		if ( ! empty( $args['sanitize_callback'] ) ) {
			add_filter( "sanitize_option_{$option_name}", [ $this, 'filter_sanitize_option' ], 10, 2 );
		}

		if ( array_key_exists( 'default', $args ) ) {
			add_filter( "default_site_option_{$option_name}", [ $this, 'filter_default_option' ], 10, 2 );
		}

		$this->registered_settings[ $option_name ] = $args;
	}

	/**
	 * Gets the registered settings and their data.
	 *
	 * @return array Array of $option_name => $data pairs.
	 */
	public function get_registered_settings() {
		return $this->registered_settings;
	}

	/**
	 * Gets the whitelisted options for a given option group.
	 *
	 * @param string $option_group Option group.
	 *
	 * @return array List of option names, or empty array if unknown option group.
	 */
	public function get_whitelist_options( $option_group ) {
		if ( ! isset( $this->whitelist_options[ $option_group ] ) ) {
			return [];
		}

		return $this->whitelist_options[ $option_group ];
	}

	/**
	 * Filters sanitization for a network option value.
	 *
	 * This method is added as a filter to `sanitize_option_{$option}` for network options that are
	 * registered with a sanitize callback.
	 *
	 * @param string $value  The sanitized option value.
	 * @param string $option The option name.
	 *
	 * @return string The filtered sanitized option value.
	 */
	public function filter_sanitize_option( $value, $option ) {

		if ( empty( $this->registered_settings[ $option ] ) ) {
			return $value;
		}

		return call_user_func( $this->registered_settings[ $option ]['sanitize_callback'], $value );
	}

	/**
	 * Filters the default value for a network option.
	 *
	 * This function is added as a filter to `default_site_option_{$option}` for network options that
	 * are registered with a default.
	 *
	 * @param mixed  $default_value Existing default value to return.
	 * @param string $option        The option name.
	 *
	 * @return mixed The filtered default value.
	 */
	public function filter_default_option( $default_value, $option ) {

		// If a default value was manually passed to the function, allow it to override.
		if ( $default_value !== false ) {
			return $default_value;
		}

		if ( empty( $this->registered_settings[ $option ] ) ) {
			return $default_value;
		}

		return $this->registered_settings[ $option ]['default'];
	}

	/**
	 * Checks whether the requirements to use this class are met.
	 *
	 * @return bool True if requirements are met, false otherwise.
	 */
	public function meets_requirements() {
		return is_multisite();
	}

	/**
	 * Gets the singleton instance of this class.
	 *
	 * @return Yoast_Network_Settings_API The singleton instance.
	 */
	public static function get() {

		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}
}
                                                                                                                                                                                                                                                                                                                                            plugins/wordpress-seo-extended/admin/class-yoast-notification-center.php                            0000644                 00000063462 15122266554 0022667 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Notifications
 */

use Yoast\WP\SEO\Presenters\Abstract_Presenter;

/**
 * Handles notifications storage and display.
 */
class Yoast_Notification_Center {

	/**
	 * Option name to store notifications on.
	 *
	 * @var string
	 */
	public const STORAGE_KEY = 'yoast_notifications';

	/**
	 * The singleton instance of this object.
	 *
	 * @var Yoast_Notification_Center
	 */
	private static $instance = null;

	/**
	 * Holds the notifications.
	 *
	 * @var Yoast_Notification[][]
	 */
	private $notifications = [];

	/**
	 * Notifications there are newly added.
	 *
	 * @var array
	 */
	private $new = [];

	/**
	 * Notifications that were resolved this execution.
	 *
	 * @var int
	 */
	private $resolved = 0;

	/**
	 * Internal storage for transaction before notifications have been retrieved from storage.
	 *
	 * @var array
	 */
	private $queued_transactions = [];

	/**
	 * Internal flag for whether notifications have been retrieved from storage.
	 *
	 * @var bool
	 */
	private $notifications_retrieved = false;

	/**
	 * Internal flag for whether notifications need to be updated in storage.
	 *
	 * @var bool
	 */
	private $notifications_need_storage = false;

	/**
	 * Construct.
	 */
	private function __construct() {

		add_action( 'init', [ $this, 'setup_current_notifications' ], 1 );

		add_action( 'all_admin_notices', [ $this, 'display_notifications' ] );

		add_action( 'wp_ajax_yoast_get_notifications', [ $this, 'ajax_get_notifications' ] );

		add_action( 'wpseo_deactivate', [ $this, 'deactivate_hook' ] );
		add_action( 'shutdown', [ $this, 'update_storage' ] );
	}

	/**
	 * Singleton getter.
	 *
	 * @return Yoast_Notification_Center
	 */
	public static function get() {

		if ( self::$instance === null ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Dismiss a notification.
	 *
	 * @return void
	 */
	public static function ajax_dismiss_notification() {
		$notification_center = self::get();

		if ( ! isset( $_POST['notification'] ) || ! is_string( $_POST['notification'] ) ) {
			die( '-1' );
		}

		$notification_id = sanitize_text_field( wp_unslash( $_POST['notification'] ) );

		if ( empty( $notification_id ) ) {
			die( '-1' );
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: We are using the variable as a nonce.
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( wp_unslash( $_POST['nonce'] ), $notification_id ) ) {
			die( '-1' );
		}

		$notification = $notification_center->get_notification_by_id( $notification_id );
		if ( ( $notification instanceof Yoast_Notification ) === false ) {

			// Permit legacy.
			$options      = [
				'id'            => $notification_id,
				'dismissal_key' => $notification_id,
			];
			$notification = new Yoast_Notification( '', $options );
		}

		if ( self::maybe_dismiss_notification( $notification ) ) {
			die( '1' );
		}

		die( '-1' );
	}

	/**
	 * Check if the user has dismissed a notification.
	 *
	 * @param Yoast_Notification $notification The notification to check for dismissal.
	 * @param int|null           $user_id      User ID to check on.
	 *
	 * @return bool
	 */
	public static function is_notification_dismissed( Yoast_Notification $notification, $user_id = null ) {

		$user_id       = self::get_user_id( $user_id );
		$dismissal_key = $notification->get_dismissal_key();

		// This checks both the site-specific user option and the meta value.
		$current_value = get_user_option( $dismissal_key, $user_id );

		// Migrate old user meta to user option on-the-fly.
		if ( ! empty( $current_value )
			&& metadata_exists( 'user', $user_id, $dismissal_key )
			&& update_user_option( $user_id, $dismissal_key, $current_value ) ) {
			delete_user_meta( $user_id, $dismissal_key );
		}

		return ! empty( $current_value );
	}

	/**
	 * Checks if the notification is being dismissed.
	 *
	 * @param Yoast_Notification $notification Notification to check dismissal of.
	 * @param string             $meta_value   Value to set the meta value to if dismissed.
	 *
	 * @return bool True if dismissed.
	 */
	public static function maybe_dismiss_notification( Yoast_Notification $notification, $meta_value = 'seen' ) {

		// Only persistent notifications are dismissible.
		if ( ! $notification->is_persistent() ) {
			return false;
		}

		// If notification is already dismissed, we're done.
		if ( self::is_notification_dismissed( $notification ) ) {
			return true;
		}

		$dismissal_key   = $notification->get_dismissal_key();
		$notification_id = $notification->get_id();

		$is_dismissing = ( $dismissal_key === self::get_user_input( 'notification' ) );
		if ( ! $is_dismissing ) {
			$is_dismissing = ( $notification_id === self::get_user_input( 'notification' ) );
		}

		// Fallback to ?dismissal_key=1&nonce=bla when JavaScript fails.
		if ( ! $is_dismissing ) {
			$is_dismissing = ( self::get_user_input( $dismissal_key ) === '1' );
		}

		if ( ! $is_dismissing ) {
			return false;
		}

		$user_nonce = self::get_user_input( 'nonce' );
		if ( wp_verify_nonce( $user_nonce, $notification_id ) === false ) {
			return false;
		}

		return self::dismiss_notification( $notification, $meta_value );
	}

	/**
	 * Dismisses a notification.
	 *
	 * @param Yoast_Notification $notification Notification to dismiss.
	 * @param string             $meta_value   Value to save in the dismissal.
	 *
	 * @return bool True if dismissed, false otherwise.
	 */
	public static function dismiss_notification( Yoast_Notification $notification, $meta_value = 'seen' ) {
		// Dismiss notification.
		return update_user_option( get_current_user_id(), $notification->get_dismissal_key(), $meta_value ) !== false;
	}

	/**
	 * Restores a notification.
	 *
	 * @param Yoast_Notification $notification Notification to restore.
	 *
	 * @return bool True if restored, false otherwise.
	 */
	public static function restore_notification( Yoast_Notification $notification ) {

		$user_id       = get_current_user_id();
		$dismissal_key = $notification->get_dismissal_key();

		// Restore notification.
		$restored = delete_user_option( $user_id, $dismissal_key );

		// Delete unprefixed user meta too for backward-compatibility.
		if ( metadata_exists( 'user', $user_id, $dismissal_key ) ) {
			$restored = delete_user_meta( $user_id, $dismissal_key ) && $restored;
		}

		return $restored;
	}

	/**
	 * Clear dismissal information for the specified Notification.
	 *
	 * When a cause is resolved, the next time it is present we want to show
	 * the message again.
	 *
	 * @param string|Yoast_Notification $notification Notification to clear the dismissal of.
	 *
	 * @return bool
	 */
	public function clear_dismissal( $notification ) {

		global $wpdb;

		if ( $notification instanceof Yoast_Notification ) {
			$dismissal_key = $notification->get_dismissal_key();
		}

		if ( is_string( $notification ) ) {
			$dismissal_key = $notification;
		}

		if ( empty( $dismissal_key ) ) {
			return false;
		}

		// Remove notification dismissal for all users.
		$deleted = delete_metadata( 'user', 0, $wpdb->get_blog_prefix() . $dismissal_key, '', true );

		// Delete unprefixed user meta too for backward-compatibility.
		$deleted = delete_metadata( 'user', 0, $dismissal_key, '', true ) || $deleted;

		return $deleted;
	}

	/**
	 * Retrieves notifications from the storage and merges in previous notification changes.
	 *
	 * The current user in WordPress is not loaded shortly before the 'init' hook, but the plugin
	 * sometimes needs to add or remove notifications before that. In such cases, the transactions
	 * are not actually executed, but added to a queue. That queue is then handled in this method,
	 * after notifications for the current user have been set up.
	 *
	 * @return void
	 */
	public function setup_current_notifications() {
		$this->retrieve_notifications_from_storage( get_current_user_id() );

		foreach ( $this->queued_transactions as $transaction ) {
			list( $callback, $args ) = $transaction;

			call_user_func_array( $callback, $args );
		}

		$this->queued_transactions = [];
	}

	/**
	 * Add notification to the cookie.
	 *
	 * @param Yoast_Notification $notification Notification object instance.
	 *
	 * @return void
	 */
	public function add_notification( Yoast_Notification $notification ) {

		$callback = [ $this, __FUNCTION__ ];
		$args     = func_get_args();
		if ( $this->queue_transaction( $callback, $args ) ) {
			return;
		}

		// Don't add if the user can't see it.
		if ( ! $notification->display_for_current_user() ) {
			return;
		}

		$notification_id = $notification->get_id();
		$user_id         = $notification->get_user_id();

		// Empty notifications are always added.
		if ( $notification_id !== '' ) {

			// If notification ID exists in notifications, don't add again.
			$present_notification = $this->get_notification_by_id( $notification_id, $user_id );
			if ( ! is_null( $present_notification ) ) {
				$this->remove_notification( $present_notification, false );
			}

			if ( is_null( $present_notification ) ) {
				$this->new[] = $notification_id;
			}
		}

		// Add to list.
		$this->notifications[ $user_id ][] = $notification;

		$this->notifications_need_storage = true;
	}

	/**
	 * Get the notification by ID and user ID.
	 *
	 * @param string   $notification_id The ID of the notification to search for.
	 * @param int|null $user_id         The ID of the user.
	 *
	 * @return Yoast_Notification|null
	 */
	public function get_notification_by_id( $notification_id, $user_id = null ) {
		$user_id = self::get_user_id( $user_id );

		$notifications = $this->get_notifications_for_user( $user_id );

		foreach ( $notifications as $notification ) {
			if ( $notification_id === $notification->get_id() ) {
				return $notification;
			}
		}

		return null;
	}

	/**
	 * Display the notifications.
	 *
	 * @param bool $echo_as_json True when notifications should be printed directly.
	 *
	 * @return void
	 */
	public function display_notifications( $echo_as_json = false ) {

		// Never display notifications for network admin.
		if ( is_network_admin() ) {
			return;
		}

		$sorted_notifications = $this->get_sorted_notifications();
		$notifications        = array_filter( $sorted_notifications, [ $this, 'is_notification_persistent' ] );

		if ( empty( $notifications ) ) {
			return;
		}

		array_walk( $notifications, [ $this, 'remove_notification' ] );

		$notifications = array_unique( $notifications );
		if ( $echo_as_json ) {
			$notification_json = [];

			foreach ( $notifications as $notification ) {
				$notification_json[] = $notification->render();
			}

			// phpcs:ignore WordPress.Security.EscapeOutput -- Reason: WPSEO_Utils::format_json_encode is safe.
			echo WPSEO_Utils::format_json_encode( $notification_json );

			return;
		}

		foreach ( $notifications as $notification ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Reason: Temporarily disabled, see: https://github.com/Yoast/wordpress-seo-premium/issues/2510 and https://github.com/Yoast/wordpress-seo-premium/issues/2511.
			echo $notification;
		}
	}

	/**
	 * Remove notification after it has been displayed.
	 *
	 * @param Yoast_Notification $notification Notification to remove.
	 * @param bool               $resolve      Resolve as fixed.
	 *
	 * @return void
	 */
	public function remove_notification( Yoast_Notification $notification, $resolve = true ) {

		$callback = [ $this, __FUNCTION__ ];
		$args     = func_get_args();
		if ( $this->queue_transaction( $callback, $args ) ) {
			return;
		}

		$index = false;

		// ID of the user to show the notification for, defaults to current user id.
		$user_id       = $notification->get_user_id();
		$notifications = $this->get_notifications_for_user( $user_id );

		// Match persistent Notifications by ID, non persistent by item in the array.
		if ( $notification->is_persistent() ) {
			foreach ( $notifications as $current_index => $present_notification ) {
				if ( $present_notification->get_id() === $notification->get_id() ) {
					$index = $current_index;
					break;
				}
			}
		}
		else {
			$index = array_search( $notification, $notifications, true );
		}

		if ( $index === false ) {
			return;
		}

		if ( $notification->is_persistent() && $resolve ) {
			++$this->resolved;
			$this->clear_dismissal( $notification );
		}

		unset( $notifications[ $index ] );
		$this->notifications[ $user_id ] = array_values( $notifications );

		$this->notifications_need_storage = true;
	}

	/**
	 * Removes a notification by its ID.
	 *
	 * @param string $notification_id The notification id.
	 * @param bool   $resolve         Resolve as fixed.
	 *
	 * @return void
	 */
	public function remove_notification_by_id( $notification_id, $resolve = true ) {
		$notification = $this->get_notification_by_id( $notification_id );

		if ( $notification === null ) {
			return;
		}

		$this->remove_notification( $notification, $resolve );
		$this->notifications_need_storage = true;
	}

	/**
	 * Get the notification count.
	 *
	 * @param bool $dismissed Count dismissed notifications.
	 *
	 * @return int Number of notifications
	 */
	public function get_notification_count( $dismissed = false ) {

		$notifications = $this->get_notifications_for_user( get_current_user_id() );
		$notifications = array_filter( $notifications, [ $this, 'filter_persistent_notifications' ] );

		if ( ! $dismissed ) {
			$notifications = array_filter( $notifications, [ $this, 'filter_dismissed_notifications' ] );
		}

		return count( $notifications );
	}

	/**
	 * Get the number of notifications resolved this execution.
	 *
	 * These notifications have been resolved and should be counted when active again.
	 *
	 * @return int
	 */
	public function get_resolved_notification_count() {

		return $this->resolved;
	}

	/**
	 * Return the notifications sorted on type and priority.
	 *
	 * @return array|Yoast_Notification[] Sorted Notifications
	 */
	public function get_sorted_notifications() {
		$notifications = $this->get_notifications_for_user( get_current_user_id() );
		if ( empty( $notifications ) ) {
			return [];
		}

		// Sort by severity, error first.
		usort( $notifications, [ $this, 'sort_notifications' ] );

		return $notifications;
	}

	/**
	 * AJAX display notifications.
	 *
	 * @return void
	 */
	public function ajax_get_notifications() {
		$echo = false;
		// phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: We are not processing form data.
		if ( isset( $_POST['version'] ) && is_string( $_POST['version'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: We are only comparing the variable in a condition.
			$echo = wp_unslash( $_POST['version'] ) === '2';
		}

		// Display the notices.
		$this->display_notifications( $echo );

		// AJAX die.
		exit;
	}

	/**
	 * Remove storage when the plugin is deactivated.
	 *
	 * @return void
	 */
	public function deactivate_hook() {

		$this->clear_notifications();
	}

	/**
	 * Returns the given user ID if it exists.
	 * Otherwise, this function returns the ID of the current user.
	 *
	 * @param int $user_id The user ID to check.
	 *
	 * @return int The user ID to use.
	 */
	private static function get_user_id( $user_id ) {
		if ( $user_id ) {
			return $user_id;
		}
		return get_current_user_id();
	}

	/**
	 * Splits the notifications on user ID.
	 *
	 * In other terms, it returns an associative array,
	 * mapping user ID to a list of notifications for this user.
	 *
	 * @param array|Yoast_Notification[] $notifications The notifications to split.
	 *
	 * @return array The notifications, split on user ID.
	 */
	private function split_on_user_id( $notifications ) {
		$split_notifications = [];
		foreach ( $notifications as $notification ) {
			$split_notifications[ $notification->get_user_id() ][] = $notification;
		}
		return $split_notifications;
	}

	/**
	 * Save persistent notifications to storage.
	 *
	 * We need to be able to retrieve these so they can be dismissed at any time during the execution.
	 *
	 * @since 3.2
	 *
	 * @return void
	 */
	public function update_storage() {

		$notifications = $this->notifications;

		/**
		 * One array of Yoast_Notifications, merged from multiple arrays.
		 *
		 * @var Yoast_Notification[] $merged_notifications
		 */
		$merged_notifications = [];
		if ( ! empty( $notifications ) ) {
			$merged_notifications = array_merge( ...$notifications );
		}

		/**
		 * Filter: 'yoast_notifications_before_storage' - Allows developer to filter notifications before saving them.
		 *
		 * @param Yoast_Notification[] $notifications
		 */
		$filtered_merged_notifications = apply_filters( 'yoast_notifications_before_storage', $merged_notifications );

		// The notifications were filtered and therefore need to be stored.
		if ( $merged_notifications !== $filtered_merged_notifications ) {
			$merged_notifications             = $filtered_merged_notifications;
			$this->notifications_need_storage = true;
		}

		$notifications = $this->split_on_user_id( $merged_notifications );

		// No notifications to store, clear storage if it was previously present.
		if ( empty( $notifications ) ) {
			$this->remove_storage();

			return;
		}

		// Only store notifications if changes are made.
		if ( $this->notifications_need_storage ) {
			array_walk( $notifications, [ $this, 'store_notifications_for_user' ] );
		}
	}

	/**
	 * Stores the notifications to its respective user's storage.
	 *
	 * @param array|Yoast_Notification[] $notifications The notifications to store.
	 * @param int                        $user_id       The ID of the user for which to store the notifications.
	 *
	 * @return void
	 */
	private function store_notifications_for_user( $notifications, $user_id ) {
		$notifications_as_arrays = array_map( [ $this, 'notification_to_array' ], $notifications );
		update_user_option( $user_id, self::STORAGE_KEY, $notifications_as_arrays );
	}

	/**
	 * Provide a way to verify present notifications.
	 *
	 * @return array|Yoast_Notification[] Registered notifications.
	 */
	public function get_notifications() {
		if ( ! $this->notifications ) {
			return [];
		}
		return array_merge( ...$this->notifications );
	}

	/**
	 * Returns the notifications for the given user.
	 *
	 * @param int $user_id The id of the user to check.
	 *
	 * @return Yoast_Notification[] The notifications for the user with the given ID.
	 */
	public function get_notifications_for_user( $user_id ) {
		if ( array_key_exists( $user_id, $this->notifications ) ) {
			return $this->notifications[ $user_id ];
		}
		return [];
	}

	/**
	 * Get newly added notifications.
	 *
	 * @return array
	 */
	public function get_new_notifications() {

		return array_map( [ $this, 'get_notification_by_id' ], $this->new );
	}

	/**
	 * Get information from the User input.
	 *
	 * Note that this function does not handle nonce verification.
	 *
	 * @param string $key Key to retrieve.
	 *
	 * @return string non-sanitized value of key if set, an empty string otherwise.
	 */
	private static function get_user_input( $key ) {
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized,WordPress.Security.NonceVerification.Missing -- Reason: We are not processing form information and only using this variable in a comparison.
		$request_method = isset( $_SERVER['REQUEST_METHOD'] ) && is_string( $_SERVER['REQUEST_METHOD'] ) ? strtoupper( wp_unslash( $_SERVER['REQUEST_METHOD'] ) ) : '';
		// phpcs:disable WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: This function does not sanitize variables.
		// phpcs:disable WordPress.Security.NonceVerification.Recommended,WordPress.Security.NonceVerification.Missing -- Reason: This function does not verify a nonce.
		if ( $request_method === 'POST' ) {
			if ( isset( $_POST[ $key ] ) && is_string( $_POST[ $key ] ) ) {
				return wp_unslash( $_POST[ $key ] );
			}
		}
		elseif ( isset( $_GET[ $key ] ) && is_string( $_GET[ $key ] ) ) {
			return wp_unslash( $_GET[ $key ] );
		}
		// phpcs:enable WordPress.Security.NonceVerification.Missing,WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		return '';
	}

	/**
	 * Retrieve the notifications from storage and fill the relevant property.
	 *
	 * @param int $user_id The ID of the user to retrieve notifications for.
	 *
	 * @return void
	 */
	private function retrieve_notifications_from_storage( $user_id ) {
		if ( $this->notifications_retrieved ) {
			return;
		}

		$this->notifications_retrieved = true;

		$stored_notifications = get_user_option( self::STORAGE_KEY, $user_id );

		// Check if notifications are stored.
		if ( empty( $stored_notifications ) ) {
			return;
		}

		if ( is_array( $stored_notifications ) ) {
			$notifications = array_map( [ $this, 'array_to_notification' ], $stored_notifications );

			// Apply array_values to ensure we get a 0-indexed array.
			$notifications = array_values( array_filter( $notifications, [ $this, 'filter_notification_current_user' ] ) );

			$this->notifications[ $user_id ] = $notifications;
		}
	}

	/**
	 * Sort on type then priority.
	 *
	 * @param Yoast_Notification $a Compare with B.
	 * @param Yoast_Notification $b Compare with A.
	 *
	 * @return int 1, 0 or -1 for sorting offset.
	 */
	private function sort_notifications( Yoast_Notification $a, Yoast_Notification $b ) {

		$a_type = $a->get_type();
		$b_type = $b->get_type();

		if ( $a_type === $b_type ) {
			return WPSEO_Utils::calc( $b->get_priority(), 'compare', $a->get_priority() );
		}

		if ( $a_type === 'error' ) {
			return -1;
		}

		if ( $b_type === 'error' ) {
			return 1;
		}

		return 0;
	}

	/**
	 * Clear local stored notifications.
	 *
	 * @return void
	 */
	private function clear_notifications() {

		$this->notifications           = [];
		$this->notifications_retrieved = false;
	}

	/**
	 * Filter out non-persistent notifications.
	 *
	 * @since 3.2
	 *
	 * @param Yoast_Notification $notification Notification to test for persistent.
	 *
	 * @return bool
	 */
	private function filter_persistent_notifications( Yoast_Notification $notification ) {

		return $notification->is_persistent();
	}

	/**
	 * Filter out dismissed notifications.
	 *
	 * @param Yoast_Notification $notification Notification to check.
	 *
	 * @return bool
	 */
	private function filter_dismissed_notifications( Yoast_Notification $notification ) {

		return ! self::maybe_dismiss_notification( $notification );
	}

	/**
	 * Convert Notification to array representation.
	 *
	 * @since 3.2
	 *
	 * @param Yoast_Notification $notification Notification to convert.
	 *
	 * @return array
	 */
	private function notification_to_array( Yoast_Notification $notification ) {

		$notification_data = $notification->to_array();

		if ( isset( $notification_data['nonce'] ) ) {
			unset( $notification_data['nonce'] );
		}

		return $notification_data;
	}

	/**
	 * Convert stored array to Notification.
	 *
	 * @param array $notification_data Array to convert to Notification.
	 *
	 * @return Yoast_Notification
	 */
	private function array_to_notification( $notification_data ) {

		if ( isset( $notification_data['options']['nonce'] ) ) {
			unset( $notification_data['options']['nonce'] );
		}

		if ( isset( $notification_data['message'] )
			&& is_subclass_of( $notification_data['message'], Abstract_Presenter::class, false )
		) {
			$notification_data['message'] = $notification_data['message']->present();
		}

		if ( isset( $notification_data['options']['user'] ) ) {
			$notification_data['options']['user_id'] = $notification_data['options']['user']->ID;
			unset( $notification_data['options']['user'] );

			$this->notifications_need_storage = true;
		}

		return new Yoast_Notification(
			$notification_data['message'],
			$notification_data['options']
		);
	}

	/**
	 * Filter notifications that should not be displayed for the current user.
	 *
	 * @param Yoast_Notification $notification Notification to test.
	 *
	 * @return bool
	 */
	private function filter_notification_current_user( Yoast_Notification $notification ) {
		return $notification->display_for_current_user();
	}

	/**
	 * Checks if given notification is persistent.
	 *
	 * @param Yoast_Notification $notification The notification to check.
	 *
	 * @return bool True when notification is not persistent.
	 */
	private function is_notification_persistent( Yoast_Notification $notification ) {
		return ! $notification->is_persistent();
	}

	/**
	 * Queues a notification transaction for later execution if notifications are not yet set up.
	 *
	 * @param callable $callback Callback that performs the transaction.
	 * @param array    $args     Arguments to pass to the callback.
	 *
	 * @return bool True if transaction was queued, false if it can be performed immediately.
	 */
	private function queue_transaction( $callback, $args ) {
		if ( $this->notifications_retrieved ) {
			return false;
		}

		$this->add_transaction_to_queue( $callback, $args );

		return true;
	}

	/**
	 * Adds a notification transaction to the queue for later execution.
	 *
	 * @param callable $callback Callback that performs the transaction.
	 * @param array    $args     Arguments to pass to the callback.
	 *
	 * @return void
	 */
	private function add_transaction_to_queue( $callback, $args ) {
		$this->queued_transactions[] = [ $callback, $args ];
	}

	/**
	 * Removes all notifications from storage.
	 *
	 * @return bool True when notifications got removed.
	 */
	protected function remove_storage() {
		if ( ! $this->has_stored_notifications() ) {
			return false;
		}

		delete_user_option( get_current_user_id(), self::STORAGE_KEY );
		return true;
	}

	/**
	 * Checks if there are stored notifications.
	 *
	 * @return bool True when there are stored notifications.
	 */
	protected function has_stored_notifications() {
		$stored_notifications = $this->get_stored_notifications();

		return ! empty( $stored_notifications );
	}

	/**
	 * Retrieves the stored notifications.
	 *
	 * @codeCoverageIgnore
	 *
	 * @return array|false Array with notifications or false when not set.
	 */
	protected function get_stored_notifications() {
		return get_user_option( self::STORAGE_KEY, get_current_user_id() );
	}
}
                                                                                                                                                                                                              plugins/wordpress-seo-extended/admin/class-yoast-notification.php                                   0000644                 00000023245 15122266554 0021404 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Notifications
 * @since   1.5.3
 */

/**
 * Implements individual notification.
 */
class Yoast_Notification {

	/**
	 * Type of capability check.
	 *
	 * @var string
	 */
	public const MATCH_ALL = 'all';

	/**
	 * Type of capability check.
	 *
	 * @var string
	 */
	public const MATCH_ANY = 'any';

	/**
	 * Notification type.
	 *
	 * @var string
	 */
	public const ERROR = 'error';

	/**
	 * Notification type.
	 *
	 * @var string
	 */
	public const WARNING = 'warning';

	/**
	 * Notification type.
	 *
	 * @var string
	 */
	public const UPDATED = 'updated';

	/**
	 * Options of this Notification.
	 *
	 * Contains optional arguments:
	 *
	 * -             type: The notification type, i.e. 'updated' or 'error'
	 * -               id: The ID of the notification
	 * -            nonce: Security nonce to use in case of dismissible notice.
	 * -         priority: From 0 to 1, determines the order of Notifications.
	 * -    dismissal_key: Option name to save dismissal information in, ID will be used if not supplied.
	 * -     capabilities: Capabilities that a user must have for this Notification to show.
	 * - capability_check: How to check capability pass: all or any.
	 * -  wpseo_page_only: Only display on wpseo page or on every page.
	 *
	 * @var array
	 */
	private $options = [];

	/**
	 * Contains default values for the optional arguments.
	 *
	 * @var array
	 */
	private $defaults = [
		'type'             => self::UPDATED,
		'id'               => '',
		'user_id'          => null,
		'nonce'            => null,
		'priority'         => 0.5,
		'data_json'        => [],
		'dismissal_key'    => null,
		'capabilities'     => [],
		'capability_check' => self::MATCH_ALL,
		'yoast_branding'   => false,
	];

	/**
	 * The message for the notification.
	 *
	 * @var string
	 */
	private $message;

	/**
	 * Notification class constructor.
	 *
	 * @param string $message Message string.
	 * @param array  $options Set of options.
	 */
	public function __construct( $message, $options = [] ) {
		$this->message = $message;
		$this->options = $this->normalize_options( $options );
	}

	/**
	 * Retrieve notification ID string.
	 *
	 * @return string
	 */
	public function get_id() {
		return $this->options['id'];
	}

	/**
	 * Retrieve the user to show the notification for.
	 *
	 * @deprecated 21.6
	 * @codeCoverageIgnore
	 *
	 * @return WP_User The user to show this notification for.
	 */
	public function get_user() {
		_deprecated_function( __METHOD__, 'Yoast SEO 21.6' );
		return null;
	}

	/**
	 * Retrieve the id of the user to show the notification for.
	 *
	 * Returns the id of the current user if not user has been sent.
	 *
	 * @return int The user id
	 */
	public function get_user_id() {
		return ( $this->options['user_id'] ?? get_current_user_id() );
	}

	/**
	 * Retrieve nonce identifier.
	 *
	 * @return string|null Nonce for this Notification.
	 */
	public function get_nonce() {
		if ( $this->options['id'] && empty( $this->options['nonce'] ) ) {
			$this->options['nonce'] = wp_create_nonce( $this->options['id'] );
		}

		return $this->options['nonce'];
	}

	/**
	 * Make sure the nonce is up to date.
	 *
	 * @return void
	 */
	public function refresh_nonce() {
		if ( $this->options['id'] ) {
			$this->options['nonce'] = wp_create_nonce( $this->options['id'] );
		}
	}

	/**
	 * Get the type of the notification.
	 *
	 * @return string
	 */
	public function get_type() {
		return $this->options['type'];
	}

	/**
	 * Priority of the notification.
	 *
	 * Relative to the type.
	 *
	 * @return float Returns the priority between 0 and 1.
	 */
	public function get_priority() {
		return $this->options['priority'];
	}

	/**
	 * Get the User Meta key to check for dismissal of notification.
	 *
	 * @return string User Meta Option key that registers dismissal.
	 */
	public function get_dismissal_key() {
		if ( empty( $this->options['dismissal_key'] ) ) {
			return $this->options['id'];
		}

		return $this->options['dismissal_key'];
	}

	/**
	 * Is this Notification persistent.
	 *
	 * @return bool True if persistent, False if fire and forget.
	 */
	public function is_persistent() {
		$id = $this->get_id();

		return ! empty( $id );
	}

	/**
	 * Check if the notification is relevant for the current user.
	 *
	 * @return bool True if a user needs to see this notification, false if not.
	 */
	public function display_for_current_user() {
		// If the notification is for the current page only, always show.
		if ( ! $this->is_persistent() ) {
			return true;
		}

		// If the current user doesn't match capabilities.
		return $this->match_capabilities();
	}

	/**
	 * Does the current user match required capabilities.
	 *
	 * @return bool
	 */
	public function match_capabilities() {
		// Super Admin can do anything.
		if ( is_multisite() && is_super_admin( $this->options['user_id'] ) ) {
			return true;
		}

		/**
		 * Filter capabilities that enable the displaying of this notification.
		 *
		 * @param array              $capabilities The capabilities that must be present for this notification.
		 * @param Yoast_Notification $notification The notification object.
		 *
		 * @return array Array of capabilities or empty for no restrictions.
		 *
		 * @since 3.2
		 */
		$capabilities = apply_filters( 'wpseo_notification_capabilities', $this->options['capabilities'], $this );

		// Should be an array.
		if ( ! is_array( $capabilities ) ) {
			$capabilities = (array) $capabilities;
		}

		/**
		 * Filter capability check to enable all or any capabilities.
		 *
		 * @param string             $capability_check The type of check that will be used to determine if an capability is present.
		 * @param Yoast_Notification $notification     The notification object.
		 *
		 * @return string self::MATCH_ALL or self::MATCH_ANY.
		 *
		 * @since 3.2
		 */
		$capability_check = apply_filters( 'wpseo_notification_capability_check', $this->options['capability_check'], $this );

		if ( ! in_array( $capability_check, [ self::MATCH_ALL, self::MATCH_ANY ], true ) ) {
			$capability_check = self::MATCH_ALL;
		}

		if ( ! empty( $capabilities ) ) {

			$has_capabilities = array_filter( $capabilities, [ $this, 'has_capability' ] );

			switch ( $capability_check ) {
				case self::MATCH_ALL:
					return $has_capabilities === $capabilities;
				case self::MATCH_ANY:
					return ! empty( $has_capabilities );
			}
		}

		return true;
	}

	/**
	 * Array filter function to find matched capabilities.
	 *
	 * @param string $capability Capability to test.
	 *
	 * @return bool
	 */
	private function has_capability( $capability ) {
		$user_id = $this->options['user_id'];
		if ( ! is_numeric( $user_id ) ) {
			return false;
		}
		$user = get_user_by( 'id', $user_id );
		if ( ! $user ) {
			return false;
		}

		return $user->has_cap( $capability );
	}

	/**
	 * Return the object properties as an array.
	 *
	 * @return array
	 */
	public function to_array() {
		return [
			'message' => $this->message,
			'options' => $this->options,
		];
	}

	/**
	 * Adds string (view) behaviour to the notification.
	 *
	 * @return string
	 */
	public function __toString() {
		return $this->render();
	}

	/**
	 * Renders the notification as a string.
	 *
	 * @return string The rendered notification.
	 */
	public function render() {
		$attributes = [];

		// Default notification classes.
		$classes = [
			'yoast-notification',
		];

		// Maintain WordPress visualisation of notifications when they are not persistent.
		if ( ! $this->is_persistent() ) {
			$classes[] = 'notice';
			$classes[] = $this->get_type();
		}

		if ( ! empty( $classes ) ) {
			$attributes['class'] = implode( ' ', $classes );
		}

		// Combined attribute key and value into a string.
		array_walk( $attributes, [ $this, 'parse_attributes' ] );

		$message = null;
		if ( $this->options['yoast_branding'] ) {
			$message = $this->wrap_yoast_seo_icon( $this->message );
		}

		if ( $message === null ) {
			$message = wpautop( $this->message );
		}

		// Build the output DIV.
		return '<div ' . implode( ' ', $attributes ) . '>' . $message . '</div>' . PHP_EOL;
	}

	/**
	 * Wraps the message with a Yoast SEO icon.
	 *
	 * @param string $message The message to wrap.
	 *
	 * @return string The wrapped message.
	 */
	private function wrap_yoast_seo_icon( $message ) {
		$out  = sprintf(
			'<img src="%1$s" height="%2$d" width="%3$d" class="yoast-seo-icon" />',
			esc_url( plugin_dir_url( WPSEO_FILE ) . 'packages/js/images/Yoast_SEO_Icon.svg' ),
			60,
			60
		);
		$out .= '<div class="yoast-seo-icon-wrap">';
		$out .= $message;
		$out .= '</div>';

		return $out;
	}

	/**
	 * Get the JSON if provided.
	 *
	 * @return false|string
	 */
	public function get_json() {
		if ( empty( $this->options['data_json'] ) ) {
			return '';
		}

		return WPSEO_Utils::format_json_encode( $this->options['data_json'] );
	}

	/**
	 * Make sure we only have values that we can work with.
	 *
	 * @param array $options Options to normalize.
	 *
	 * @return array
	 */
	private function normalize_options( $options ) {
		$options = wp_parse_args( $options, $this->defaults );

		// Should not exceed 0 or 1.
		$options['priority'] = min( 1, max( 0, $options['priority'] ) );

		// Set default capabilities when not supplied.
		if ( empty( $options['capabilities'] ) || $options['capabilities'] === [] ) {
			$options['capabilities'] = [ 'wpseo_manage_options' ];
		}

		// Set to the id of the current user if not supplied.
		if ( $options['user_id'] === null ) {
			$options['user_id'] = get_current_user_id();
		}

		return $options;
	}

	/**
	 * Format HTML element attributes.
	 *
	 * @param string $value Attribute value.
	 * @param string $key   Attribute name.
	 *
	 * @return void
	 */
	private function parse_attributes( &$value, $key ) {
		$value = sprintf( '%s="%s"', sanitize_key( $key ), esc_attr( $value ) );
	}
}
                                                                                                                                                                                                                                                                                                                                                           plugins/wordpress-seo-extended/admin/class-yoast-notifications.php                                  0000644                 00000017204 15122266554 0021565 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin\Notifications
 */

/**
 * Class Yoast_Notifications.
 */
class Yoast_Notifications {

	/**
	 * Holds the admin page's ID.
	 *
	 * @var string
	 */
	public const ADMIN_PAGE = 'wpseo_dashboard';

	/**
	 * Total notifications count.
	 *
	 * @var int
	 */
	private static $notification_count = 0;

	/**
	 * All error notifications.
	 *
	 * @var array
	 */
	private static $errors = [];

	/**
	 * Active errors.
	 *
	 * @var array
	 */
	private static $active_errors = [];

	/**
	 * Dismissed errors.
	 *
	 * @var array
	 */
	private static $dismissed_errors = [];

	/**
	 * All warning notifications.
	 *
	 * @var array
	 */
	private static $warnings = [];

	/**
	 * Active warnings.
	 *
	 * @var array
	 */
	private static $active_warnings = [];

	/**
	 * Dismissed warnings.
	 *
	 * @var array
	 */
	private static $dismissed_warnings = [];

	/**
	 * Yoast_Notifications constructor.
	 */
	public function __construct() {

		$this->add_hooks();
	}

	/**
	 * Add hooks
	 *
	 * @return void
	 */
	private function add_hooks() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
		if ( isset( $_GET['page'] ) && is_string( $_GET['page'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
			$page = sanitize_text_field( wp_unslash( $_GET['page'] ) );
			if ( $page === self::ADMIN_PAGE ) {
				add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
			}
		}

		// Needed for adminbar and Notifications page.
		add_action( 'admin_init', [ self::class, 'collect_notifications' ], 99 );

		// Add AJAX hooks.
		add_action( 'wp_ajax_yoast_dismiss_notification', [ $this, 'ajax_dismiss_notification' ] );
		add_action( 'wp_ajax_yoast_restore_notification', [ $this, 'ajax_restore_notification' ] );
	}

	/**
	 * Enqueue assets.
	 *
	 * @return void
	 */
	public function enqueue_assets() {
		$asset_manager = new WPSEO_Admin_Asset_Manager();

		$asset_manager->enqueue_style( 'notifications' );
	}

	/**
	 * Handle ajax request to dismiss a notification.
	 *
	 * @return void
	 */
	public function ajax_dismiss_notification() {

		$notification = $this->get_notification_from_ajax_request();
		if ( $notification ) {
			$notification_center = Yoast_Notification_Center::get();
			$notification_center->maybe_dismiss_notification( $notification );

			$this->output_ajax_response( $notification->get_type() );
		}

		wp_die();
	}

	/**
	 * Handle ajax request to restore a notification.
	 *
	 * @return void
	 */
	public function ajax_restore_notification() {

		$notification = $this->get_notification_from_ajax_request();
		if ( $notification ) {
			$notification_center = Yoast_Notification_Center::get();
			$notification_center->restore_notification( $notification );

			$this->output_ajax_response( $notification->get_type() );
		}

		wp_die();
	}

	/**
	 * Create AJAX response data.
	 *
	 * @param string $type Notification type.
	 *
	 * @return void
	 */
	private function output_ajax_response( $type ) {

		$html = $this->get_view_html( $type );
		// phpcs:disable WordPress.Security.EscapeOutput -- Reason: WPSEO_Utils::format_json_encode is safe.
		echo WPSEO_Utils::format_json_encode(
			[
				'html'  => $html,
				'total' => self::get_active_notification_count(),
			]
		);
		// phpcs:enable -- Reason: WPSEO_Utils::format_json_encode is safe.
	}

	/**
	 * Get the HTML to return in the AJAX request.
	 *
	 * @param string $type Notification type.
	 *
	 * @return bool|string
	 */
	private function get_view_html( $type ) {

		switch ( $type ) {
			case 'error':
				$view = 'errors';
				break;

			case 'warning':
			default:
				$view = 'warnings';
				break;
		}

		// Re-collect notifications.
		self::collect_notifications();

		/**
		 * Stops PHPStorm from nagging about this variable being unused. The variable is used in the view.
		 *
		 * @noinspection PhpUnusedLocalVariableInspection
		 */
		$notifications_data = self::get_template_variables();

		ob_start();
		include WPSEO_PATH . 'admin/views/partial-notifications-' . $view . '.php';
		$html = ob_get_clean();

		return $html;
	}

	/**
	 * Extract the Yoast Notification from the AJAX request.
	 *
	 * This function does not handle nonce verification.
	 *
	 * @return Yoast_Notification|null A Yoast_Notification on success, null on failure.
	 */
	private function get_notification_from_ajax_request() {
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Reason: This function does not handle nonce verification.
		if ( ! isset( $_POST['notification'] ) || ! is_string( $_POST['notification'] ) ) {
			return null;
		}
		// phpcs:ignore WordPress.Security.NonceVerification.Missing -- Reason: This function does not handle nonce verification.
		$notification_id = sanitize_text_field( wp_unslash( $_POST['notification'] ) );

		if ( empty( $notification_id ) ) {
			return null;
		}
		$notification_center = Yoast_Notification_Center::get();
		return $notification_center->get_notification_by_id( $notification_id );
	}

	/**
	 * Collect the notifications and group them together.
	 *
	 * @return void
	 */
	public static function collect_notifications() {

		$notification_center = Yoast_Notification_Center::get();

		$notifications            = $notification_center->get_sorted_notifications();
		self::$notification_count = count( $notifications );

		self::$errors           = array_filter( $notifications, [ self::class, 'filter_error_notifications' ] );
		self::$dismissed_errors = array_filter( self::$errors, [ self::class, 'filter_dismissed_notifications' ] );
		self::$active_errors    = array_diff( self::$errors, self::$dismissed_errors );

		self::$warnings           = array_filter( $notifications, [ self::class, 'filter_warning_notifications' ] );
		self::$dismissed_warnings = array_filter( self::$warnings, [ self::class, 'filter_dismissed_notifications' ] );
		self::$active_warnings    = array_diff( self::$warnings, self::$dismissed_warnings );
	}

	/**
	 * Get the variables needed in the views.
	 *
	 * @return array
	 */
	public static function get_template_variables() {

		return [
			'metrics'  => [
				'total'    => self::$notification_count,
				'active'   => self::get_active_notification_count(),
				'errors'   => count( self::$errors ),
				'warnings' => count( self::$warnings ),
			],
			'errors'   => [
				'dismissed' => self::$dismissed_errors,
				'active'    => self::$active_errors,
			],
			'warnings' => [
				'dismissed' => self::$dismissed_warnings,
				'active'    => self::$active_warnings,
			],
		];
	}

	/**
	 * Get the number of active notifications.
	 *
	 * @return int
	 */
	public static function get_active_notification_count() {

		return ( count( self::$active_errors ) + count( self::$active_warnings ) );
	}

	/**
	 * Filter out any non-errors.
	 *
	 * @param Yoast_Notification $notification Notification to test.
	 *
	 * @return bool
	 */
	private static function filter_error_notifications( Yoast_Notification $notification ) {

		return $notification->get_type() === 'error';
	}

	/**
	 * Filter out any non-warnings.
	 *
	 * @param Yoast_Notification $notification Notification to test.
	 *
	 * @return bool
	 */
	private static function filter_warning_notifications( Yoast_Notification $notification ) {

		return $notification->get_type() !== 'error';
	}

	/**
	 * Filter out any dismissed notifications.
	 *
	 * @param Yoast_Notification $notification Notification to test.
	 *
	 * @return bool
	 */
	private static function filter_dismissed_notifications( Yoast_Notification $notification ) {

		return Yoast_Notification_Center::is_notification_dismissed( $notification );
	}
}

class_alias( Yoast_Notifications::class, 'Yoast_Alerts' );
                                                                                                                                                                                                                                                                                                                                                                                            plugins/wordpress-seo-extended/admin/class-yoast-plugin-conflict.php                                0000644                 00000024520 15122266554 0022010 0                                                                                                    ustar 00                                                                                                                                                                                                                                                       <?php
/**
 * WPSEO plugin file.
 *
 * @package WPSEO\Admin
 * @since   1.7.0
 */

/**
 * Base class for handling plugin conflicts.
 */
class Yoast_Plugin_Conflict {

	/**
	 * The plugins must be grouped per section.
	 *
	 * It's possible to check for each section if there are conflicting plugins.
	 *
	 * @var array
	 */
	protected $plugins = [];

	/**
	 * All the current active plugins will be stored in this private var.
	 *
	 * @var array
	 */
	protected $all_active_plugins = [];

	/**
	 * After searching for active plugins that are in $this->plugins the active plugins will be stored in this
	 * property.
	 *
	 * @var array
	 */
	protected $active_conflicting_plugins = [];

	/**
	 * Property for holding instance of itself.
	 *
	 * @var Yoast_Plugin_Conflict
	 */
	protected static $instance;

	/**
	 * For the use of singleton pattern. Create instance of itself and return this instance.
	 *
	 * @param string $class_name Give the classname to initialize. If classname is
	 *                           false (empty) it will use it's own __CLASS__.
	 *
	 * @return Yoast_Plugin_Conflict
	 */
	public static function get_instance( $class_name = '' ) {

		if ( is_null( self::$instance ) ) {
			if ( ! is_string( $class_name ) || $class_name === '' ) {
				$class_name = self::class;
			}

			self::$instance = new $class_name();
		}

		return self::$instance;
	}

	/**
	 * Setting instance, all active plugins and search for active plugins.
	 *
	 * Protected constructor to prevent creating a new instance of the
	 * *Singleton* via the `new` operator from outside this class.
	 */
	protected function __construct() {
		// Set active plugins.
		$this->all_active_plugins = get_option( 'active_plugins' );

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reason: We are not processing form information.
		if ( isset( $_GET['action'] ) && is_string( $_GET['action'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended,WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Reason: We are not processing form information and only comparing the variable in a condition.
			$action = wp_unslash( $_GET['action'] );
			if ( $action === 'deactivate' ) {
				$this->remove_deactivated_plugin();
			}
		}

		// Search for active plugins.
		$this->search_active_plugins();
	}

	/**
	 * Check if there are conflicting plugins for given $plugin_section.
	 *
	 * @param string $plugin_section Type of plugin conflict (such as Open Graph or sitemap).
	 *
	 * @return bool
	 */
	public function check_for_conflicts( $plugin_section ) {

		static $sections_checked;

		// Return early if there are no active conflicting plugins at all.
		if ( empty( $this->active_conflicting_plugins ) ) {
			return false;
		}

		if ( $sections_checked === null ) {
			$sections_checked = [];
		}

		if ( ! in_array( $plugin_section, $sections_checked, true ) ) {
			$sections_checked[] = $plugin_section;
			return ( ! empty( $this->active_conflicting_plugins[ $plugin_section ] ) );
		}

		return false;
	}

	/**
	 * Checks for given $plugin_sections for conflicts.
	 *
	 * @param array $plugin_sections Set of sections.
	 *
	 * @return void
	 */
	public function check_plugin_conflicts( $plugin_sections ) {
		foreach ( $plugin_sections as $plugin_section => $readable_plugin_section ) {
			// Check for conflicting plugins and show error if there are conflicts.
			if ( $this->check_for_conflicts( $plugin_section ) ) {
				$this->set_error( $plugin_section, $readable_plugin_section );
			}
		}

		// List of all active sections.
		$sections = array_keys( $plugin_sections );
		// List of all sections.
		$all_plugin_sections = array_keys( $this->plugins );

		/*
		 * Get all sections that are inactive.
		 * These plugins need to be cleared.
		 *
		 * This happens when Sitemaps or OpenGraph implementations toggle active/disabled.
		 */
		$inactive_sections = array_diff( $all_plugin_sections, $sections );
		if ( ! empty( $inactive_sections ) ) {
			foreach ( $i