<?php
/**
 * Ef Practical functions and definitions
 *
 * @package Ef Practical
 */

if ( ! function_exists( 'practical_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function practical_setup() {
    
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on practical, use a find and replace
	 * to change 'ef-practical' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'ef-practical', get_template_directory() . '/languages' );
    
	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );
    
	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );
    
	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link http://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );
    
    // This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu', 'ef-practical' ),
	) );
    
	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'comment-form',
		'gallery',
		'caption',
	) );
    
    /*
	 * Enable support for Post Formats.
	 * See http://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'post-formats', array(
		'aside',
		'image',
		'video',
		'quote',
		'link',
		'gallery',
		'status',
		'audio',
		'chat',
	) );
    
	// Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'practical_custom_background_args', array(
        'default-image'      => get_template_directory_uri() . '/images/bg.png',
		'default-color'      => 'ffffff',
        'default-position-x' => 'center',
        'default-position-y' => 'center',
        'default-repeat'     => 'repeat',
        'default-attachment' => 'fixed',
	) ) );
    
    // Set up the WordPress core custom logo feature.
    add_theme_support( 'custom-logo', apply_filters( 'practical_custom_logo_args', array(
        'height'      => 70,
        'width'       => 300,
        'flex-height' => true,
        'flex-width'  => true,
    ) ) );
    
    // Add theme support for selective refresh for widgets.
    add_theme_support( 'customize-selective-refresh-widgets' );
    
}
endif;
add_action( 'after_setup_theme', 'practical_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function practical_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'practical_content_width', 1240 );
}
add_action( 'after_setup_theme', 'practical_content_width', 0 );

/**
 * Register widget area.
 *
 * @link http://codex.wordpress.org/Function_Reference/register_sidebar
 */
function practical_widgets_init() {
    
    // Left sidebar
    register_sidebar( array(
		'name' 			=> esc_html__( 'Left Sidebar', 'ef-practical' ),
		'id' 			=> 'sidebar-left',
		'description'   => esc_html__( 'Shows widgets in left sidebar.', 'ef-practical' ),
		'before_widget' => '<section id="%1$s" class="widget cf %2$s">',
		'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>'
	) );
    
    // Right sidebar
    register_sidebar( array(
		'name' 			=> esc_html__( 'Right Sidebar', 'ef-practical' ),
		'id' 			=> 'sidebar-right',
		'description'   => esc_html__( 'Shows widgets in right sidebar.', 'ef-practical' ),
		'before_widget' => '<section id="%1$s" class="widget cf %2$s">',
		'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>'
	) );
    
    // Footer sidebar
    $footer_widget_areas = 4;
    for ( $i=1; $i <= $footer_widget_areas; $i++ ) {
        register_sidebar( array(
            'name'          => esc_html__( 'Footer Widget ', 'ef-practical' ) . $i,
            'id'            => 'footer-widget-'. $i,
            'description'   => esc_html__( 'Shows widgets on footer area.', 'ef-practical' ),
            'before_widget' => '<section id="%1$s" class="widget cf %2$s">',
            'after_widget'  => '</section>',
            'before_title'  => '<h3 class="widget-title">',
            'after_title'   => '</h3>'
        ) );
    }
    
    register_widget( 'Practical_Recent_Posts_Widget' );
    register_widget( 'Practical_Social_Widget' );
    
}
add_action( 'widgets_init', 'practical_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function practical_scripts() {
    
    $theme = wp_get_theme( 'ef-practical' );
    $version = $theme->get( 'Version' );
    
    wp_enqueue_style( 'practical-fonts', '//fonts.googleapis.com/css?family=PT+Serif:400,400i,700,700i|Playfair+Display:400,400i,700,700i' );
    
    wp_enqueue_style( 'practical-font-awesome', get_template_directory_uri() . '/css/font-awesome.min.css', array(), $version );
    
    wp_enqueue_style( 'fancybox-css', get_template_directory_uri() . '/css/fancybox.min.css', array(), $version );
    
    wp_enqueue_style( 'practical-bootstrap-grid', get_template_directory_uri() . '/css/bootstrap-grid.min.css', array(), $version );
    
    wp_enqueue_style( 'practical-style', get_stylesheet_uri() );
    
    wp_enqueue_script( 'jquery-fancybox', get_template_directory_uri() . '/js/fancybox.min.js', array( 'jquery' ), $version, true );
    
    if ( 0 != get_theme_mod( 'practical_animation', 1 ) ) {
        wp_enqueue_style( 'practical-animate', get_template_directory_uri() .'/css/animate.min.css', array(), $version );
        wp_enqueue_script( 'practical-wow', get_template_directory_uri() . '/js/wow.min.js', array(), $version, true );
    }
    
    if ( 0 != get_theme_mod( 'practical_smoothscroll', 1 ) ) {
        wp_enqueue_script( 'practical-smoothscroll', get_template_directory_uri() . '/js/SmoothScroll.js', array(), $version, true );
    }
    
    $practical_js_settings = array(
        'sticky_header'                => esc_attr( get_theme_mod( 'practical_sticky_header', 0 ) ),
        'animation'                    => esc_attr( get_theme_mod( 'practical_animation', 1 ) ),
        'go_top_button'                => esc_attr( get_theme_mod( 'practical_go_top_button', 1 ) ),
        'sticky_footer'                => esc_attr( get_theme_mod( 'practical_sticky_footer', 1 ) ),
        'enable_lightbox'              => esc_attr( get_theme_mod( 'practical_enable_lightbox', 0 ) ),
        'featured_img_lightbox'        => esc_attr( get_theme_mod( 'practical_lightbox_featured_img', 0 ) ),
        'content_img_lightbox'         => esc_attr( get_theme_mod( 'practical_lightbox_content_area_img', 0 ) ),
        'animation_effect_lightbox'    => esc_attr( get_theme_mod( 'practical_lightbox_animation_effect', 'zoom-in-out' ) ),
        'transition_effect_lightbox'   => esc_attr( get_theme_mod( 'practical_lightbox_transition_effect', 'circular' ) ),
        'transition_duration_lightbox' => absint( get_theme_mod( 'practical_lightbox_transition_duration', 1000 ) ),
        'lightbox_auto_play_lightbox'  => esc_attr( get_theme_mod( 'practical_lightbox_auto_play', 0 ) ),
        'transition_speed_lightbox'    => absint( get_theme_mod( 'practical_lightbox_slideshow_speed', 4000 ) ),
        
    );
    
    wp_enqueue_script( 'practical-main', get_template_directory_uri() . '/js/main.js', array( 'jquery' ), $version, true );
    
    wp_enqueue_script( 'practical-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), $version, true );
    
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
	}
    
    wp_localize_script( 'jquery', 'practical_js_settings', $practical_js_settings );
    
}
add_action( 'wp_enqueue_scripts', 'practical_scripts' );

// Enqueue the Dashicons script
function practical_dashicons_front_end() {
    wp_enqueue_style( 'dashicons' );
}
add_action( 'wp_enqueue_scripts', 'practical_dashicons_front_end' );

// Excerpt length
if ( ! function_exists( 'practical_excerpt_length' ) ) {
    
    function practical_excerpt_length( $length ) {
		
		$custom_length = absint( get_theme_mod( 'practical_excerpt_length', $length ) );
        
        if ( is_admin() ) {
			return $length;
        } else {
			return $length = $custom_length;
		}
		
    }
    
    add_filter( 'excerpt_length', 'practical_excerpt_length', 999 );
    
}

// Remove [...] from excerpt
if ( ! function_exists( 'practical_excerpt_more' ) ) {
    
    function practical_excerpt_more( $more ) {
        
		if ( is_admin() ) {
			return $more;
        } else {
			return '...';
		}
		
    }
    
    add_filter( 'excerpt_more', 'practical_excerpt_more' );
    
}

if ( ! function_exists( 'practical_post_format_icon' ) ) :
    /**
     * Post Format Icons
     */
    function practical_post_format_icon() {
        if ( has_post_format( 'aside' ) ) { echo '<span class="dashicons dashicons-format-aside"></span>'; }
        elseif ( has_post_format( 'chat' ) ) { echo '<span class="dashicons dashicons-format-chat"></span>'; }
        elseif ( has_post_format( 'gallery' ) ) { echo '<span class="dashicons dashicons-format-gallery"></span>'; }
        elseif ( has_post_format( 'image' ) ) { echo '<span class="dashicons dashicons-format-image"></span>'; }
        elseif ( has_post_format( 'link' ) ) { echo '<span class="dashicons dashicons-admin-links"></span>'; }
        elseif ( has_post_format( 'quote' ) ) { echo '<span class="dashicons dashicons-format-quote"></span>'; }
        elseif ( has_post_format( 'status' ) ) { echo '<span class="dashicons dashicons-format-status"></span>'; }
        elseif ( has_post_format( 'video' ) ) { echo '<span class="dashicons dashicons-format-video"></span>'; }
        elseif ( has_post_format( 'audio' ) ) { echo '<span class="dashicons dashicons-format-audio"></span>'; }
        else { echo ''; }
    }
endif;

/**
 * Modifies tag cloud widget arguments to have all tags in the widget same font size.
 */
function practical_widget_tag_cloud_args( $args ) {
	$args['largest'] = 12;
	$args['smallest'] = 12;
	$args['unit'] = 'px';
	return $args;
}
add_filter( 'widget_tag_cloud_args', 'practical_widget_tag_cloud_args' );

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Kirki Toolkit.
 */
require get_template_directory() . '/inc/class-practical-kirki.php';

require get_template_directory() . '/inc/kirki/kirki.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Add meta boxes.
 */
require get_template_directory() . '/inc/metaboxes/meta-boxes.php';

/**
 * Custom Widgets.
 */
require get_template_directory() . '/inc/widgets/recent-posts.php';
require get_template_directory() . '/inc/widgets/social-widget.php';

/**
 * Load Jetpack compatibility file.
 */
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}